//! Milter commands.

use crate::{
    macros::Stage,
    message::{Message, Version},
    proto_util::{Actions, ProtoOpts, SocketInfo},
    session::State,
};
use bytes::{Buf, Bytes};
use std::{
    error::Error,
    ffi::{CStr, CString},
    fmt::{self, Display, Formatter},
    net::{SocketAddrV4, SocketAddrV6},
};

#[derive(Copy, Clone, Debug, PartialEq, Eq)]
pub enum CommandKind {
    Abort,
    BodyChunk,
    ConnInfo,
    DefMacros,
    BodyEnd,
    Helo,
    QuitNc,
    Header,
    Mail,
    Eoh,
    OptNeg,
    Quit,
    Rcpt,
    Data,
    Unknown,
}

impl CommandKind {
    pub fn as_state(&self) -> Option<State> {
        match self {
            Self::Abort => Some(State::Abort),
            Self::BodyChunk => Some(State::Body),
            Self::ConnInfo => Some(State::Conn),
            Self::DefMacros => None,
            Self::BodyEnd => Some(State::Eom),
            Self::Helo => Some(State::Helo),
            Self::QuitNc => Some(State::QuitNc),
            Self::Header => Some(State::Header),
            Self::Mail => Some(State::Mail),
            Self::Eoh => Some(State::Eoh),
            Self::OptNeg => Some(State::Opts),
            Self::Quit => Some(State::Quit),
            Self::Rcpt => Some(State::Rcpt),
            Self::Data => Some(State::Data),
            Self::Unknown => Some(State::Unknown),
        }
    }
}

impl From<CommandKind> for u8 {
    fn from(kind: CommandKind) -> Self {
        match kind {
            CommandKind::Abort => b'A',
            CommandKind::BodyChunk => b'B',
            CommandKind::ConnInfo => b'C',
            CommandKind::DefMacros => b'D',
            CommandKind::BodyEnd => b'E',
            CommandKind::Helo => b'H',
            CommandKind::QuitNc => b'K',
            CommandKind::Header => b'L',
            CommandKind::Mail => b'M',
            CommandKind::Eoh => b'N',
            CommandKind::OptNeg => b'O',
            CommandKind::Quit => b'Q',
            CommandKind::Rcpt => b'R',
            CommandKind::Data => b'T',
            CommandKind::Unknown => b'U',
        }
    }
}

impl TryFrom<u8> for CommandKind {
    type Error = CommandError;  // TODO error type

    fn try_from(value: u8) -> Result<Self, Self::Error> {
        match value {
            b'A' => Ok(Self::Abort),
            b'B' => Ok(Self::BodyChunk),
            b'C' => Ok(Self::ConnInfo),
            b'D' => Ok(Self::DefMacros),
            b'E' => Ok(Self::BodyEnd),
            b'H' => Ok(Self::Helo),
            b'K' => Ok(Self::QuitNc),
            b'L' => Ok(Self::Header),
            b'M' => Ok(Self::Mail),
            b'N' => Ok(Self::Eoh),
            b'O' => Ok(Self::OptNeg),
            b'Q' => Ok(Self::Quit),
            b'R' => Ok(Self::Rcpt),
            b'T' => Ok(Self::Data),
            b'U' => Ok(Self::Unknown),
            _ => Err(CommandError::UnknownCommandKind),
        }
    }
}

#[derive(Debug)]
pub struct CommandMessage {
    pub kind: CommandKind,
    pub buffer: Bytes,
}

impl TryFrom<Message> for CommandMessage {
    type Error = CommandError;

    fn try_from(msg: Message) -> Result<Self, Self::Error> {
        let kind = msg.kind.try_into()?;
        let buffer = msg.buffer;

        Ok(Self { kind, buffer })
    }
}

// TODO actually CommandParseError
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq)]
pub enum CommandError {
    // related to buffer only:
    NotNulTerminated,
    NoCStringFound,
    NoU8Found,
    NoU16Found,
    UnknownMacroStage,
    InvalidSocketAddr,
    UnknownFamily,
    MissingOptneg,
    UnsupportedProtocolVersion,
    EmptyCString,

    // related to kind only:
    UnknownCommandKind,

    // related to stage only:
    UnknownStage,
}

impl Display for CommandError {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        write!(f, "{:?}", self)
    }
}

impl Error for CommandError {}

/// A command.
#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub enum Command {
    /// The `A` command.
    Abort,
    /// The `B` command.
    BodyChunk(BodyPayload),
    /// The `C` command.
    ConnInfo(ConnInfoPayload),
    /// The `D` command.
    DefMacros(MacroPayload),
    /// The `E` command.
    BodyEnd(BodyPayload),
    /// The `H` command.
    Helo(HeloPayload),
    /// The `K` command.
    QuitNc,
    /// The `L` command.
    Header(HeaderPayload),
    /// The `M` command.
    Mail(EnvAddrPayload),
    /// The `N` command.
    Eoh,
    /// The `O` command.
    OptNeg(OptNegPayload),
    /// The `Q` command.
    Quit,
    /// The `R` command.
    Rcpt(EnvAddrPayload),
    /// The `T` command.
    Data,
    /// The `U` command.
    Unknown(UnknownPayload),
}

// Not currently needed, because the session separates command kind and payload.
impl Command {
    pub fn from_message(msg: CommandMessage) -> Result<Self, CommandError> {
        Ok(match msg.kind {
            CommandKind::Abort => Self::Abort,
            CommandKind::BodyChunk => Self::BodyChunk(BodyPayload::parse_buffer(msg.buffer)?),
            CommandKind::ConnInfo => Self::ConnInfo(ConnInfoPayload::parse_buffer(msg.buffer)?),
            CommandKind::DefMacros => Self::DefMacros(MacroPayload::parse_buffer(msg.buffer)?),
            CommandKind::BodyEnd => Self::BodyEnd(BodyPayload::parse_buffer(msg.buffer)?),
            CommandKind::Helo => Self::Helo(HeloPayload::parse_buffer(msg.buffer)?),
            CommandKind::QuitNc => Self::QuitNc,
            CommandKind::Header => Self::Header(HeaderPayload::parse_buffer(msg.buffer)?),
            CommandKind::Mail => Self::Mail(EnvAddrPayload::parse_buffer(msg.buffer)?),
            CommandKind::Eoh => Self::Eoh,
            CommandKind::OptNeg => Self::OptNeg(OptNegPayload::parse_buffer(msg.buffer)?),
            CommandKind::Quit => Self::Quit,
            CommandKind::Rcpt => Self::Rcpt(EnvAddrPayload::parse_buffer(msg.buffer)?),
            CommandKind::Data => Self::Data,
            CommandKind::Unknown => Self::Unknown(UnknownPayload::parse_buffer(msg.buffer)?),
        })
    }
}

// TODO this is not actually needed?
/// A body chunk payload.
#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub struct BodyPayload {
    pub chunk: Vec<u8>,
}

impl BodyPayload {
    pub fn parse_buffer(buf: Bytes) -> Result<Self, CommandError> {
        let chunk = Vec::from(buf.as_ref());

        Ok(Self { chunk })
    }
}

/// A `ConnInfo` command payload.
#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub struct ConnInfoPayload {
    pub hostname: CString,
    pub socket_info: Option<SocketInfo>,
}

enum Family {
    Unknown,
    Ipv4,
    Ipv6,
    Unix,
}

impl TryFrom<u8> for Family {
    type Error = CommandError;

    fn try_from(value: u8) -> Result<Self, Self::Error> {
        match value {
            b'U' => Ok(Self::Unknown),
            b'4' => Ok(Self::Ipv4),
            b'6' => Ok(Self::Ipv6),
            b'L' => Ok(Self::Unix),
            _ => Err(CommandError::UnknownFamily),
        }
    }
}

impl ConnInfoPayload {
    pub fn parse_buffer(mut buf: Bytes) -> Result<Self, CommandError> {
        let hostname = get_c_string(&mut buf)?;

        let family = get_u8(&mut buf)?.try_into()?;

        let socket_info = match family {
            Family::Unknown => None,
            Family::Ipv4 => {
                let port = get_u16(&mut buf)?;

                ensure_nul_terminated(&buf)?;

                let addr = get_c_string(&mut buf)?;
                let addr = addr
                    .into_string()
                    .map_err(|_| CommandError::InvalidSocketAddr)?
                    .parse()
                    .map_err(|_| CommandError::InvalidSocketAddr)?;

                Some(SocketInfo::Inet(SocketAddrV4::new(addr, port).into()))
            }
            Family::Ipv6 => {
                let port = get_u16(&mut buf)?;

                ensure_nul_terminated(&buf)?;

                let addr = get_c_string(&mut buf)?;
                let addr = addr
                    .into_string()
                    .map_err(|_| CommandError::InvalidSocketAddr)?
                    .parse()
                    .map_err(|_| CommandError::InvalidSocketAddr)?;

                Some(SocketInfo::Inet(SocketAddrV6::new(addr, port, 0, 0).into()))
            }
            Family::Unix => {
                let _unused = get_u16(&mut buf)?;

                ensure_nul_terminated(&buf)?;

                let path = get_c_string(&mut buf)?;

                Some(SocketInfo::Unix(path))
            }
        };

        Ok(Self {
            hostname,
            socket_info,
        })
    }
}

/// A `DefMacros` command payload.
#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub struct MacroPayload {
    pub stage: Stage,
    pub macros: Vec<CString>,  // key/value pairs, non-empty
}

impl MacroPayload {
    pub fn parse_buffer(mut buf: Bytes) -> Result<Self, CommandError> {
        let stage = get_u8(&mut buf)?
            .try_into()
            .map_err(|_| CommandError::UnknownMacroStage)?;

        let mut macros = vec![get_c_string(&mut buf)?];
        while let Ok(s) = get_c_string(&mut buf) {
            macros.push(s);
        }

        Ok(Self { stage, macros })
    }
}

/// A `Helo` command payload.
#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub struct HeloPayload {
    pub hostname: CString,
}

impl HeloPayload {
    pub fn parse_buffer(mut buf: Bytes) -> Result<Self, CommandError> {
        ensure_nul_terminated(&buf)?;

        let hostname = get_c_string(&mut buf)?;

        Ok(Self { hostname })
    }
}

/*
pub trait ParseBuf<T> {
    fn parse_buf(self) -> Result<T, CommandError>;
}
impl ParseBuf<HeloPayload> for Bytes {
    fn parse_buf(mut self) -> Result<HeloPayload, CommandError> {
        ensure_nul_terminated(&self)?;

        let hostname = get_c_string(&mut self)?;

        Ok(HeloPayload { hostname })
    }
}
*/

/// A `Header` command payload.
#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub struct HeaderPayload {
    pub name: CString,  // non-empty
    pub value: CString,
}

impl HeaderPayload {
    pub fn parse_buffer(mut buf: Bytes) -> Result<Self, CommandError> {
        ensure_nul_terminated(&buf)?;

        let name = get_c_string(&mut buf)?;
        if name.as_bytes().is_empty() {
            return Err(CommandError::EmptyCString);
        }

        let value = get_c_string(&mut buf)?;

        Ok(Self { name, value })
    }
}

/// An envelope address payload.
#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub struct EnvAddrPayload {
    pub args: Vec<CString>,  // non-empty
}

impl EnvAddrPayload {
    pub fn parse_buffer(mut buf: Bytes) -> Result<Self, CommandError> {
        let mut args = vec![get_c_string(&mut buf)?];

        while let Ok(s) = get_c_string(&mut buf) {
            args.push(s);
        }

        Ok(Self { args })
    }
}

/// An `OptNeg` command payload.
#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub struct OptNegPayload {
    pub version: Version,
    pub actions: Actions,
    pub opts: ProtoOpts,
}

impl OptNegPayload {
    pub fn parse_buffer(mut buf: Bytes) -> Result<Self, CommandError> {
        if buf.remaining() < 12 {
            return Err(CommandError::MissingOptneg);
        }

        let version = buf.get_u32();
        if version < 2 {
            return Err(CommandError::UnsupportedProtocolVersion);
        }

        let actions = Actions::from_bits_truncate(buf.get_u32());
        let opts = ProtoOpts::from_bits_truncate(buf.get_u32());

        Ok(Self {
            version,
            actions,
            opts,
        })
    }
}

/// An `Unknown` command payload.
#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub struct UnknownPayload {
    pub arg: CString,
}

impl UnknownPayload {
    pub fn parse_buffer(mut buf: Bytes) -> Result<Self, CommandError> {
        let arg = get_c_string(&mut buf)?;

        Ok(Self { arg })
    }
}

fn ensure_nul_terminated(bytes: &[u8]) -> Result<(), CommandError> {
    if !bytes.ends_with(&[0]) {
        return Err(CommandError::NotNulTerminated);
    }
    Ok(())
}

fn get_u8(buf: &mut Bytes) -> Result<u8, CommandError> {
    if !buf.has_remaining() {
        return Err(CommandError::NoU8Found);
    }
    Ok(buf.get_u8())
}

fn get_u16(buf: &mut Bytes) -> Result<u16, CommandError> {
    if buf.remaining() < 2 {
        return Err(CommandError::NoU16Found);
    }
    Ok(buf.get_u16())
}

fn get_c_string(buf: &mut Bytes) -> Result<CString, CommandError> {
    if let Some(i) = buf.iter().position(|&x| x == 0) {
        let b = buf.split_to(i + 1);
        return Ok(CStr::from_bytes_with_nul(b.as_ref()).unwrap().into());
    }
    Err(CommandError::NoCStringFound)
}

#[cfg(test)]
mod tests {
    use super::*;
    use byte_strings::c_str;

    #[test]
    fn header_works() {
        assert_eq!(
            HeaderPayload::parse_buffer(Bytes::from_static(b"name\0value\0")),
            Ok(HeaderPayload {
                name: c_str!("name").into(),
                value: c_str!("value").into(),
            })
        );
        assert!(HeaderPayload::parse_buffer(Bytes::new()).is_err());
        assert!(HeaderPayload::parse_buffer(Bytes::from_static(b"name")).is_err());
    }

    #[test]
    fn helo_works() {
        assert_eq!(
            HeloPayload::parse_buffer(Bytes::from_static(b"hello\0")),
            Ok(HeloPayload { hostname: c_str!("hello").into() })
        );
        assert!(HeloPayload::parse_buffer(Bytes::new()).is_err());
        assert!(HeloPayload::parse_buffer(Bytes::from_static(b"hello")).is_err());

        // undocumented:
        assert!(HeloPayload::parse_buffer(Bytes::from_static(b"hello\0excess")).is_err());
        assert_eq!(
            HeloPayload::parse_buffer(Bytes::from_static(b"hello\0excess\0")),
            Ok(HeloPayload { hostname: c_str!("hello").into() })
        );
    }
}
