//! Milter protocol messages.

use bytes::Bytes;
use std::{
    ascii,
    fmt::{self, Debug, Formatter, Write},
};

pub mod commands;
pub mod replies;

pub type Version = u32;

// TODO Rename PROTO_VERSION?
/// Milter protocol version.
pub const MILTER_VERSION: Version = 6;

#[derive(Clone, Eq, Hash, PartialEq)]
pub struct Message {
    pub kind: u8,
    pub buffer: Bytes,
}

impl Message {
    // TODO but see _FFR feature to do with `Maxdatasize`: limits size of the buffer, may be negotiated
    // Limit length of message buffer (somewhat arbitrarily) to 1MB -- else if someone
    // erroneously sends a jumbo message size we will spend a lot of time and memory
    // accumulating it all.
    pub const MAX_BUFFER_LEN: usize = 1024 * 1024 - 1;

    pub fn new(kind: impl Into<u8>, buffer: impl Into<Bytes>) -> Self {
        Self {
            kind: kind.into(),
            buffer: buffer.into(),
        }
    }
}

impl Debug for Message {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        f.debug_struct("Message")
            .field("kind", &Byte(self.kind))
            .field("buffer", &self.buffer)
            .finish()
    }
}

struct Byte(u8);

impl Debug for Byte {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        f.write_str("b'")?;

        match self.0 {
            b'\0' => f.write_str("\\0")?,
            b'"' => f.write_str("\"")?,
            byte => {
                for c in ascii::escape_default(byte) {
                    f.write_char(c.into())?;
                }
            }
        }

        f.write_str("'")?;

        Ok(())
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn byte_debug_ok() {
        assert_eq!(format!("{:?}", Byte(b'\0')), r#"b'\0'"#);
        assert_eq!(format!("{:?}", Byte(b'\n')), r#"b'\n'"#);
        assert_eq!(format!("{:?}", Byte(b'"')), r#"b'"'"#);
        assert_eq!(format!("{:?}", Byte(b'a')), r#"b'a'"#);
        assert_eq!(format!("{:?}", Byte(b' ')), r#"b' '"#);
        assert_eq!(format!("{:?}", Byte(b'\x07')), r#"b'\x07'"#);
        assert_eq!(format!("{:?}", Byte(b'\xef')), r#"b'\xef'"#);
    }

    #[test]
    fn message_debug_ok() {
        assert_eq!(
            format!("{:?}", Message::new(b'A', &b"abc\0"[..])),
            r#"Message { kind: b'A', buffer: b"abc\0" }"#
        );
        assert_eq!(
            format!("{:?}", Message::new(b'\0', &b""[..])),
            r#"Message { kind: b'\0', buffer: b"" }"#
        );
    }
}
