use crate::message::commands::CommandError;
use std::{
    collections::HashMap,
    ffi::{CStr, CString},
};

#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq)]
pub enum Stage {
    Connect,
    Helo,
    Mail,
    Rcpt,
    Data,
    Eoh,
    Eom,
}

impl Stage {
    pub fn all_stages() -> impl DoubleEndedIterator<Item = Self> {
        use Stage::*;
        [Connect, Helo, Mail, Rcpt, Data, Eoh, Eom].into_iter()
    }
}

impl From<Stage> for i32 {
    fn from(stage: Stage) -> Self {
        match stage {
            Stage::Connect => 0,
            Stage::Helo => 1,
            Stage::Mail => 2,
            Stage::Rcpt => 3,
            Stage::Data => 4,
            Stage::Eoh => 5,
            Stage::Eom => 6,
        }
    }
}

impl From<Stage> for u8 {
    fn from(stage: Stage) -> Self {
        match stage {
            Stage::Connect => b'C',
            Stage::Helo => b'H',
            Stage::Mail => b'M',
            Stage::Rcpt => b'R',
            Stage::Data => b'T',
            Stage::Eoh => b'N',
            Stage::Eom => b'E',
        }
    }
}

impl TryFrom<i32> for Stage {
    // TODO not CommandError
    type Error = CommandError;

    fn try_from(value: i32) -> Result<Self, Self::Error> {
        match value {
            0 => Ok(Self::Connect),
            1 => Ok(Self::Helo),
            2 => Ok(Self::Mail),
            3 => Ok(Self::Rcpt),
            4 => Ok(Self::Data),
            5 => Ok(Self::Eoh),
            6 => Ok(Self::Eom),
            _ => Err(CommandError::UnknownStage),
        }
    }
}

impl TryFrom<u8> for Stage {
    // TODO not CommandError
    type Error = CommandError;

    fn try_from(value: u8) -> Result<Self, Self::Error> {
        match value {
            b'C' => Ok(Self::Connect),
            b'H' => Ok(Self::Helo),
            b'M' => Ok(Self::Mail),
            b'R' => Ok(Self::Rcpt),
            b'T' => Ok(Self::Data),
            b'N' => Ok(Self::Eoh),
            b'E' => Ok(Self::Eom),
            _ => Err(CommandError::UnknownStage),
        }
    }
}

// TODO *not* Default or Clone: this is an encapsulating type that is not meant
// to be created outside this crate
/// Currently defined macros.
#[derive(Debug)]
pub struct Macros(HashMap<Stage, HashMap<CString, CString>>);

impl Macros {
    pub(crate) fn new() -> Self {
        Self(
            Stage::all_stages()
                .map(|stage| (stage, Default::default()))
                .collect(),
        )
    }

    pub(crate) fn duplicate(&self) -> Self {
        Self(self.0.clone())
    }

    pub fn get(&self, key: &CStr) -> Option<&CStr> {
        let alt_key;
        let alt_key = match *key.to_bytes() {
            [x] => {
                alt_key = [b'{', x, b'}', 0];
                Some(CStr::from_bytes_with_nul(&alt_key[..4]).unwrap())
            }
            [b'{', x, b'}'] => {
                alt_key = [x, 0, 0, 0];
                Some(CStr::from_bytes_with_nul(&alt_key[..2]).unwrap())
            }
            _ => None,
        };

        Stage::all_stages().rev().find_map(|stage| {
            let m = &self.0[&stage];
            m.get(key)
                .or_else(|| alt_key.and_then(|k| m.get(k)))
                .map(|v| v.as_ref())
        })
    }

    pub fn to_hash_map(&self) -> HashMap<CString, CString> {
        Stage::all_stages()
            .flat_map(|stage| self.0[&stage].clone())
            .collect()
    }

    pub(crate) fn insert(&mut self, key: Stage, entries: HashMap<CString, CString>) {
        self.0.insert(key, entries);
    }

    pub(crate) fn clear(&mut self) {
        for s in Stage::all_stages() {
            self.0.get_mut(&s).unwrap().clear();
        }
    }

    pub(crate) fn clear_after(&mut self, stage: Stage) {
        for s in Stage::all_stages().rev().take_while(|&s| s != stage) {
            self.0.get_mut(&s).unwrap().clear();
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use byte_strings::c_str;

    #[test]
    fn macro_lookup() {
        let mut macros = Macros::new();

        macros.insert(
            Stage::Connect,
            HashMap::from([
                (c_str!("a").into(), c_str!("b").into()),
                (c_str!("{x}").into(), c_str!("y").into()),
                (c_str!("{name}").into(), c_str!("value").into()),
            ]),
        );

        assert_eq!(macros.get(c_str!("n")), None);
        assert_eq!(macros.get(c_str!("{none}")), None);
        assert_eq!(macros.get(c_str!("{name}")), Some(c_str!("value")));
        assert_eq!(macros.get(c_str!("a")), Some(c_str!("b")));
        assert_eq!(macros.get(c_str!("{a}")), Some(c_str!("b")));
        assert_eq!(macros.get(c_str!("x")), Some(c_str!("y")));
        assert_eq!(macros.get(c_str!("{x}")), Some(c_str!("y")));
    }
}
