use std::io;
use tokio::net::{TcpListener, UnixListener};

/// Conversion to a listener.
pub trait IntoListener {
    fn into_listener(self) -> io::Result<Listener>;
}

impl IntoListener for std::net::TcpListener {
    fn into_listener(self) -> io::Result<Listener> {
        TcpListener::from_std(self).map(Listener::Tcp)
    }
}

impl IntoListener for std::os::unix::net::UnixListener {
    fn into_listener(self) -> io::Result<Listener> {
        UnixListener::from_std(self).map(Listener::Unix)
    }
}

impl IntoListener for TcpListener {
    fn into_listener(self) -> io::Result<Listener> {
        Ok(Listener::Tcp(self))
    }
}

impl IntoListener for UnixListener {
    fn into_listener(self) -> io::Result<Listener> {
        Ok(Listener::Unix(self))
    }
}

impl IntoListener for Listener {
    fn into_listener(self) -> io::Result<Listener> {
        Ok(self)
    }
}

/// A Tokio-based listener.
pub enum Listener {
    Tcp(TcpListener),
    Unix(UnixListener),
}
