//! A library for writing asynchronous milter applications.
//!
//! This library provides an API for the sendmail mail filter protocol, also
//! known as *libmilter*.

mod callbacks;
mod config;
mod connection;
mod context;
mod ffi_util;
mod listener;
mod macros;
pub mod message;
mod proto_util;
mod session;
mod stream;

use crate::session::Session;
pub use crate::{
    callbacks::{CallbackFuture, Callbacks, Status},
    config::Config,
    context::{
        Context, ContextActions, ContextError, EomActions, EomContext, NegotiateContext,
        SetErrorReply, SmtpReply,
    },
    ffi_util::IntoCString,
    listener::{IntoListener, Listener},
    macros::{Macros, Stage},
    proto_util::{Actions, ProtoOpts, SocketInfo},
};
use std::{future::Future, io, sync::Arc};
use tokio::{
    io::{AsyncRead, AsyncWrite},
    select,
    sync::{watch, OwnedSemaphorePermit, Semaphore},
};
use tracing::{debug, error, trace};

// TODO Tracing levels policy:
// When this library encounters an unanticipated failure condition (programming
// error) it panics. No error logging is done in such a case.
//
// For all other error conditions the following policy is used.
// As a library, the general principle is not to log about library operation
// above debug level.
//
// `error!`: the milter library fails to provide service, eg when no new
// connections can be accepted due to I/O problem. Needs administrator attention
// `warn!`: when the user-provided milter is detected to misbehave (user error),
// eg when `Noreply` status is returned but was not negotiated beforehand
// `info!`: not used
// `debug!`: events related to milter library operation of more general interest
// `trace!`: fine-grained operation events
// TODO or perhaps *only* `trace`?

/// Runs a milter task that handles MTA connections until it is shut down.
///
/// # Cancellation
///
/// For graceful termination, the milter should be shut down by letting the
/// `shutdown` future complete. If the `Future` returned by this function is
/// simply dropped, currently active, spawned sessions may not exit immediately.
///
/// # Errors
///
/// To do.
///
/// # Example
///
/// The following example shows the simplest possible, no-op milter.
///
/// ```
/// # async fn f() -> std::io::Result<()> {
/// use indymilter::Callbacks;
/// use std::{future, net::TcpListener};
///
/// let listener = TcpListener::bind("127.0.0.1:3000")?;
/// listener.set_nonblocking(true)?;
/// let callbacks = Callbacks::<()>::new();
/// let config = Default::default();
/// let shutdown = future::pending::<()>();
///
/// indymilter::run(listener, callbacks, config, shutdown).await
/// # }
/// ```
pub async fn run<T>(
    listener: impl IntoListener,
    callbacks: Callbacks<T>,
    config: Config,
    shutdown: impl Future,
) -> io::Result<()>
where
    T: Send + 'static,
{
    debug!("milter starting");

    let listener = listener.into_listener()?;

    // The supplied shutdown_milter causes the main connection/session spawn in
    // the `select!` below to exit.
    // At the same time, multiple sessions may have been spawned (detached) and
    // be busy: those need to be notified of shutdown too, via the
    // `shutdown_sessions` handle, subscribed to by each session.
    let shutdown_milter = shutdown;
    let (shutdown_sessions, _) = watch::channel(false);

    // The invocation of `run_milter` never returns normally. It has an infinite
    // loop that is only broken when the listener cannot accept any new
    // connections.
    // When the shutdown future completes, the `run_milter` future is simply
    // dropped in the middle of whatever it is doing.

    let result = select! {
        res = run_milter(&listener, callbacks, config, &shutdown_sessions) => {
            let e = res.unwrap_err();
            error!("milter exited with error, shutting down: {}", e);
            Err(e)
        }
        _ = shutdown_milter => {
            debug!("milter shutting down");
            Ok(())
        }
    };

    // Spawned, currently active sessions need to be notified of the shutdown,
    // and exit gracefully. Await session termination.

    let _ = shutdown_sessions.send(true);
    let _ = shutdown_sessions.closed().await;

    result
}

// Main loop spawning session tasks, that handle commands coming in on a
// session. Runs for ever.

// TODO However, there are internal error conditions that *should* result in
// loop exit and Err result: for example if the listener somehow breaks and
// cannot be used to obtain connections any more. In that case the fault is not
// of some individual connection (which is not treated as an error here), but
// ours, and should be propagated.

// Arguments are moved into the `run_milter` future, so that when it is dropped,
// associated resources (except the ones only borrowed) are dropped at the same
// time, too.
async fn run_milter<T: Send + 'static>(
    listener: &Listener,
    callbacks: Callbacks<T>,
    config: Config,
    shutdown_sender: &watch::Sender<bool>,
) -> io::Result<()> {
    let callbacks = Arc::new(callbacks);
    let config = Arc::new(config);

    let conn_permits = Arc::new(Semaphore::new(config.max_connections));

    // This is a diverging function, ie the following loop runs for ever. Since
    // this function is called in a `select!` together with the shutdown signal,
    // the future produced by this function including all captured data will
    // simply be dropped on shutdown (at one of the following await points).

    loop {
        // Spawn new sessions continuously, but make sure that no more than the
        // max connections limit are in flight at the same time.
        // After this point, permission to handle a new connection is available.

        let permit = conn_permits.clone().acquire_owned().await.unwrap();

        // Wait for a connection. Then, ready to go: synchronously spawn a new
        // session to tokio and resume looping.

        match listener {
            Listener::Tcp(listener) => {
                let (stream, _) = listener.accept().await?;

                spawn_session(stream, shutdown_sender, &callbacks, &config, permit);
            }
            Listener::Unix(listener) => {
                let (stream, _) = listener.accept().await?;

                spawn_session(stream, shutdown_sender, &callbacks, &config, permit);
            }
        };
    }
}

fn spawn_session<S, T>(
    stream: S,
    shutdown_sender: &watch::Sender<bool>,
    callbacks: &Arc<Callbacks<T>>,
    config: &Config,
    permit: OwnedSemaphorePermit,
) where
    S: AsyncRead + AsyncWrite + Unpin + Send + 'static,
    T: Send + 'static,
{
    let callbacks = callbacks.clone();

    let shutdown = shutdown_sender.subscribe();

    let session = Session::new(stream, shutdown, callbacks, config);

    tokio::spawn(async move {
        trace!("session beginning processing commands");

        // Every sessions runs until it either exits regularly (eg client issues
        // the "quit" command), or it exits with an error. An error is only
        // logged but not otherwise propagated.

        match session.process_commands().await {
            Ok(()) => {
                trace!("session done processing commands");
            }
            Err(e) => {
                trace!("error in session while processing commands: {}", e);
            }
        }

        // At this point the session and the stream that it contains are already
        // dropped. The final thing to do is to drop the permit to allow another
        // client to connect.

        drop(permit);
    });
}
