use bytes::{Buf, BufMut, Bytes, BytesMut};
use indymilter::{
    command::{Command, CommandKind, OptNegPayload},
    frame::{Frame, FrameError},
    reply::{Reply, ReplyKind},
    Actions, ProtoOpts,
};
use std::{
    collections::HashMap,
    convert::{TryFrom, TryInto},
    io::{self, Cursor, ErrorKind},
};
use tokio::{
    io::{AsyncReadExt, AsyncWriteExt},
    net::{TcpStream, ToSocketAddrs},
};

pub struct Client {
    stream: TcpStream,
    buffer: BytesMut,
}

impl Client {
    pub async fn connect<A: ToSocketAddrs>(addr: A) -> io::Result<Self> {
        let stream = TcpStream::connect(addr).await?;

        Ok(Self {
            stream,
            buffer: BytesMut::new(),
        })
    }

    pub async fn write_command(&mut self, cmd: Command) -> io::Result<()> {
        let frame = match cmd {
            Command::OptNeg(OptNegPayload { version, actions, popts }) => {
                let mut buf = BytesMut::new();

                buf.put_u32(version);
                buf.put_u32(actions.bits());
                buf.put_u32(popts.bits());

                Frame::new(CommandKind::OptNeg, buf)
            }
            Command::Quit => Frame::new(CommandKind::Quit, Bytes::new()),
            Command::Data => Frame::new(CommandKind::Data, Bytes::new()),
            _ => todo!(),
        };

        self.write_frame(frame).await?;

        Ok(())
    }

    pub async fn write_frame(&mut self, frame: Frame) -> io::Result<()> {
        let buflen = u32::try_from(frame.buffer.len()).unwrap();
        let buflen = buflen.checked_add(1).unwrap();

        self.stream.write_u32(buflen).await?;
        self.stream.write_u8(frame.kind).await?;
        self.stream.write_all(frame.buffer.as_ref()).await?;
        self.stream.flush().await?;

        Ok(())
    }

    pub async fn read_reply(&mut self) -> io::Result<Option<Reply>> {
        let frame = self.read_frame().await?;

        match frame {
            Some(frame) => match parse_reply(frame) {
                Ok(reply) => Ok(Some(reply)),
                Err(_e) => Err(ErrorKind::InvalidData.into()),
            },
            None => Ok(None),
        }
    }

    async fn read_frame(&mut self) -> io::Result<Option<Frame>> {
        loop {
            if let Some(cmd) = self.parse_frame()? {
                return Ok(Some(cmd));
            }

            if self.stream.read_buf(&mut self.buffer).await? == 0 {
                return if self.buffer.is_empty() {
                    Ok(None)
                } else {
                    Err(ErrorKind::ConnectionReset.into())
                };
            }
        }
    }

    fn parse_frame(&mut self) -> io::Result<Option<Frame>> {
        let mut buf = Cursor::new(&self.buffer[..]);

        match Frame::parse(&mut buf) {
            Ok(frame) => {
                let len = buf.position() as usize;

                self.buffer.advance(len);

                Ok(Some(frame))
            }
            Err(FrameError::Incomplete) => Ok(None),
            Err(_e) => Err(ErrorKind::InvalidData.into()),
        }
    }

    // Note: consumes and therefore drops this client and connection.
    pub async fn disconnect(mut self) -> io::Result<()> {
        self.stream.shutdown().await
    }
}

// TODO implement properly
use std::error::Error;
fn parse_reply(frame: Frame) -> Result<Reply, Box<dyn Error + Send + Sync>> {
    let kind = frame.kind.try_into().unwrap();
    match kind {
        ReplyKind::Continue => Ok(Reply::Continue),
        ReplyKind::OptNeg => {
            let mut buf = frame.buffer;

            let version = buf.get_u32();
            let actions = Actions::from_bits(buf.get_u32()).unwrap();
            let popts = ProtoOpts::from_bits(buf.get_u32()).unwrap();
            let requested_macros = HashMap::new();
            // TODO macros

            Ok(Reply::OptNeg {
                version,
                actions,
                popts,
                requested_macros,
            })
        }
        _ => todo!(),
    }
}
