use crate::{
    callbacks::{Callbacks, Status},
    command::{
        Actions, BodyPayload, CommandError, CommandFrame, CommandKind, ConnInfoPayload,
        EnvAddrPayload, HeaderPayload, HeloPayload, MacroPayload, OptNegPayload, ProtoOpts,
        UnknownPayload,
    },
    config::Config,
    connection::Connection,
    context::{Context, EomContext, NegotiateContext},
    macros::Stage,
    reply::Reply,
};
use bytes::Bytes;
use std::{
    collections::HashMap,
    convert::TryFrom,
    error::Error,
    ffi::CString,
    fmt::{self, Display, Formatter},
    io, mem,
    sync::Arc,
};
use tokio::{
    io::{AsyncRead, AsyncWrite},
    select,
    sync::watch,
};
use tracing::{debug, trace};

#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq)]
pub enum State {
    Init,
    Opts,
    Conn,
    Helo,
    Mail,
    Rcpt,
    Data,
    Header,
    Eoh,
    Body,
    Eom,
    Quit,
    Abort,
    Unknown,
    QuitNc,
}

impl State {
    fn all_states() -> impl DoubleEndedIterator<Item = Self> {
        use State::*;
        [
            Init, Opts, Conn, Helo, Mail, Rcpt, Data, Header, Eoh, Body, Eom, Quit, Abort, Unknown,
            QuitNc,
        ]
        .iter()
        .copied()
    }

    fn is_mail_transaction(&self) -> bool {
        use State::*;
        matches!(self, Mail | Rcpt | Data | Header | Eoh | Body)
    }

    fn can_reach(&self, target: Self, popts: &ProtoOpts) -> bool {
        let can_be_skipped = |s: &Self| match s {
            Self::Conn => popts.contains(ProtoOpts::NOCONNECT),
            Self::Helo => popts.contains(ProtoOpts::NOHELO),
            Self::Mail => popts.contains(ProtoOpts::NOMAIL),
            Self::Rcpt => popts.contains(ProtoOpts::NORCPT),
            Self::Header => popts.contains(ProtoOpts::NOHDRS),
            Self::Eoh => popts.contains(ProtoOpts::NOEOH),
            Self::Body => popts.contains(ProtoOpts::NOBODY),
            Self::Data => popts.contains(ProtoOpts::NODATA),
            Self::Unknown => popts.contains(ProtoOpts::NOUNKNOWN),
            _ => false,
        };

        // First, check if target state can be reached directly.
        // Then, check if target state can be reached from some subsequent,
        // skippable state.

        if self.has_transition_to(target) {
            return true;
        }

        self.all_remaining()
            .skip(1)
            .take_while(can_be_skipped)
            .any(|s| s.has_transition_to(target))
    }

    fn has_transition_to(&self, next: Self) -> bool {
        use State::*;
        match self {
            Init => matches!(next, Opts),
            Opts | QuitNc => matches!(next, Conn | Unknown),
            Conn | Helo => matches!(next, Helo | Mail | Unknown),
            Mail => matches!(next, Rcpt | Abort | Unknown),
            Rcpt => matches!(
                next,
                Header | Eoh | Data | Body | Eom | Rcpt | Abort | Unknown
            ),
            Data | Header => matches!(next, Eoh | Header | Abort),
            Eoh | Body => matches!(next, Body | Eom | Abort),
            Eom => matches!(next, Quit | Mail | Unknown | QuitNc),
            Quit | Abort => false,
            Unknown => matches!(
                next,
                Helo | Mail | Rcpt | Data | Body | Unknown | Abort | Quit | QuitNc
            ),
        }
    }

    fn all_remaining(&self) -> impl Iterator<Item = Self> + '_ {
        Self::all_states().skip_while(move |s| s != self)
    }
}

fn callbacks_to_popts<T: Send>(callbacks: &Callbacks<T>) -> ProtoOpts {
    let mut popts = ProtoOpts::empty();
    popts.set(ProtoOpts::NOCONNECT, callbacks.connect.is_none());
    popts.set(ProtoOpts::NOHELO, callbacks.helo.is_none());
    popts.set(ProtoOpts::NOMAIL, callbacks.mail.is_none());
    popts.set(ProtoOpts::NORCPT, callbacks.rcpt.is_none());
    popts.set(ProtoOpts::NOHDRS, callbacks.header.is_none());
    popts.set(ProtoOpts::NOEOH, callbacks.eoh.is_none());
    popts.set(ProtoOpts::NOBODY, callbacks.body.is_none());
    popts.set(ProtoOpts::NODATA, callbacks.data.is_none());
    popts.set(ProtoOpts::NOUNKNOWN, callbacks.unknown.is_none());
    popts
}

pub type SessionResult = Result<(), SessionError>;

// Reasons for abnormal session termination.
#[derive(Debug)]
pub enum SessionError {
    Io(io::Error),
    ParseCommand(CommandError),
    ConnectionClosed,    // unexpected closing of connection mid-conversation
    RequestedActionsNotSupported, // requested actions cannot be fulfilled
    RequestedProtoOptsNotSupported,
    InvalidStatusInNegotiate(Status),
    BufferEmpty,  // empty buffer not allowed
    UnknownCommand,
    // TODO...
    // TODO timeout?
}

impl Display for SessionError {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        write!(f, "{:?}", self)
    }
}

impl Error for SessionError {}

impl From<io::Error> for SessionError {
    fn from(error: io::Error) -> Self {
        Self::Io(error)
    }
}

impl From<CommandError> for SessionError {
    fn from(error: CommandError) -> Self {
        Self::ParseCommand(error)
    }
}

pub struct Session<T: Send> {
    conn: Connection,

    context: Context<T>,

    callbacks: Arc<Callbacks<T>>,

    shutdown: watch::Receiver<bool>,

    state: State,

    // TODO negotiation

    initial_actions: Actions,
    initial_popts: ProtoOpts,

    actions: Actions,
    popts: ProtoOpts,
}

// TODO initialization: note that with QuitNc command, this session can again
// begin at the negotiate stage: ie need to make sure all fields are reset to
// initial values
impl<T: Send> Session<T> {
    pub fn new<S>(
        stream: S,
        shutdown: watch::Receiver<bool>,
        callbacks: Arc<Callbacks<T>>,
        config: Arc<Config>,
    ) -> Self
    where
        S: AsyncRead + AsyncWrite + Unpin + Send + 'static,
    {
        let initial_actions = config.actions;
        let conn_timeout = config.connection_timeout;

        let initial_popts = callbacks_to_popts(&callbacks);
        let conn = Connection::new(stream, conn_timeout);

        Self {
            conn,

            context: Context::new(),

            callbacks,
            shutdown,

            state: State::Init,

            initial_actions,
            initial_popts,
            actions: initial_actions,
            popts: initial_popts,
        }
    }

    // This takes session by value, as it is called only once. Therefore, when
    // the invocation returns in the task that calls it, the session and
    // associated values are already dropped.
    pub async fn process_commands(mut self) -> SessionResult {
        let result = self.process().await;

        if result.is_err() {
            if self.state.is_mail_transaction() {
                if let Some(abort) = &self.callbacks.abort {
                    let _ = abort(&mut self.context).await;
                }
            }
            if let Some(close) = &self.callbacks.close {
                let _ = close(&mut self.context).await;
            }
        }

        result
    }

    async fn process(&mut self) -> SessionResult {
        // For ever process this session’s connection by handling commands until
        // the quit command is received, or the shutdown signal is received.

        while self.state != State::Quit && !*self.shutdown.borrow() {
            // Read the next frame from the connection. Always also check if the
            // shutdown flag was touched, and reread its value, else proceed
            // with handling the command.

            let frame = select! {
                frame = self.conn.read_frame() => frame?,
                _ = self.shutdown.changed() => continue,
            };

            // TODO Note that if read_frame exits regularly (peer closes),
            // then still need to call abort and close callbacks ...
            // This is actually an error condition: while expecting a
            // new command, the client closed the connection -- w/o
            // issuing the quit/close command.
            let frame = frame.ok_or(SessionError::ConnectionClosed)?;

            trace!(?frame, "got next frame");

            // At this point we need a 'CommandFrame': a valid command frame,
            // but not yet fully parsed: the parsing should only happen once it
            // is determined that a callback will be called.

            let frame = CommandFrame::try_from(frame)
                .map_err(|_| SessionError::UnknownCommand)?;

            let cmd = frame.kind;

            // Check if desired state change from the CommandFrame is feasible,
            // taking into account the current set of protocol options.

            if let Some(next_state) = cmd.as_state() {
                if !self.state.can_reach(next_state, &self.popts) {
                    // When the desired state cannot be reached, abort the
                    // current mail transaction, and retry starting from the
                    // HELO state.

                    if self.state.is_mail_transaction() {
                        if let Some(abort) = &self.callbacks.abort {
                            let _ = abort(&mut self.context).await;
                        }
                    }

                    self.set_state(State::Helo);

                    if !self.state.can_reach(next_state, &self.popts) {
                        // The desired state cannot be reached; but if it's a
                        // quit anyway then exit regularly. Else, *ignore* the
                        // invalid command.
                        if next_state == State::Quit {
                            break;
                        } else {
                            debug!("ignoring unexpected command");
                            continue;
                        }
                    }
                }
            }

            // The state transition is acceptable. Ensure the given command
            // buffer is, too, then transition to the new state, and handle the
            // command.

            ensure_buffer_present(cmd, &frame.buffer)?;

            if let Some(next_state) = cmd.as_state() {
                self.set_state(next_state);
            }

            self.handle_command(frame).await?;
        }

        // TODO still call abort? may either be triggered by `?` and returning
        // early, or by receiving the shutdown signal and landing here!

        // We exited the loop in a different state than quit: for example if the
        // quit command was received in the middle of a conversation. Call the
        // close callback.

        if self.state != State::Quit {
            if let Some(close) = &self.callbacks.close {
                let _ = close(&mut self.context).await;
            }
        }

        Ok(())
    }

    fn set_state(&mut self, state: State) {
        // TODO however, note distinction curstate vs ctx_state?
        trace!(?state, "transitioned to new state");
        self.state = state;
    }

    async fn handle_command(&mut self, frame: CommandFrame) -> SessionResult {
        // Handling a command depends on the command kind. The command buffer is
        // only parsed (and parsing can only fail) if a callback is available.
        // The exception are `OptNeg` and `DefMacros` where the buffer is parsed
        // unconditionally and special handling applies.

        let status = match frame.kind {
            CommandKind::OptNeg => {
                self.handle_opt_neg_command(frame.buffer).await?;

                return Ok(());
            }
            CommandKind::DefMacro => {
                self.handle_def_macro_command(frame.buffer);

                return Ok(());
            }
            CommandKind::ConnInfo => {
                self.context.clear_macros_after(Stage::Connect);

                if let Some(connect) = &self.callbacks.connect {
                    let ConnInfoPayload {
                        hostname,
                        socket_info,
                    } = ConnInfoPayload::parse_buffer(frame.buffer)?;
                    connect(&mut self.context, hostname, socket_info).await
                } else {
                    Status::Continue
                }
            }
            CommandKind::Helo => {
                self.context.clear_macros_after(Stage::Helo);

                if let Some(helo) = &self.callbacks.helo {
                    let HeloPayload { hostname } = HeloPayload::parse_buffer(frame.buffer)?;
                    helo(&mut self.context, hostname).await
                } else {
                    Status::Continue
                }
            }
            CommandKind::Mail => {
                self.context.clear_macros_after(Stage::Mail);

                if let Some(mail) = &self.callbacks.mail {
                    let EnvAddrPayload { args } = EnvAddrPayload::parse_buffer(frame.buffer)?;
                    mail(&mut self.context, args).await
                } else {
                    Status::Continue
                }
            }
            CommandKind::Rcpt => {
                self.context.clear_macros_after(Stage::Rcpt);

                if let Some(rcpt) = &self.callbacks.rcpt {
                    let EnvAddrPayload { args } = EnvAddrPayload::parse_buffer(frame.buffer)?;
                    rcpt(&mut self.context, args).await
                } else {
                    Status::Continue
                }
            }
            CommandKind::Data => {
                if let Some(data) = &self.callbacks.data {
                    data(&mut self.context).await
                } else {
                    Status::Continue
                }
            }
            CommandKind::Header => {
                if let Some(header) = &self.callbacks.header {
                    let HeaderPayload { name, value } = HeaderPayload::parse_buffer(frame.buffer)?;
                    header(&mut self.context, name, value).await
                } else {
                    Status::Continue
                }
            }
            CommandKind::Eoh => {
                if let Some(eoh) = &self.callbacks.eoh {
                    eoh(&mut self.context).await
                } else {
                    Status::Continue
                }
            }
            CommandKind::BodyChunk => {
                if let Some(body) = &self.callbacks.body {
                    let BodyPayload { chunk } = BodyPayload::parse_buffer(frame.buffer)?;
                    body(&mut self.context, chunk).await
                } else {
                    Status::Continue
                }
            }
            CommandKind::BodyEnd => {
                let mut status = Status::Continue;

                if let Some(body) = &self.callbacks.body {
                    if !frame.buffer.is_empty() {
                        let BodyPayload { chunk } = BodyPayload::parse_buffer(frame.buffer)?;
                        status = body(&mut self.context, chunk).await;

                        if status != Status::Continue {
                            self.write_reply(status).await?;
                        }
                    }
                }

                if status == Status::Continue {
                    if let Some(eom) = &self.callbacks.eom {
                        let mut ctx = EomContext::new(
                            self.conn.clone(),
                            self.context.data.take(),
                            self.context.macros.duplicate(),
                            self.context.reply.duplicate(),
                            self.actions,
                        );

                        status = eom(&mut ctx).await;

                        // TODO restore context:
                        self.context.restore_from_eom(ctx);
                    }
                }

                status
            }
            CommandKind::Abort => {
                if let Some(abort) = &self.callbacks.abort {
                    abort(&mut self.context).await
                } else {
                    Status::Continue
                }
            }
            CommandKind::Quit | CommandKind::QuitNc => {
                let status = if let Some(close) = &self.callbacks.close {
                    close(&mut self.context).await
                } else {
                    Status::Continue
                };

                self.context.clear_macros();

                status
            }
            CommandKind::Unknown => {
                if let Some(unknown) = &self.callbacks.unknown {
                    let UnknownPayload { arg } = UnknownPayload::parse_buffer(frame.buffer)?;
                    unknown(&mut self.context, arg).await
                } else {
                    Status::Continue
                }
            }
        };

        self.write_reply(status).await?;

        // If a callback returns a status that terminates processing at some
        // stage, reset state to HELO.

        if status == Status::Accept
            || matches!(status, Status::Reject | Status::Discard | Status::Tempfail)
                && !matches!(self.state, State::Rcpt | State::Unknown)
        {
            self.set_state(State::Helo);
        }
        // TODO abort? not possible from callback I think -- check
        // can *only* happen in bodyend when sending reply after body callback (?)
        // and also when negotiate aborts, eg cannot satisfy milter reqs

        Ok(())
    }

    async fn handle_opt_neg_command(&mut self, buffer: Bytes) -> SessionResult {
        self.context.clear_macros();

        let OptNegPayload {
            version,
            actions: action_flags,
            popts: protocol_flags,
        } = OptNegPayload::parse_buffer(buffer)?;

        // TODO update session object with info

        let mta_version = version;

        // `mta_actions` are received in negotiate, and checked for
        // compliance, but not *used* anywhere later. However they are
        // used for `actions`.
        let mut mta_actions = action_flags;
        let mut mta_protocol_opts = protocol_flags;

        // clear macros > neg

        assert!(mta_version >= 6);

        if mta_actions.is_empty() {
            mta_actions = Actions::from_bits_truncate(0xf);
        }

        if mta_protocol_opts.is_empty() {
            mta_protocol_opts = ProtoOpts::from_bits_truncate(0x3f);
        }

        self.actions = self.initial_actions;

        let popts2mta;
        let mut requested_macros = Default::default();

        // If a callback is available, ...
        if let Some(negotiate) = &self.callbacks.negotiate {
            // TODO more stuff ...

            let mut ctx = NegotiateContext::new(
                // self.context.macros.duplicate(),
                self.context.reply.duplicate(),
                mta_actions,
                self.initial_popts,
            );

            let status = negotiate(&mut ctx, mta_actions, mta_protocol_opts).await;

            let requested_actions = ctx.requested_actions;
            let requested_popts = ctx.requested_proto_opts;
            requested_macros = mem::take(&mut ctx.requested_macros);

            // TODO restore context
            self.context.data = ctx.data;
            self.context.reply = ctx.reply;

            match status {
                Status::AllOpts => {
                    self.actions = mta_actions;
                    popts2mta = self.initial_popts;
                    // ...
                }
                Status::Continue => {
                    self.actions = requested_actions;
                    popts2mta = requested_popts;
                    // ...
                }
                status => {
                    return Err(SessionError::InvalidStatusInNegotiate(status));
                }
            }

            // TODO ...
        } else {
            popts2mta = self.initial_popts;

            // TODO ...
        }

        // TODO else, update session a bit more, check requirements etc. (may error)
        // especially update stages (some may be disabled even though available)!

        if !mta_actions.contains(self.actions) {
            return Err(SessionError::RequestedActionsNotSupported);
        }

        if !mta_protocol_opts.contains(popts2mta) {
            return Err(SessionError::RequestedProtoOptsNotSupported);
        }

        // TODO fix_stm

        self.write_optneg_reply(self.actions, popts2mta, requested_macros)
            .await?;

        Ok(())
    }

    fn handle_def_macro_command(&mut self, buffer: Bytes) {
        // Macro definitions are handled leniently. Failures are silent, eg when
        // the buffer contains unusable data, there are empty keys, or if there
        // is not an even number of macro keys/values.
        // TODO though note that some of this isn't really "failure": eg Postfix
        // will send empty macros ("") which cannot be parsed ...

        let MacroPayload { stage, macros } = match MacroPayload::parse_buffer(buffer) {
            Ok(payload) => payload,
            Err(e) => {
                // Errors in a macro command are ignored. Resume loop.
                debug!("invalid macro command received: {}", e);
                return;
            }
        };

        // TODO Note: convert to str, drop isolated excess value, remove empty keys
        let pairs = macros
            .chunks(2)
            .map(|chunk| match chunk {
                [k, v] => (k.to_owned(), v.to_owned()),
                _ => unreachable!(),
            })
            .filter(|(k, _)| !k.as_bytes().is_empty())
            .collect();

        debug!(?stage, ?pairs, "inserted new macros");

        self.context.insert_macros(stage, pairs);
    }

    async fn write_optneg_reply(
        &mut self,
        requested_actions: Actions,
        requested_popts: ProtoOpts,
        requested_macros: HashMap<Stage, CString>,
    ) -> io::Result<()> {
        let c = Reply::OptNeg {
            version: 0x6,
            actions: requested_actions, //Actions::from_bits_truncate(0x1ff),
            popts: requested_popts,
            requested_macros,
        };

        self.conn.write_reply(c).await
    }

    async fn write_reply(&mut self, status: Status) -> io::Result<()> {
        if matches!(self.state, State::Quit | State::QuitNc | State::Abort) {
            // no reply if regularly received quit or quit_nc command
            return Ok(());
        }

        fn reject_or_tempfail<T: Send>(session: &mut Session<T>, initial: u8, r: Reply) -> Reply {
            session
                .context
                .reply
                .take_reply_if_init_eq(initial)
                .map_or(r, |reply| Reply::ReplyCode { reply })
        }

        let c = match status {
            Status::Accept => Reply::Accept,
            Status::Continue => Reply::Continue,
            Status::Reject => reject_or_tempfail(self, b'5', Reply::Reject),
            Status::Tempfail => reject_or_tempfail(self, b'4', Reply::Tempfail),
            Status::Discard => Reply::Discard,
            Status::Skip => Reply::Skip,
            Status::Noreply => {
                // TODO write continue if not supported
                return Ok(());
            }
            Status::AllOpts => {
                return Ok(());
            }
        };

        self.conn.write_reply(c).await
    }
}

fn ensure_buffer_present(cmd: CommandKind, buf: &[u8]) -> SessionResult {
    use CommandKind::*;
    if matches!(
        cmd,
        DefMacro | BodyChunk | ConnInfo | Helo | Header | Mail | OptNeg | Rcpt | Unknown
    ) && buf.is_empty()
    {
        return Err(SessionError::BufferEmpty);
    }

    Ok(())
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn can_reach_ok() {
        use State::*;

        let popts = ProtoOpts::NOHELO
            | ProtoOpts::NOMAIL
            | ProtoOpts::NOHDRS
            | ProtoOpts::NOEOH
            | ProtoOpts::NOBODY
            | ProtoOpts::NODATA
            | ProtoOpts::NOUNKNOWN;

        assert!(Init.can_reach(Opts, &popts));
        assert!(Opts.can_reach(Conn, &popts));
        assert!(!Conn.can_reach(Conn, &popts));
        assert!(!Opts.can_reach(Helo, &popts));
        assert!(Conn.can_reach(Helo, &popts));
        assert!(Conn.can_reach(Mail, &popts));
        assert!(Conn.can_reach(Rcpt, &popts));
        assert!(!Conn.can_reach(Data, &popts));
    }
}
