//! Milter replies.

use crate::{
    command::{Actions, ProtoOpts},
    macros::Stage,
};
use std::{
    collections::HashMap,
    convert::TryFrom,
    error::Error,
    ffi::CString,
    fmt::{self, Display, Formatter},
};

#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq)]
pub enum ReplyKind {
    AddRcpt,
    DeleteRcpt,
    AddRcptExt,
    OptNeg,
    Accept,
    ReplaceBody,
    Continue,
    Discard,
    ChangeSender,
    AddHeader,
    InsertHeader,
    ChangeHeader,
    Progress,
    Quarantine,
    Reject,
    Skip,
    Tempfail,
    ReplyCode,
}

impl From<ReplyKind> for u8 {
    fn from(kind: ReplyKind) -> Self {
        match kind {
            ReplyKind::AddRcpt => b'+',
            ReplyKind::DeleteRcpt => b'-',
            ReplyKind::AddRcptExt => b'2',
            ReplyKind::OptNeg => b'O',
            ReplyKind::Accept => b'a',
            ReplyKind::ReplaceBody => b'b',
            ReplyKind::Continue => b'c',
            ReplyKind::Discard => b'd',
            ReplyKind::ChangeSender => b'e',
            ReplyKind::AddHeader => b'h',
            ReplyKind::InsertHeader => b'i',
            ReplyKind::ChangeHeader => b'm',
            ReplyKind::Progress => b'p',
            ReplyKind::Quarantine => b'q',
            ReplyKind::Reject => b'r',
            ReplyKind::Skip => b's',
            ReplyKind::Tempfail => b't',
            ReplyKind::ReplyCode => b'y',
        }
    }
}

impl TryFrom<u8> for ReplyKind {
    type Error = ReplyError;

    fn try_from(value: u8) -> Result<Self, Self::Error> {
        match value {
            b'+' => Ok(Self::AddRcpt),
            b'-' => Ok(Self::DeleteRcpt),
            b'2' => Ok(Self::AddRcptExt),
            b'O' => Ok(Self::OptNeg),
            b'a' => Ok(Self::Accept),
            b'b' => Ok(Self::ReplaceBody),
            b'c' => Ok(Self::Continue),
            b'd' => Ok(Self::Discard),
            b'e' => Ok(Self::ChangeSender),
            b'h' => Ok(Self::AddHeader),
            b'i' => Ok(Self::InsertHeader),
            b'm' => Ok(Self::ChangeHeader),
            b'p' => Ok(Self::Progress),
            b'q' => Ok(Self::Quarantine),
            b'r' => Ok(Self::Reject),
            b's' => Ok(Self::Skip),
            b't' => Ok(Self::Tempfail),
            b'y' => Ok(Self::ReplyCode),
            _ => Err(ReplyError::UnknownReplyKind),
        }
    }
}

// TODO
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq)]
pub enum ReplyError {
    UnknownReplyKind,
}

impl Display for ReplyError {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        write!(f, "{:?}", self)
    }
}

impl Error for ReplyError {}

/// A reply.
#[derive(Clone, Debug, Eq, PartialEq)]
pub enum Reply {
    /// The `e` reply.
    ChangeSender {
        mail: CString,  // non-empty
        args: Option<CString>,
    },
    /// The `+` reply.
    AddRcpt {
        rcpt: CString,  // non-empty
    },
    /// The `2` reply.
    AddRcptExt {
        rcpt: CString,  // non-empty
        args: Option<CString>,
    },
    /// The `-` reply.
    DeleteRcpt {
        rcpt: CString,  // non-empty
    },
    /// The `O` reply.
    OptNeg {
        version: u32,
        actions: Actions,
        popts: ProtoOpts,
        requested_macros: HashMap<Stage, CString>,
    },
    /// The `a` reply.
    Accept,
    /// The `b` reply.
    ReplaceBody {
        chunk: Vec<u8>,
    },
    /// The `c` reply.
    Continue,
    /// The `d` reply.
    Discard,
    /// The `h` reply.
    AddHeader {
        name: CString,  // non-empty
        value: CString,
    },
    /// The `i` reply.
    InsertHeader {
        index: i32,  // non-negative
        name: CString,  // non-empty
        value: CString,
    },
    /// The `m` reply.
    ChangeHeader {
        name: CString,  // non-empty
        index: i32,  // non-negative
        value: CString,
    },
    /// The `p` reply.
    Progress,
    /// The `q` reply.
    Quarantine {
        reason: CString,  // non-empty
    },
    /// The `r` reply.
    Reject,
    /// The `s` reply.
    Skip,
    /// The `t` reply.
    Tempfail,
    /// The `y` reply.
    ReplyCode {
        reply: CString,  // conforms to reply as produced in Context, eg "550 ..."
    },
}
