use crate::command::CommandError;
use std::{
    collections::HashMap,
    convert::TryFrom,
    ffi::{CStr, CString},
};

// Note: Different from libmilter: In libmilter, stages Eoh and Eom are in the
// wrong order (Eom before Eoh).
#[derive(Clone, Copy, Debug, Eq, Hash, PartialEq)]
pub enum Stage {
    Connect,
    Helo,
    Mail,
    Rcpt,
    Data,
    Eoh,
    Eom,
}

impl Stage {
    pub fn all_stages() -> impl DoubleEndedIterator<Item = Stage> {
        use Stage::*;
        [Connect, Helo, Mail, Rcpt, Data, Eoh, Eom].iter().copied()
    }
}

impl From<Stage> for i32 {
    fn from(stage: Stage) -> Self {
        match stage {
            Stage::Connect => 0,
            Stage::Helo => 1,
            Stage::Mail => 2,
            Stage::Rcpt => 3,
            Stage::Data => 4,
            Stage::Eoh => 6,
            Stage::Eom => 5,
        }
    }
}

impl TryFrom<u8> for Stage {
    type Error = CommandError;

    fn try_from(value: u8) -> Result<Self, Self::Error> {
        match value {
            b'C' => Ok(Self::Connect),
            b'H' => Ok(Self::Helo),
            b'M' => Ok(Self::Mail),
            b'R' => Ok(Self::Rcpt),
            b'T' => Ok(Self::Data),
            b'N' => Ok(Self::Eoh),
            b'E' => Ok(Self::Eom),
            _ => Err(CommandError::UnknownStage),
        }
    }
}

// TODO *not* Default or Clone: this is an encapsulating type that is not meant
// to be created outside this crate
// TODO Rename: MacrosHandle MacroList MacroLookup MacroValues MacroStore Macros
#[derive(Debug)]
pub struct MacroMap(HashMap<Stage, HashMap<CString, CString>>);

impl MacroMap {
    pub(crate) fn new() -> Self {
        Self(
            Stage::all_stages()
                .map(|stage| (stage, Default::default()))
                .collect(),
        )
    }

    pub(crate) fn duplicate(&self) -> Self {
        Self(self.0.clone())
    }

    pub fn get(&self, key: &CStr) -> Option<&CStr> {
        Stage::all_stages()
            .rev()
            .find_map(|stage| self.0[&stage].get(key).map(|v| v.as_ref()))
    }

    pub fn to_hash_map(&self) -> HashMap<CString, CString> {
        Stage::all_stages()
            .flat_map(|stage| self.0[&stage].clone())
            .collect()
    }

    pub(crate) fn insert(&mut self, key: Stage, entries: HashMap<CString, CString>) {
        self.0.insert(key, entries);
    }

    pub(crate) fn clear(&mut self) {
        for s in Stage::all_stages() {
            self.0.get_mut(&s).unwrap().clear();
        }
    }

    pub(crate) fn clear_after(&mut self, stage: Stage) {
        for s in Stage::all_stages().rev().take_while(|&s| s != stage) {
            self.0.get_mut(&s).unwrap().clear();
        }
    }
}

// TODO delete: should not be public
impl Default for MacroMap {
    fn default() -> Self {
        Self::new()
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use byte_strings::c_str;

    #[test]
    fn test_macro_map() {
        let mut macros = MacroMap::new();

        macros.insert(Stage::Connect, {
            let mut m = HashMap::new();
            m.insert(c_str!("a").into(), c_str!("b").into());
            m
        });

        assert_eq!(macros.get(c_str!("bla")), None);
        assert_eq!(macros.get(c_str!("a")), Some(c_str!("b")));
    }
}
