use bytes::{Buf, Bytes};
use std::{convert::TryFrom, io::Cursor};

// Limit length of frame buffer (somewhat arbitrarily) to 1MB -- else if someone
// erroneously sends a jumbo frame size we will spend a lot of time and memory
// accumulating it all.
const MAX_BUFFER_LEN: usize = 1024 * 1024 - 1;

#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub struct Frame {
    pub kind: u8,
    pub buffer: Bytes,
}

impl Frame {
    pub fn new(kind: impl Into<u8>, buffer: impl Into<Bytes>) -> Self {
        Self {
            kind: kind.into(),
            buffer: buffer.into(),
        }
    }

    pub fn parse(src: &mut Cursor<&[u8]>) -> Result<Frame, FrameError> {
        // A frame always consists of a 4-byte length (= buffer length + 1)
        // followed by a 1-byte command or reply kind.

        if src.remaining() < 5 {
            return Err(FrameError::Incomplete);
        }

        let len = usize::try_from(src.get_u32())
            .expect("unsupported pointer size")
            .saturating_sub(1);

        let kind = src.get_u8();

        if len > MAX_BUFFER_LEN {
            return Err(FrameError::TooLarge);
        }
        if src.remaining() < len {
            return Err(FrameError::Incomplete);
        }

        let buffer = src.copy_to_bytes(len);

        Ok(Frame { kind, buffer })
    }
}

#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub enum FrameError {
    Incomplete,
    TooLarge,
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn parse_ok() {
        let bytes = vec![0, 0, 0, 2, b'A', b'\0'];
        let mut cursor = Cursor::new(&bytes[..]);
        let f = Frame::parse(&mut cursor).unwrap();
        assert_eq!(f, Frame::new(b'A', &[0][..]));

        let bytes = vec![0, 0, 0, 5, b'A', b'\0'];
        let mut cursor = Cursor::new(&bytes[..]);
        assert_eq!(Frame::parse(&mut cursor), Err(FrameError::Incomplete));
    }
}
