use crate::{
    command::{Actions, ProtoOpts},
    connection::Connection,
    ffi_util::IntoCString,
    macros::{MacroMap, Stage},
    reply::Reply,
};
use async_trait::async_trait;
use std::{
    collections::HashMap,
    error::Error,
    ffi::CString,
    fmt::{self, Display, Formatter},
    io::{self, Write},
    str::FromStr,
};

pub trait SetErrorReply {
    fn set_error_reply<I, T>(
        &mut self,
        rcode: &str,
        xcode: Option<&str>,
        message: I,
    ) -> Result<(), ContextError>
    where
        I: IntoIterator<Item = T>,
        T: IntoCString;
}

#[async_trait]
pub trait ContextActions {
    async fn add_header<'cx, 'k, 'v>(
        &'cx self,
        name: impl IntoCString + Send + 'k,
        value: impl IntoCString + Send + 'v,
    ) -> Result<(), ContextError>;

    async fn insert_header<'cx, 'k, 'v>(
        &'cx self,
        index: i32,
        name: impl IntoCString + Send + 'k,
        value: impl IntoCString + Send + 'v,
    ) -> Result<(), ContextError>;

    async fn change_header<'cx, 'k, 'v>(
        &'cx self,
        name: impl IntoCString + Send + 'k,
        index: i32,
        value: Option<impl IntoCString + Send + 'v>,
    ) -> Result<(), ContextError>;

    async fn change_sender<'cx, 'a, 'b>(
        &'cx self,
        mail: impl IntoCString + Send + 'a,
        args: Option<impl IntoCString + Send + 'b>,
    ) -> Result<(), ContextError>;

    async fn add_recipient<'cx, 'a>(
        &'cx self,
        rcpt: impl IntoCString + Send + 'a,
    ) -> Result<(), ContextError>;

    async fn add_recipient_ext<'cx, 'a, 'b>(
        &'cx self,
        rcpt: impl IntoCString + Send + 'a,
        args: Option<impl IntoCString + Send + 'b>,
    ) -> Result<(), ContextError>;

    async fn delete_recipient<'cx, 'a>(
        &'cx self,
        rcpt: impl IntoCString + Send + 'a,
    ) -> Result<(), ContextError>;

    async fn replace_body<'cx, 'a>(&'cx self, chunk: &'a [u8]) -> Result<(), ContextError>;

    async fn progress<'cx>(&'cx self) -> Result<(), ContextError>;

    async fn quarantine<'cx, 'a>(
        &'cx self,
        reason: impl IntoCString + Send + 'a,
    ) -> Result<(), ContextError>;
}

#[derive(Debug)]
pub enum ContextError {
    ActionUnavailable,
    InvalidReply,
    Io(io::Error),

    // TODO separate out into separate type:
    InvalidReplyCode,
    InvalidEnhancedStatusCode,
    InvalidReplyText,
}

impl Display for ContextError {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        write!(f, "{:?}", self)
    }
}

impl Error for ContextError {}

impl From<io::Error> for ContextError {
    fn from(error: io::Error) -> Self {
        Self::Io(error)
    }
}

pub struct NegotiateContext<T: Send> {
    pub data: Option<T>,

    // note: there are no macros at this stage
    pub reply: ReplyHandle,

    pub requested_actions: Actions,
    pub requested_proto_opts: ProtoOpts,
    pub requested_macros: HashMap<Stage, CString>,
}

impl<T: Send> NegotiateContext<T> {
    pub(crate) fn new(
        reply: ReplyHandle,
        requested_actions: Actions,
        requested_proto_opts: ProtoOpts,
    ) -> Self {
        Self {
            data: None,  // always `None` initially
            reply,
            requested_actions,
            requested_proto_opts,
            requested_macros: HashMap::new(),
        }
    }
}

pub struct Context<T: Send> {
    pub data: Option<T>,

    // TODO if we provide a &mut reference to the context then this is all open,
    // and can be replaced by users; must somehow be read-only/protected/uninstantiable
    pub macros: MacroMap,
    // TODO alternatively, we could also expose ErrorReply directly -- then test code
    // can mock it as `&mut Option<ErrorReply>`?
    pub reply: ReplyHandle,
}

impl<T: Send> Context<T> {
    pub(crate) fn new() -> Self {
        Self {
            data: None,
            macros: MacroMap::new(),
            reply: ReplyHandle::new(),
        }
    }

    pub(crate) fn clear_macros(&mut self) {
        self.macros.clear();
    }

    pub(crate) fn clear_macros_after(&mut self, stage: Stage) {
        self.macros.clear_after(stage);
    }

    pub(crate) fn insert_macros(&mut self, stage: Stage, entries: HashMap<CString, CString>) {
        self.macros.insert(stage, entries);
    }

    pub(crate) fn restore_from_eom(&mut self, cx: EomContext<T>) {
        self.data = cx.data;
        self.reply = cx.reply;
    }
}

pub struct EomContext<T: Send> {
    pub data: Option<T>,

    pub macros: MacroMap,
    pub reply: ReplyHandle,
    pub actions: EomContextActions,
}

impl<T: Send> EomContext<T> {
    pub fn new(
        conn: Connection,
        data: Option<T>,
        macros: MacroMap,
        reply: ReplyHandle,
        available_actions: Actions,
    ) -> Self {
        Self {
            data,
            macros,
            reply,
            actions: EomContextActions {
                conn,
                available_actions,
            },
        }
    }
}

#[derive(Debug)]
pub struct ReplyHandle {
    reply: Option<ErrorReply>,
}

impl ReplyHandle {
    pub(crate) fn new() -> Self {
        Self { reply: None }
    }

    pub(crate) fn duplicate(&self) -> Self {
        Self { reply: self.reply.clone() }
    }

    // TODO revisit
    pub(crate) fn take_reply_if_init_eq(&mut self, initial: u8) -> Option<CString> {
        if let Some(reply) = self.reply.as_ref() {
            if reply.rcode.as_ref().as_bytes()[0] == initial {
                let r = self.reply.take().unwrap();
                return Some(r.make_error_reply());
            }
        }
        None
    }
}

impl SetErrorReply for ReplyHandle {
    fn set_error_reply<I, T>(
        &mut self,
        rcode: &str,
        xcode: Option<&str>,
        message: I,
    ) -> Result<(), ContextError>
    where
        I: IntoIterator<Item = T>,
        T: IntoCString,
    {
        let rcode = rcode.parse().map_err(|_| ContextError::InvalidReplyCode)?;

        let xcode = xcode
            .map(str::parse)
            .transpose()
            .map_err(|_| ContextError::InvalidEnhancedStatusCode)?;

        let mut message_lines = Vec::new();
        for (i, line) in message.into_iter().enumerate() {
            if i >= 32 {
                return Err(ContextError::InvalidReplyText);
            }
            let line = line.into_c_string();
            if line.as_bytes().len() > 980 {
                return Err(ContextError::InvalidReplyText);
            }
            if line.as_bytes().iter().any(|&b| matches!(b, b'\r' | b'\n')) {
                return Err(ContextError::InvalidReplyText);
            }
            message_lines.push(line);
        }

        self.reply = Some(ErrorReply {
            rcode,
            xcode,
            message: message_lines,
        });

        Ok(())
    }
}

#[derive(Clone, Debug)]
struct ErrorReply {
    rcode: ReplyCode,
    xcode: Option<EnhancedStatusCode>,
    message: Vec<CString>,
}

impl ErrorReply {
    // TODO single line vs multi line:
    // single:  always "r "
    //          then may have "x "
    //          then may have "m"
    //          => ie minimal is just "550 "
    //          => regex:  /r (x|m|x m)?/
    // multi: always "r x "
    //        then may have no or multiple lines of text...
    // in other words need API that supports following possibs:
    // "r "
    // "r x"
    // "r m"
    // "r x m"
    // "r x m..."
    pub(crate) fn make_error_reply(&self) -> CString {
        // TODO revise

        let s = if self.message.len() <= 1 {
            let mut s = Vec::<u8>::new();
            write!(s, "{} ", self.rcode.as_ref()).unwrap();
            if let Some(xcode) = &self.xcode {
                write!(s, "{}", xcode.as_ref()).unwrap();
            }
            if let Some(m) = self.message.iter().next() {
                if self.xcode.is_some() {
                    write!(s, " ").unwrap();
                }
                s.write_all(m.as_bytes()).unwrap();
            }
            s
        } else {
            match &self.message[..] {
                [] | [_] => unreachable!(),
                [xs @ .., x] => {
                    let rcode = &self.rcode;
                    let xcode = self.xcode.as_ref().map_or(match rcode {
                        ReplyCode::Transient(_) => "4.0.0",
                        ReplyCode::Permanent(_) => "5.0.0",
                    }, |x| x.as_ref());

                    let mut s = Vec::<u8>::new();

                    for x in xs {
                        write!(s, "{}-{} ", rcode.as_ref(), xcode).unwrap();
                        s.write_all(x.as_bytes()).unwrap();
                        s.write_all(b"\r\n").unwrap();
                    }

                    write!(s, "{} {} ", rcode.as_ref(), xcode).unwrap();
                    s.write_all(x.as_bytes()).unwrap();
                    s
                }
            }
        };

        CString::new(s).expect("ill-formed error reply text")
    }
}

pub struct EomContextActions {
    conn: Connection,
    available_actions: Actions,
}

#[async_trait]
impl ContextActions for EomContextActions {
    async fn add_header<'cx, 'k, 'v>(
        &'cx self,
        name: impl IntoCString + Send + 'k,
        value: impl IntoCString + Send + 'v,
    ) -> Result<(), ContextError> {
        if !self.available_actions.contains(Actions::ADDHDRS) {
            return Err(ContextError::ActionUnavailable);
        }

        let name = name.into_c_string();
        if name.to_bytes().is_empty() {
            return Err(ContextError::InvalidReply);
        }

        let value = value.into_c_string();

        self.conn.write_reply(Reply::AddHeader { name, value }).await?;

        Ok(())
    }

    async fn insert_header<'cx, 'k, 'v>(
        &'cx self,
        index: i32,
        name: impl IntoCString + Send + 'k,
        value: impl IntoCString + Send + 'v,
    ) -> Result<(), ContextError> {
        if !self.available_actions.contains(Actions::ADDHDRS) {
            return Err(ContextError::ActionUnavailable);
        }

        if index < 0 {
            return Err(ContextError::InvalidReply);
        }

        let name = name.into_c_string();
        if name.to_bytes().is_empty() {
            return Err(ContextError::InvalidReply);
        }

        let value = value.into_c_string();

        self.conn.write_reply(Reply::InsertHeader { index, name, value }).await?;

        Ok(())
    }

    async fn change_header<'cx, 'k, 'v>(
        &'cx self,
        name: impl IntoCString + Send + 'k,
        index: i32,
        value: Option<impl IntoCString + Send + 'v>,
    ) -> Result<(), ContextError> {
        if !self.available_actions.contains(Actions::CHGHDRS) {
            return Err(ContextError::ActionUnavailable);
        }

        let name = name.into_c_string();
        if name.to_bytes().is_empty() {
            return Err(ContextError::InvalidReply);
        }

        if index < 0 {
            return Err(ContextError::InvalidReply);
        }

        let value = value.map_or_else(Default::default, |v| v.into_c_string());

        self.conn.write_reply(Reply::ChangeHeader { name, index, value }).await?;

        Ok(())
    }

    async fn change_sender<'cx, 'a, 'b>(
        &'cx self,
        mail: impl IntoCString + Send + 'a,
        args: Option<impl IntoCString + Send + 'b>,
    ) -> Result<(), ContextError> {
        if !self.available_actions.contains(Actions::CHGFROM) {
            return Err(ContextError::ActionUnavailable);
        }

        let mail = mail.into_c_string();
        if mail.to_bytes().is_empty() {
            return Err(ContextError::InvalidReply);
        }

        let args = args.map(|v| v.into_c_string());

        self.conn.write_reply(Reply::ChangeSender { mail, args }).await?;

        Ok(())
    }

    async fn add_recipient<'cx, 'a>(
        &'cx self,
        rcpt: impl IntoCString + Send + 'a,
    ) -> Result<(), ContextError> {
        if !self.available_actions.contains(Actions::ADDRCPT) {
            return Err(ContextError::ActionUnavailable);
        }

        let rcpt = rcpt.into_c_string();
        if rcpt.to_bytes().is_empty() {
            return Err(ContextError::InvalidReply);
        }

        self.conn.write_reply(Reply::AddRcpt { rcpt }).await?;

        Ok(())
    }

    async fn add_recipient_ext<'cx, 'a, 'b>(
        &'cx self,
        rcpt: impl IntoCString + Send + 'a,
        args: Option<impl IntoCString + Send + 'b>,
    ) -> Result<(), ContextError> {
        if !self.available_actions.contains(Actions::ADDRCPT_PAR) {
            return Err(ContextError::ActionUnavailable);
        }

        let rcpt = rcpt.into_c_string();
        if rcpt.to_bytes().is_empty() {
            return Err(ContextError::InvalidReply);
        }

        let args = args.map(|v| v.into_c_string());

        self.conn.write_reply(Reply::AddRcptExt { rcpt, args }).await?;

        Ok(())
    }

    async fn delete_recipient<'cx, 'a>(
        &'cx self,
        rcpt: impl IntoCString + Send + 'a,
    ) -> Result<(), ContextError> {
        if !self.available_actions.contains(Actions::DELRCPT) {
            return Err(ContextError::ActionUnavailable);
        }

        let rcpt = rcpt.into_c_string();
        if rcpt.to_bytes().is_empty() {
            return Err(ContextError::InvalidReply);
        }

        self.conn.write_reply(Reply::DeleteRcpt { rcpt }).await?;

        Ok(())
    }

    async fn replace_body<'cx, 'a>(&'cx self, chunk: &'a [u8]) -> Result<(), ContextError> {
        if !self.available_actions.contains(Actions::CHGBODY) {
            return Err(ContextError::ActionUnavailable);
        }

        // TODO revisit

        if chunk.is_empty() {
            let reply = Reply::ReplaceBody { chunk: vec![] };

            self.conn.write_reply(reply).await?;

            return Ok(());
        }

        let milter_chunk_size = 65535;

        let chunks = chunk.chunks(milter_chunk_size);

        for c in chunks {
            let reply = Reply::ReplaceBody { chunk: c.to_vec() };

            self.conn.write_reply(reply).await?;
        }

        Ok(())
    }

    async fn progress<'cx>(&'cx self) -> Result<(), ContextError> {
        self.conn.write_reply(Reply::Progress).await?;

        Ok(())
    }

    async fn quarantine<'cx, 'a>(
        &'cx self,
        reason: impl IntoCString + Send + 'a,
    ) -> Result<(), ContextError> {
        if !self.available_actions.contains(Actions::QUARANTINE) {
            return Err(ContextError::ActionUnavailable);
        }

        let reason = reason.into_c_string();
        if reason.to_bytes().is_empty() {
            return Err(ContextError::InvalidReply);
        }

        self.conn.write_reply(Reply::Quarantine { reason }).await?;

        Ok(())
    }
}

#[derive(Clone, Copy, Debug, Default, Eq, Hash, PartialEq)]
pub struct ParseStatusCodeError;

impl Error for ParseStatusCodeError {}

impl Display for ParseStatusCodeError {
    fn fmt(&self, f: &mut Formatter<'_>) -> fmt::Result {
        write!(f, "failed to parse status code")
    }
}

#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub enum ReplyCode {
    Transient(String),
    Permanent(String),
}

impl AsRef<str> for ReplyCode {
    fn as_ref(&self) -> &str {
        match self {
            Self::Transient(s) | Self::Permanent(s) => s,
        }
    }
}

impl FromStr for ReplyCode {
    type Err = ParseStatusCodeError;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s.as_bytes() {
            [x, y, z]
                if matches!(x, b'4' | b'5')
                    && matches!(y, b'0'..=b'9')
                    && matches!(z, b'0'..=b'9') =>
            {
                Ok(match x {
                    b'4' => Self::Transient(s.into()),
                    b'5' => Self::Permanent(s.into()),
                    _ => unreachable!(),
                })
            }
            _ => Err(ParseStatusCodeError),
        }
    }
}

#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub enum EnhancedStatusCode {
    Transient(String),
    Permanent(String),
}

impl AsRef<str> for EnhancedStatusCode {
    fn as_ref(&self) -> &str {
        match self {
            Self::Transient(s) | Self::Permanent(s) => s,
        }
    }
}

impl FromStr for EnhancedStatusCode {
    type Err = ParseStatusCodeError;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        fn is_three_digits(s: &str) -> bool {
            s == "0"
                || matches!(s.len(), 1..=3)
                    && s.chars().all(|c| c.is_ascii_digit())
                    && !s.starts_with('0')
        }

        let mut iter = s.splitn(3, '.');
        match (iter.next(), iter.next(), iter.next()) {
            (Some(class), Some(subject), Some(detail))
                if matches!(class, "4" | "5")
                    && is_three_digits(subject)
                    && is_three_digits(detail) =>
            {
                Ok(match class {
                    "4" => Self::Transient(s.into()),
                    "5" => Self::Permanent(s.into()),
                    _ => unreachable!(),
                })
            }
            _ => Err(ParseStatusCodeError),
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use byte_strings::c_str;

    #[test]
    fn error_reply_ok() {
        let e = ErrorReply {
            rcode: "550".parse().unwrap(),
            xcode: Some("5.0.0".parse().unwrap()),
            message: vec![c_str!("fail").into()],
        };

        let s = e.make_error_reply();

        assert_eq!(s, c_str!("550 5.0.0 fail").into());
    }

    #[test]
    fn error_reply_multi_ok() {
        let e = ErrorReply {
            rcode: "550".parse().unwrap(),
            xcode: Some("5.0.0".parse().unwrap()),
            message: vec![c_str!("fail").into(), c_str!("totally").into()],
        };

        let s = e.make_error_reply();

        assert_eq!(s, c_str!("550-5.0.0 fail\r\n550 5.0.0 totally").into());
    }
}
