use crate::algorithms::ptd::{PostType, Type};
use crate::algorithms::Properties;
use crate::client::{self, ResponseValue};
use crate::indieauth::AccessToken;
use crate::mf2::types::Item;
use serde::de::{self, Visitor};
use serde::ser::SerializeSeq;
use serde::{Deserialize, Deserializer, Serialize};
use std::fmt;
use std::str::FromStr;
use std::string::ToString;

#[derive(Debug, Clone, serde::Serialize, serde::Deserialize, PartialEq)]
#[serde(untagged, rename_all = "kebab-case")]
pub enum PostStatus {
    Published,
    Drafted,
    Other(String),
}

impl Default for PostStatus {
    fn default() -> Self {
        PostStatus::Drafted
    }
}

impl ToString for PostStatus {
    fn to_string(&self) -> String {
        match self {
            PostStatus::Published => "published".to_owned(),
            PostStatus::Drafted => "draft".to_owned(),
            PostStatus::Other(status) => format!("x-{}", status),
        }
    }
}

impl FromStr for PostStatus {
    type Err = std::convert::Infallible;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        if s.is_empty() {
            return Ok(Self::Drafted);
        }
        match s.to_lowercase().trim_matches('"') {
            "published" => Ok(Self::Published),
            "draft" => Ok(Self::Drafted),
            status => Ok(Self::Other(status.split_at(2).1.to_owned())),
        }
    }
}

#[derive(Copy, Clone, Debug, PartialEq)]
pub enum Order {
    Ascending,
    Descending,
}

impl Default for Order {
    fn default() -> Self {
        Self::Descending
    }
}

impl FromStr for Order {
    type Err = std::convert::Infallible;

    fn from_str(v: &str) -> Result<Self, Self::Err> {
        match v.to_lowercase().as_str() {
            "ascending" | "asc" | "a" => Ok(Self::Ascending),
            "descending" | "desc" | "d" | _ => Ok(Self::Descending),
        }
    }
}

impl ToString for Order {
    fn to_string(&self) -> String {
        match self {
            Self::Descending => "desc".to_owned(),
            Self::Ascending => "asc".to_owned(),
        }
    }
}

impl Serialize for Order {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        self.to_string().serialize(serializer)
    }
}

impl<'de> Deserialize<'de> for Order {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        <&str as serde::Deserialize>::deserialize(deserializer)
            .and_then(|s| Self::from_str(s).map_err(de::Error::custom))
    }
}

#[derive(Copy, Clone, Debug, PartialEq)]
pub enum Visibility {
    Public,
    Unlisted,
    Private,
}

impl Default for Visibility {
    fn default() -> Visibility {
        Visibility::Public
    }
}

impl FromStr for Visibility {
    type Err = std::convert::Infallible;

    fn from_str(v: &str) -> Result<Self, Self::Err> {
        match v.to_lowercase().as_str() {
            "private" => Ok(Visibility::Private),
            "unlisted" => Ok(Visibility::Unlisted),
            "public" | _ => Ok(Visibility::Public),
        }
    }
}

impl ToString for Visibility {
    fn to_string(&self) -> String {
        match self {
            Visibility::Public => "public".into(),
            Visibility::Unlisted => "unlisted".into(),
            Visibility::Private => "private".into(),
        }
    }
}

impl Serialize for Visibility {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        self.to_string().serialize(serializer)
    }
}

impl<'de> Deserialize<'de> for Visibility {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        String::deserialize(deserializer)
            .and_then(|vs| Self::from_str(&vs).map_err(de::Error::custom))
    }
}

#[derive(Clone, Debug, PartialEq, Serialize, Deserialize)]
#[serde(untagged, rename_all = "kebab-case")]
pub enum ChannelForm {
    Expanded {
        uid: String,
        name: String,

        #[serde(flatten, default)]
        properties: Properties,
    },
    Simple(String),
}

impl Default for ChannelForm {
    fn default() -> Self {
        Self::Simple("default".to_owned())
    }
}

impl ChannelForm {
    pub fn name(&self) -> String {
        match self {
            Self::Simple(name) => name.to_owned(),
            Self::Expanded { name, .. } => name.to_owned(),
        }
    }

    pub fn uid(&self) -> String {
        match self {
            Self::Simple(name) => name.to_owned(),
            Self::Expanded { uid, .. } => uid.to_owned(),
        }
    }
}

impl FromStr for ChannelForm {
    type Err = crate::Error;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        Ok(Self::Simple(s.to_string()))
    }
}

/// Represents the normalized form of pagination in Micropub queries.
#[derive(Deserialize, Serialize, Clone, Debug, PartialEq, Default)]
#[serde(rename_all = "kebab-case", deny_unknown_fields)]
pub struct Pagination {
    #[serde(skip_serializing_if = "Option::is_none")]
    pub limit: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    pub offset: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    pub filter: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    pub before: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    pub after: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    pub order: Option<Order>,
}

#[derive(Serialize, Deserialize, PartialEq, Debug, Clone)]
#[serde(rename_all = "kebab-case", deny_unknown_fields)]
pub struct PaginatedResponse {
    /// A list of JSON-serialized objects.
    pub items: Vec<serde_json::Value>,

    /// Defines the current paging options used.
    #[serde(default)]
    pub paging: Pagination,

    /// An optional representation of a cursor pointing to the most recent items for this query.
    #[serde(alias = "_latest")]
    pub latest: Option<String>,

    /// An optional representation of a cursor pointing to the earliest items for this query.
    #[serde(alias = "_earliest")]
    pub earliest: Option<String>,
}

#[derive(Serialize, Deserialize, PartialEq, Debug, Clone)]
#[serde(rename_all = "kebab-case", deny_unknown_fields)]
pub struct ConfigurationResponse {
    #[serde(default)]
    pub q: Vec<String>,

    #[serde(default)]
    pub channels: Vec<ChannelForm>,

    #[serde(default)]
    pub media_endpoint: Option<url::Url>,

    #[serde(default)]
    pub post_types: Vec<PostType>,

    #[serde(default)]
    pub syndicate_to: Vec<String>,
}

#[derive(Serialize, Deserialize, PartialEq, Debug, Clone)]
#[serde(rename_all = "kebab-case")]
pub struct SourceResponse {
    /// The hinted post type.
    #[serde(default, rename = "post-type", with = "as_string_or_list")]
    pub post_type: Vec<Type>,

    #[serde(flatten)]
    pub item: Item,
}

#[derive(PartialEq, Debug, Clone, Serialize, Deserialize)]
#[serde(rename_all = "kebab-case", deny_unknown_fields)]
pub struct ChannelResponse {
    pub channels: Vec<ChannelForm>,
}

/// Represents the response for a query.
// FIXME: Add the representation of syndication targets.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone)]
#[serde(untagged, rename_all = "kebab-case")]
pub enum QueryResponse {
    Source(SourceResponse),
    Channel(ChannelResponse),
    Configuration(ConfigurationResponse),
    Paginated(PaginatedResponse),
}

/// Used to convert a value or a list of values into a normalized list of values.
pub mod as_string_or_list {
    use super::*;

    pub fn deserialize<'de, Item, D>(deserializer: D) -> Result<Vec<Item>, D::Error>
    where
        Item: Deserialize<'de> + FromStr<Err = crate::Error> + fmt::Debug,
        D: Deserializer<'de>,
    {
        struct CocereIntoList;

        impl<'de> Visitor<'de> for CocereIntoList {
            type Value = Vec<String>;

            fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
                formatter.write_str(
                    "expecting no value, a string or a list of values, all to be made into a list",
                )
            }

            fn visit_str<E>(self, v: &str) -> Result<Self::Value, E>
            where
                E: de::Error,
            {
                if v.is_empty() {
                    Ok(Vec::default())
                } else {
                    Ok(vec![v.to_string()])
                }
            }

            fn visit_seq<A>(self, mut seq: A) -> Result<Self::Value, A::Error>
            where
                A: de::SeqAccess<'de>,
            {
                let mut values = Vec::with_capacity(seq.size_hint().unwrap_or_default());

                while let Some(value) = seq.next_element()? {
                    values.push(value);
                }

                Ok(values)
            }
        }

        deserializer
            .deserialize_any(CocereIntoList)
            .and_then(|strings| {
                strings.into_iter().try_fold(
                    Vec::default(),
                    |mut acc, string| match Item::from_str(&string) {
                        Ok(v) => {
                            acc.push(v);
                            Ok(acc)
                        }
                        Err(e) => Err(de::Error::custom(e)),
                    },
                )
            })
    }

    pub fn serialize<Item, S>(list: &Vec<Item>, serializer: S) -> Result<S::Ok, S::Error>
    where
        Item: ToString + Serialize,
        S: serde::Serializer,
    {
        if let Some(value) = list.first().filter(|_| list.len() == 1) {
            serializer.serialize_str(value.to_string().as_str())
        } else {
            let mut seq = serializer.serialize_seq(Some(list.len()))?;

            for v in list {
                seq.serialize_element(v)?;
            }

            seq.end()
        }
    }
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "kebab-case")]
pub struct Query {
    /// A representation of the base querying components.
    #[serde(flatten, rename = "q")]
    pub kind: QueryKind,

    /// Pagination options to modify where in the list to look
    #[serde(flatten, default)]
    pub pagination: Pagination,
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "kebab-case", tag = "q")]
pub enum QueryKind {
    /// Pulls the configuration of the Micropub server.
    #[serde(rename = "config")]
    Configuration,

    /// Represents a call to `?q=source` for post information
    #[serde(rename_all = "kebab-case")]
    Source {
        #[serde(skip_serializing_if = "Option::is_none", default)]
        url: Option<url::Url>,

        /// Represents one or many post types to ask for when filtering this query.
        #[serde(
            with = "as_string_or_list",
            default,
            skip_serializing_if = "Vec::is_empty"
        )]
        post_type: Vec<Type>,

        /// Represents one or many channels to ask for when filtering this query.
        #[serde(skip_serializing_if = "Vec::is_empty", default)]
        channel: Vec<ChannelForm>,
    },

    /// Represents a call to `?q=channel` to fetch channels.
    /// https://github.com/indieweb/micropub-extensions/issues/40
    #[serde(rename_all = "kebab-case")]
    Channel,

    /// Represents a call to `?q=syndicate-to` to fetch syndication targets.
    #[serde(rename_all = "kebab-case")]
    SyndicateTo {
        /// Represents a single or many post types to ask for when filtering this query.
        #[serde(with = "as_string_or_list", skip_serializing_if = "Vec::is_empty")]
        post_type: Vec<Type>,
    },

    /// Represents a call to `?q=rel` to fetch relationship links.
    #[serde(alias = "rel")]
    Relation {
        /// The relations to explicitly look for.
        #[serde(default, skip_serializing_if = "Vec::is_empty")]
        rel: Vec<String>,

        /// The identity to look up relationships for.
        url: url::Url,
    },
}

impl Query {
    /// Sends this request over to the requested Micropub server.
    pub fn send(
        &self,
        endpoint: &url::Url,
        access_token: &AccessToken,
    ) -> Result<ResponseValue<QueryResponse>, crate::Error> {
        let mut url = endpoint.clone();
        let self_string = serde_qs::to_string(&self).map_err(crate::Error::Qs)?;
        if url.query().is_none() {
            url.set_query(Some(self_string.as_str()));
        } else {
            let self_params = url::form_urlencoded::parse(self_string.as_bytes());
            url.query_pairs_mut().extend_pairs(self_params);
        }

        log::trace!("Sending request to {:?}", url);
        let req = client::Request {
            headers: client::Headers::from_iter(vec![
                ("accept", "application/json"),
                ("content-type", "application/json"),
                (
                    "authorization",
                    &format!("bearer {}", access_token.secret()),
                ),
            ]),
            body: vec![],
            url,
        };

        req.send("GET").and_then(client::Response::into_json)
    }
}

impl ToString for Query {
    fn to_string(&self) -> String {
        serde_qs::to_string(self).unwrap_or_default()
    }
}

#[cfg(test)]
#[allow(non_snake_case)]
mod tests {
    use microformats::types::{Class, KnownClass};
    use serde_json::json;

    use super::*;

    #[test]
    fn query_from_qs_str() {
        assert_eq!(
            Some(vec![Type::Note, Type::Read]),
            serde_qs::from_str::<Query>("q=source&post-type[]=note&post-type[]=read")
                .ok()
                .map(|q| q.kind)
                .map(|k| match k {
                    QueryKind::Source { post_type, .. } => post_type,
                    _ => vec![],
                }),
            "provides deserialization of the source query with multiple post type asked for"
        );
        assert_eq!(
            Some(vec![Type::Video, Type::Read]),
            serde_qs::Config::new(1, false)
                .deserialize_str::<Query>(
                    "q=source&post-type%5B0%5D=video&post-type%5B1%5D=read&limit=5"
                )
                .ok()
                .map(|q| q.kind)
                .map(|k| match k {
                    QueryKind::Source { post_type, .. } => post_type,
                    _ => vec![],
                }),
            "provides deserialization of the source query with multiple post type asked for"
        );
        assert_eq!(
            None,
            serde_qs::from_str::<Query>("q=source&post-type=")
                .err()
                .map(|e| e.to_string()),
            "ignores if the post type value is 'empty'"
        );
        assert_eq!(
            None,
            serde_qs::from_str::<Query>("q=config")
                .err()
                .map(|e| e.to_string()),
            "provides deserialization of the config query"
        );
    }

    #[test]
    fn query_response_for_channels() {
        assert_eq!(
            Some(QueryResponse::Channel(ChannelResponse {
                channels: vec![ChannelForm::Expanded {
                    uid: "magic".to_string(),
                    name: "Magic".to_string(),
                    properties: Properties::try_from(json!({"grr": "bark"})).unwrap()
                }],
            })),
            serde_json::from_value(json!({
                "channels": [{
                    "uid": "magic",
                    "name": "Magic",
                    "grr": "bark"
                }]
            }))
            .ok()
        )
    }

    #[test]
    fn query_response_for_configuration() {
        assert_eq!(
            Some(QueryResponse::Configuration(ConfigurationResponse {
                q: vec!["channels".to_owned()],
                channels: vec![ChannelForm::Simple("all".to_owned())],
                media_endpoint: None,
                post_types: vec![PostType::Simple(Type::Note)],
                syndicate_to: vec![]
            })),
            serde_json::from_value(json!({
                "q": ["channels"], "post-types": ["note"], "channels": ["all"]
            }))
            .ok()
        )
    }

    #[test]
    fn query_response_for_source() {
        crate::test::init();
        let mut item = Item::default();
        item.r#type = vec![Class::Known(KnownClass::Entry)];
        assert_eq!(
            Ok(QueryResponse::Source(SourceResponse {
                post_type: vec![],
                item: item.clone()
            })),
            serde_json::from_value(json!({
                "type": ["h-entry"],
                "properties": {}
            }))
            .map_err(|e| e.to_string())
        );

        assert_eq!(
            serde_json::from_value::<QueryResponse>(json!(
            {
                "post-type": [
                    "article"
                ],
                "properties": {
                    "audience": [],
                    "category": [],
                    "channel": [
                        "all"
                    ],
                    "content": {
                        "html": "<p>well-here-we-go</p>"
                    },
                    "name": "magic-omg",
                    "post-status": [
                        "published"
                    ],
                    "published": [
                        "2022-02-12T23:22:27+00:00"
                    ],
                    "slug": [
                        "Gzg043ii"
                    ],
                    "syndication": [],
                    "updated": [
                        "2022-02-12T23:22:27+00:00"
                    ],
                    "url": [
                        "http://localhost:3112/Gzg043ii"
                    ],
                    "visibility": [
                        "public"
                    ]
                },
                "type": [
                    "h-entry"
                ]
            }
                    ))
            .map_err(|e| e.to_string())
            .err(),
            None,
        )
    }

    #[test]
    fn post_status() {
        assert_eq!(
            Some(PostStatus::Drafted),
            PostStatus::from_str("draft").ok()
        );
    }
}
