use crate::client;
use anyhow::Context;
use scraper::{Html, Selector};

fn get_rels_from_body(html: String, rel: &str) -> Vec<String> {
    let link_rel_redirect_uri_selector =
        Selector::parse(&format!("link[rel=\"{}\"]", rel)).unwrap();
    let a_rel_redirect_uri_selector = Selector::parse(&format!("a[rel=\"{}\"]", rel)).unwrap();

    let html_document_fragment = Html::parse_fragment(html.as_str());

    let mut endpoints = Vec::new();

    endpoints.extend(
        html_document_fragment
            .select(&link_rel_redirect_uri_selector)
            .map(|element| element.value().attr("href").unwrap_or_default().to_owned()),
    );

    endpoints.extend(
        html_document_fragment
            .select(&a_rel_redirect_uri_selector)
            .map(|element| element.value().attr("href").unwrap_or_default().to_owned()),
    );

    endpoints
        .iter()
        .filter(|href| !href.is_empty())
        .cloned()
        .collect()
}

fn get_rels_from_header(headers: &client::Headers, rel: &str) -> Vec<String> {
    client::find_header_by_name(headers, "link")
        .into_iter()
        .map(|redirect_uri_header_value| {
            redirect_uri_header_value
                .split(',')
                .collect::<Vec<&str>>()
                .iter()
                .map(|link_rel: &&str| {
                    let mut parameters = link_rel.split("; ").collect::<Vec<&str>>();
                    let uri_value: String = parameters.remove(0).to_owned();
                    (uri_value, parameters)
                })
                .filter(|(_uri, parameters)| {
                    parameters.contains(&format!("rel=\"{}\"", rel).as_str())
                })
                .filter_map(|(uri, _parameters)| {
                    uri.strip_prefix("<")
                        .map(|u| u.to_string())
                        .unwrap_or_default()
                        .strip_suffix(">")
                        .map(|u| u.to_string())
                        .to_owned()
                })
                .collect::<Vec<String>>()
        })
        .flatten()
        .collect::<Vec<_>>()
}

/// Resolves all of the relating links for a particular URL.
pub fn for_url(url: &str, rel: &str) -> Result<Vec<String>, crate::Error> {
    let page_url = url::Url::parse(url)
        .context("Could not unravel the provided URL string as a valid URL.")
        .map_err(crate::Error::Other)?;
    let req = client::Request {
        url: page_url.clone(),
        headers: client::Headers::from_iter(vec![("accept", "text/html")]),
        body: vec![],
    };

    req.send("GET").and_then(|resp| {
        let rels_from_header = get_rels_from_header(&resp.headers, rel);
        let rels_from_body = get_rels_from_body(resp.body_as_string()?, rel);

        let mut all_links: Vec<String> = [rels_from_header, rels_from_body]
            .concat()
            .into_iter()
            .filter_map(
                |resolved_url| match url::Url::parse(resolved_url.as_str()) {
                    Ok(u) => Some(u.to_string()),
                    Err(_) => page_url
                        .clone()
                        .join(resolved_url.as_str())
                        .map(|url| url.to_string())
                        .ok(),
                },
            )
            .collect();

        // FIXME: Make this list unique
        all_links.dedup();
        Ok(all_links)
    })
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::test;
    use mockito::mock;

    #[test]
    fn for_url_returns_empty_list_on_error() {
        let cfg = test::init();
        let urls_for_rel = for_url("https://example.com", "payment");
        assert_eq!(urls_for_rel.map(|u| u.len()).ok(), Some(0));
    }

    #[test]
    fn for_url_provides_bad_urls_as_is() {
        let cfg = test::init();
        let url = format!("{}/bad-urls", mockito::server_url());
        let urls = vec![
            "well-dang".to_owned(),
            "https://paypal.not-me/a-user".to_owned(),
        ];
        let expected_urls = vec![
            format!("{}/well-dang", mockito::server_url()),
            urls.get(1).unwrap().clone(),
        ];
        let link_str = urls
            .iter()
            .map(|u| format!(r#"<link rel="payment" href={} />"#, u))
            .collect::<Vec<String>>()
            .join("\n");
        let _m = mock("GET", "/bad-urls")
            .with_status(200)
            .with_header("Content-Type", "text/html")
            .with_body(format!("<html>{}</html>", link_str))
            .create();
        let urls_for_rel = for_url(&url, "payment");
        assert_eq!(urls_for_rel.ok(), Some(expected_urls));
    }

    #[test]
    fn for_url_extracts_from_headers() {
        let cfg = test::init();
        let url = format!("{}/bad-urls", mockito::server_url());
        let urls = vec![
            "well-dang".to_owned(),
            "https://paypal.not-me/a-user".to_owned(),
        ];
        let expected_urls = vec![
            format!("{}/well-dang", mockito::server_url()),
            urls.get(1).unwrap().clone(),
        ];
        let link_str = urls
            .iter()
            .map(|u| format!(r#"<{}>; rel="payment""#, u))
            .collect::<Vec<String>>()
            .join(",");
        let _m = mock("GET", "/bad-urls")
            .with_status(200)
            .with_header("Content-Type", "text/html")
            .with_header("Link", &link_str)
            .with_body(String::default())
            .create();
        let urls_for_rel = for_url(&url, "payment");
        assert_eq!(urls_for_rel.ok(), Some(expected_urls));
    }

    #[test]
    fn for_url_extracts_from_html_body() {
        let cfg = test::init();
        let url = format!("{}/urls", mockito::server_url());
        let urls = vec![
            "/well-dang".to_owned(),
            "https://paypal.not-me/a-user".to_owned(),
        ];
        let expected_urls = vec![
            format!("{}/well-dang", mockito::server_url()),
            urls.get(1).unwrap().clone(),
        ];
        let link_str = urls
            .iter()
            .map(|u| format!(r#"<link rel="payment" href={} />"#, u))
            .collect::<Vec<String>>()
            .join("\n");
        let _m = mock("GET", "/urls")
            .with_status(200)
            .with_header("Content-Type", "text/html")
            .with_body(format!("<html>{}</html>", link_str))
            .create();
        let urls_for_rel = for_url(&url, "payment");
        assert_eq!(urls_for_rel.ok(), Some(expected_urls));
    }
}
