pub mod algorithms;
pub mod client;
pub mod indieauth;
pub mod link_rel;
pub mod micropub;
pub mod traits;
pub mod webmention;

#[cfg(test)]
mod test;

use mf2::types::Fragment;
pub use microformats as mf2;
pub use oauth2;
use url::Url;

/// Represents all of the error states of this crate.
#[derive(thiserror::Error, Debug)]
pub enum Error {
    #[error("No {rel:?} endpoints were found at {url:?}")]
    NoEndpointsFound { rel: String, url: String },

    #[error(
        "The Webmention endpoint at {url:?} returned a unexpected status code of {status_code:?})"
    )]
    WebmentionUnsupportedStatusCode { status_code: u16, url: String },

    #[error(transparent)]
    Client(#[from] ureq::Transport),

    #[error(transparent)]
    JSON(#[from] serde_json::Error),

    #[error(transparent)]
    Other(#[from] anyhow::Error),

    #[error(transparent)]
    Url(#[from] url::ParseError),

    #[error(transparent)]
    Qs(#[from] serde_qs::Error),

    #[error(transparent)]
    Microformats(#[from] mf2::Error),
}

impl PartialEq for Error {
    fn eq(&self, other: &Self) -> bool {
        std::mem::discriminant(self) == std::mem::discriminant(other)
    }
}

/// Converts into a concrete representation of text.
///
/// # Examples
/// ```
/// # use indieweb::mf2::types::Fragment;
///
/// assert_eq!(
///     Ok(Fragment { value: "wow".to_string(), ..Default::default() }),
///     parse_content_value(serde_json::Value::String("wow".to_string()), "https://indieweb.org".parse().unwrap())
/// );
pub fn parse_content_value<V>(value: V, url: &Url) -> Vec<Fragment>
where
    V: Into<serde_json::Value>,
{
    use serde_json::Value;
    match value.into() {
        Value::String(text) => vec![Fragment {
            value: mf2::resolve_text_from_html(&text, url.to_owned()).unwrap_or(text.clone()),
            html: text,
            ..Default::default()
        }],
        Value::Object(obj) => {
            vec![serde_json::from_value::<Fragment>(Value::Object(obj)).unwrap_or_default()]
        }
        Value::Array(values) => values
            .into_iter()
            .map(|v| parse_content_value(v, url))
            .flatten()
            .collect::<Vec<_>>(),
        _ => vec![],
    }
}
