/// PTD, or [post type discovery](https://indieweb.org/ptd) is a means of resolving the
/// semantically relevant post type of a provided [microformats::types::Item].
use std::collections::HashMap;
use std::iter::FromIterator;
use std::str::FromStr;

/// A canonical list of the recognized post types.
#[derive(Debug, Clone, serde::Serialize, serde::Deserialize)]
#[serde(rename_all = "kebab-case")]
pub enum Type {
    Like,
    Bookmark,
    Reply,
    Repost,
    Note,
    Article,
    Photo,
    Video,
    Audio,
    Media,
    Quote,
    GamePlay,
    #[serde(rename = "rsvp")]
    RSVP,
    #[serde(rename = "checkin")]
    CheckIn,
    Listen,
    Watch,
    Review,
    Read,
    Jam,
    Follow,
    Event,
    Other(String),
}

impl ToString for Type {
    fn to_string(&self) -> String {
        serde_json::to_value(self)
            .unwrap_or_default()
            .as_str()
            .unwrap_or_default()
            .to_owned()
    }
}

impl PartialEq for Type {
    fn eq(&self, other: &Self) -> bool {
        self.to_string() == other.to_string()
    }
}

impl Default for Type {
    fn default() -> Type {
        Type::Note
    }
}

/// Represents the potential forms of defining a post type. The similar form
/// as represented by a single string is the one conventionally used. The
/// expanded form is one that's being experimented on to allow for the definition
/// of the constraints that a type can set.
#[derive(Debug, Clone, serde::Serialize, serde::Deserialize)]
#[serde(untagged, rename_all = "kebab-case")]
pub enum PostType {
    /// Represents a simpler (textual) form of a post type.
    Simple(Type),

    /// Represents an expanded way to describe a post type.
    Expanded {
        /// The presentational name of the post type.
        name: String,

        /// The known post type being expanded.
        #[serde(alias = "type")]
        kind: Type,

        /// The container type represented as a [microformats::types::Class]
        #[serde(default = "default_class")]
        h: microformats::types::Class,

        /// Recognized properties for this post type.
        #[serde(default)]
        properties: Vec<String>,

        /// Properties for this post type that are required.
        #[serde(default)]
        required_properties: Vec<String>,
    },
}

fn default_class() -> microformats::types::Class {
    use crate::mf2;
    mf2::types::Class::Known(mf2::types::KnownClass::Entry)
}

impl From<PostType> for Type {
    fn from(post_type: PostType) -> Type {
        match post_type {
            PostType::Simple(kind) => kind,
            PostType::Expanded { kind, .. } => kind,
        }
    }
}

impl From<&PostType> for Type {
    fn from(post_type: &PostType) -> Type {
        match post_type {
            PostType::Simple(kind) => kind.clone(),
            PostType::Expanded { kind, .. } => kind.clone(),
        }
    }
}

impl FromStr for Type {
    type Err = crate::Error;

    fn from_str(value: &str) -> Result<Self, Self::Err> {
        serde_json::from_value::<Self>(serde_json::Value::String(value.to_string()))
            .map_err(crate::Error::JSON)
    }
}

impl PartialEq for PostType {
    fn eq(&self, other: &Self) -> bool {
        let ltype: Type = self.into();
        let rtype: Type = other.into();

        ltype == rtype
    }
}

impl PostType {
    pub fn name(&self) -> String {
        match self {
            Self::Simple(simple_type) => simple_type.to_string(),
            Self::Expanded { ref name, .. } => name.to_string(),
        }
    }

    pub fn kind(&self) -> String {
        match self {
            Self::Simple(simple_type) => simple_type.to_string(),
            Self::Expanded { ref name, .. } => name.to_string(),
        }
    }
}

fn properties_from_type() -> HashMap<String, Type> {
    HashMap::from_iter(
        vec![
            ("like-of".to_owned(), Type::Like),
            ("in-reply-to".to_owned(), Type::Reply),
            ("bookmark-of".to_owned(), Type::Bookmark),
            ("repost-of".to_owned(), Type::Repost),
            ("quote-of".to_owned(), Type::Quote),
            ("gameplay-of".to_owned(), Type::GamePlay),
            ("follow-of".to_owned(), Type::Follow),
            ("jam-of".to_owned(), Type::Jam),
            ("listen-of".to_owned(), Type::Listen),
            ("rsvp".to_owned(), Type::RSVP),
            ("photo".to_owned(), Type::Photo),
            ("video".to_owned(), Type::Video),
            ("audio".to_owned(), Type::Audio),
            ("checkin".to_owned(), Type::CheckIn),
            ("read-of".to_owned(), Type::Read),
            ("media".to_owned(), Type::Media),
        ]
        .iter()
        .cloned(),
    )
}

/// Aims to resolve the known post type of the provided [microformats::types::Item].
///
/// ```
/// # use microformats::algorithms::ptd::*;
/// # use serde_json::*;
///
/// resolve_from_object(json!({
///     "properties": {
///         "like-of": ["https://indieweb.org/like"]
///     }
/// }))
/// ```
pub fn resolve_from_object<V>(into_item_mf2: V) -> Option<Type>
where
    V: TryInto<microformats::types::Item>,
{
    if let Ok(item_mf2) = into_item_mf2.try_into() {
        let props = item_mf2.properties.borrow();
        let mut types: Vec<Type> = vec![];
        let has_content = props.contains_key(&"content".to_owned());
        let has_name = props.contains_key(&"name".to_owned());

        properties_from_type().iter().for_each(|(key, val)| {
            if props.contains_key(&key.to_owned()) {
                types.push(val.clone());
            }
        });

        if has_name && has_content {
            types.push(Type::Article)
        } else if !has_name && has_content {
            types.push(Type::Note)
        }

        combinatory_type(types.clone())
            .or(types.first().cloned())
            .or(Some(Type::Note))
    } else {
        None
    }
}

fn combinatory_type(types: Vec<Type>) -> Option<Type> {
    [
        (Type::RSVP, vec![Type::Reply, Type::RSVP]),
        (Type::Photo, vec![Type::Photo, Type::Note]),
        (Type::Video, vec![Type::Video, Type::Photo, Type::Note]),
        (
            Type::Media,
            vec![Type::Audio, Type::Video, Type::Photo, Type::Note],
        ),
    ]
    .iter()
    .find_map(|(combined_type, expected_types)| {
        if expected_types
            .iter()
            .all(|post_type| types.contains(post_type))
        {
            Some(combined_type.to_owned())
        } else {
            None
        }
    })
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn post_type_from_json() {
        assert_eq!(
            serde_json::from_str::<PostType>(
                r#"
                {
                    "name": "Note",
                    "type": "note"
                }
                "#
            )
            .ok(),
            Some(PostType::Expanded {
                name: "Note".to_string(),
                kind: Type::Note,
                h: default_class(),
                properties: Vec::default(),
                required_properties: Vec::default()
            })
        );

        assert_eq!(
            serde_json::from_str::<Vec<PostType>>(
                r#"
                [{
                    "name": "Note",
                    "type": "note"
                }, "like"]
                "#
            )
            .ok(),
            Some(vec![
                PostType::Expanded {
                    name: "Note".to_string(),
                    kind: Type::Note,
                    h: default_class(),
                    properties: Vec::default(),
                    required_properties: Vec::default()
                },
                PostType::Simple(Type::Like)
            ])
        );
    }

    #[test]
    fn post_type_partial_eq() {
        assert_eq!(
            PostType::Simple(Type::Like),
            PostType::Expanded {
                kind: Type::Like,
                name: "Like".to_string(),
                h: default_class(),
                properties: Vec::default(),
                required_properties: Vec::default()
            }
        );
    }

    // FIXME: Refactor to read HTML and JSON information from disk and generate tests from that.
    // This will let us use https://github.com/microformats/tests as a shared source of testing.
    #[test]
    fn type_insure_string_form() {
        assert_eq!(Type::Note.to_string(), "note".to_owned());
        assert_eq!(serde_json::from_str(r#""note""#).ok(), Some(Type::Note));
        assert_eq!(
            serde_json::from_str(r#""checkin""#).ok(),
            Some(Type::CheckIn)
        );
        assert_eq!(
            serde_json::from_str(r#""note""#)
                .ok()
                .map(|k: Type| k.to_string()),
            Some("note".to_owned())
        );
    }
}
