use super::{Request, Response, Status};
use crate::{client, Error};
use anyhow::{Context, Result};

/// Resolves the Webmention endpoints for the provided URL.
pub fn endpoints_for(url: &str) -> Result<Vec<String>, Error> {
    let urls = crate::link_rel::for_url(url, "webmention")?;
    if urls.is_empty() {
        Err(Error::NoEndpointsFound {
            url: url.to_owned(),
            rel: "webmention".to_owned(),
        })
    } else {
        Ok(urls)
    }
}

/// Sends a Webmention.
pub fn send(req: &Request) -> Result<Response, Error> {
    let req = client::Request {
        headers: client::Headers::from_iter(vec![
            ("accept", "text/html,application/json"),
            ("content-type", "application/x-www-form-urlencoded"),
        ]),
        body: serde_qs::to_string(&req)
            .map_err(|e| Error::Other(e.into()))?
            .into_bytes(),
        url: url::Url::parse(&req.endpoint)
            .context("Failed to parse the URL to be sent in a Webmention sending request.")
            .map_err(crate::Error::Other)?,
    };
    let resp = req.send("POST")?;

    match resp.status_code {
        204 | 200 | 201 => {
            let location = client::find_header_by_name(&resp.headers, "location")
                .first()
                .cloned();
            let body = Some(resp.body_as_string()?);
            let status = match resp.status_code {
                _ => Status::Accepted,
            };

            Ok(super::Response {
                endpoint: req.url,
                body,
                location,
                status,
            })
        }
        code => Err(Error::WebmentionUnsupportedStatusCode {
            status_code: code,
            url: req.url.to_string(),
        }),
    }
}

/// Dispatchs a Webmention.
pub fn dispatch(
    source_url: &str,
    target_url: &str,
    additional_options: Option<Request>,
) -> Vec<Result<Response, Error>> {
    // FIXME: Use `try_flatten_stream` here in the future.
    endpoints_for(target_url)
        .unwrap_or_default()
        .into_iter()
        .map(|endpoint| {
            let mut req = additional_options.clone().unwrap_or_default();
            req.endpoint = endpoint;
            req.source = source_url.to_owned();
            req.target = target_url.to_owned();

            send(&req)
        })
        .collect()
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::test;
    use mockito::{mock, Matcher};
    use url::Url;

    #[test]
    fn send_successful_sync() {
        let cfg = test::init();
        let endpoint_mock = mock("POST", "/endpoint")
            .match_header("content-type", "application/x-www-form-urlencoded")
            .with_header("content-type", "text/plain")
            .with_status(200)
            .with_body("")
            .create();
        let target = format!("{}/web-page", &mockito::server_url());
        let request = super::Request {
            endpoint: format!("{}/endpoint", mockito::server_url()),
            source: format!("{}/source", mockito::server_url()),
            target,
            vouch: None,
        };

        let result = super::send(&request);
        assert_eq!(
            result.ok(),
            Some(super::Response {
                body: None,
                location: None,
                status: Status::Sent,
                endpoint: request.endpoint.parse().unwrap()
            })
        );
        endpoint_mock.assert();
    }

    #[test]
    fn send_captures_async_response() {
        let cfg = test::init();
        let endpoint_mock = mock("POST", "/endpoint-async-response")
            .match_header("content-type", "application/x-www-form-urlencoded")
            .with_header("content-type", "text/plain")
            .with_status(200)
            .with_body("")
            .create();
        let target = format!("{}/web-page", &mockito::server_url());
        let request = super::Request {
            endpoint: format!("{}/endpoint-async-response", mockito::server_url()),
            source: format!("{}/source", mockito::server_url()),
            target,
            vouch: None,
        };
        let result = super::send(&request);
        assert_eq!(
            result.ok(),
            Some(super::Response {
                body: None,
                location: None,
                endpoint: request.endpoint.parse().unwrap(),
                status: Status::Sent
            })
        );
        endpoint_mock.assert();
    }

    #[test]
    fn send_captures_resulting_url() {
        let cfg = test::init();
        let expected_location = "https://theresult.com/wow".to_owned();
        let endpoint_mock = mock("POST", "/endpoint-captures-url")
            .match_header("content-type", "application/x-www-form-urlencoded")
            .with_header("content-type", "text/plain")
            .with_header("location", &expected_location)
            .with_status(200)
            .with_body("")
            .create();
        let target = format!("{}/web-page", &mockito::server_url());
        let request = super::Request {
            endpoint: format!("{}/endpoint-captures-url", mockito::server_url()),
            source: format!("{}/source", mockito::server_url()),
            target,
            vouch: None,
        };

        let result = super::send(&request);
        assert_eq!(
            result.ok(),
            Some(super::Response {
                body: None,
                location: Some(expected_location),
                endpoint: request.endpoint.parse().unwrap(),
                status: Status::Sent
            })
        );
        endpoint_mock.assert();
    }

    #[test]
    fn send_fails_if_status_code_not_in_success_values() {
        let cfg = test::init();
        let endpoint_mock = mock("POST", "/endpoint-crashes")
            .match_header("content-type", "application/x-www-form-urlencoded")
            .with_header("content-type", "application/json")
            .with_status(500)
            .with_body(serde_json::json!({"error":"fake"}).to_string())
            .create();
        let target = format!("{}/web-page", &mockito::server_url());
        let request = super::Request {
            endpoint: format!("{}/endpoint-crashes", mockito::server_url()),
            source: format!("{}/source", mockito::server_url()),
            target,
            vouch: None,
        };
        let result = super::send(&request);
        assert!(result.is_err());
        assert_eq!(
            result.err(),
            Some(Error::WebmentionUnsupportedStatusCode {
                status_code: 500,
                url: request.endpoint.clone()
            })
        );
        endpoint_mock.assert();
    }

    #[test]
    fn dispatch_sends_webmentions_to_every_endpoint() {
        let cfg = test::init();
        let source_url = "https://jacky.wtf/";
        let max_count = 15;
        let webpage_link_html = std::ops::Range {
            start: 0,
            end: max_count,
        }
        .into_iter()
        .map(|i| format!("<link rel='webmention' href='/endpoint/{}' />", i))
        .collect::<Vec<String>>()
        .join("\n");
        let target_url = format!("{}/web-page", &mockito::server_url());
        let endpoint_mock = mock("POST", Matcher::Regex(r"/endpoint/\d".into()))
            .match_header("content-type", "application/x-www-form-urlencoded")
            .match_header("user-agent", Matcher::Any)
            .match_header("accept", Matcher::Any)
            .match_body(Matcher::UrlEncoded("source".into(), source_url.into()))
            .with_header("content-type", "application/json")
            .with_status(201)
            .expect(max_count)
            .create();

        let webpage_mock = mock("GET", "/web-page")
            .with_header("content-type", "text/html")
            .with_status(200)
            .with_body(format!(
                r#"
                <html>
                    <head>
                    {}
                    </head>
                </html>
        "#,
                webpage_link_html
            ))
            .expect(1)
            .create();

        let request = super::Request::default();
        let endpoint: Url = request.endpoint.clone().parse().unwrap();
        let result = super::dispatch(&source_url, &target_url, Some(request));

        webpage_mock.assert();
        endpoint_mock.assert();
        assert_eq!(
            result,
            std::iter::repeat_with(|| Ok(Response {
                status: Status::Accepted,
                endpoint: endpoint.clone(),
                body: None,
                location: None
            }))
            .take(max_count)
            .collect::<Vec<Result<Response, crate::Error>>>()
        );
    }
}
