use serde_json::json;
use std::fmt::{self, Formatter};
use std::str::FromStr;
use log::*;
use std::time::Duration;

use anyhow::{anyhow, Context};
use oauth2::basic::BasicTokenIntrospectionResponse;
use oauth2::revocation::StandardRevocableToken;
use oauth2::{basic::BasicRevocationErrorResponse, RedirectUrl};
use oauth2::{
    basic::{BasicErrorResponse, BasicTokenType},
    Scope,
};
use oauth2::{reqwest::http_client, TokenResponse};
pub use oauth2::{AccessToken, RefreshToken};
use oauth2::{
    AuthUrl, AuthorizationCode, ClientId, CsrfToken, PkceCodeChallenge, PkceCodeVerifier, TokenUrl,
};

use crate::client::{Headers, Request, ResponseValue};

type Client = oauth2::Client<
    BasicErrorResponse,
    Token,
    BasicTokenType,
    BasicTokenIntrospectionResponse,
    StandardRevocableToken,
    BasicRevocationErrorResponse,
>;

// This module holds abstractions for interacting with IndieAuth servers,
// effectively making this a 'client' module.

#[derive(Clone, Debug, Default, PartialEq)]
pub struct Scopes(Vec<Scope>);

impl ToString for Scopes {
    fn to_string(&self) -> String {
        self.0
            .iter()
            .map(|s| s.to_string())
            .collect::<Vec<String>>()
            .join(" ")
    }
}

impl FromStr for Scopes {
    type Err = std::convert::Infallible;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        Ok(Self(
            s.split(" ")
                .map(|st| Scope::new(st.to_string()))
                .collect::<Vec<Scope>>(),
        ))
    }
}

impl Scopes {
    pub fn is_empty(&self) -> bool {
        self.0.is_empty()
    }

    pub fn has(&self, scope: &str) -> bool {
        self.0.iter().any(|s| s.to_string().starts_with(scope))
    }

    pub fn as_vec(&self) -> &Vec<Scope> {
        &self.0
    }
}

impl serde::Serialize for Scopes {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        serializer.serialize_str(&self.to_string())
    }
}

struct ScopesVisitor;

impl<'de> serde::de::Visitor<'de> for ScopesVisitor {
    type Value = Scopes;

    fn expecting(&self, formatter: &mut Formatter) -> fmt::Result {
        formatter.write_str("a list of strings or a string of space-separated values")
    }

    fn visit_str<E>(self, s: &str) -> Result<Self::Value, E>
    where
        E: serde::de::Error,
    {
        Ok(Scopes(
            s.split(" ")
                .map(|scope_str| Scope::new(scope_str.to_string()))
                .collect::<Vec<_>>(),
        ))
    }
}

impl<'de> serde::Deserialize<'de> for Scopes {
    fn deserialize<D>(d: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        d.deserialize_str(ScopesVisitor)
    }
}

#[derive(serde::Serialize, serde::Deserialize, Debug, Clone)]
pub struct RedeemPayload {
    pub code: String,
    pub client_id: url::Url,
    pub redirect_uri: url::Url,
    pub code_verifier: Option<String>,
}

#[derive(serde::Deserialize, serde::Serialize, Debug, Clone)]
#[serde(tag = "grant_type", rename_all = "snake_case")]
pub enum GrantPayload {
    AuthorizationCode(RedeemPayload),
}

#[derive(serde::Serialize, serde::Deserialize, Debug, Clone)]
pub struct Profile {
    pub name: Option<String>,
    pub url: Option<url::Url>,
    pub photo: Option<url::Url>,
    pub email: Option<String>,
}

#[derive(serde::Serialize, serde::Deserialize, Debug, Clone)]
pub struct CodeResponse {
    me: url::Url,
    profile: Option<Profile>,
}

// #[derive(Default, Debug)]
// pub struct DiscoveryResult {
//     authorization_endpoints: Vec<url::Url>,
//     token_endpoints: Vec<url::Url>,
//     ticket_endpoints: Vec<url::Url>
// }

pub struct AuthenticationRequest {
    client: Box<Client>,
    me: String,
    scopes: Vec<String>,
    verifier: PkceCodeVerifier,
    challenge: PkceCodeChallenge,
    redirect_uri: Option<String>,
    client_id: String,
}

impl AuthenticationRequest {
    pub fn add_scope(&mut self, scope: &str) {
        self.scopes.push(scope.to_owned());
    }

    pub fn set_querying_identity(&mut self, me: &str) {
        self.me = me.to_string();
    }

    pub fn set_pkce_challenge(&mut self, verifier: &str) {
        self.verifier = PkceCodeVerifier::new(verifier.to_string());
    }

    pub fn get_pkce_verifier(&self) -> String {
        self.verifier.secret().clone()
    }

    /// Creates a new request to obtain authentication information from a IndieAuth server.
    pub fn new(
        me: &str,
        client_id: &str,
        authorization_endpoint: &str,
        redirect_uri: Option<String>,
        scopes: Option<Vec<String>>,
    ) -> anyhow::Result<Self> {
        let client = Box::new(Client::new(
            ClientId::new(client_id.to_string()),
            None,
            AuthUrl::new(authorization_endpoint.to_string())
                .map_err(|e| anyhow!("Failed to parse the authorization endpoint URL: {:#?}", e))?,
            None,
        ));
        let (challenge, verifier) = PkceCodeChallenge::new_random_sha256_len(48);

        Ok(Self {
            client,
            client_id: client_id.to_string(),
            me: me.to_string(),
            scopes: scopes.unwrap_or_default(),
            verifier,
            challenge,
            redirect_uri,
        })
    }

    pub fn construct_url(&self) -> anyhow::Result<(String, String)> {
        let redirect_uri = self
            .redirect_uri
            .clone()
            .or(Some(self.client_id.clone()))
            .ok_or(anyhow::format_err!(
                "No redirect URI or client ID was provied."
            ))?;
        let redirect_url = RedirectUrl::from_url(url::Url::from_str(&redirect_uri)?);
        let request = self
            .client
            .authorize_url(|| CsrfToken::new_random_len(48))
            .set_pkce_challenge(self.challenge.clone())
            .add_extra_param("me", self.me.clone())
            .add_scope(Scope::new(self.scopes.join(" ")))
            .set_redirect_uri(std::borrow::Cow::Borrowed(&redirect_url));
        let (url, state) = request.url();

        Ok((url.to_string(), state.secret().clone()))
    }
}

pub struct CodeVerificationRequest {
    pub code: String,
    pub pkce: Option<PkceCodeVerifier>,
    pub client_id: String,
    pub redirect_uri: String,
    pub authorization_endpoint: String,
    pub token_endpoint: String,
}

#[derive(Debug, Clone, serde::Serialize, serde::Deserialize)]
pub struct Token {
    pub me: url::Url,
    pub access_token: AccessToken,

    #[serde(rename = "scope")]
    pub scopes: Scopes,

    pub expires_in: Option<u64>,
    pub refresh_token: Option<RefreshToken>,
    pub profile: Option<Profile>,
}

impl TokenResponse<BasicTokenType> for Token {
    fn scopes(&self) -> Option<&Vec<Scope>> {
        Some(self.scopes.as_vec())
    }

    fn access_token(&self) -> &AccessToken {
        &self.access_token
    }

    fn refresh_token(&self) -> Option<&RefreshToken> {
        self.refresh_token.as_ref()
    }

    fn expires_in(&self) -> Option<Duration> {
        self.expires_in.map(|s| Duration::from_secs(s))
    }

    fn token_type(&self) -> &BasicTokenType {
        &BasicTokenType::Bearer
    }
}

impl CodeVerificationRequest {
    pub fn redeem(&self) -> Result<Token, crate::Error> {
        let client = Client::new(
            ClientId::new(self.client_id.clone()),
            None,
            AuthUrl::new(self.authorization_endpoint.clone())
                .context("Failed to parse the authorization URL")
                .map_err(crate::Error::Other)?,
            Some(
                TokenUrl::new(self.token_endpoint.clone())
                    .context("Failed to parse the authorization URL")
                    .map_err(crate::Error::Other)?,
            ),
        );

        let mut request = client.exchange_code(AuthorizationCode::new(self.code.clone()));

        request = match self.pkce.as_ref() {
            Some(verifier) => {
                request.set_pkce_verifier(PkceCodeVerifier::new(verifier.secret().to_string()))
            }
            None => request,
        };

        // FIXME: Replace this error with a local error for a failed verification attempt.
        request
            .request(http_client)
            .with_context(|| {
                format!("Failed to obtain a response from the token endpoint {:?} about the code {:?} for a token.", self.token_endpoint, self.code)
            })
            .map_err(crate::Error::Other)
    }
}

// pub async fn discover(remote_url: &str) -> Option<DiscoveryResult>
// {
//     unimplemented!("write logic to find IndieAuth endpoint discovery.");
// }

/// Confirms that the provided token is valid with the provided endpoint.
pub fn verify(endpoint: &str, access_token: &str) -> Result<Token, crate::Error>
where
{
    let req = Request {
        url: url::Url::from_str(endpoint).unwrap(),
        headers: Headers::from_iter([
            ("accept", "application/json".to_owned().into_bytes()),
            (
                "authorization",
                format!("bearer {}", access_token).into_bytes(),
            ),
        ]),
        body: vec![],
    };

    let resp = req.send("GET")?;

    match resp.into_json::<Token>()? {
        ResponseValue::Okay(t) => Ok(t),
        ResponseValue::Failure(f) => Err(crate::Error::Other(anyhow!(f.to_string()))),
    }
}

pub fn redeem<'a, R>(endpoint: &str, redeem_payload: RedeemPayload) -> Result<R, crate::Error>
where
    R: serde::de::DeserializeOwned + std::fmt::Debug,
{
    let req = Request {
        url: url::Url::from_str(endpoint).unwrap(),
        headers: Headers::from_iter(vec![
            ("accept", "application/json"),
            ("content-type", "application/json"),
        ]),
        body: json!({
            "grant_type": "authorization_code",
            "code": redeem_payload.code,
            "client_id": redeem_payload.client_id,
            "redirect_uri": redeem_payload.redirect_uri,
            "code_verifier": redeem_payload.code_verifier
        })
        .to_string()
        .into_bytes(),
    };

    let resp = req.send("POST")?;
    match resp.into_json::<R>()? {
        ResponseValue::Okay(t) => {
            trace!("The code was redeemed for a token (or token-compatible value)");
            Ok(t)
        }
        ResponseValue::Failure(f) => {
            warn!("The code could not be redeemed.");
            Err(crate::Error::Other(anyhow!(f.to_string())))
        }
    }
}

#[cfg(test)]
#[allow(non_snake_case)]
mod tests {
    use super::*;
    use crate::test;
    use oauth2::{PkceCodeChallenge, PkceCodeVerifier};

    #[test]
    fn AuthenticationRequest_construct_url_has_expected_params() {
        let me = "https://jacky.wtf".to_string();
        let client_id = "http://sele.black.af".to_string();
        let endpoint = "https://sele.black.af/endpoints/wow".to_string();
        let redirect_uri =
            "https://special-domain-name.endpoints.sele.black.af/testing_page".to_string();
        let result = AuthenticationRequest::new(
            &me,
            &client_id,
            &endpoint,
            Some(redirect_uri.clone()),
            None,
        );

        assert!(matches!(result, Ok(_)));
        let mut req = result.unwrap();

        req.add_scope("read");
        req.add_scope("create");

        let url_and_state = req.construct_url();

        assert!(url_and_state.is_ok());
        let (url, state) = url_and_state.unwrap();
        let challenge = PkceCodeChallenge::from_code_verifier_sha256(&PkceCodeVerifier::new(
            req.get_pkce_verifier(),
        ));
        let challenge_str = challenge.as_str();

        use std::collections::HashMap;

        let params: HashMap<String, String> =
            dbg!(serde_qs::from_str(url.parse::<url::Url>().unwrap().query().unwrap()).unwrap());
        assert!(params.contains_key(&"redirect_uri".to_string()));
        assert_eq!(
            params.get("scope").cloned(),
            Some("read create".to_string())
        );
        assert_eq!(
            params.get("code_challenge").cloned(),
            Some(challenge_str.to_string())
        );
        assert_eq!(
            params.get("response_type").cloned(),
            Some("code".to_string())
        );
        assert_eq!(
            params.get("code_challenge_method").cloned(),
            Some("S256".to_string())
        );
        assert_eq!(params.get("state").cloned(), Some(state));
        assert_eq!(params.get("me").cloned(), Some(me));
        assert_eq!(params.get("client_id").cloned(), Some(client_id));
        assert_eq!(params.get("redirect_uri").cloned(), Some(redirect_uri));
    }

    #[test]
    fn CodeVerificationRequest_redeem_success() {
        let (_challenge, verifier) = PkceCodeChallenge::new_random_sha256();
        let code = "a-valid-code".to_owned();
        let authorization_endpoint = format!("{}/auth-endpoint", mockito::server_url());
        let token_endpoint = format!("{}/token-endpoint", mockito::server_url());

        let req = super::CodeVerificationRequest {
            code,
            pkce: Some(verifier),
            client_id: "https://sele.black.af".to_owned(),
            redirect_uri: "https://sele.black.af/bounce-back".to_owned(),
            authorization_endpoint,
            token_endpoint,
        };
        let body = serde_json::json!({
            "me": "https://jacky.wtf/",
            "token_type": "bearer",
            "scope": "read profile email create",
            "access_token": "the-access-token",
            "profile": {
                "name": "Jacky Alciné",
                "url": "https://jacky.wtf/",
                "photo": "https://jacky.wtf/photo.jpeg",
                "email": "yo+codespam@jacky.wtf"
            }
        })
        .to_string();

        let endpoint_mock = mockito::mock("POST", "/token-endpoint")
            .with_status(200)
            .with_body(body)
            .expect_at_least(1)
            .create();

        let result = dbg!(req.redeem());
        endpoint_mock.assert();
        assert!(matches!(result, Ok(_)));

        let token = result.unwrap();
        assert!(token.profile.is_some());
        let profile = token.profile.unwrap();
        assert_eq!(profile.url, "https://jacky.wtf/".parse().ok());
    }

    #[test]
    fn verify_confirms_valid_token() {
        let cfg = test::init();
        let endpoint = format!("{}/endpoint", mockito::server_url());
        let access_token = "access-token";
        let body = serde_json::json!({
            "access_token": "a-token",
            "me": "https://jacky.wtf/",
            "scope": "read profile email"
        })
        .to_string();
        let endpoint_mock = mockito::mock("GET", "/endpoint")
            .with_body(&body)
            .expect_at_most(1)
            .create();

        endpoint_mock.assert();
        let result = super::verify(&endpoint, access_token);
        assert!(result.is_ok());

        let token = result.unwrap();
        assert_eq!(token.me, "https://jacky.wtf/".parse().unwrap());
        assert_eq!(token.access_token.secret().clone(), "a-token".to_owned());
        assert_eq!(token.scopes.to_string(), "read profile email".to_owned());
    }

    #[test]
    fn redeem_resolves_responses() {
        let cfg = test::init();
        let endpoint = format!("{}/endpoint", mockito::server_url());
        let body = serde_json::json!({
            "me": "https://jacky.wtf/",
            "token_type": "bearer",
            "scope": "read profile email create",
            "access_token": "the-access-token",
            "profile": {
                "name": "Jacky Alciné",
                "url": "https://jacky.wtf/",
                "photo": "https://jacky.wtf/photo.jpeg",
                "email": "yo+codespam@jacky.wtf"
            }

        })
        .to_string();

        let endpoint_mock = mockito::mock("POST", "/endpoint")
            .with_body(&body)
            .expect_at_most(2)
            .create();
        let payload = RedeemPayload {
            code: "a-code".to_owned(),
            code_verifier: Some("a-code-verifier".to_owned()),
            client_id: "https://indieweb.org".parse().unwrap(),
            redirect_uri: "https://indieweb.org/back".parse().unwrap(),
        };

        let optional_token: Result<super::Token, crate::Error> =
            super::redeem(&endpoint, payload.clone());
        let optional_code: Result<super::CodeResponse, crate::Error> =
            super::redeem(&endpoint, payload.clone());
        endpoint_mock.assert();

        assert!(optional_code.is_ok());
        assert!(optional_token.is_ok());
    }

    #[test]
    fn Scopes_assert_presence() -> Result<(), anyhow::Error> {
        assert!(!Scopes::from_str("read profile")?.has("update"));
        assert!(Scopes::from_str("read profile")?.has("read"));
        Ok(())
    }
}
