use anyhow::Context;
use log::*;
use std::{fmt::Debug, io::Read};
use ureq::OrAnyStatus;

/// Represents headers.
pub type Header = (String, Vec<u8>);

#[derive(Debug, Clone)]
pub struct Headers(Vec<Header>);

impl<'a> FromIterator<(&'a str, Vec<u8>)> for Headers {
    fn from_iter<L: IntoIterator<Item = (&'a str, Vec<u8>)>>(items: L) -> Self {
        Self(
            items
                .into_iter()
                .map(|(name, value)| (name.to_string(), value))
                .collect::<Vec<_>>(),
        )
    }
}

impl<'a> FromIterator<(&'a str, &'a str)> for Headers {
    fn from_iter<L: IntoIterator<Item = (&'a str, &'a str)>>(items: L) -> Self {
        Self(
            items
                .into_iter()
                .map(|(name, value)| (name.to_string(), value.into()))
                .collect(),
        )
    }
}

impl FromIterator<Header> for Headers {
    fn from_iter<L: IntoIterator<Item = Header>>(items: L) -> Self {
        Self(items.into_iter().collect::<Vec<Header>>())
    }
}

/// A thin representation of a network request.
/// This is here to allow people to bring their own HTTP clients.
#[derive(Debug)]
pub struct Request {
    pub url: url::Url,
    pub headers: Headers,
    pub body: Vec<u8>,
}

/// Represents a returned response from a server.
pub struct Response {
    pub status_code: u16,
    pub headers: Headers,
    pub body: Vec<u8>,
}

pub fn find_header_by_name<'a>(headers: &Headers, expected_name: &str) -> Vec<String> {
    headers
        .0
        .iter()
        .filter_map(|(name, value)| {
            if name == expected_name {
                String::from_utf8(value.clone()).ok()
            } else {
                None
            }
        })
        .collect::<Vec<_>>()
}

impl Request {
    pub fn send(&self, method: &str) -> Result<Response, crate::Error> {
        let mut req = ureq::Agent::new().request(method, self.url.as_str());
        for (name, value) in &self.headers.0 {
            req = req.set(
                &name,
                &String::from_utf8(value.clone())
                    .context("Failed to convert header value into a string")
                    .map_err(crate::Error::Other)?,
            );
        }

        req.send_bytes(&self.body)
            .or_any_status()
            .map_err(crate::Error::Client)
            .map(|resp| Response {
                status_code: resp.status(),
                headers: resp
                    .headers_names()
                    .into_iter()
                    .map(|header| {
                        let header_value = if let Some(v) = resp.header(&header) {
                            v.as_bytes().into_iter().cloned().collect::<Vec<_>>()
                        } else {
                            Vec::default()
                        };
                        (header, header_value)
                    })
                    .collect::<Headers>(),
                body: resp
                    .into_reader()
                    .bytes()
                    .filter_map(Result::ok)
                    .collect::<Vec<u8>>(),
            })
    }
}

#[derive(Debug)]
pub enum ResponseValue<ExpectedResponse: Debug> {
    Okay(ExpectedResponse),
    Failure(serde_json::Value),
}

impl Response {
    pub fn into_json<Body>(self) -> Result<ResponseValue<Body>, crate::Error>
    where
        Body: serde::de::DeserializeOwned + Debug,
    {
        trace!(
            "Serializing {:#?} as JSON for a {}..",
            String::from_utf8(self.body.clone()),
            self.status_code
        );

        serde_json::from_slice::<Body>(&self.body)
            .map_err(crate::Error::JSON)
            .and_then(|json_body| {
                if self.status_code >= 200 && self.status_code <= 399 {
                    Ok(ResponseValue::Okay(json_body))
                } else {
                    serde_json::from_slice::<serde_json::Value>(&self.body)
                        .map_err(crate::Error::JSON)
                        .map(ResponseValue::Failure)
                }
            })
    }

    pub fn body_as_string(&self) -> Result<String, crate::Error> {
        String::from_utf8(self.body.clone())
            .context("Failed to convert body into a UTF-8 string.")
            .map_err(crate::Error::Other)
    }

    // FIXME: Implement a method for converting non-2xx or non-3xx into an error.
}
