use crate::mf2::types::Item;
use std::ops;

use serde::{Deserialize, Serialize};

pub mod authorship;
pub mod ptd;

#[derive(Clone, Debug, Serialize, Deserialize, Default, PartialEq)]
pub struct Properties(pub serde_json::Map<String, serde_json::Value>);

impl ops::Deref for Properties {
    type Target = serde_json::Map<String, serde_json::Value>;

    fn deref(&self) -> &Self::Target {
        &self.0
    }
}

impl ops::DerefMut for Properties {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.0
    }
}

impl TryFrom<serde_json::Value> for Properties {
    type Error = serde_json::error::Category;

    fn try_from(value: serde_json::Value) -> Result<Self, Self::Error> {
        if let serde_json::Value::Object(props) = value {
            Ok(Self(props))
        } else {
            Err(serde_json::error::Category::Data)
        }
    }
}

impl Into<serde_json::Value> for Properties {
    fn into(self) -> serde_json::Value {
        serde_json::Value::Object(self.0)
    }
}

impl Properties {
    /// Creates a copy of this set of properties that's been normalized.
    ///
    /// # Examples
    /// ```
    /// # use indieweb::algorithms::Properties;
    /// # use serde_json::json;
    ///
    /// let j = json!({
    ///     "properties": {
    ///         "actual": "value"
    ///     }
    /// });
    ///
    /// let og_props = Properties::try_from(j).unwrap_or_default();
    ///
    /// assert_ne!(
    ///     og_props,
    ///     og_props.normalize(),
    ///     "'og_props' isn't normalized (contains 'properties' key)."
    /// );
    ///
    /// let k = json!({
    ///     "actual": "value"
    /// });
    ///
    /// let cool_props = Properties::try_from(j).unwrap_or_default();
    ///
    /// assert_eq!(
    ///     cool_props,
    ///     cool_props.normalize(),
    ///     "'cool_props' is normalized (no 'properties' key)."
    /// );
    ///
    /// ```
    pub fn normalize(&self) -> Properties {
        if self.contains_key("properties") {
            Properties(
                self.get("properties")
                    .and_then(|p| p.as_object())
                    .unwrap()
                    .clone(),
            )
        } else {
            self.clone()
        }
    }
}

pub fn extract_urls(_item: Item) -> Vec<url::Url> {
    Vec::default()
}
