use crate::algorithms::ptd::{PostType, Type};
use crate::algorithms::Properties;
use crate::client::{self, ResponseValue};
use crate::indieauth::AccessToken;
use serde::{Deserialize, Serialize};
use std::str::FromStr;
use std::string::ToString;

#[derive(Debug, Clone, serde::Serialize, serde::Deserialize, PartialEq)]
#[serde(untagged, rename_all = "kebab-case")]
pub enum PostStatus {
    Published,
    Drafted,
    Other(String),
}

impl Default for PostStatus {
    fn default() -> Self {
        PostStatus::Drafted
    }
}

impl ToString for PostStatus {
    fn to_string(&self) -> String {
        match self {
            PostStatus::Published => "published".to_owned(),
            PostStatus::Drafted => "draft".to_owned(),
            PostStatus::Other(status) => format!("x-{}", status),
        }
    }
}

impl FromStr for PostStatus {
    type Err = std::convert::Infallible;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        if s.is_empty() {
            return Ok(Self::Drafted);
        }
        match s.to_lowercase().trim_matches('"') {
            "published" => Ok(Self::Published),
            "draft" => Ok(Self::Drafted),
            status => Ok(Self::Other(status.split_at(2).1.to_owned())),
        }
    }
}

#[derive(Copy, Clone, Debug, PartialEq)]
pub enum Order {
    Ascending,
    Descending,
}

impl Default for Order {
    fn default() -> Self {
        Self::Descending
    }
}

impl From<String> for Order {
    fn from(v: String) -> Self {
        let lv = v.to_lowercase();
        if lv == "descending" || lv == "desc" {
            Self::Descending
        } else if lv == "ascending" || lv == "asc" {
            Self::Ascending
        } else {
            Self::Descending
        }
    }
}

impl ToString for Order {
    fn to_string(&self) -> String {
        match self {
            Self::Descending => "desc".to_owned(),
            Self::Ascending => "asc".to_owned(),
        }
    }
}

impl Serialize for Order {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        self.to_string().serialize(serializer)
    }
}

impl<'de> Deserialize<'de> for Order {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        String::deserialize(deserializer).map(|vs| vs.into())
    }
}

#[derive(Copy, Clone, Debug, PartialEq)]
pub enum Visibility {
    Public,
    Unlisted,
    Private,
}

impl Default for Visibility {
    fn default() -> Visibility {
        Visibility::Public
    }
}

impl From<String> for Visibility {
    fn from(v: String) -> Self {
        let lv = v.to_lowercase();
        if lv == "public" {
            Visibility::Public
        } else if lv == "private" {
            Visibility::Private
        } else if lv == "unlisted" {
            Visibility::Unlisted
        } else {
            Visibility::Public
        }
    }
}

impl ToString for Visibility {
    fn to_string(&self) -> String {
        match self {
            Visibility::Public => "public".into(),
            Visibility::Unlisted => "unlisted".into(),
            Visibility::Private => "private".into(),
        }
    }
}

impl Serialize for Visibility {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        self.to_string().serialize(serializer)
    }
}

impl<'de> Deserialize<'de> for Visibility {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        String::deserialize(deserializer).map(|vs| vs.into())
    }
}

/// Represents the known high-level content containers in Microformats.
#[derive(Clone, Debug, PartialEq)]
pub enum Kind {
    /// https://microformats.org/wiki/h-entry
    Entry,

    /// https://microformats.org/wiki/h-event
    Event,

    /// https://microformats.org/wiki/h-card
    Card,

    /// https://microformats.org/wiki/h-review
    Review,

    /// https://microformats.org/wiki/h-resume
    Resume,

    /// https://microformats.org/wiki/h-app
    /// https://indieweb.org/h-app
    Application,

    Other(String),
}

impl ToString for Kind {
    /// Returns this kind as its known Microformat2 representation.
    fn to_string(&self) -> String {
        match self {
            Kind::Entry => "h-entry".to_string(),
            Kind::Event => "h-event".to_string(),
            Kind::Card => "h-card".to_string(),
            Kind::Review => "h-review".to_string(),
            Kind::Resume => "h-resume".to_string(),
            Kind::Application => "h-app".to_string(),
            Kind::Other(other) => format!("h-x-{}", other),
        }
    }
}

impl FromStr for Kind {
    type Err = std::convert::Infallible;
    /// Parses a string value that could a potential Microformats2 kind representation into its
    /// kind. Even if it doesn't recognize the type, it'll take the permissive approach by
    /// Microformats2 and convert it into an entry.
    fn from_str(s: &str) -> Result<Self, Self::Err> {
        use Kind::*;
        Ok(match s {
            "event" => Event,
            "h-event" => Event,
            "h-x-app" => Application,
            "app" => Application,
            "h-app" => Application,
            "card" => Card,
            "h-card" => Card,
            "resume" => Resume,
            "h-resume" => Resume,
            "h-entry" => Entry,
            "entry" => Entry,
            other => Other(other.to_string()),
        })
    }
}

impl<'de> Deserialize<'de> for Kind {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        let s = dbg!(String::deserialize(deserializer)?);
        Ok(Kind::from_str(&s).unwrap_or_default())
    }
}

impl Serialize for Kind {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        self.to_string().serialize(serializer)
    }
}

impl Default for Kind {
    fn default() -> Kind {
        Self::Entry
    }
}

#[derive(Clone, Debug, PartialEq, Serialize, Deserialize)]
#[serde(untagged, rename_all = "kebab-case")]
pub enum Channel {
    Expanded {
        uid: String,
        name: String,

        #[serde(flatten, default)]
        properties: Properties,
    },
    Simple(String),
}

impl Default for Channel {
    fn default() -> Self {
        Self::Simple("default".to_owned())
    }
}

impl Channel {
    pub fn name(&self) -> String {
        match self {
            Self::Simple(name) => name.to_owned(),
            Self::Expanded { name, .. } => name.to_owned(),
        }
    }

    pub fn uid(&self) -> String {
        match self {
            Self::Simple(name) => name.to_owned(),
            Self::Expanded { uid, .. } => uid.to_owned(),
        }
    }
}

impl FromStr for Channel {
    type Err = std::convert::Infallible;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        Ok(Self::Simple(s.to_string()))
    }
}

/// Represents the normalized form of pagination in Micropub queries.
#[derive(Deserialize, Serialize, Clone, Debug, PartialEq, Default)]
#[serde(rename_all = "kebab-case")]
pub struct Pagination {
    #[serde(skip_serializing_if = "Option::is_none")]
    pub limit: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    pub offset: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    pub filter: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    pub before: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    pub after: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    pub order: Option<Order>,
}

#[derive(Serialize, Deserialize, PartialEq, Debug, Clone)]
#[serde(rename_all = "kebab-case", deny_unknown_fields)]
pub struct PaginatedResponse {
    /// A list of JSON-serialized objects.
    pub items: Vec<serde_json::Value>,

    /// Defines the current paging options used.
    #[serde(default)]
    pub paging: Pagination,

    /// An optional representation of a cursor pointing to the most recent items for this query.
    #[serde(alias = "_latest")]
    pub latest: Option<String>,

    /// An optional representation of a cursor pointing to the earliest items for this query.
    #[serde(alias = "_earliest")]
    pub earliest: Option<String>,
}

#[derive(Serialize, Deserialize, PartialEq, Debug, Clone)]
#[serde(rename_all = "kebab-case", deny_unknown_fields)]
pub struct ConfigurationResponse {
    #[serde(default)]
    pub q: Vec<String>,

    #[serde(default)]
    pub channels: Vec<Channel>,

    #[serde(default)]
    pub media_endpoint: Option<url::Url>,

    #[serde(default)]
    pub post_types: Vec<PostType>,

    #[serde(default)]
    pub syndicate_to: Vec<String>,
}

#[derive(Serialize, Deserialize, PartialEq, Debug, Clone)]
#[serde(rename_all = "kebab-case", deny_unknown_fields)]
pub struct SourceResponse {
    /// If provided/filled, this is the hinted post type.
    #[serde(default)]
    pub post_type: Vec<Type>,

    /// This is the kind of container object this Microformats Item is.
    #[serde(rename = "type")]
    pub kind: Vec<Kind>,

    pub properties: Properties,
}

#[derive(PartialEq, Debug, Clone, Serialize, Deserialize)]
#[serde(rename_all = "kebab-case", deny_unknown_fields)]
pub struct ChannelResponse {
    pub channels: Vec<Channel>,
}

/// Represents the response for a query.
// FIXME: Add the representation of syndication targets.
#[derive(Serialize, Deserialize, PartialEq, Debug, Clone)]
#[serde(untagged, rename_all = "kebab-case", deny_unknown_fields)]
pub enum QueryResponse {
    Source(SourceResponse),
    Channel(ChannelResponse),
    Configuration(ConfigurationResponse),
    Paginated(PaginatedResponse),
}

#[derive(Serialize, Deserialize)]
#[serde(rename_all = "kebab-case", tag = "q")]
pub enum Query {
    /// Pulls the configuration of the Micropub server.
    #[serde(rename = "config")]
    Configuration,

    /// Represents a call to `?q=source` for post information
    #[serde(rename_all = "kebab-case")]
    Source {
        #[serde(skip_serializing_if = "Option::is_none")]
        url: Option<url::Url>,

        #[serde(skip_serializing_if = "Option::is_none")]
        post_type: Option<Type>,

        #[serde(skip_serializing_if = "Option::is_none")]
        channel: Option<Channel>,

        #[serde(flatten)]
        pagination: Pagination,
    },

    /// Represents a call to `?q=channel` to fetch all channels.
    /// https://github.com/indieweb/micropub-extensions/issues/40
    #[serde(rename_all = "kebab-case")]
    Channel {
        #[serde(flatten)]
        pagination: Pagination,
    },

    #[serde(rename_all = "kebab-case")]
    SyndicateTo {
        #[serde(default, skip_serializing_if = "Option::is_none")]
        post_type: Option<Type>,

        #[serde(flatten)]
        pagination: Pagination,
    },
}

impl Query {
    /// Sends this request over to the requested Micropub server.
    pub fn send(
        &self,
        endpoint: &url::Url,
        access_token: &AccessToken,
    ) -> Result<ResponseValue<QueryResponse>, crate::Error> {
        let mut url = endpoint.clone();
        let params = serde_json::to_value(self.clone())
            .map_err(crate::Error::JSON)?
            .as_object()
            .cloned()
            .unwrap_or_default();
        url.query_pairs_mut()
            .extend_pairs(params.into_iter().map(|(key, value)| {
                (
                    key,
                    value.as_str().map(|s| s.to_string()).unwrap_or_default(),
                )
            }));
        let req = client::Request {
            headers: client::Headers::from_iter(vec![
                ("accept", "application/json"),
                ("content-type", "application/json"),
                (
                    "authorization",
                    &format!("bearer {}", access_token.secret()),
                ),
            ]),
            body: vec![],
            url,
        };

        req.send("get").and_then(client::Response::into_json)
    }
}

#[cfg(test)]
#[allow(non_snake_case)]
mod tests {
    use serde_json::json;

    use super::*;

    #[test]
    fn query_send_formats_query_result_correctly_for_a_single_url() {}

    #[test]
    fn query_send_formats_query_result_correctly_for_post_type_list() {}

    #[test]
    fn query_send_looks_up_configuration() {}

    #[test]
    fn query_send_looks_up_channels() {}

    #[test]
    fn pagination_from_string() {}

    #[test]
    fn query_response_for_pagination() {}

    #[test]
    fn query_response_for_channels() {
        assert_eq!(
            Some(QueryResponse::Channel(ChannelResponse {
                channels: vec![Channel::Expanded {
                    uid: "magic".to_string(),
                    name: "Magic".to_string(),
                    properties: Properties::try_from(json!({"grr": "bark"})).unwrap()
                }],
            })),
            serde_json::from_value(json!({
                "channels": [{
                    "uid": "magic",
                    "name": "Magic",
                    "grr": "bark"
                }]
            }))
            .ok()
        )
    }

    #[test]
    fn query_response_for_configuration() {
        assert_eq!(
            Some(QueryResponse::Configuration(ConfigurationResponse {
                q: vec!["channels".to_owned()],
                channels: vec![Channel::Simple("all".to_owned())],
                media_endpoint: None,
                post_types: vec![PostType::Simple(Type::Note)],
                syndicate_to: vec![]
            })),
            serde_json::from_value(json!({
                "q": ["channels"], "post-types": ["note"], "channels": ["all"]
            }))
            .ok()
        )
    }

    #[test]
    fn query_response_for_source() {
        assert_eq!(
            Some(QueryResponse::Source(SourceResponse {
                post_type: vec![Type::Note],
                kind: vec![Kind::Entry],
                properties: Properties::default()
            })),
            serde_json::from_value(json!({
                "type": ["h-entry"],
                "properties": {}
            }))
            .ok()
        );
    }

    #[test]
    fn post_status() {
        assert_eq!(
            Some(PostStatus::Drafted),
            PostStatus::from_str("draft").ok()
        );
    }
}
