pub mod algorithms;
pub mod client;
pub mod indieauth;
pub mod link_rel;
pub mod micropub;
pub mod webmention;

#[cfg(test)]
mod test;

// FIXME: Make this into a parent collection of errors from each module.
#[derive(thiserror::Error, Debug)]
pub enum Error {
    #[error("No {rel:?} endpoints were found at {url:?}")]
    NoEndpointsFound { rel: String, url: String },

    #[error(
        "The Webmention endpoint at {url:?} returned a unexpected status code of {status_code:?})"
    )]
    WebmentionUnsupportedStatusCode { status_code: u16, url: String },

    #[error(transparent)]
    Client(#[from] ureq::Error),

    #[error(transparent)]
    JSON(#[from] serde_json::Error),

    #[error(transparent)]
    Other(#[from] anyhow::Error),
}

impl PartialEq for Error {
    fn eq(&self, other: &Self) -> bool {
        std::mem::discriminant(self) == std::mem::discriminant(other)
    }
}

fn parse_content_value(value: &serde_json::Value) -> Option<String> {
    use serde_json::Value;
    match value {
        Value::String(text) => Some("<p>".to_owned() + text + "</p>"),
        Value::Object(obj) => obj.get("html").map(|s| s.as_str().unwrap_or("").to_owned()),
        Value::Array(values) => Some(
            values
                .iter()
                .map(parse_content_value)
                .map(|v| v.unwrap())
                .collect::<Vec<String>>()
                .join(""),
        ),
        _ => None,
    }
}

// FIXME: Move this into the 'indieweb' crate.
pub fn resolve_content(properties: algorithms::Properties) -> Option<String> {
    properties.get("content").and_then(parse_content_value)
}
