use std::ops;

use serde::{Deserialize, Serialize};

pub mod authorship;
pub mod ptd;

#[derive(Clone, Debug, Serialize, Deserialize, Default, PartialEq)]
pub struct Properties(pub serde_json::Map<String, serde_json::Value>);

impl ops::Deref for Properties {
    type Target = serde_json::Map<String, serde_json::Value>;

    fn deref(&self) -> &Self::Target {
        &self.0
    }
}

impl ops::DerefMut for Properties {
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.0
    }
}

impl TryFrom<serde_json::Value> for Properties {
    type Error = serde_json::error::Category;

    fn try_from(value: serde_json::Value) -> Result<Self, Self::Error> {
        if let serde_json::Value::Object(props) = value {
            Ok(Self(props))
        } else {
            Err(serde_json::error::Category::Data)
        }
    }
}

impl Into<serde_json::Value> for Properties {
    fn into(self) -> serde_json::Value {
        serde_json::Value::Object(self.0)
    }
}

pub fn normalize_properties(properties: Properties) -> Properties {
    if properties.contains_key("properties") {
        Properties(
            properties
                .get("properties")
                .and_then(|p| p.as_object())
                .unwrap()
                .clone(),
        )
    } else {
        properties
    }
}

pub fn resolve_item_from_mf2(
    mf2_items: Vec<Properties>,
    item_url: &url::Url,
) -> Option<Properties> {
    if mf2_items.is_empty() {
        return None;
    };

    mf2_items
        .iter()
        .cloned()
        .filter(|item| item.contains_key(&"properties".to_owned()))
        .find(|mf2: &Properties| {
            let no_values: Vec<serde_json::Value> = vec![];
            let expected_url = serde_json::Value::String(item_url.to_string());
            let url_val = mf2["properties"].get("url").cloned().unwrap_or_default();
            let urls = url_val.as_array().unwrap_or(&no_values);
            urls.contains(&expected_url)
        })
}

pub fn into_properties(value: &serde_json::Value) -> Option<Properties> {
    if let serde_json::Value::Object(v) = value {
        Some(Properties(v.to_owned()))
    } else {
        None
    }
}
