use std::time::Duration;

use anyhow::anyhow;
use oauth2::basic::BasicRevocationErrorResponse;
use oauth2::basic::BasicTokenIntrospectionResponse;
use oauth2::revocation::StandardRevocableToken;
use oauth2::{
  basic::{BasicErrorResponse, BasicTokenType},
  AccessToken, RefreshToken, Scope,
};
use oauth2::{reqwest::http_client, TokenResponse};
use oauth2::{AuthUrl, AuthorizationCode, ClientId, CsrfToken, PkceCodeChallenge, PkceCodeVerifier, TokenUrl};

type Client = oauth2::Client<
  BasicErrorResponse,
  Token,
  BasicTokenType,
  BasicTokenIntrospectionResponse,
  StandardRevocableToken,
  BasicRevocationErrorResponse,
>;

// This module holds abstractions for interacting with IndieAuth servers,
// effectively making this a 'client' module.

#[derive(Default, Debug)]
pub struct DiscoveryResult {
  authorization_endpoints: Vec<String>,
  token_endpoints: Vec<String>,
}

/// Provides the basis of making a IndieAuth request.
pub struct AuthenticationRequest {
  client: Box<Client>,
  me: String,
  scopes: Vec<String>,
  verifier: PkceCodeVerifier,
}

impl AuthenticationRequest {
  pub fn add_scope(&mut self, scope: &str) {
    self.scopes.push(scope.to_owned());
  }

  pub fn set_querying_identity(&mut self, me: &str) {
    self.me = me.to_string();
  }

  pub fn set_pkce_challenge(&mut self, verifier: &str) {
    self.verifier = PkceCodeVerifier::new(verifier.to_string());
  }

  pub fn get_pkce_verifier(&self) -> String {
    self.verifier.secret().clone()
  }

  pub fn new(
    me: &str,
    client_id: &str,
    authorization_endpoint: &str,
    _redirect_url: Option<String>,
    scopes: Option<Vec<String>>,
  ) -> anyhow::Result<Self> {
    let client = Box::new(Client::new(
      ClientId::new(client_id.to_string()),
      None,
      AuthUrl::new(authorization_endpoint.to_string()).map_err(|e| anyhow!("{:#?}", e))?,
      None,
    ));
    let verifier = PkceCodeChallenge::new_random_sha256().1;

    Ok(Self {
      client,
      me: me.to_string(),
      scopes: scopes.unwrap_or_default(),
      verifier,
    })
  }

  pub fn construct_url(&self) -> anyhow::Result<(String, String)> {
    let request = self
      .client
      .authorize_url(CsrfToken::new_random)
      .set_pkce_challenge(PkceCodeChallenge::from_code_verifier_sha256(&self.verifier))
      .add_extra_param("me", self.me.clone())
      .add_scope(Scope::new(self.scopes.join(" ")));

    let (url, state) = request.url();
    Ok((url.to_string(), state.secret().clone()))
  }
}

pub struct CodeVerificationRequest {
  pub code: String,
  pub pkce: Option<PkceCodeVerifier>,
  pub client_id: String,
  pub redirect_uri: String,
  pub authorization_endpoint: String,
  pub token_endpoint: String,
}

#[derive(Debug, Clone, serde::Serialize, serde::Deserialize)]
pub struct Token {
  #[serde(rename = "scope")]
  pub scopes: Vec<Scope>,
  pub access_token: AccessToken,
  pub expires_in: Option<Duration>,
  pub refresh_token: Option<RefreshToken>,

  pub me: String,
  pub profile: Option<std::collections::HashMap<String, String>>,
}

impl TokenResponse<BasicTokenType> for Token {
  fn scopes(&self) -> Option<&Vec<Scope>> {
    if self.scopes.is_empty() {
      None
    } else {
      Some(&self.scopes)
    }
  }
  fn access_token(&self) -> &AccessToken {
    &self.access_token
  }
  fn refresh_token(&self) -> Option<&RefreshToken> {
    self.refresh_token.as_ref()
  }
  fn expires_in(&self) -> Option<Duration> {
    self.expires_in
  }
  fn token_type(&self) -> &BasicTokenType {
    &BasicTokenType::Bearer
  }
}

impl CodeVerificationRequest {
  pub fn redeem(&self) -> anyhow::Result<Token> {
    let client = Client::new(
      ClientId::new(self.client_id.clone()),
      None,
      AuthUrl::new(self.authorization_endpoint.clone())?,
      Some(TokenUrl::new(self.token_endpoint.clone())?),
    );

    let mut request = client.exchange_code(AuthorizationCode::new(self.code.clone()));

    request = match self.pkce.as_ref() {
      Some(verifier) => request.set_pkce_verifier(PkceCodeVerifier::new(verifier.secret().to_string())),
      None => request,
    };

    request.request(http_client).map_err(|e| anyhow!("{:#?}", e))
  }
}

pub async fn verify(cfg: &crate::Configuration, endpoint: &str, access_token: &str) -> Option<Token> {
  let resp = cfg
    .request(Some("Requesting information about a token".to_owned()))
    .request(reqwest::Method::POST, endpoint.to_owned())
    .header("accept", "application/json")
    .bearer_auth(access_token)
    .send()
    .await
    .ok();

  match resp
    .expect("The endpoint didn't provide a valid response.")
    .error_for_status()
  {
    Ok(resp) => resp
      .text()
      .await
      .ok()
      .map(|json_text| dbg!(serde_json::from_str(&json_text)).ok())
      .flatten(),
    Err(_err) => None,
  }
}

#[cfg(test)]
#[allow(non_snake_case)]
mod tests {
  use crate::test;
  use oauth2::{PkceCodeChallenge, PkceCodeVerifier};

  #[test]
  fn AuthenticationRequest_construct_url_has_expected_params() {
    let me = "https://jacky.wtf";
    let client_id = "http://sele.black.af";
    let endpoint = "https://sele.black.af/endpoints/wow";
    let redirect_uri = "https://special-domain-name.endpoints.sele.black.af/testing_page";
    let result = super::AuthenticationRequest::new(me, client_id, endpoint, Some(redirect_uri.to_owned()), None);

    assert!(matches!(result, Ok(_)));
    let mut req = result.unwrap();

    req.add_scope("read");
    req.add_scope("create");

    let url_and_state = req.construct_url();

    assert!(url_and_state.is_ok());
    let (url, state) = url_and_state.unwrap();
    let challenge = PkceCodeChallenge::from_code_verifier_sha256(&PkceCodeVerifier::new(req.get_pkce_verifier()));
    let challenge_str = challenge.as_str();

    // assert!(url.contains(
    //   serde_urlencoded::to_string(&[("redirect_uri", &redirect_uri)])
    //     .unwrap()
    //     .as_str()
    // ));
    assert!(url.contains(
      serde_urlencoded::to_string(&[("code_challenge", challenge_str), ("code_challenge_method", "S256"),])
        .unwrap()
        .as_str()
    ));

    assert!(url.contains(
      serde_urlencoded::to_string(&[("client_id", &client_id)])
        .unwrap()
        .as_str()
    ));
    assert!(url.contains(serde_urlencoded::to_string(&[("me", &me)]).unwrap().as_str()));
    assert!(url.contains(serde_urlencoded::to_string(&[("state", &state)]).unwrap().as_str()));
    assert!(url.contains(
      dbg!(serde_urlencoded::to_string(&[("scope", "read create")]))
        .unwrap()
        .as_str()
    ));
  }

  #[test]
  fn CodeVerificationRequest_redeem_success() {
    let (_challenge, verifier) = PkceCodeChallenge::new_random_sha256();
    let code = "a-valid-code".to_owned();
    let authorization_endpoint = format!("{}/auth-endpoint", mockito::server_url());
    let token_endpoint = format!("{}/token-endpoint", mockito::server_url());

    let req = super::CodeVerificationRequest {
      code,
      pkce: Some(verifier),
      client_id: "https://sele.black.af".to_owned(),
      redirect_uri: "https://sele.black.af/bounce-back".to_owned(),
      authorization_endpoint,
      token_endpoint,
    };
    let body = serde_json::json!({
        "me": "https://jacky.wtf/",
        "token_type": "bearer",
        "scope": ["read", "profile", "email", "create"],
        "access_token": "the-access-token",
        "profile": {
            "name": "Jacky Alciné",
            "url": "https://jacky.wtf/",
            "photo": "https://jacky.wtf/photo.jpeg",
            "email": "yo+codespam@jacky.wtf"
        }
    })
    .to_string();

    let endpoint_mock = mockito::mock("POST", "/token-endpoint")
      .with_status(200)
      .with_body(body)
      .expect_at_least(1)
      .create();

    let result = dbg!(req.redeem());
    endpoint_mock.assert();
    assert!(matches!(result, Ok(_)));

    let token = result.unwrap();
    assert_eq!(token.me, "https://jacky.wtf/".to_owned());
    assert!(matches!(token.profile, Some(_)));

    let profile = token.profile.unwrap();
    assert_eq!(profile.get("url"), Some(&"https://jacky.wtf/".to_owned()));
  }

  #[tokio::test(flavor = "multi_thread")]
  async fn verify_confirms_valid_token() {
    let cfg = test::init();
    let endpoint = format!("{}/endpoint", mockito::server_url());
    let access_token = "access-token";
    let body = serde_json::json!({
        "access_token": "a-token",
        "me": "https://jacky.wtf/",
        "scope": ["read", "profile", "email"]
    })
    .to_string();
    let endpoint_mock = mockito::mock("POST", "/endpoint")
      .with_body(&body)
      .expect_at_most(1)
      .create();

    endpoint_mock.assert();
    let result = super::verify(&cfg, &endpoint, access_token).await;
    assert!(result.is_some());

    let token = result.unwrap();
    assert_eq!(token.me, "https://jacky.wtf/".to_owned());
    assert_eq!(token.access_token.secret().clone(), "a-token".to_owned());
    assert_eq!(
      token.scopes.iter().map(|s| s.as_str()).collect::<Vec<&str>>(),
      vec!["read", "profile", "email"]
    );
  }
}
