use crate::IConfiguration;
use serde_json::json;
use std::str::FromStr;
use std::time::Duration;

use anyhow::anyhow;
use oauth2::basic::BasicTokenIntrospectionResponse;
use oauth2::revocation::StandardRevocableToken;
use oauth2::{basic::BasicRevocationErrorResponse, RedirectUrl};
use oauth2::{
    basic::{BasicErrorResponse, BasicTokenType},
    Scope,
};
use oauth2::{reqwest::http_client, TokenResponse};
pub use oauth2::{AccessToken, RefreshToken};
use oauth2::{
    AuthUrl, AuthorizationCode, ClientId, CsrfToken, PkceCodeChallenge, PkceCodeVerifier, TokenUrl,
};
use serde::{de::Visitor, Deserialize, Deserializer, Serialize, Serializer};

type Client = oauth2::Client<
    BasicErrorResponse,
    Token,
    BasicTokenType,
    BasicTokenIntrospectionResponse,
    StandardRevocableToken,
    BasicRevocationErrorResponse,
>;

// This module holds abstractions for interacting with IndieAuth servers,
// effectively making this a 'client' module.

#[derive(Clone, Debug, Default)]
pub struct Scopes {
    scopes: Vec<Scope>,
}

#[derive(serde::Serialize, serde::Deserialize, Debug, Clone)]
pub struct RedeemPayload {
    pub code: String,
    pub client_id: url::Url,
    pub redirect_uri: url::Url,
    pub code_verifier: String,
}

#[derive(serde::Serialize, serde::Deserialize, Debug, Clone)]
pub struct Profile {
    pub name: Option<String>,
    pub url: Option<url::Url>,
    pub photo: Option<url::Url>,
    pub email: Option<String>,
}

#[derive(serde::Serialize, serde::Deserialize, Debug, Clone)]
pub struct CodeResponse {
    me: url::Url,
    profile: Option<Profile>,
}

struct ScopesVisitor {}

impl<'de> Visitor<'de> for ScopesVisitor {
    type Value = Scopes;

    fn expecting(&self, formatter: &mut std::fmt::Formatter) -> std::fmt::Result {
        formatter.write_str("a string of words")
    }

    fn visit_str<E>(self, v: &str) -> Result<Self::Value, E> {
        Ok(Scopes::from_str(v).unwrap())
    }
}

impl<'de> Deserialize<'de> for Scopes {
    fn deserialize<D>(deserializer: D) -> Result<Scopes, D::Error>
    where
        D: Deserializer<'de>,
    {
        deserializer.deserialize_str(ScopesVisitor {})
    }
}

impl Serialize for Scopes {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        serializer.serialize_str(&self.to_string())
    }
}

impl ToString for Scopes {
    fn to_string(&self) -> String {
        self.scopes
            .iter()
            .map(|s| s.to_string())
            .collect::<Vec<String>>()
            .join(" ")
    }
}

impl FromStr for Scopes {
    type Err = std::convert::Infallible;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        Ok(Self {
            scopes: s
                .split(" ")
                .map(|st| Scope::new(st.to_string()))
                .collect::<Vec<Scope>>(),
        })
    }
}

impl Scopes {
    pub fn is_empty(&self) -> bool {
        self.scopes.is_empty()
    }

    pub fn has(&self, scope: &str) -> bool {
        self.scopes.iter().any(|s| s.to_string().starts_with(scope))
    }

    pub fn to_vec(&self) -> Vec<Scope> {
        self.scopes.clone()
    }
}

#[derive(Default, Debug)]
pub struct DiscoveryResult {
    authorization_endpoints: Vec<String>,
    token_endpoints: Vec<String>,
}

/// Provides the basis of making a IndieAuth request.
pub struct AuthenticationRequest {
    client: Box<Client>,
    me: String,
    scopes: Vec<String>,
    verifier: PkceCodeVerifier,
    redirect_uri: Option<String>,
    client_id: String,
}

impl AuthenticationRequest {
    pub fn add_scope(&mut self, scope: &str) {
        self.scopes.push(scope.to_owned());
    }

    pub fn set_querying_identity(&mut self, me: &str) {
        self.me = me.to_string();
    }

    pub fn set_pkce_challenge(&mut self, verifier: &str) {
        self.verifier = PkceCodeVerifier::new(verifier.to_string());
    }

    pub fn get_pkce_verifier(&self) -> String {
        self.verifier.secret().clone()
    }

    pub fn new(
        me: &str,
        client_id: &str,
        authorization_endpoint: &str,
        redirect_uri: Option<String>,
        scopes: Option<Vec<String>>,
    ) -> anyhow::Result<Self> {
        let client = Box::new(Client::new(
            ClientId::new(client_id.to_string()),
            None,
            AuthUrl::new(authorization_endpoint.to_string()).map_err(|e| anyhow!("{:#?}", e))?,
            None,
        ));
        let verifier = PkceCodeChallenge::new_random_sha256().1;

        Ok(Self {
            client,
            client_id: client_id.to_string(),
            me: me.to_string(),
            scopes: scopes.unwrap_or_default(),
            verifier,
            redirect_uri,
        })
    }

    pub fn construct_url(&self) -> anyhow::Result<(String, String)> {
        let redirect_uri = self
            .redirect_uri
            .clone()
            .or(Some(self.client_id.clone()))
            .ok_or(anyhow::format_err!(
                "No redirect URI or client ID was provied."
            ))?;
        let redirect_url = RedirectUrl::from_url(url::Url::from_str(&redirect_uri)?);
        let request = self
            .client
            .authorize_url(CsrfToken::new_random)
            .set_pkce_challenge(PkceCodeChallenge::from_code_verifier_sha256(&self.verifier))
            .add_extra_param("me", self.me.clone())
            .add_scope(Scope::new(self.scopes.join(" ")))
            .set_redirect_uri(std::borrow::Cow::Borrowed(&redirect_url));
        let (url, state) = request.url();
        Ok((url.to_string(), state.secret().clone()))
    }
}

pub struct CodeVerificationRequest {
    pub code: String,
    pub pkce: Option<PkceCodeVerifier>,
    pub client_id: String,
    pub redirect_uri: String,
    pub authorization_endpoint: String,
    pub token_endpoint: String,
}

#[derive(Debug, Clone, serde::Serialize, serde::Deserialize)]
pub struct Token {
    pub me: url::Url,
    pub access_token: AccessToken,

    #[serde(rename = "scope")]
    pub scopes: Scopes,

    pub expires_in: Option<u64>,
    pub refresh_token: Option<RefreshToken>,
    pub profile: Option<Profile>,
}

impl TokenResponse<BasicTokenType> for Token {
    fn scopes(&self) -> Option<&Vec<Scope>> {
        // FIXME: This knows too much - is there a better way to do this?
        Some(&self.scopes.scopes)
    }
    fn access_token(&self) -> &AccessToken {
        &self.access_token
    }
    fn refresh_token(&self) -> Option<&RefreshToken> {
        self.refresh_token.as_ref()
    }
    fn expires_in(&self) -> Option<Duration> {
        self.expires_in.map(|s| Duration::from_secs(s))
    }
    fn token_type(&self) -> &BasicTokenType {
        &BasicTokenType::Bearer
    }
}

impl CodeVerificationRequest {
    pub fn redeem(&self) -> anyhow::Result<Token> {
        let client = Client::new(
            ClientId::new(self.client_id.clone()),
            None,
            AuthUrl::new(self.authorization_endpoint.clone())?,
            Some(TokenUrl::new(self.token_endpoint.clone())?),
        );

        let mut request = client.exchange_code(AuthorizationCode::new(self.code.clone()));

        request = match self.pkce.as_ref() {
            Some(verifier) => {
                request.set_pkce_verifier(PkceCodeVerifier::new(verifier.secret().to_string()))
            }
            None => request,
        };

        request
            .request(http_client)
            .map_err(|e| anyhow!("{:#?}", e))
    }
}

/// Confirms that the provided token is valid with the provided endpoint.
pub async fn verify<C>(cfg: &C, endpoint: &str, access_token: &str) -> Option<Token>
where
    C: IConfiguration + Send + Sync,
{
    let resp = cfg
        .request(Some("Requesting information about a token".to_owned()))
        .request(reqwest::Method::POST, endpoint.to_owned())
        .header("accept", "application/json")
        .bearer_auth(access_token)
        .send()
        .await
        .ok();

    match resp
        .expect("The endpoint didn't provide a valid response.")
        .error_for_status()
    {
        Ok(resp) => resp
            .text()
            .await
            .ok()
            .map(|json_text| dbg!(serde_json::from_str(&json_text)).ok())
            .flatten(),
        Err(_err) => None,
    }
}

pub async fn redeem<'a, C, R>(
    cfg: &C,
    endpoint: &str,
    redeem_payload: RedeemPayload,
) -> Result<R, crate::Error>
where
    C: IConfiguration + Send + Sync,
    R: for<'de> Deserialize<'de>,
{
    let endpoint_resp = cfg
        .request(Some(
            "Requesting information about an authorization request".to_owned(),
        ))
        .request(reqwest::Method::POST, endpoint.to_owned())
        .header("accept", "application/json")
        .body(
            json!({
                "grant_type": "authorization_code",
                "code": redeem_payload.code,
                "client_id": redeem_payload.client_id,
                "redirect_uri": redeem_payload.redirect_uri,
                "code_verifier": redeem_payload.code_verifier
            })
            .to_string(),
        )
        .send()
        .await
        .map_err(|e| crate::Error::Other(anyhow::anyhow!(e)))?;

    let resp = endpoint_resp
        .error_for_status()
        .map_err(|e| anyhow::anyhow!(e))?;
    log::info!("Handling response from endpoint.");
    let text = resp
        .text()
        .await
        .map_err(|e| crate::Error::Other(anyhow::anyhow!(e)))?;

    serde_json::from_str(text.as_str()).map_err(|e| crate::Error::Other(anyhow::anyhow!(e)))
}

#[cfg(test)]
#[allow(non_snake_case)]
mod tests {
    use super::*;
    use crate::test;
    use oauth2::{PkceCodeChallenge, PkceCodeVerifier};

    #[test]
    fn AuthenticationRequest_construct_url_has_expected_params() {
        let me = "https://jacky.wtf";
        let client_id = "http://sele.black.af";
        let endpoint = "https://sele.black.af/endpoints/wow";
        let redirect_uri = "https://special-domain-name.endpoints.sele.black.af/testing_page";
        let result = AuthenticationRequest::new(
            me,
            client_id,
            endpoint,
            Some(redirect_uri.to_owned()),
            None,
        );

        assert!(matches!(result, Ok(_)));
        let mut req = result.unwrap();

        req.add_scope("read");
        req.add_scope("create");

        let url_and_state = req.construct_url();

        assert!(url_and_state.is_ok());
        let (url, state) = url_and_state.unwrap();
        let challenge = PkceCodeChallenge::from_code_verifier_sha256(&PkceCodeVerifier::new(
            req.get_pkce_verifier(),
        ));
        let challenge_str = challenge.as_str();

        assert!(dbg!(url.clone()).contains(
            serde_urlencoded::to_string(&[("redirect_uri", &redirect_uri)])
                .unwrap()
                .as_str()
        ));
        assert!(url.contains(
            serde_urlencoded::to_string(&[
                ("code_challenge", challenge_str),
                ("code_challenge_method", "S256"),
            ])
            .unwrap()
            .as_str()
        ));

        assert!(url.contains(
            serde_urlencoded::to_string(&[("client_id", &client_id)])
                .unwrap()
                .as_str()
        ));
        assert!(url.contains(
            serde_urlencoded::to_string(&[("me", &me)])
                .unwrap()
                .as_str()
        ));
        assert!(url.contains(
            serde_urlencoded::to_string(&[("state", &state)])
                .unwrap()
                .as_str()
        ));
        assert!(url.contains(
            dbg!(serde_urlencoded::to_string(&[("scope", "read create")]))
                .unwrap()
                .as_str()
        ));
    }

    #[test]
    fn CodeVerificationRequest_redeem_success() {
        let (_challenge, verifier) = PkceCodeChallenge::new_random_sha256();
        let code = "a-valid-code".to_owned();
        let authorization_endpoint = format!("{}/auth-endpoint", mockito::server_url());
        let token_endpoint = format!("{}/token-endpoint", mockito::server_url());

        let req = super::CodeVerificationRequest {
            code,
            pkce: Some(verifier),
            client_id: "https://sele.black.af".to_owned(),
            redirect_uri: "https://sele.black.af/bounce-back".to_owned(),
            authorization_endpoint,
            token_endpoint,
        };
        let body = serde_json::json!({
            "me": "https://jacky.wtf/",
            "token_type": "bearer",
            "scope": "read profile email create",
            "access_token": "the-access-token",
            "profile": {
                "name": "Jacky Alciné",
                "url": "https://jacky.wtf/",
                "photo": "https://jacky.wtf/photo.jpeg",
                "email": "yo+codespam@jacky.wtf"
            }
        })
        .to_string();

        let endpoint_mock = mockito::mock("POST", "/token-endpoint")
            .with_status(200)
            .with_body(body)
            .expect_at_least(1)
            .create();

        let result = dbg!(req.redeem());
        endpoint_mock.assert();
        assert!(matches!(result, Ok(_)));

        let token = result.unwrap();
        assert!(token.profile.is_some());
        let profile = token.profile.unwrap();
        assert_eq!(profile.url, "https://jacky.wtf/".parse().ok());
    }

    #[tokio::test(flavor = "multi_thread")]
    async fn verify_confirms_valid_token() {
        let cfg = test::init();
        let endpoint = format!("{}/endpoint", mockito::server_url());
        let access_token = "access-token";
        let body = serde_json::json!({
            "access_token": "a-token",
            "me": "https://jacky.wtf/",
            "scope": "read profile email"
        })
        .to_string();
        let endpoint_mock = mockito::mock("POST", "/endpoint")
            .with_body(&body)
            .expect_at_most(1)
            .create();

        endpoint_mock.assert();
        let result = super::verify(&cfg, &endpoint, access_token).await;
        assert!(result.is_some());

        let token = result.unwrap();
        assert_eq!(token.me, "https://jacky.wtf/".parse().unwrap());
        assert_eq!(token.access_token.secret().clone(), "a-token".to_owned());
        assert_eq!(token.scopes.to_string(), "read profile email".to_owned());
    }

    #[tokio::test(flavor = "multi_thread")]
    async fn redeem_resolves_responses() {
        let cfg = test::init();
        let endpoint = format!("{}/endpoint", mockito::server_url());
        let body = serde_json::json!({
            "me": "https://jacky.wtf/",
            "token_type": "bearer",
            "scope": "read profile email create",
            "access_token": "the-access-token",
            "profile": {
                "name": "Jacky Alciné",
                "url": "https://jacky.wtf/",
                "photo": "https://jacky.wtf/photo.jpeg",
                "email": "yo+codespam@jacky.wtf"
            }

        })
        .to_string();

        let endpoint_mock = mockito::mock("POST", "/endpoint")
            .with_body(&body)
            .expect_at_most(2)
            .create();
        let payload = RedeemPayload {
            code: "a-code".to_owned(),
            code_verifier: "a-code-verifier".to_owned(),
            client_id: "https://indieweb.org".parse().unwrap(),
            redirect_uri: "https://indieweb.org/back".parse().unwrap(),
        };

        let optional_token: Result<super::Token, crate::Error> =
            super::redeem(&cfg, &endpoint, payload.clone()).await;
        let optional_code: Result<super::CodeResponse, crate::Error> =
            super::redeem(&cfg, &endpoint, payload.clone()).await;
        endpoint_mock.assert();

        assert!(optional_code.is_ok());
        assert!(optional_token.is_ok());
    }

    #[test]
    fn Scopes_assert_presence() -> Result<(), anyhow::Error> {
        assert!(!Scopes::from_str("read profile")?.has("update"));
        assert!(Scopes::from_str("read profile")?.has("read"));
        Ok(())
    }
}
