use crate::select2::grouped::{Group, Groupable, GroupableRecord, GroupedResponse};
use crate::select2::{Pagination, Record, Request};
use std::clone::Clone;
use std::cmp::{Eq, PartialEq};
use std::collections::BTreeMap;
use std::fmt::{Debug, Display};
use std::hash::Hash;
use std::string::ToString;

// -----------------------------------------------------------------------------

impl Request {

    /// This function does not perform the `term` or `q` search for the client
    /// request. The search much be performed by the caller using the
    /// `search_select2` method. The search results are passed to this function
    /// to build the response.
    ///
    /// If no search is requested, the caller can pass the entire collection (in
    /// the form of a slice) to this function to be processed into the `Select2`
    /// format.

    #[tracing::instrument(level = "trace", name = "Build Grouped Response", skip(self, search_results_keys, search_results_values))]
    pub fn grouped_response<K: Clone + Debug + Display + Eq + Hash + PartialEq + ToString, G: Groupable>(
        &self,
        items_per_page: &Option<usize>,
        selected_record: &Option<String>,
        search_results_keys: &[&K],
        search_results_values: &[G]
    ) -> GroupedResponse {

        // Ensure that there are the same number of keys and values:

        if search_results_keys.len() != search_results_values.len() {
            tracing::error!(
                "Caller supplied {} keys and {} values. \
                The number of keys and values should be the same. \
                Returning empty response.",
                search_results_keys.len(),
                search_results_values.len(),
            ); // error!
            return GroupedResponse::default()
        } else if search_results_keys.is_empty() {
            tracing::debug!(
                "List of keys and values is empty. \
                Returning empty response.",
            ); // debug!
            return GroupedResponse::default()
        } // if

        // The `search_select2` method will return references to the keys. The
        // caller needs to look up the full records and return their references
        // to this method (the `*_results` methods) because we don't necessarily
        // know how to do this. Here, we zip the keys and records together into
        // a tuple:

        let search_results: Vec<(&K, &G)> = search_results_keys
            .iter()
            .cloned()
            .zip(search_results_values.iter())
            .collect();

        // Observe pagination. If the caller specifies a maximum number of items
        // per page, then consider pagination turned on:

        // self.request_type == Some("query_append".to_string())
        let groupable_results: (bool, Vec<&(&K, &G)>) = if let Some(items_per_page) = items_per_page {

            // Paginated response:

            // Get the `page` number from the request:
            let page: usize = self.page_number();

            // This function works on the resolved output of a search, or the
            // records dumped from a key-value store:
            let paginated_results: Vec<&(&K, &G)> = search_results
                // Iterate over each passed record:
                .iter()
                // Skip records so we start at beginning of specified `page`:
                .skip(items_per_page * (page - 1))
                // Only take a page's worth of records:
                .take(*items_per_page)
                // Collect all Select2 records into a `Vec<GroupableRecord>`:
                .collect();

            // Return pagination status and results to outer scope:
            (items_per_page * page < search_results.len(), paginated_results)

        } else {

            // Unpaginated response:

            // This function works on the resolved output of a search, or the
            // records dumped from a key-value store:
            let unpaginated_results: Vec<&(&K, &G)> = search_results
                // Iterate over each passed record:
                .iter()
                // Collect all select2 records into a `Vec<GroupableRecord>`:
                .collect();

            // Return pagination status and results to outer scope:
            (false, unpaginated_results)

        }; // if

        // This `BTreeMap` is used to organize the records into their groups.
        // The `key` represents the group, and the `value` represents the
        // records in the group:

        let mut grouped_results: BTreeMap<String, Vec<Record>> = BTreeMap::new();

        // Iterate over the results and insert them into their respective
        // groups:
        groupable_results.1
            // Iterate over the results records:
            .iter()
            // For each record in the results:
            .for_each(|(key, value)| {
                // Convert the record from a `&G` into a `GroupableRecord`:
                let groupable_record: GroupableRecord = value.record();
                // Convert the `GroupableRecord` to a `Select2` `Record`:
                let mut record = groupable_record.to_record(key);
                // Check if the `selected_record` was set...
                if let Some(selected_record) = selected_record {
                    // ...was set. Update record with comparison result and
                    // return record:
                    record.selected = record.id == *selected_record;
                } // if
                // Attempt to get mutuable reference to the group entry in
                // the B-tree map:
                match grouped_results.get_mut(&groupable_record.group) {
                    // If group exists in hash map, add record to the group:
                    Some(group) => group.push(record),
                    // If group does not exist, initialize with this record:
                    None => { grouped_results.insert(
                        groupable_record.group.to_owned(),
                        vec![record]
                    ); } // None
                } // match
            }); // for_each

        // Convert `BTreeMap<String, Vec<Record>>` structure used to organize
        // records into `Vec<Group>` which will be returned as the response:

        let grouped_results: Vec<Group> = grouped_results
            .iter()
            .map(|(group, records)| Group {
                text: group.to_string(),
                children: records.to_owned(),
            }) // map
            .collect();

        // Return Select2 `GroupedResponse` to caller:

        GroupedResponse {
            results: grouped_results,
            pagination: Pagination {
                more: groupable_results.0,
            }, // Pagination
        } // GroupedResponse

    } // fn

} // impl