use {
    anchor_lang::{
        prelude::*,
        solana_program::system_program
    },
    crate::state,
    std::mem::size_of
};

#[derive(Accounts)]
#[instruction(
    reference: Pubkey,
    item_bump: u8,
    proof_bump: u8,
)]
pub struct CreateItem<'info> {
    #[account(
        mut, 
        seeds = [
            state::SEED_INDEX, 
            index.owner.key().as_ref(), 
            index.namespace.as_ref()
        ], 
        bump = index.bump, 
        constraint = index.owner == signer.key(),
    )]
    pub index: Account<'info, state::Index>,

    #[account(
        init,
        seeds = [
            state::SEED_ITEM,
            index.key().as_ref(),
            index.item_count.to_string().as_bytes(),
        ],
        bump = item_bump,
        payer = signer,
        space = 8 + size_of::<state::Item>(),
    )]
    pub item: Account<'info, state::Item>,

    #[account(
        init,
        seeds = [
            state::SEED_PROOF,
            index.key().as_ref(),
            reference.as_ref(),
        ],
        bump = proof_bump,
        payer = signer,
        space = 8 + size_of::<state::Proof>(),
    )]
    pub proof: Account<'info, state::Proof>,

    #[account(mut)]
    pub signer: Signer<'info>,
    
    #[account(address = system_program::ID)]
    pub system_program: Program<'info, System>,
}

pub fn handler(
    ctx: Context<CreateItem>,
    reference: Pubkey,
    item_bump: u8,
    proof_bump: u8,
) -> ProgramResult {
    // Get accounts.
    let index = &mut ctx.accounts.index;
    let item = &mut ctx.accounts.item;
    let proof = &mut ctx.accounts.proof;

    // Initialize item account.
    item.id = index.item_count;
    item.reference = reference;
    item.bump = item_bump;

    // Initialize proof account.
    proof.bump = proof_bump;

    // Increment index counter.
    index.item_count += 1;
    
    return Ok(());
}
