use {
    anchor_lang::{
        prelude::*,
        solana_program::system_program
    },
    crate::state,
    std::mem::size_of
};

#[derive(Accounts)]
#[instruction(
    account: Pubkey,
    record_bump: u8,
    reference_bump: u8,
)]
pub struct CreateReference<'info> {
    #[account(
        mut, 
        seeds = [
            state::SEED_INDEX, 
            owner.key().as_ref(), 
            index.namespace.as_ref()
        ], 
        bump = index.bump, 
        has_one = owner,
    )]
    pub index: Account<'info, state::Index>,

    #[account(mut)]
    pub owner: Signer<'info>,

    #[account(
        init,
        seeds = [
            state::SEED_RECORD,
            index.key().as_ref(),
            account.as_ref(),
        ],
        bump = record_bump,
        payer = owner,
        space = 8 + size_of::<state::Record>(),
    )]
    pub record: Account<'info, state::Record>,

    #[account(
        init,
        seeds = [
            state::SEED_REFERENCE,
            index.key().as_ref(),
            index.count.to_string().as_bytes(),
        ],
        bump = reference_bump,
        payer = owner,
        space = 8 + size_of::<state::Reference>(),
    )]
    pub reference: Account<'info, state::Reference>,
    
    #[account(address = system_program::ID)]
    pub system_program: Program<'info, System>,
}

pub fn handler(
    ctx: Context<CreateReference>, 
    account: Pubkey,
    record_bump: u8,
    reference_bump: u8,
) -> ProgramResult {
    // Get accounts.
    let index = &mut ctx.accounts.index;
    let reference = &mut ctx.accounts.reference;
    let record = &mut ctx.accounts.record;

    // Initialize reference account.
    reference.id = index.count;
    reference.account = account;
    reference.bump = reference_bump;

    // Initialize record account.
    record.bump = record_bump;

    // Increment index counter.
    index.count += 1;
    
    return Ok(());
}
