use std::error;
use std::fmt;

#[derive(Debug, thiserror::Error)]
pub enum Error {
	#[error("I/O error")]
	IO(#[from] std::io::Error),
	#[error("codec error")]
	Codec(#[from] Box<bincode::ErrorKind>),
	#[error("key {0} not found")]
	NotFound(u64),
	#[error("missing value")]
	MissingValue
}

/// Wraps both [indexkv errors](Error) and generic errors that could end up in the stream passed to
/// [Store::write](crate::Store::write)
#[derive(Debug)]
pub enum StreamError<E: error::Error> {
	Internal(Error),
	Caller(E)
}

impl<E: error::Error> fmt::Display for StreamError<E> {
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		match self {
			Self::Internal(e) => write!(f, "indexkv error writing stream: {}", e),
			Self::Caller(e) => write!(f, "external error writing stream: {}", e)
		}
	}
}

impl<E: error::Error + 'static> error::Error for StreamError<E> {
	fn source(&self) -> Option<&(dyn error::Error + 'static)> {
		match self {
			Self::Internal(e) => Some(e),
			Self::Caller(e) => Some(e)
		}
	}
}

impl<E: error::Error, T> From<T> for StreamError<E> where Error: From<T> {
	fn from(inner: T) -> Self {
		Self::Internal(Error::from(inner))
	}
}

impl<E: error::Error> StreamError<E> {
	pub(crate) fn from_external(inner: E) -> Self {
		Self::Caller(inner)
	}
}

