use std::fmt::{Arguments, Formatter, Write};
use std::ops::{AddAssign, Deref, SubAssign};

pub struct IndentFormatter<'i> {
    raw: Formatter<'i>,
    pub indent_level: usize,
    pub indent_chars: &'i str,
}

impl<'i> From<Formatter<'i>> for IndentFormatter<'i> {
    fn from(f: Formatter<'i>) -> Self {
        Self::new(f, "    ")
    }
}

impl<'i> Deref for IndentFormatter<'i> {
    type Target = Formatter<'i>;

    fn deref(&self) -> &Self::Target {
        &self.raw
    }
}

impl<'i> AddAssign<usize> for IndentFormatter<'i> {
    fn add_assign(&mut self, rhs: usize) {
        self.indent_level = self.indent_level.saturating_add(rhs);
    }
}

impl<'i> SubAssign<usize> for IndentFormatter<'i> {
    fn sub_assign(&mut self, rhs: usize) {
        self.indent_level = self.indent_level.saturating_sub(rhs)
    }
}

impl<'i> Write for IndentFormatter<'i> {
    fn write_str(&mut self, s: &str) -> std::fmt::Result {
        self.raw.write_str(s)
    }
    fn write_char(&mut self, c: char) -> std::fmt::Result {
        self.raw.write_char(c)
    }
    fn write_fmt(self: &mut Self, args: Arguments<'_>) -> std::fmt::Result {
        self.raw.write_fmt(args)
    }
}

impl<'i> IndentFormatter<'i> {
    pub fn new(f: Formatter<'i>, indent: &'i str) -> Self {
        Self {
            raw: f,
            indent_level: 0,
            indent_chars: indent,
        }
    }
    pub fn write_indent(&mut self) -> std::fmt::Result {
        for _ in 0..self.indent_level {
            self.raw.write_str(self.indent_chars)?;
        }
        Ok(())
    }
    pub fn write_lines(&mut self, s: &str) -> std::fmt::Result {
        for line in s.lines() {
            self.write_indent()?;
            self.raw.write_str(line)?;
        }
        Ok(())
    }
}

pub trait IndentDisplay {
    fn indent_fmt(&self, f: IndentFormatter) -> std::fmt::Result;
}
