use std::ops::DerefMut;
use std::sync::{Arc, RwLock};
use tokio::sync::mpsc::{unbounded_channel, UnboundedSender};

use lapin::{options::*, BasicProperties, Channel, Connection, ConnectionProperties, ConnectionState};
use tracing::Level;
use super::error::Error;

use super::message::{LogMessage, LogMessage_Severity};

pub struct RabbitMqClient {
    _ampq_url: String,
    _conn: Arc<RwLock<Connection>>,
    queue_sender: UnboundedSender<LogMessage>,
    _reconnection_max: u8,
    service_name: String,
}

impl RabbitMqClient {
    pub async fn new(ampq_url: String, service_name: String) -> Result<Self, Error> {
        let conn = Arc::new(RwLock::new(
            RabbitMqClient::connect(ampq_url.clone()).await?,
        ));
        let (tx, mut rx) = unbounded_channel::<LogMessage>();
        let lock_conn = conn.read()?;
        let mut channel = lock_conn.create_channel().await?;
        drop(lock_conn);

        let conn_move = conn.clone();
        let ampq_url_move = ampq_url.clone();
        let reconnection_max = 4;
        let reconnection_max_move = reconnection_max;
        tokio::spawn(async move {

            while let Some(message) = rx.recv().await {

                let mut retry: u8 = 0;
                loop {
                    if retry > reconnection_max_move {
                        break
                    }

                    if let Err(_err) = RabbitMqClient::publish_message_static(&channel, message.clone()).await {
                        let new_conn = match RabbitMqClient::connect(ampq_url_move.clone()).await {
                            Ok(conn) => conn,
                            Err(_err) => {
                                retry += 1;
                                println!("RabbitMQ reconnection failed");
                                continue
                            }
                        };
                        channel = match new_conn.create_channel().await {
                            Ok(channel) => channel,
                            Err(_err) => {
                                retry += 1;
                                println!("RabbitMQ unable to establish a channel");
                                continue
                            }
                        };
                        let new_conn_arc = conn_move.clone();
                        {
                            match new_conn_arc.write() {
                                Ok(mut write_guard) => {
                                    let write_ref = write_guard.deref_mut();
                                    *write_ref = new_conn;
                                    println!("Rabbit connection restored");
                                    break
                                }
                                Err(_err) => {
                                    retry += 1;
                                    println!("RabbitMQ poisoned connection");
                                    continue
                                }
                            };
                        }
                    }
                    else {
                        break
                    }
                }
            }
        });

        Ok(Self {
            _ampq_url: ampq_url,
            _conn: conn,
            queue_sender: tx,
            _reconnection_max: reconnection_max,
            service_name,
        })
    }

    async fn connect(ampq_url: String) -> Result<Connection, Error> {
        let conn = Connection::connect(&ampq_url, ConnectionProperties::default()).await?;
        Ok(conn)
    }

    pub fn check_status(&self) -> bool {
        match self.status() {
            Ok(state) => {
                use ConnectionState::*;
                match state {
                    Initial | Connecting | Connected => true,
                    Closing | Closed | Error => false,
                }
            },
            Err(_err) => false
        }
    }

    pub fn status(&self) -> Result<ConnectionState, Error> {
        Ok(self._conn.read()?.status().state())
    }

    async fn publish_message_static(channel: &Channel, message: LogMessage) -> Result<(), Error> {
        channel.basic_publish(
                "imsub.logger",
                "logger",
                BasicPublishOptions::default(),
                serde_json::to_string(&message)?.as_bytes(),
                BasicProperties::default(),
            )
            .await?
            .await?;
        // assert_eq!(confirm, Confirmation::NotRequested);
        Ok(())
    }

    pub fn send_log(
        &self,
        level: &Level,
        user_id: Option<String>,
        method_name: Option<String>,
        message: Option<String>,
        details: Option<String>,
    ) {
        let message = LogMessage {
            service_name: self.service_name.clone(),
            telegram_id: user_id.unwrap_or_else(|| "-1".to_string()),
            method_name: method_name.unwrap_or_else(|| "".to_string()),
            severity: {
                match *level {
                    Level::ERROR => LogMessage_Severity::Error,
                    Level::WARN => LogMessage_Severity::Warning,
                    Level::INFO => LogMessage_Severity::Info,
                    Level::DEBUG => LogMessage_Severity::Debug,
                    Level::TRACE => LogMessage_Severity::Debug,
                }
            },
            message: message.unwrap_or_else(|| "".to_string()),
            details: details.unwrap_or_else(|| "".to_string()),
            ..Default::default()
        };
        if let Err(err) = self.queue_sender.send(message) {
            println!("{:?}", err);
        }
    }
}
