mod message;
mod rabbitmq;
mod error;

use std::collections::HashMap;

use tracing::{Subscriber};
use tracing_subscriber::Layer;

pub use rabbitmq::RabbitMqClient;
use error::Error;

#[derive(Default, Debug)]
struct FieldsHandler {
    telegram_id: Option<String>,
    target: String,
    name: String,
    method_name: Option<String>,
    message: Option<String>,
    details: HashMap<String, String>,
}

impl tracing::field::Visit for FieldsHandler {
    fn record_f64(&mut self, field: &tracing::field::Field, value: f64) {
        self.dispatch_field_value(field.name(), value.to_string())
    }

    fn record_i64(&mut self, field: &tracing::field::Field, value: i64) {
        self.dispatch_field_value(field.name(), value.to_string())
    }

    fn record_u64(&mut self, field: &tracing::field::Field, value: u64) {
        self.dispatch_field_value(field.name(), value.to_string())
    }

    fn record_bool(&mut self, field: &tracing::field::Field, value: bool) {
        self.dispatch_field_value(field.name(), value.to_string())
    }

    fn record_str(&mut self, field: &tracing::field::Field, value: &str) {
        self.dispatch_field_value(field.name(), value.to_string())
    }

    fn record_error(
        &mut self,
        field: &tracing::field::Field,
        value: &(dyn std::error::Error + 'static),
    ) {
        self.dispatch_field_value(field.name(), value.to_string());
    }

    fn record_debug(&mut self, field: &tracing::field::Field, value: &dyn std::fmt::Debug) {
        self.dispatch_field_value(field.name(), format!("{:#?}", value));
    }
}

impl FieldsHandler {
    pub fn new(target: &str, name: &str) -> Self {
        Self {
            target: target.to_string(),
            name: name.to_string(),
            ..Default::default()
        }
    }

    fn dispatch_field_value(&mut self, field_name: &str, field_value: String) {
        match field_name {
            "telegram_id" => self.telegram_id = Some(field_value),
            "message" => self.message = Some(field_value),
            "fn_path" => self.set_fn_path(field_value),
            field_name => {
                self.details.insert(field_name.to_string(), field_value);
            }
        };
    }

    pub fn get_details_to_string(&self) -> String {
        self.details.iter().fold("".to_string(), |acc, (k, v)| {
            format!("{}{} = {}\n", acc, k, v)
        })
    }

    pub fn set_fn_path(&mut self, fn_path: String) {
        let fn_path = format!("{}::{}", self.target, fn_path);
        self.method_name = Some(fn_path);
    }

    pub fn get_method_name(&self) -> String {
        match &self.method_name {
            None => self.name.clone(),
            Some(value) => value.to_string(),
        }
    }
}

pub struct RabbitLayer {
    rabbit_client: RabbitMqClient,
}

impl RabbitLayer {
    pub async fn new(rabbit_url: String, service_name: String) -> Result<Self, Error> {
        let rabbit_client = RabbitMqClient::new(rabbit_url, service_name).await?;
        Ok(Self { rabbit_client })
    }
}

impl<S: Subscriber> Layer<S> for RabbitLayer {
    fn on_event(
        &self,
        event: &tracing::Event<'_>,
        _ctx: tracing_subscriber::layer::Context<'_, S>,
    ) {
        let mut visitor = FieldsHandler::new(event.metadata().target(), event.metadata().name());
        event.record(&mut visitor);

        self.rabbit_client.send_log(
            event.metadata().level(),
            visitor.telegram_id.clone(),
            Some(visitor.get_method_name()),
            visitor.message.clone(),
            Some(visitor.get_details_to_string()),
        );
    }
}
