use crate::{
    flavor::ImperatorFlavor,
    models::{HeaderBorrowed, HeaderOwned, Save},
    tokens::TokenLookup,
    FailedResolveStrategy, ImperatorError, ImperatorErrorKind,
};
use jomini::{BinaryDeserializer, TextDeserializer, TextTape, TokenResolver};
use serde::de::{Deserialize, DeserializeOwned};
use std::io::{Cursor, Read, Seek, SeekFrom};
use zip::{result::ZipError, ZipArchive};

/// Describes the format of the save before decoding
#[derive(Debug, Clone, Copy, PartialEq)]
pub enum Encoding {
    /// Save had been generated by the game started with `-debug_mode`
    Debug,

    /// A standard ironman or normal save
    Standard,
}

/// The memory allocation strategy for handling zip files
///
/// When the `mmap` feature is enabled, the extractor can use
/// an anonymous memory map
#[derive(Debug, Clone, Copy)]
pub enum Extraction {
    /// Extract the zip data into memory
    InMemory,

    /// Extract the zip data into a anonymous memory map
    #[cfg(feature = "mmap")]
    MmapTemporaries,
}

/// Customize how a save is extracted
#[derive(Debug, Clone)]
pub struct ImperatorExtractorBuilder {
    extraction: Extraction,
    on_failed_resolve: FailedResolveStrategy,
}

impl Default for ImperatorExtractorBuilder {
    fn default() -> Self {
        ImperatorExtractorBuilder::new()
    }
}

impl ImperatorExtractorBuilder {
    /// Create a new extractor with default values: extract zips into memory
    // and ignore unknown binary tokens
    pub fn new() -> Self {
        ImperatorExtractorBuilder {
            extraction: Extraction::InMemory,
            on_failed_resolve: FailedResolveStrategy::Ignore,
        }
    }

    /// Set the memory allocation extraction behavior for when a zip is encountered
    pub fn with_extraction(mut self, extraction: Extraction) -> Self {
        self.extraction = extraction;
        self
    }

    /// Set the behavior for when an unresolved binary token is encountered
    pub fn with_on_failed_resolve(mut self, strategy: FailedResolveStrategy) -> Self {
        self.on_failed_resolve = strategy;
        self
    }

    /// Extract the header from the save.
    pub fn extract_header_owned(
        &self,
        data: &[u8],
    ) -> Result<(HeaderOwned, Encoding), ImperatorError> {
        self.extract_header_as(data)
    }

    /// Extract the header from the save with zero copy deserialization.
    pub fn extract_header_borrowed<'a>(
        &self,
        data: &'a [u8],
    ) -> Result<(HeaderBorrowed<'a>, Encoding), ImperatorError> {
        self.extract_header_as(data)
    }

    /// Extract the header from the save as a custom type
    pub fn extract_header_with_tokens_as<'de, T, Q>(
        &self,
        data: &'de [u8],
        resolver: &'de Q,
    ) -> Result<(T, Encoding), ImperatorError>
    where
        T: Deserialize<'de>,
        Q: TokenResolver,
    {
        let data = skip_save_prefix(data);
        let mut cursor = Cursor::new(data);
        let offset = match detect_encoding(&mut cursor)? {
            BodyEncoding::Plain => data.len(),
            BodyEncoding::Zip(zip) => zip.offset() as usize,
        };

        let data = &data[..offset];
        if sniff_is_binary(data) {
            let res = BinaryDeserializer::builder_flavor(ImperatorFlavor)
                .on_failed_resolve(self.on_failed_resolve)
                .from_slice(data, resolver)?;
            Ok((res, Encoding::Standard))
        } else {
            // allow uncompressed text as TextZip even though the game doesn't produce said format
            let res = TextDeserializer::from_utf8_slice(data)?;
            Ok((res, Encoding::Debug))
        }
    }

    /// Extract the header from the save as a custom type
    pub fn extract_header_as<'de, T>(
        &self,
        data: &'de [u8],
    ) -> Result<(T, Encoding), ImperatorError>
    where
        T: Deserialize<'de>,
    {
        self.extract_header_with_tokens_as(data, &TokenLookup)
    }

    /// Extract all info from a save
    pub fn extract_save<R>(&self, reader: R) -> Result<(Save, Encoding), ImperatorError>
    where
        R: Read + Seek,
    {
        self.extract_save_with_tokens(reader, &TokenLookup)
    }

    /// Extract all info from a save
    pub fn extract_save_with_tokens<R, Q>(
        &self,
        reader: R,
        resolver: &Q,
    ) -> Result<(Save, Encoding), ImperatorError>
    where
        R: Read + Seek,
        Q: TokenResolver,
    {
        self.extract_save_as(reader, resolver)
    }

    // todo, customize deserialize type
    fn extract_save_as<R, Q>(
        &self,
        mut reader: R,
        resolver: &Q,
    ) -> Result<(Save, Encoding), ImperatorError>
    where
        R: Read + Seek,
        Q: TokenResolver,
    {
        let mut buffer = Vec::new();
        match detect_encoding(&mut reader)? {
            BodyEncoding::Plain => {
                // Ensure we are at the start of the stream
                reader.seek(SeekFrom::Start(0))?;

                // So we can get the length
                let len = reader.seek(SeekFrom::End(0))?;
                reader.seek(SeekFrom::Start(0))?;
                buffer.reserve(len as usize);
                reader.read_to_end(&mut buffer)?;

                let data = skip_save_prefix(&buffer);
                let tape = TextTape::from_slice(data)?;
                let header = TextDeserializer::from_utf8_tape(&tape)?;
                let gamestate = TextDeserializer::from_utf8_tape(&tape)?;
                Ok((Save { header, gamestate }, Encoding::Debug))
            }
            BodyEncoding::Zip(mut zip) => {
                let res = match self.extraction {
                    Extraction::InMemory => melt_in_memory(
                        &mut buffer,
                        "gamestate",
                        &mut zip,
                        self.on_failed_resolve,
                        resolver,
                    ),
                    #[cfg(feature = "mmap")]
                    Extraction::MmapTemporaries => {
                        melt_with_temporary("gamestate", &mut zip, self.on_failed_resolve, resolver)
                    }
                }?;

                Ok((res, Encoding::Standard))
            }
        }
    }
}

/// Logic container for extracting data from an Imperator save
#[derive(Debug, Clone)]
pub struct ImperatorExtractor {}

impl ImperatorExtractor {
    /// Create a customized container
    pub fn builder() -> ImperatorExtractorBuilder {
        ImperatorExtractorBuilder::new()
    }

    /// Extract just the header from the save
    pub fn extract_header(data: &[u8]) -> Result<(HeaderOwned, Encoding), ImperatorError> {
        Self::builder().extract_header_owned(data)
    }

    /// Extract all info from a save
    pub fn extract_save<R>(reader: R) -> Result<(Save, Encoding), ImperatorError>
    where
        R: Read + Seek,
    {
        Self::builder().extract_save(reader)
    }
}

fn melt_in_memory<T, R, Q>(
    buffer: &mut Vec<u8>,
    name: &'static str,
    zip: &mut zip::ZipArchive<R>,
    on_failed_resolve: FailedResolveStrategy,
    resolver: &Q,
) -> Result<T, ImperatorError>
where
    R: Read + Seek,
    T: DeserializeOwned,
    Q: TokenResolver,
{
    buffer.clear();
    let mut zip_file = zip
        .by_name(name)
        .map_err(|e| ImperatorErrorKind::ZipMissingEntry(name, e))?;

    // protect against excessively large uncompressed data
    if zip_file.size() > 1024 * 1024 * 200 {
        return Err(ImperatorErrorKind::ZipSize(name).into());
    }

    buffer.reserve(zip_file.size() as usize);
    zip_file
        .read_to_end(buffer)
        .map_err(|e| ImperatorErrorKind::ZipExtraction(name, e))?;

    let res = BinaryDeserializer::builder_flavor(ImperatorFlavor)
        .on_failed_resolve(on_failed_resolve)
        .from_slice(buffer, resolver)
        .map_err(|e| ImperatorErrorKind::Deserialize {
            part: Some(name.to_string()),
            err: e,
        })?;

    Ok(res)
}

#[cfg(feature = "mmap")]
fn melt_with_temporary<T, R, Q>(
    name: &'static str,
    zip: &mut zip::ZipArchive<R>,
    on_failed_resolve: FailedResolveStrategy,
    resolver: &Q,
) -> Result<T, ImperatorError>
where
    R: Read + Seek,
    T: DeserializeOwned,
    Q: TokenResolver,
{
    let mut zip_file = zip
        .by_name(name)
        .map_err(|e| ImperatorErrorKind::ZipMissingEntry(name, e))?;

    // protect against excessively large uncompressed data
    if zip_file.size() > 1024 * 1024 * 200 {
        return Err(ImperatorErrorKind::ZipSize(name).into());
    }

    let mut mmap = memmap::MmapMut::map_anon(zip_file.size() as usize)?;
    std::io::copy(&mut zip_file, &mut mmap.as_mut())
        .map_err(|e| ImperatorErrorKind::ZipExtraction(name, e))?;
    let buffer = &mmap[..];

    let res = BinaryDeserializer::builder_flavor(ImperatorFlavor)
        .on_failed_resolve(on_failed_resolve)
        .from_slice(buffer, resolver)
        .map_err(|e| ImperatorErrorKind::Deserialize {
            part: Some(name.to_string()),
            err: e,
        })?;

    Ok(res)
}

/// Throwaway the first line
fn skip_save_prefix(data: &[u8]) -> &[u8] {
    let id_line_idx = data
        .iter()
        .position(|&x| x == b'\n')
        .map(|x| x + 1)
        .unwrap_or(0);
    &data[id_line_idx..]
}

/// We guess from the initial data (after the save id line) that the save is
/// binary if the 3rd and 4th byte are the binary equals token, which should
/// not occur in a plaintext save
fn sniff_is_binary(data: &[u8]) -> bool {
    data.get(2..4).map_or(false, |x| x == [0x01, 0x00])
}

pub(crate) enum BodyEncoding<'a, R>
where
    R: Read + Seek,
{
    Zip(ZipArchive<&'a mut R>),
    Plain,
}

pub(crate) fn detect_encoding<R>(reader: &mut R) -> Result<BodyEncoding<R>, ImperatorError>
where
    R: Read + Seek,
{
    let zip_attempt = zip::ZipArchive::new(reader);

    match zip_attempt {
        Ok(x) => Ok(BodyEncoding::Zip(x)),
        Err(ZipError::InvalidArchive(_)) => Ok(BodyEncoding::Plain),
        Err(e) => Err(ImperatorError::new(
            ImperatorErrorKind::ZipCentralDirectory(e),
        )),
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_skip_save_prefix() {
        let data = b"abc\n123";
        let result = skip_save_prefix(&data[..]);
        assert_eq!(result, b"123");
    }
}
