use core::sync::atomic::{AtomicBool, Ordering};

use crate::*;

mod test_simple_cyclic {
    use super::*;

    struct Test(Gc<Test>);

    impl Drop for Test {
        fn drop(&mut self) {
            DROPPED.store(true, Ordering::SeqCst);
        }
    }

    unsafe impl Trace for Test {
        unsafe_field_trace! {0}
    }

    static DROPPED: AtomicBool = AtomicBool::new(false);

    #[test]
    fn test_simple_cyclic() {
        assert!(!DROPPED.load(Ordering::SeqCst));
        {
            let _cyclic = Gc::new_cyclic(|v| Test(v));
        }
        assert!(DROPPED.load(Ordering::SeqCst), "cyclic wasn't dropped");
    }
}

mod test_better_cyclic {
    use super::*;

    //   +--------------+
    //   v              |
    // Test1 -> Test2 --+
    //   ^              |
    //   +--------------+

    // We also have two external pointers to each of test1 and test2

    struct Test1(Gc<Test2>);

    impl Drop for Test1 {
        fn drop(&mut self) {
            DROPPED_1.store(true, Ordering::SeqCst);
        }
    }

    unsafe impl Trace for Test1 {
        unsafe_field_trace! {0}
    }

    struct Test2(Gc<Test1>, Gc<Test1>);

    impl Drop for Test2 {
        fn drop(&mut self) {
            DROPPED_2.store(true, Ordering::SeqCst);
        }
    }

    unsafe impl Trace for Test2 {
        unsafe_field_trace! {0, 1}
    }

    static DROPPED_1: AtomicBool = AtomicBool::new(false);
    static DROPPED_2: AtomicBool = AtomicBool::new(false);

    #[test]
    fn test_better_cyclic() {
        assert!(!DROPPED_1.load(Ordering::SeqCst));
        assert!(!DROPPED_2.load(Ordering::SeqCst));
        {
            let mut _gc_test2_1 = None;
            let _gc_test1_1 = Gc::new_cyclic(|gc_1: Gc<Test1>| {
                let gc_2 = Gc::clone(&gc_1);
                let test2 = Gc::new(Test2(gc_1, gc_2));
                _gc_test2_1 = Some(test2.clone());
                Test1(test2)
            });
            let _gc_test2_2 = _gc_test2_1.as_ref().unwrap().clone();
            let _gc_test1_2 = _gc_test1_1.clone();
            drop(_gc_test1_1);
            drop(_gc_test1_2);
            drop(_gc_test2_1);
            drop(_gc_test2_2);
        }
        assert!(DROPPED_1.load(Ordering::SeqCst), "cyclic wasn't dropped");
        assert!(DROPPED_2.load(Ordering::SeqCst), "cyclic wasn't dropped");
    }
}
