use std::str::FromStr;

use arrayvec::ArrayVec;
use serde::de::Unexpected;
use serde::Deserializer;
use serde::Deserialize;
use serde_repr::Deserialize_repr;
use serde_repr::Serialize_repr;

#[derive(Clone, Copy, Debug, Deserialize_repr, Eq, PartialEq, Serialize_repr, strum::IntoStaticStr)]
#[repr(u8)]
/// Identifies the genre of a [Title](crate::Title).  Often, a title will have multiple.
pub enum Genre /* {{{ */ {
	Action = 0,
	Adult = 1,
	Adventure = 2,
	Animation = 3,
	Biography = 4,
	Comedy = 5,
	Crime = 6,
	Documentary = 7,
	Drama = 8,
	Family = 9,
	Fantasy = 10,
	FilmNoir = 11,
	GameShow = 12,
	History = 13,
	Horror = 14,
	Music = 15,
	Musical = 16,
	Mystery = 17,
	News = 18,
	RealityTv = 19,
	Romance = 20,
	SciFi = 21,
	Short = 22,
	Sport = 23,
	TalkShow = 24,
	Thriller = 25,
	War = 26,
	Western = 27
} // }}}

impl FromStr for Genre {
	type Err = crate::Error;
	fn from_str(s: &str) -> Result<Self, Self::Err> {
		Ok(match s {
			"Action" => Self::Action,
			"Adult" => Self::Adult,
			"Adventure" => Self::Adventure,
			"Animation" => Self::Animation,
			"Biography" => Self::Biography,
			"Comedy" => Self::Comedy,
			"Crime" => Self::Crime,
			"Documentary" => Self::Documentary,
			"Drama" => Self::Drama,
			"Family" => Self::Family,
			"Fantasy" => Self::Fantasy,
			"Film-Noir" => Self::FilmNoir,
			"Game-Show" => Self::GameShow,
			"History" => Self::History,
			"Horror" => Self::Horror,
			"Music" => Self::Music,
			"Musical" => Self::Musical,
			"Mystery" => Self::Mystery,
			"News" => Self::News,
			"Reality-TV" => Self::RealityTv,
			"Romance" => Self::Romance,
			"Sci-Fi" => Self::SciFi,
			"Short" => Self::Short,
			"Sport" => Self::Sport,
			"Talk-Show" => Self::TalkShow,
			"Thriller" => Self::Thriller,
			"War" => Self::War,
			"Western" => Self::Western,
			s => return Err(crate::Error::InvalidGenre(s.into()))
		})
	}
}

pub(crate) fn deserialize<'de, D: Deserializer<'de>>(deserializer: D) -> Result<ArrayVec<Genre, 3>, D::Error> {
	let input: &str = Deserialize::deserialize(deserializer)?;
	if(input == "\\N") {
		return Ok(ArrayVec::new());
	}
	match input.split(',').map(Genre::from_str).collect::<Result<ArrayVec<_, 3>, crate::Error>>() {
		Ok(v) => Ok(v),
		Err(_) => Err(serde::de::Error::invalid_value(
			Unexpected::Str(input),
			&"'Action', 'Adult', 'Adventure', 'Animation', 'Biography', 'Comedy', 'Crime', 'Documentary', 'Drama', 'Family', 'Fantasy', 'Film-Noir', 'GameShow', 'History', 'Horror', 'Music', 'Musical', 'Mystery', 'News', 'Reality-TV', 'Romance', 'Sci-Fi', 'Short', 'Sport', 'Talk-Show', 'Thriller', 'War', 'Western'"
		))
	}
}

