use serde::de::Error;
use serde::de::Unexpected;
use serde::Deserialize;
use serde::Deserializer;

const IMDB_ID_EXPECT: &str = "a string starting with 'tt', followed by a positive integer";

pub fn parse_imdb_id<'de, D: Deserializer<'de>>(deserializer: D) -> Result<u32, D::Error> {
	let s: &str = Deserialize::deserialize(deserializer)?;
	if(s.len() < 2 || &s[0..=1] != "tt") {
		return Err(Error::invalid_value(Unexpected::Str(s), &IMDB_ID_EXPECT));
	}
	match s[2..].parse() {
		Ok(v) => Ok(v),
		Err(_) => Err(Error::invalid_value(Unexpected::Str(s), &IMDB_ID_EXPECT))
	}
}

pub fn parse_janky_tsv_option<'de, D: Deserializer<'de>>(deserializer: D) -> Result<Option<u16>, D::Error> {
	let s: &str = Deserialize::deserialize(deserializer)?;
	if(s == "\\N") {
		return Ok(None);
	}
	match s.parse() {
		Ok(v) => Ok(Some(v)),
		Err(_) => Err(Error::invalid_type(Unexpected::Str(s), &"an unsigned 16-bit integer, or '\\N'"))
	}
}

pub fn parse_janky_tsv_option_u32<'de, D: Deserializer<'de>>(deserializer: D) -> Result<Option<u32>, D::Error> {
	let s: &str = Deserialize::deserialize(deserializer)?;
	if(s == "\\N") {
		return Ok(None);
	}
	match s.parse() {
		Ok(v) => Ok(Some(v)),
		Err(_) => Err(Error::invalid_type(Unexpected::Str(s), &"an unsigned 32-bit integer, or '\\N'"))
	}
}

