use std::cmp::Ordering;
use std::convert::TryFrom;

use smartstring::alias::String;

use crate::EpisodeLink;
use crate::Error;
use crate::Title;
use crate::TitleType;

#[derive(Eq, PartialEq)]
/// Represents a TV series from IMDB.
pub struct Show {
	pub imdb_id: u32,
	pub title: String,
	pub is_adult: bool,
	pub start_year: u16,
	pub end_year: Option<u16>,
	pub runtime_minutes: Option<u16>,
	pub genres: Vec<String>,
	pub episodes: Vec<Episode>
}

impl std::fmt::Debug for Show /* {{{ */ {
	fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
		f.debug_struct("Show")
			.field("imdb_id", &self.imdb_id)
			.field("title", &self.title)
			.field("is_adult", &self.is_adult)
			.field("start_year", &self.start_year)
			.field("end_year", &self.end_year)
			.field("runtime_minutes", &self.runtime_minutes)
			.field("genres", &self.genres)
			.field("episodes", &self.episodes.len())
			.finish()
	}
} // }}}

impl Show {
	#[allow(dead_code)] // Used in tests
	#[allow(clippy::too_many_arguments)] // Only a convenience function for tests; this is fine
	pub(crate) fn new(imdb_id: u32, title: &str, is_adult: bool, start_year: u16, end_year: Option<u16>, runtime_minutes: Option<u16>, genres: &[&str], episodes: &[Episode]) -> Self /* {{{ */ {
		Self{
			imdb_id,
			title: String::from(title),
			is_adult,
			start_year,
			end_year,
			runtime_minutes,
			genres: genres.iter().map(|s| String::from(*s)).collect(),
			episodes: episodes.to_vec()
		}
	} // }}}

	pub(crate) fn from_wrapped_title(input: Result<Title, Error>) -> Result<Self, Error> /* {{{ */ {
		Self::try_from(input?)
	} // }}}
}

impl TryFrom<Title> for Show /* {{{ */ {
	type Error = Error;
	fn try_from(input: Title) -> Result<Self, Error> {
		match input.title_type {
			TitleType::TVSeries => Ok(Self{
				imdb_id: input.imdb_id,
				title: input.primary_title,
				is_adult: input.is_adult,
				start_year: match input.start_year {
					Some(v) => v,
					None => return Err(Error::YearMissing)
				},
				end_year: input.end_year,
				runtime_minutes: input.runtime_minutes,
				genres: input.genres,
				episodes: Vec::new()
			}),
			_ => Err(Error::WrongMediaType(input.title_type.into(), "Show"))
		}
	}
} // }}}

#[derive(Clone, Debug, Eq, PartialEq)]
/// Represents an episode of a TV series from IMDB.  Pared down from [Title] and EpisodeLink based on fields that make sense for an individual episode.
pub struct Episode {
	pub season: u16,
	pub episode: u16,
	pub imdb_id: u32,
	pub title: String,
	pub runtime_minutes: Option<u16>
}

impl PartialOrd for Episode /* {{{ */ {
	fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
		Some(self.cmp(other))
	}
} // }}}

impl Ord for Episode /* {{{ */ {
	fn cmp(&self, other: &Self) -> Ordering {
		match self.season.cmp(&other.season) {
			Ordering::Equal => self.episode.cmp(&other.episode),
			v => v
		}
	}
} // }}}

impl Episode {
	#[allow(dead_code)] // Used in tests
	pub(crate) fn new(season: u16, episode: u16, imdb_id: u32, title: &str, runtime_minutes: Option<u16>) -> Self /* {{{ */ {
		Self{
			season,
			episode,
			imdb_id,
			title: String::from(title),
			runtime_minutes
		}
	} // }}}

	pub(crate) fn from_title_and_link(title: Title, link: EpisodeLink) -> Result<Self, Error> /* {{{ */ {
		match title.title_type {
			TitleType::Episode => Ok(Self{
				season: match link.season {
					Some(v) => v,
					None => return Err(Error::SeasonMissing)
				},
				episode: match link.episode {
					Some(v) => v,
					None => return Err(Error::EpisodeMissing)
				},
				imdb_id: title.imdb_id,
				title: title.primary_title,
				runtime_minutes: title.runtime_minutes
			}),
			_ => Err(Error::WrongMediaType(title.title_type.into(), "Episode"))
		}
	} // }}}
}

pub(crate) fn title_matches_show_name_and_year(title: &Result<Title, Error>, name: &str, year: u16) -> bool /* {{{ */ {
	if let Ok(title) = title {
		if(title.title_type != TitleType::TVSeries) {
			return false;
		}
		if let Some(title_year) = title.start_year {
			if(title_year != year) {
				return false;
			}
		} else {
			return false;
		}
		if(title.primary_title != name && title.original_title != name) {
			return false;
		}
		return true;
	}
	false
} // }}}

