use std::collections::HashMap;
use std::convert::TryInto;
use std::path::Path;
use std::path::PathBuf;
use std::time::Duration;

use futures::future::ready;
use futures::io::AsyncRead;
use futures::stream::Stream;
use futures::stream::StreamExt;
use futures::stream::TryStreamExt;

use async_compression::futures::bufread::GzipDecoder;
use csv_async::AsyncReaderBuilder;
use tracing::instrument;

use indexkv::Store;

use crate::Error;

use crate::Episode;
use crate::episode::EpisodeLinkFromImdb;
use crate::EpisodeLink;
use crate::Movie;
use crate::Show;
use crate::Title;
use crate::title::TitleFromImdb;

use crate::movie::title_matches_movie_name_and_year;
use crate::show::title_matches_show_name_and_year;

pub(crate) const BASE_URL: &str = "https://datasets.imdbws.com/";

pub(crate) const EPISODES_FILE: &str = "title.episode";
pub(crate) const TITLES_FILE: &str = "title.basics";

/// A client to IMDB's dataset
pub struct Client {
	cache_invalidation_time: Duration,
	cache_directory: PathBuf,
	title_cache: Store<Title>,
	episode_cache: Store<Vec<EpisodeLink>>,
	http: reqwest::Client
}

impl Client {
	// TODO:  Allow custom reqwest::Client options
	/// Instantiates a new Client
	pub async fn new(cache_invalidation_time: Duration, cache_directory: impl AsRef<Path>) -> Result<Self, Error> /* {{{ */ {
		if let Err(err) = tokio::fs::create_dir_all(&cache_directory).await {
			if(err.kind() != std::io::ErrorKind::AlreadyExists) {
				return Err(err.into());
			}
		}
		Ok(Self{
			cache_invalidation_time,
			cache_directory: cache_directory.as_ref().to_path_buf(),
			title_cache: Store::new(Self::cache_path(&cache_directory, TITLES_FILE)).await?,
			episode_cache: Store::new(Self::cache_path(&cache_directory, EPISODES_FILE)).await?,
			http: reqwest::Client::new()
		})
	} // }}}

	fn make_url(file: &str) -> String /* {{{ */ {
		format!("{}{}.tsv.gz", BASE_URL, file)
	} // }}}

	fn cache_path(cache_directory: impl AsRef<Path>, file: &str) -> PathBuf /* {{{ */ {
		cache_directory.as_ref().join(&format!("{}.db", file))
	} // }}}

	#[instrument(err, level = "info", skip(self))]
	async fn start_download(&self, file: &str) -> Result<impl AsyncRead, Error> /* {{{ */ {
		let net = self.http.get(&Self::make_url(file)).send().await?
			.bytes_stream()
			.map_err(|e| std::io::Error::new(std::io::ErrorKind::Other, e))
			.into_async_read();
		let decompressed = GzipDecoder::new(net);
		Ok(decompressed)
	} // }}}

	async fn should_invalidate_cache(&self, file: &str) -> Result<bool, Error> /* {{{ */ {
		let metadata = match tokio::fs::metadata(Self::cache_path(&self.cache_directory, file)).await {
			Ok(v) => v,
			Err(_e) => {
				// TODO:  Check specific error type
				return Ok(true);
			}
		};
		let mtime = metadata.modified()?.elapsed()?;
		Ok(metadata.len() == 0 || mtime >= self.cache_invalidation_time)
	} // }}}

	async fn ensure_title_cache(&mut self) -> Result<(), Error> /* {{{ */ {
		if(self.title_cache.is_valid().await? && !self.should_invalidate_cache(TITLES_FILE).await?) {
			return Ok(());
		}
		let raw = self.start_download(TITLES_FILE).await?;
		let stream = AsyncReaderBuilder::new()
			.delimiter(b'\t')
			.quoting(false)
			.create_deserializer(raw)
			.into_deserialize::<TitleFromImdb>();
		self.title_cache.write(
			stream
				.err_into::<Error>()
				.map_ok(|t| (t.imdb_id, t.into()))
		).await?;
		Ok(())
	} // }}}

	async fn ensure_episode_cache(&mut self) -> Result<(), Error> /* {{{ */ {
		if(self.episode_cache.is_valid().await? && !self.should_invalidate_cache(EPISODES_FILE).await?) {
			return Ok(());
		}
		let raw = self.start_download(EPISODES_FILE).await?;
		let mut stream = AsyncReaderBuilder::new()
			.delimiter(b'\t')
			.create_deserializer(raw)
			.into_deserialize::<EpisodeLinkFromImdb>();
		let mut episodes_by_show: HashMap<u64, Vec<EpisodeLink>, fnv::FnvBuildHasher> = HashMap::with_capacity_and_hasher(151825, fnv::FnvBuildHasher::default()); // Capacity chosen by looking at number of unique parentTconst values in IMDB title.episode.tsv.gz at time of writing
		while let Some(result) = stream.next().await {
			let episode = result?;
			let episodes_for_this_show = episodes_by_show.entry(episode.series_imdb_id).or_insert_with(Vec::new);
			episodes_for_this_show.push(episode.into());
		}
		let write_stream = futures::stream::iter(episodes_by_show.into_iter());
		self.episode_cache.write_infallible(write_stream).await?;
		Ok(())
	} // }}}

	/// Returns a single [Title] by ID.
	#[instrument(err, level = "debug", skip(self))]
	#[inline]
	pub async fn get_title(&mut self, id: u64) -> Result<Title, Error> /* {{{ */ {
		self.ensure_title_cache().await?;
		let result = self.title_cache.get(id).await;
		match result {
			Ok(v) => Ok(v),
			Err(indexkv::Error::NotFound(id)) => Err(Error::NotFound(id)),
			Err(e) => Err(e.into())
		}
	} // }}}

	/// Returns a list of [Titles](Title) given a list of IDs.  Missing IDs are ignored;
	/// [Error::NotFound] is not returned.
	#[instrument(err, level = "debug", skip(self))]
	#[inline]
	pub async fn get_titles(&mut self, ids: &[u64]) -> Result<Vec<Title>, Error> /* {{{ */ {
		self.ensure_title_cache().await?;
		let mut map = self.title_cache.get_many(ids).await?;
		Ok(ids.iter().filter_map(|id| map.remove(id)).collect())
	} // }}}

	/// Streams all [Titles](Title) from persistent storage.  In relational database parlance, this
	/// is analogous to a tablescan.
	#[instrument(err, level = "debug", skip(self))]
	#[inline]
	pub async fn stream_titles(&mut self) -> Result<impl Stream<Item = Result<Title, Error>>, Error> /* {{{ */ {
		self.ensure_title_cache().await?;
		Ok(self.title_cache.stream_values().await?.err_into())
	} // }}}

	/// Looks up a [Movie] by name and year.  This is slow, as the cache is not keyed on those
	/// fields; a tablescan is performed.  Returns Ok(None) if no match is found, not
	/// [Error::NotFound].
	#[instrument(err, level = "debug", skip(self))]
	pub async fn get_movie_by_name_and_year(&mut self, name: &str, year: u16) -> Result<Option<Movie>, Error> /* {{{ */ {
		let mut stream = self.stream_titles().await?
			.filter(|r| ready(title_matches_movie_name_and_year(r, name, year)))
			.take(1);
		let movie = match stream.next().await {
			Some(r) => match Movie::from_wrapped_title(r) {
				Ok(v) => v,
				Err(Error::WrongMediaType(_, _)) => return Ok(None),
				Err(e) => return Err(e)
			},
			None => return Ok(None)
		};
		Ok(Some(movie))
	} // }}}

	/// Returns a single [Movie] by ID.
	#[instrument(err, level = "debug", skip(self))]
	#[inline]
	pub async fn get_movie(&mut self, id: u64) -> Result<Movie, Error> /* {{{ */ {
		self.get_title(id).await?.try_into()
	} // }}}

	/// Returns a list of [Movies](Movie) given a list of IDs.  Missing IDs are ignored;
	/// [Error::NotFound] is not returned.
	#[instrument(err, level = "debug", skip(self))]
	#[inline]
	pub async fn get_movies(&mut self, ids: &[u64]) -> Result<Vec<Movie>, Error> /* {{{ */ {
		Ok(self.get_titles(ids).await?.into_iter().filter_map(|t| match t.try_into() {
			Ok(m) => Some(m),
			Err(_) => None
		}).collect())
	} // }}}

	/// Streams all [Movies](Movie) from persistnet storage.  In relational database parlance, this
	/// is analogous to a tablescan.
	#[instrument(err, level = "debug", skip(self))]
	#[inline]
	pub async fn stream_movies(&mut self) -> Result<impl Stream<Item = Result<Movie, Error>>, Error> /* {{{ */ {
		let stream = self.stream_titles().await?;
		Ok(stream.filter_map(|r| ready(match Movie::from_wrapped_title(r) {
			Ok(v) => Some(Ok(v)),
			Err(Error::WrongMediaType(_, _)) => None,
			Err(e) => Some(Err(e))
		})))
	} // }}}

	/// Looks up a [Show] by name and year.  This is slow, as the cache is not keyed on those
	/// fields; a tablescan is performed.  Returns Ok(None) if not match is found, not
	/// [Error::NotFound].
	#[instrument(err, level = "debug", skip(self))]
	pub async fn get_show_by_name_and_year(&mut self, name: &str, year: u16) -> Result<Option<Show>, Error> /* {{{ */ {
		let mut stream = self.stream_titles().await?
			.filter(|r| ready(title_matches_show_name_and_year(r, name, year)))
			.take(1);
		let mut show = match stream.next().await {
			Some(r) => match Show::from_wrapped_title(r) {
				Ok(v) => v,
				Err(Error::WrongMediaType(_, _)) => return Ok(None),
				Err(e) => return Err(e)
			},
			None => return Ok(None)
		};
		show.episodes = self.get_episodes(show.imdb_id).await?;
		Ok(Some(show))
	} // }}}

	/// Returns a list of [Shows](Show) given a list of IDs.  Missing IDs are ignored;
	/// [Error::NotFound] is not returned.
	#[instrument(err, level = "debug", skip(self))]
	#[inline]
	pub async fn get_shows(&mut self, ids: &[u64]) -> Result<Vec<Show>, Error> /* {{{ */ {
		// Not making this a wrapper around self.get_titles() for now to keep the get_episodes()
		// bit simple
		let mut shows = Vec::with_capacity(ids.len());
		for id in ids {
			let show = self.get_show(*id).await?;
			shows.push(show);
		}
		Ok(shows)
	} // }}}

	/// Returns a single [Show] by ID.
	#[instrument(err, level = "debug", skip(self))]
	#[inline]
	pub async fn get_show(&mut self, id: u64) -> Result<Show, Error> /* {{{ */ {
		let mut show: Show = self.get_title(id).await?.try_into()?;
		show.episodes = self.get_episodes(show.imdb_id).await?;
		Ok(show)
	} // }}}

	/// Returns a map<ID, [Show]> given a list of IDs.  Missing IDs are ignored; [Error::NotFound]
	/// is not returned.
	#[instrument(err, level = "debug", skip(self))]
	#[inline]
	pub async fn get_shows_by_id(&mut self, ids: &[u64]) -> Result<HashMap<u64, Show, fnv::FnvBuildHasher>, Error> /* {{{ */ {
		Ok(self.get_shows(ids).await?.into_iter().map(|s| (s.imdb_id, s)).collect())
	} // }}}

	#[instrument(err, level = "debug", skip(self))]
	async fn get_episodes(&mut self, show_id: u64) -> Result<Vec<Episode>, Error> /* {{{ */ {
		self.ensure_episode_cache().await?;
		let result = self.episode_cache.get(show_id).await;
		let links = match result {
			Ok(v) => v,
			Err(indexkv::Error::NotFound(show_id)) => return Err(Error::NotFound(show_id)),
			Err(e) => return Err(e.into())
		};
		let mut episodes = Vec::with_capacity(links.len());
		for link in links.into_iter() {
			let title = self.get_title(link.imdb_id).await?;
			match Episode::from_title_and_link(title, link) {
				Ok(episode) => episodes.push(episode),
				Err(Error::WrongMediaType(_, _)) => (),
				Err(Error::SeasonMissing) => (),
				Err(Error::EpisodeMissing) => (),
				Err(e) => return Err(e)
			};
		}
		episodes.sort_unstable();
		Ok(episodes)
	} // }}}

	/// Returns a map<Show ID, Vec<[Episode]>> given a list of [Show] IDs.  Missing IDs are
	/// ignored; [Error::NotFound] is not returned.
	#[instrument(err, level = "debug", skip(self))]
	pub async fn get_episodes_by_show(&mut self, show_ids: &[u64]) -> Result<HashMap<u64, Vec<Episode>, fnv::FnvBuildHasher>, Error> /* {{{ */ {
		let mut episodes_by_show = HashMap::with_capacity_and_hasher(show_ids.len(), fnv::FnvBuildHasher::default());
		for show_id in show_ids {
			let episodes_in = self.get_episodes(*show_id).await?;
			let episodes = episodes_by_show.entry(*show_id).or_insert_with(Vec::new);
			for episode in episodes_in.into_iter() {
				episodes.push(episode);
			}
		}
		Ok(episodes_by_show)
	} // }}}
}

#[cfg(test)]
mod tests {
	use std::collections::HashMap;
	use std::time::Duration;
	use crate::*;

	async fn client() -> (tempfile::TempDir, Client) {
		let dir = tempfile::TempDir::new().unwrap();
		let client = Client::new(Duration::from_secs(86400), dir.path()).await.unwrap();
		(dir, client)
	}

	#[tokio::test]
	async fn movies_single() /* {{{ */ {
		let (_dir, mut client) = client().await;
		for i in 0..5 {
			println!(">>> {}", i);
			assert_eq!(client.get_movie(26972).await.unwrap(), Movie::new(26972, "Scrooge", false, 1935, Some(78), &["Drama", "Family", "Fantasy"]));
			assert_eq!(client.get_movie(12766346).await.unwrap(), Movie::new(12766346, "If I Only Had Christmas", false, 2020, Some(84), &["Drama", "Romance"]));
		}
	} // }}}

	#[tokio::test]
	async fn movies_multi() /* {{{ */ {
		let (_dir, mut client) = client().await;
		for i in 0..5 {
			println!(">>> {}", i);
			let mut movies = client.get_movies(&[280380, 4767950, 10919380]).await.unwrap().into_iter().map(|m| (m.imdb_id, m)).collect::<HashMap<_, _>>();
			assert_eq!(movies.len(), 3);
			assert_eq!(movies.remove(&280380), Some(Movie::new(280380, "Trapped", false, 2002, Some(106), &["Action", "Crime", "Drama"])));
			assert_eq!(movies.remove(&4767950), Some(Movie::new(4767950, "A Christmas Detour", false, 2015, Some(85), &["Comedy", "Romance"])));
			assert_eq!(movies.remove(&10919380), Some(Movie::new(10919380, "Freaky", false, 2020, Some(102), &["Comedy", "Horror", "Thriller"])));
		}
	} // }}}

	#[tokio::test]
	async fn episodes_by_show_single() /* {{{ */ {
		let (_dir, mut client) = client().await;

		for i in 0..5 {
			println!(">>> {}", i);
			let episodes = client.get_episodes(163507).await.unwrap();
			assert_eq!(episodes.len(), 220); // {{{
			assert_eq!(episodes[0], Episode::new(1, 1, 747998, "Show No. 103", None));
			assert_eq!(episodes[1], Episode::new(1, 2, 748009, "Show No. 104", None));
			assert_eq!(episodes[2], Episode::new(1, 3, 748011, "Show No. 106", None));
			assert_eq!(episodes[3], Episode::new(1, 4, 748012, "Show No. 101", None));
			assert_eq!(episodes[4], Episode::new(1, 5, 748013, "Show No. 105", None));
			assert_eq!(episodes[5], Episode::new(1, 6, 748014, "Show No. 102", None));
			assert_eq!(episodes[6], Episode::new(1, 7, 748015, "Show No. 107", None));
			assert_eq!(episodes[7], Episode::new(1, 8, 748016, "Show No. 111", None));
			assert_eq!(episodes[8], Episode::new(1, 9, 748017, "Show No. 108", None));
			assert_eq!(episodes[9], Episode::new(1, 10, 747999, "Show No. 117", None));
			assert_eq!(episodes[10], Episode::new(1, 11, 748000, "Show No. 113", None));
			assert_eq!(episodes[11], Episode::new(1, 12, 748001, "Show No. 119", None));
			assert_eq!(episodes[12], Episode::new(1, 13, 748002, "Show No. 120", None));
			assert_eq!(episodes[13], Episode::new(1, 14, 748003, "Show No. 114", None));
			assert_eq!(episodes[14], Episode::new(1, 15, 748004, "Show No. 118", None));
			assert_eq!(episodes[15], Episode::new(1, 16, 748005, "Show No. 121", None));
			assert_eq!(episodes[16], Episode::new(1, 17, 748006, "Show No. 116", None));
			assert_eq!(episodes[17], Episode::new(1, 18, 748007, "Show No. 115", None));
			assert_eq!(episodes[18], Episode::new(1, 19, 748008, "Show No. 110", None));
			assert_eq!(episodes[19], Episode::new(1, 20, 748010, "Show No. 109", None));
			assert_eq!(episodes[20], Episode::new(2, 1, 748018, "Show No. 209", None));
			assert_eq!(episodes[21], Episode::new(2, 2, 748029, "Show No. 205", None));
			assert_eq!(episodes[22], Episode::new(2, 3, 748040, "Show No. 206", None));
			assert_eq!(episodes[23], Episode::new(2, 4, 748051, "Show No. 201", None));
			assert_eq!(episodes[24], Episode::new(2, 5, 748052, "Show No. 220", None));
			assert_eq!(episodes[25], Episode::new(2, 6, 748053, "Show No. 203", None));
			assert_eq!(episodes[26], Episode::new(2, 7, 748054, "Show No. 202", None));
			assert_eq!(episodes[27], Episode::new(2, 8, 748055, "Show No. 212", None));
			assert_eq!(episodes[28], Episode::new(2, 9, 748056, "Show No. 221", None));
			assert_eq!(episodes[29], Episode::new(2, 10, 748019, "Show No. 204", None));
			assert_eq!(episodes[30], Episode::new(2, 11, 748020, "Show No. 222", None));
			assert_eq!(episodes[31], Episode::new(2, 12, 748021, "Show No. 211", None));
			assert_eq!(episodes[32], Episode::new(2, 13, 748022, "Show No. 223", None));
			assert_eq!(episodes[33], Episode::new(2, 14, 748023, "Show No. 213", None));
			assert_eq!(episodes[34], Episode::new(2, 15, 748024, "Show No.207", None));
			assert_eq!(episodes[35], Episode::new(2, 16, 748025, "Show No. 224", None));
			assert_eq!(episodes[36], Episode::new(2, 17, 748026, "Show No. 208", None));
			assert_eq!(episodes[37], Episode::new(2, 18, 748027, "Show No. 225", None));
			assert_eq!(episodes[38], Episode::new(2, 19, 748028, "Show No. 210", None));
			assert_eq!(episodes[39], Episode::new(2, 20, 748030, "Show No. 215", None));
			assert_eq!(episodes[40], Episode::new(2, 21, 748031, "Show No. 226", None));
			assert_eq!(episodes[41], Episode::new(2, 22, 748032, "Show No. 227", None));
			assert_eq!(episodes[42], Episode::new(2, 23, 748033, "Show No. 216", None));
			assert_eq!(episodes[43], Episode::new(2, 24, 748034, "Show No. 228", None));
			assert_eq!(episodes[44], Episode::new(2, 25, 748035, "Show No. 229", None));
			assert_eq!(episodes[45], Episode::new(2, 26, 748036, "Show No. 230", None));
			assert_eq!(episodes[46], Episode::new(2, 27, 748037, "Show No. 231", None));
			assert_eq!(episodes[47], Episode::new(2, 28, 748038, "Show No. 217", None));
			assert_eq!(episodes[48], Episode::new(2, 29, 748039, "Show No. 232", None));
			assert_eq!(episodes[49], Episode::new(2, 30, 748041, "Show No. 233", None));
			assert_eq!(episodes[50], Episode::new(2, 31, 748042, "Show No. 234", None));
			assert_eq!(episodes[51], Episode::new(2, 32, 748043, "Show No. 235", None));
			assert_eq!(episodes[52], Episode::new(2, 33, 748044, "Show No. 214", None));
			assert_eq!(episodes[53], Episode::new(2, 34, 748045, "Show No. 218", None));
			assert_eq!(episodes[54], Episode::new(2, 35, 748046, "Show No. 236", None));
			assert_eq!(episodes[55], Episode::new(2, 36, 748047, "Show No. 219", None));
			assert_eq!(episodes[56], Episode::new(2, 37, 748048, "Show No. 237", None));
			assert_eq!(episodes[57], Episode::new(2, 38, 748049, "Show No. 238", None));
			assert_eq!(episodes[58], Episode::new(2, 39, 748050, "Show No. 239", None));
			assert_eq!(episodes[59], Episode::new(3, 0, 918399, "The Best of Whose Line Is It Anyway?", None));
			assert_eq!(episodes[60], Episode::new(3, 1, 748057, "Show No. 302", None));
			assert_eq!(episodes[61], Episode::new(3, 2, 748068, "Show No. 303", None));
			assert_eq!(episodes[62], Episode::new(3, 3, 748079, "Show No. 312", None));
			assert_eq!(episodes[63], Episode::new(3, 4, 748090, "Show No. 305", None));
			assert_eq!(episodes[64], Episode::new(3, 5, 748091, "Show No. 319", None));
			assert_eq!(episodes[65], Episode::new(3, 6, 748092, "Show No. 306", None));
			assert_eq!(episodes[66], Episode::new(3, 7, 748093, "Show No. 304", None));
			assert_eq!(episodes[67], Episode::new(3, 8, 748094, "Show No. 320", None));
			assert_eq!(episodes[68], Episode::new(3, 9, 748095, "Show No. 313", None));
			assert_eq!(episodes[69], Episode::new(3, 10, 748058, "Show No. 341", None));
			assert_eq!(episodes[70], Episode::new(3, 11, 748059, "Show No. 340", None));
			assert_eq!(episodes[71], Episode::new(3, 12, 748060, "Show No. 309", None));
			assert_eq!(episodes[72], Episode::new(3, 13, 748061, "Show No. 342", None));
			assert_eq!(episodes[73], Episode::new(3, 14, 748062, "Show No. 301", None));
			assert_eq!(episodes[74], Episode::new(3, 15, 748063, "Show No. 321", None));
			assert_eq!(episodes[75], Episode::new(3, 16, 748064, "Show No. 322", None));
			assert_eq!(episodes[76], Episode::new(3, 17, 748065, "Show No. 323", None));
			assert_eq!(episodes[77], Episode::new(3, 18, 748066, "Show No. 327", None));
			assert_eq!(episodes[78], Episode::new(3, 19, 748067, "Show No. 326", None));
			assert_eq!(episodes[79], Episode::new(3, 20, 748069, "Show No. 325", None));
			assert_eq!(episodes[80], Episode::new(3, 21, 748070, "Show No. 307", None));
			assert_eq!(episodes[81], Episode::new(3, 22, 748071, "Show No. 328", None));
			assert_eq!(episodes[82], Episode::new(3, 23, 748072, "Show No. 333", None));
			assert_eq!(episodes[83], Episode::new(3, 24, 748073, "Show No. 310", None));
			assert_eq!(episodes[84], Episode::new(3, 25, 748074, "Show No. 329", None));
			assert_eq!(episodes[85], Episode::new(3, 26, 748075, "Show No. 335", None));
			assert_eq!(episodes[86], Episode::new(3, 27, 748076, "Show No. 311", None));
			assert_eq!(episodes[87], Episode::new(3, 28, 748077, "Show No. 330", None));
			assert_eq!(episodes[88], Episode::new(3, 29, 748078, "Show No. 336", None));
			assert_eq!(episodes[89], Episode::new(3, 30, 748080, "Show No. 334", None));
			assert_eq!(episodes[90], Episode::new(3, 31, 748081, "Show No. 332", None));
			assert_eq!(episodes[91], Episode::new(3, 32, 748082, "Show No. 331", None));
			assert_eq!(episodes[92], Episode::new(3, 33, 748083, "Show No. 338", None));
			assert_eq!(episodes[93], Episode::new(3, 34, 748084, "Show No. 343", None));
			assert_eq!(episodes[94], Episode::new(3, 35, 748085, "Show No. 339", None));
			assert_eq!(episodes[95], Episode::new(3, 36, 748086, "Show No. 344", None));
			assert_eq!(episodes[96], Episode::new(3, 37, 748087, "Show No. 410", None));
			assert_eq!(episodes[97], Episode::new(3, 38, 748088, "Show No. 308", None));
			assert_eq!(episodes[98], Episode::new(3, 39, 748089, "Show No. 407", None));
			assert_eq!(episodes[99], Episode::new(4, 1, 748096, "Show No. 401", None));
			assert_eq!(episodes[100], Episode::new(4, 2, 748107, "Show No. 405", None));
			assert_eq!(episodes[101], Episode::new(4, 3, 748118, "Show No. 324", None));
			assert_eq!(episodes[102], Episode::new(4, 4, 748121, "Show No. 420", None));
			assert_eq!(episodes[103], Episode::new(4, 5, 748122, "Show No. 317", None));
			assert_eq!(episodes[104], Episode::new(4, 6, 748123, "Show No. 413", None));
			assert_eq!(episodes[105], Episode::new(4, 7, 748124, "Show No. 315", None));
			assert_eq!(episodes[106], Episode::new(4, 8, 748125, "Show No. 414", None));
			assert_eq!(episodes[107], Episode::new(4, 9, 748126, "Show No. 318", None));
			assert_eq!(episodes[108], Episode::new(4, 10, 748097, "Show No. 419", None));
			assert_eq!(episodes[109], Episode::new(4, 11, 748098, "Show No. 422", None));
			assert_eq!(episodes[110], Episode::new(4, 12, 748099, "Show No. 337", None));
			assert_eq!(episodes[111], Episode::new(4, 13, 748100, "Show No. 418", None));
			assert_eq!(episodes[112], Episode::new(4, 14, 748101, "Show No. 507", None));
			assert_eq!(episodes[113], Episode::new(4, 15, 748102, "Show No. 503", None));
			assert_eq!(episodes[114], Episode::new(4, 16, 748103, "Show No. 417", None));
			assert_eq!(episodes[115], Episode::new(4, 17, 748104, "Show No. 412", None));
			assert_eq!(episodes[116], Episode::new(4, 18, 748105, "Show No. 421", None));
			assert_eq!(episodes[117], Episode::new(4, 19, 748106, "Show No. 411", None));
			assert_eq!(episodes[118], Episode::new(4, 20, 748108, "Show No. 415", None));
			assert_eq!(episodes[119], Episode::new(4, 21, 748109, "Show No. 409", None));
			assert_eq!(episodes[120], Episode::new(4, 22, 748110, "Show No. 423", None));
			assert_eq!(episodes[121], Episode::new(4, 23, 748111, "Show No. 416", None));
			assert_eq!(episodes[122], Episode::new(4, 24, 748112, "Show No. 508", None));
			assert_eq!(episodes[123], Episode::new(4, 25, 748113, "Show No. 403", None));
			assert_eq!(episodes[124], Episode::new(4, 26, 748114, "Show No. 404", None));
			assert_eq!(episodes[125], Episode::new(4, 27, 748115, "Show No. 432", None));
			assert_eq!(episodes[126], Episode::new(4, 28, 748116, "Show No. 431", None));
			assert_eq!(episodes[127], Episode::new(4, 29, 748117, "Show No. 402", None));
			assert_eq!(episodes[128], Episode::new(4, 30, 748119, "Show No. 429", None));
			assert_eq!(episodes[129], Episode::new(4, 31, 748120, "Show No. 427", None));
			assert_eq!(episodes[130], Episode::new(5, 1, 748127, "Show No. 511", None));
			assert_eq!(episodes[131], Episode::new(5, 2, 748138, "Show No. 509", None));
			assert_eq!(episodes[132], Episode::new(5, 3, 748149, "Show No. 406", None));
			assert_eq!(episodes[133], Episode::new(5, 4, 748155, "Show No. 512", None));
			assert_eq!(episodes[134], Episode::new(5, 5, 748156, "Show No. 425", None));
			assert_eq!(episodes[135], Episode::new(5, 6, 748157, "Show No. 433", None));
			assert_eq!(episodes[136], Episode::new(5, 7, 748158, "Show No. 501", None));
			assert_eq!(episodes[137], Episode::new(5, 8, 748159, "Show No. 506", None));
			assert_eq!(episodes[138], Episode::new(5, 9, 748160, "Show No. 515", None));
			assert_eq!(episodes[139], Episode::new(5, 10, 748128, "Show No. 504", None));
			assert_eq!(episodes[140], Episode::new(5, 11, 748129, "Show No. 517", None));
			assert_eq!(episodes[141], Episode::new(5, 12, 748130, "Show No. 510", None));
			assert_eq!(episodes[142], Episode::new(5, 13, 748131, "Show No. 408", None));
			assert_eq!(episodes[143], Episode::new(5, 14, 748132, "Show No. 513", None));
			assert_eq!(episodes[144], Episode::new(5, 15, 748133, "Show No. 514", None));
			assert_eq!(episodes[145], Episode::new(5, 16, 748134, "Show No. 518", None));
			assert_eq!(episodes[146], Episode::new(5, 17, 748135, "Show No. 521", None));
			assert_eq!(episodes[147], Episode::new(5, 18, 748136, "Show No. 609", None));
			assert_eq!(episodes[148], Episode::new(5, 19, 748137, "Show No. 610", None));
			assert_eq!(episodes[149], Episode::new(5, 20, 748139, "Show No. 603", None));
			assert_eq!(episodes[150], Episode::new(5, 21, 748140, "Show No. 611", None));
			assert_eq!(episodes[151], Episode::new(5, 22, 748141, "Show No. 607", None));
			assert_eq!(episodes[152], Episode::new(5, 23, 748142, "Show No. 519", None));
			assert_eq!(episodes[153], Episode::new(5, 24, 748143, "Show No. 316", None));
			assert_eq!(episodes[154], Episode::new(5, 25, 748144, "Show No. 502", None));
			assert_eq!(episodes[155], Episode::new(5, 26, 748145, "Show No. 516", None));
			assert_eq!(episodes[156], Episode::new(5, 27, 748146, "Show No. 601", None));
			assert_eq!(episodes[157], Episode::new(5, 28, 748147, "Show No. 608", None));
			assert_eq!(episodes[158], Episode::new(5, 29, 748148, "Show No. 606", None));
			assert_eq!(episodes[159], Episode::new(5, 30, 748150, "Show No. 612", None));
			assert_eq!(episodes[160], Episode::new(5, 31, 748151, "Show No. 520", None));
			assert_eq!(episodes[161], Episode::new(5, 32, 748152, "Show No. 426", None));
			assert_eq!(episodes[162], Episode::new(5, 33, 748153, "Show No. 605", None));
			assert_eq!(episodes[163], Episode::new(5, 34, 748154, "Show No. 604", None));
			assert_eq!(episodes[164], Episode::new(6, 1, 748161, "Show No. 602", None));
			assert_eq!(episodes[165], Episode::new(6, 2, 748164, "Show No. 424", None));
			assert_eq!(episodes[166], Episode::new(6, 3, 748165, "Show No. 348", None));
			assert_eq!(episodes[167], Episode::new(6, 4, 748166, "Show No. 347", None));
			assert_eq!(episodes[168], Episode::new(6, 5, 748167, "Show No. 345", None));
			assert_eq!(episodes[169], Episode::new(6, 6, 748168, "Show No. 505", None));
			assert_eq!(episodes[170], Episode::new(6, 7, 748169, "Show No. 430", None));
			assert_eq!(episodes[171], Episode::new(6, 8, 748170, "Show No. 314", None));
			assert_eq!(episodes[172], Episode::new(6, 9, 748171, "Show No. 346", None));
			assert_eq!(episodes[173], Episode::new(6, 10, 748162, "Show No. 428", None));
			assert_eq!(episodes[174], Episode::new(7, 1, 748172, "Show No. 7001", None));
			assert_eq!(episodes[175], Episode::new(7, 2, 748181, "Show No. 7002", None));
			assert_eq!(episodes[176], Episode::new(7, 3, 748186, "Show No. 7003", None));
			assert_eq!(episodes[177], Episode::new(7, 4, 748187, "Show No. 7004", None));
			assert_eq!(episodes[178], Episode::new(7, 5, 748188, "Show No. 7005", None));
			assert_eq!(episodes[179], Episode::new(7, 6, 795028, "Show No. 711", None));
			assert_eq!(episodes[180], Episode::new(7, 7, 748189, "Show No. 7007", None));
			assert_eq!(episodes[181], Episode::new(7, 8, 748190, "Show No. 7008", None));
			assert_eq!(episodes[182], Episode::new(7, 9, 748191, "Show No. 7009", None));
			assert_eq!(episodes[183], Episode::new(7, 10, 748173, "Show No. 7010", None));
			assert_eq!(episodes[184], Episode::new(7, 11, 748174, "Show No. 7011", None));
			assert_eq!(episodes[185], Episode::new(7, 12, 748175, "Show No. 7012", None));
			assert_eq!(episodes[186], Episode::new(7, 13, 1119576, "Show No. 7013", None));
			assert_eq!(episodes[187], Episode::new(7, 14, 748176, "Show No. 7014", None));
			assert_eq!(episodes[188], Episode::new(7, 15, 748177, "Show No. 7015", None));
			assert_eq!(episodes[189], Episode::new(7, 16, 1119577, "Show No. 7016", None));
			assert_eq!(episodes[190], Episode::new(7, 17, 748178, "Show No. 7017", None));
			assert_eq!(episodes[191], Episode::new(7, 18, 748179, "Show No. 7018", None));
			assert_eq!(episodes[192], Episode::new(7, 19, 748180, "Show No. 7019", None));
			assert_eq!(episodes[193], Episode::new(7, 20, 748182, "Show No. 7020", None));
			assert_eq!(episodes[194], Episode::new(7, 21, 748183, "Show No. 7021", None));
			assert_eq!(episodes[195], Episode::new(7, 22, 748184, "Show No. 7022", None));
			assert_eq!(episodes[196], Episode::new(7, 23, 795027, "Show No. 7023", None));
			assert_eq!(episodes[197], Episode::new(7, 24, 1119578, "Show No. 7024", None));
			assert_eq!(episodes[198], Episode::new(7, 25, 748185, "Show No. 7025", None));
			assert_eq!(episodes[199], Episode::new(8, 1, 748192, "Show No. 8001", None));
			assert_eq!(episodes[200], Episode::new(8, 2, 748193, "Show No. 8002", None));
			assert_eq!(episodes[201], Episode::new(8, 3, 748194, "Show No. 8003", None));
			assert_eq!(episodes[202], Episode::new(8, 4, 748195, "Show No. 8004", None));
			assert_eq!(episodes[203], Episode::new(8, 5, 748196, "Show No. 8005", None));
			assert_eq!(episodes[204], Episode::new(8, 6, 748197, "Show No. 8006", None));
			assert_eq!(episodes[205], Episode::new(8, 7, 918397, "Show No. 8007", None));
			assert_eq!(episodes[206], Episode::new(8, 8, 918393, "Show No. 8008", None));
			assert_eq!(episodes[207], Episode::new(8, 9, 918394, "Show No. 8009", None));
			assert_eq!(episodes[208], Episode::new(8, 10, 918392, "Show No. 8010", None));
			assert_eq!(episodes[209], Episode::new(8, 11, 918398, "Show No. 8011", None));
			assert_eq!(episodes[210], Episode::new(8, 12, 918395, "Show No. 8012", None));
			assert_eq!(episodes[211], Episode::new(8, 13, 918391, "Show No. 8013", None));
			assert_eq!(episodes[212], Episode::new(8, 14, 918389, "Show No. 8014", None));
			assert_eq!(episodes[213], Episode::new(8, 15, 918396, "Show No. 8015", None));
			assert_eq!(episodes[214], Episode::new(8, 16, 918390, "Show No. 8016", None));
			assert_eq!(episodes[215], Episode::new(8, 17, 2015197, "Show No. 8017", None));
			assert_eq!(episodes[216], Episode::new(8, 18, 1910467, "Show No. 8018", None));
			assert_eq!(episodes[217], Episode::new(8, 19, 1914143, "Show No. 8019", None));
			assert_eq!(episodes[218], Episode::new(8, 20, 1918587, "Show No. 8020", None));
			assert_eq!(episodes[219], Episode::new(8, 21, 1922480, "Show No. 8021", None));
			// }}}

			let episodes = client.get_episodes(2364582).await.unwrap();
			assert_eq!(episodes.len(), 136); // {{{
			assert_eq!(episodes[0], Episode::new(1, 1, 2912308, "Pilot", Some(44)));
			assert_eq!(episodes[1], Episode::new(1, 2, 3130816, "0-8-4", Some(42)));
			assert_eq!(episodes[2], Episode::new(1, 3, 3130186, "The Asset", Some(43)));
			assert_eq!(episodes[3], Episode::new(1, 4, 3218318, "Eye Spy", Some(43)));
			assert_eq!(episodes[4], Episode::new(1, 5, 3229400, "Girl in the Flower Dress", Some(43)));
			assert_eq!(episodes[5], Episode::new(1, 6, 3229404, "FZZT", Some(43)));
			assert_eq!(episodes[6], Episode::new(1, 7, 3229406, "The Hub", Some(43)));
			assert_eq!(episodes[7], Episode::new(1, 8, 3252298, "The Well", Some(43)));
			assert_eq!(episodes[8], Episode::new(1, 9, 3252332, "Repairs", Some(43)));
			assert_eq!(episodes[9], Episode::new(1, 10, 3252318, "The Bridge", Some(42)));
			assert_eq!(episodes[10], Episode::new(1, 11, 3307564, "The Magical Place", Some(43)));
			assert_eq!(episodes[11], Episode::new(1, 12, 3307568, "Seeds", Some(43)));
			assert_eq!(episodes[12], Episode::new(1, 13, 3307566, "T.R.A.C.K.S.", Some(43)));
			assert_eq!(episodes[13], Episode::new(1, 14, 3476430, "T.A.H.I.T.I.", Some(45)));
			assert_eq!(episodes[14], Episode::new(1, 15, 3473818, "Yes Men", Some(45)));
			assert_eq!(episodes[15], Episode::new(1, 16, 3539224, "End of the Beginning", Some(45)));
			assert_eq!(episodes[16], Episode::new(1, 17, 3539406, "Turn, Turn, Turn", Some(44)));
			assert_eq!(episodes[17], Episode::new(1, 18, 3609126, "Providence", Some(45)));
			assert_eq!(episodes[18], Episode::new(1, 19, 3609154, "The Only Light in the Darkness", Some(41)));
			assert_eq!(episodes[19], Episode::new(1, 20, 3609164, "Nothing Personal", Some(41)));
			assert_eq!(episodes[20], Episode::new(1, 21, 3609168, "Ragtag", Some(42)));
			assert_eq!(episodes[21], Episode::new(1, 22, 3609170, "Beginning of the End", Some(44)));
			assert_eq!(episodes[22], Episode::new(2, 1, 3721000, "Shadows", Some(43)));
			assert_eq!(episodes[23], Episode::new(2, 2, 3997450, "Heavy Is the Head", Some(42)));
			assert_eq!(episodes[24], Episode::new(2, 3, 3997456, "Making Friends and Influencing People", Some(43)));
			assert_eq!(episodes[25], Episode::new(2, 4, 3997452, "Face My Enemy", Some(42)));
			assert_eq!(episodes[26], Episode::new(2, 5, 3970174, "A Hen in the Wolf House", Some(44)));
			assert_eq!(episodes[27], Episode::new(2, 6, 4078580, "A Fractured House", Some(41)));
			assert_eq!(episodes[28], Episode::new(2, 7, 4082652, "The Writing on the Wall", Some(42)));
			assert_eq!(episodes[29], Episode::new(2, 8, 4082656, "The Things We Bury", Some(44)));
			assert_eq!(episodes[30], Episode::new(2, 9, 4120698, "Ye Who Enter Here", Some(43)));
			assert_eq!(episodes[31], Episode::new(2, 10, 4120702, "What They Become", Some(43)));
			assert_eq!(episodes[32], Episode::new(2, 11, 4120704, "Aftershocks", Some(43)));
			assert_eq!(episodes[33], Episode::new(2, 12, 4120714, "Who You Really Are", Some(42)));
			assert_eq!(episodes[34], Episode::new(2, 13, 4120718, "One of Us", Some(43)));
			assert_eq!(episodes[35], Episode::new(2, 14, 4120720, "Love in the Time of Hydra", Some(44)));
			assert_eq!(episodes[36], Episode::new(2, 15, 4120722, "One Door Closes", Some(42)));
			assert_eq!(episodes[37], Episode::new(2, 16, 4120726, "Afterlife", Some(44)));
			assert_eq!(episodes[38], Episode::new(2, 17, 4120730, "Melinda", Some(42)));
			assert_eq!(episodes[39], Episode::new(2, 18, 4120738, "Frenemy of My Enemy", Some(43)));
			assert_eq!(episodes[40], Episode::new(2, 19, 4120734, "The Dirty Half Dozen", Some(42)));
			assert_eq!(episodes[41], Episode::new(2, 20, 4122454, "Scars", Some(44)));
			assert_eq!(episodes[42], Episode::new(2, 21, 4122456, "S.O.S. Part 1", Some(43)));
			assert_eq!(episodes[43], Episode::new(2, 22, 4110822, "S.O.S. Part 2", Some(43)));
			assert_eq!(episodes[44], Episode::new(3, 1, 4667540, "Laws of Nature", Some(44)));
			assert_eq!(episodes[45], Episode::new(3, 2, 4706470, "Purpose in the Machine", Some(43)));
			assert_eq!(episodes[46], Episode::new(3, 3, 4727950, "A Wanted (Inhu)Man", Some(42)));
			assert_eq!(episodes[47], Episode::new(3, 4, 4732884, "Devils You Know", Some(43)));
			assert_eq!(episodes[48], Episode::new(3, 5, 4742110, "4,722 Hours", Some(43)));
			assert_eq!(episodes[49], Episode::new(3, 6, 4760490, "Among Us Hide...", Some(42)));
			assert_eq!(episodes[50], Episode::new(3, 7, 4765060, "Chaos Theory", Some(43)));
			assert_eq!(episodes[51], Episode::new(3, 8, 4769592, "Many Heads, One Tale", Some(43)));
			assert_eq!(episodes[52], Episode::new(3, 9, 4778366, "Closure", Some(43)));
			assert_eq!(episodes[53], Episode::new(3, 10, 4786258, "Maveth", Some(42)));
			assert_eq!(episodes[54], Episode::new(3, 11, 4794580, "Bouncing Back", Some(43)));
			assert_eq!(episodes[55], Episode::new(3, 12, 4800060, "The Inside Man", Some(43)));
			assert_eq!(episodes[56], Episode::new(3, 13, 4809604, "Parting Shot", Some(42)));
			assert_eq!(episodes[57], Episode::new(3, 14, 4819384, "Watchdogs", Some(42)));
			assert_eq!(episodes[58], Episode::new(3, 15, 4829172, "Spacetime", Some(43)));
			assert_eq!(episodes[59], Episode::new(3, 16, 4838204, "Paradise Lost", Some(43)));
			assert_eq!(episodes[60], Episode::new(3, 17, 4842854, "The Team", Some(43)));
			assert_eq!(episodes[61], Episode::new(3, 18, 4857186, "The Singularity", Some(42)));
			assert_eq!(episodes[62], Episode::new(3, 19, 4871764, "Failed Experiments", Some(41)));
			assert_eq!(episodes[63], Episode::new(3, 20, 4877412, "Emancipation", Some(43)));
			assert_eq!(episodes[64], Episode::new(3, 21, 4883818, "Absolution", Some(43)));
			assert_eq!(episodes[65], Episode::new(3, 22, 4899794, "Ascension", Some(43)));
			assert_eq!(episodes[66], Episode::new(4, 1, 5518974, "The Ghost", Some(43)));
			assert_eq!(episodes[67], Episode::new(4, 2, 5916264, "Meet the New Boss", Some(43)));
			assert_eq!(episodes[68], Episode::new(4, 3, 5916458, "Uprising", Some(42)));
			assert_eq!(episodes[69], Episode::new(4, 4, 5916482, "Let Me Stand Next to Your Fire", Some(43)));
			assert_eq!(episodes[70], Episode::new(4, 5, 5916512, "Lockup", Some(42)));
			assert_eq!(episodes[71], Episode::new(4, 6, 5916528, "The Good Samaritan", Some(41)));
			assert_eq!(episodes[72], Episode::new(4, 7, 5916746, "Deals with Our Devils", Some(43)));
			assert_eq!(episodes[73], Episode::new(4, 8, 5916754, "The Laws of Inferno Dynamics", Some(42)));
			assert_eq!(episodes[74], Episode::new(4, 9, 5916768, "Broken Promises", Some(43)));
			assert_eq!(episodes[75], Episode::new(4, 10, 5916774, "The Patriot", Some(42)));
			assert_eq!(episodes[76], Episode::new(4, 11, 5916778, "Wake Up", Some(42)));
			assert_eq!(episodes[77], Episode::new(4, 12, 5916782, "Hot Potato Soup", Some(42)));
			assert_eq!(episodes[78], Episode::new(4, 13, 5916792, "BOOM", Some(43)));
			assert_eq!(episodes[79], Episode::new(4, 14, 5916818, "The Man Behind the Shield", Some(42)));
			assert_eq!(episodes[80], Episode::new(4, 15, 5916824, "Self Control", Some(43)));
			assert_eq!(episodes[81], Episode::new(4, 16, 5916832, "What If...", Some(42)));
			assert_eq!(episodes[82], Episode::new(4, 17, 5916836, "Identity and Change", Some(43)));
			assert_eq!(episodes[83], Episode::new(4, 18, 5916840, "No Regrets", Some(43)));
			assert_eq!(episodes[84], Episode::new(4, 19, 5916846, "All the Madame\'s Men", Some(43)));
			assert_eq!(episodes[85], Episode::new(4, 20, 5916856, "Farewell, Cruel World!", Some(41)));
			assert_eq!(episodes[86], Episode::new(4, 21, 5916876, "The Return", Some(43)));
			assert_eq!(episodes[87], Episode::new(4, 22, 5916882, "World\'s End", Some(43)));
			assert_eq!(episodes[88], Episode::new(5, 1, 6878538, "Orientation: Part 1", Some(42)));
			assert_eq!(episodes[89], Episode::new(5, 2, 7178426, "Orientation: Part 2", Some(43)));
			assert_eq!(episodes[90], Episode::new(5, 3, 7183060, "A Life Spent", Some(42)));
			assert_eq!(episodes[91], Episode::new(5, 4, 7183068, "A Life Earned", Some(43)));
			assert_eq!(episodes[92], Episode::new(5, 5, 7183072, "Rewind", Some(42)));
			assert_eq!(episodes[93], Episode::new(5, 6, 7183078, "Fun & Games", Some(42)));
			assert_eq!(episodes[94], Episode::new(5, 7, 7183084, "Together or Not at All", Some(40)));
			assert_eq!(episodes[95], Episode::new(5, 8, 7183086, "The Last Day", Some(42)));
			assert_eq!(episodes[96], Episode::new(5, 9, 7183090, "Best Laid Plans", Some(41)));
			assert_eq!(episodes[97], Episode::new(5, 10, 7183092, "Past Life", Some(42)));
			assert_eq!(episodes[98], Episode::new(5, 11, 7183100, "All the Comforts of Home", Some(42)));
			assert_eq!(episodes[99], Episode::new(5, 12, 7183104, "The Real Deal", Some(43)));
			assert_eq!(episodes[100], Episode::new(5, 13, 7183106, "Principia", Some(42)));
			assert_eq!(episodes[101], Episode::new(5, 14, 7183112, "The Devil Complex", Some(42)));
			assert_eq!(episodes[102], Episode::new(5, 15, 7183116, "Rise and Shine", Some(43)));
			assert_eq!(episodes[103], Episode::new(5, 16, 7183120, "Inside Voices", Some(42)));
			assert_eq!(episodes[104], Episode::new(5, 17, 7183126, "The Honeymoon", Some(42)));
			assert_eq!(episodes[105], Episode::new(5, 18, 7183130, "All Roads Lead...", Some(41)));
			assert_eq!(episodes[106], Episode::new(5, 19, 7183140, "Option Two", Some(42)));
			assert_eq!(episodes[107], Episode::new(5, 20, 7183150, "The One Who Will Save Us All", Some(42)));
			assert_eq!(episodes[108], Episode::new(5, 21, 7183156, "The Force of Gravity", Some(43)));
			assert_eq!(episodes[109], Episode::new(5, 22, 7183160, "The End", Some(43)));
			assert_eq!(episodes[110], Episode::new(6, 1, 8416608, "Missing Pieces", Some(42)));
			assert_eq!(episodes[111], Episode::new(6, 2, 8416896, "Window of Opportunity", Some(43)));
			assert_eq!(episodes[112], Episode::new(6, 3, 8416904, "Fear and Loathing on the Planet of Kitson", Some(43)));
			assert_eq!(episodes[113], Episode::new(6, 4, 8416910, "Code Yellow", Some(41)));
			assert_eq!(episodes[114], Episode::new(6, 5, 8416912, "The Other Thing", Some(42)));
			assert_eq!(episodes[115], Episode::new(6, 6, 8416914, "Inescapable", Some(42)));
			assert_eq!(episodes[116], Episode::new(6, 7, 8416916, "Toldja", Some(42)));
			assert_eq!(episodes[117], Episode::new(6, 8, 8422650, "Collision Course (Part I)", Some(42)));
			assert_eq!(episodes[118], Episode::new(6, 9, 8422652, "Collision Course (Part II)", Some(42)));
			assert_eq!(episodes[119], Episode::new(6, 10, 8422656, "Leap", Some(42)));
			assert_eq!(episodes[120], Episode::new(6, 11, 8422660, "From the Ashes", Some(42)));
			assert_eq!(episodes[121], Episode::new(6, 12, 8422664, "The Sign", Some(41)));
			assert_eq!(episodes[122], Episode::new(6, 13, 8422666, "New Life", Some(43)));
			assert_eq!(episodes[123], Episode::new(7, 1, 9292314, "The New Deal", Some(43)));
			assert_eq!(episodes[124], Episode::new(7, 2, 9307248, "Know Your Onions", Some(42)));
			assert_eq!(episodes[125], Episode::new(7, 3, 9307250, "Alien Commies from the Future!", Some(42)));
			assert_eq!(episodes[126], Episode::new(7, 4, 9307252, "Out of the Past", Some(43)));
			assert_eq!(episodes[127], Episode::new(7, 5, 9307254, "A Trout in the Milk", Some(42)));
			assert_eq!(episodes[128], Episode::new(7, 6, 9307256, "Adapt or Die", Some(43)));
			assert_eq!(episodes[129], Episode::new(7, 7, 9307258, "The Totally Excellent Adventures of Mack and the D", Some(43)));
			assert_eq!(episodes[130], Episode::new(7, 8, 9307260, "After, Before", Some(42)));
			assert_eq!(episodes[131], Episode::new(7, 9, 9307262, "As I Have Always Been", Some(43)));
			assert_eq!(episodes[132], Episode::new(7, 10, 9307264, "Stolen", Some(41)));
			assert_eq!(episodes[133], Episode::new(7, 11, 9307266, "Brand New Day", Some(43)));
			assert_eq!(episodes[134], Episode::new(7, 12, 9307268, "The End Is at Hand", Some(43)));
			assert_eq!(episodes[135], Episode::new(7, 13, 9535202, "What We\'re Fighting For", Some(43)));
			// }}}
		}
	} // }}}

	#[tokio::test]
	async fn episodes_by_show_multi() /* {{{ */ {
		let (_dir, mut client) = client().await;

		for i in 0..5 {
			println!(">>> {}", i);
			let mut episodes_by_show = client.get_episodes_by_show(&[1442437, 944947, 460681]).await.unwrap();
			assert_eq!(episodes_by_show.len(), 3);

			let mut episodes = episodes_by_show.remove(&1442437).unwrap();
			episodes.sort_unstable();
			assert_eq!(episodes.len(), 250); // {{{
			assert_eq!(episodes[0], Episode::new(1, 1, 1444504, "Pilot", Some(23)));
			assert_eq!(episodes[1], Episode::new(1, 2, 1492032, "The Bicycle Thief", Some(21)));
			assert_eq!(episodes[2], Episode::new(1, 3, 1520618, "Come Fly with Me", Some(21)));
			assert_eq!(episodes[3], Episode::new(1, 4, 1523089, "The Incident", Some(21)));
			assert_eq!(episodes[4], Episode::new(1, 5, 1530593, "Coal Digger", Some(21)));
			assert_eq!(episodes[5], Episode::new(1, 6, 1523088, "Run for Your Wife", Some(21)));
			assert_eq!(episodes[6], Episode::new(1, 7, 1530594, "En Garde", Some(20)));
			assert_eq!(episodes[7], Episode::new(1, 8, 1530595, "Great Expectations", Some(21)));
			assert_eq!(episodes[8], Episode::new(1, 9, 1528496, "Fizbo", Some(21)));
			assert_eq!(episodes[9], Episode::new(1, 10, 1536241, "Undeck the Halls", Some(21)));
			assert_eq!(episodes[10], Episode::new(1, 11, 1536765, "Up All Night", Some(21)));
			assert_eq!(episodes[11], Episode::new(1, 12, 1581054, "Not in My House", Some(21)));
			assert_eq!(episodes[12], Episode::new(1, 13, 1569106, "Fifteen Percent", Some(21)));
			assert_eq!(episodes[13], Episode::new(1, 14, 1580159, "Moon Landing", Some(21)));
			assert_eq!(episodes[14], Episode::new(1, 15, 1583904, "My Funky Valentine", Some(21)));
			assert_eq!(episodes[15], Episode::new(1, 16, 1594366, "Fears", Some(21)));
			assert_eq!(episodes[16], Episode::new(1, 17, 1596476, "Truth Be Told", Some(21)));
			assert_eq!(episodes[17], Episode::new(1, 18, 1615677, "Starry Night", Some(21)));
			assert_eq!(episodes[18], Episode::new(1, 19, 1617483, "Game Changer", Some(21)));
			assert_eq!(episodes[19], Episode::new(1, 20, 1625484, "Benched", Some(21)));
			assert_eq!(episodes[20], Episode::new(1, 21, 1625485, "Travels with Scout", Some(22)));
			assert_eq!(episodes[21], Episode::new(1, 22, 1643884, "Airport 2010", Some(21)));
			assert_eq!(episodes[22], Episode::new(1, 23, 1643885, "Hawaii", Some(21)));
			assert_eq!(episodes[23], Episode::new(1, 24, 1624700, "Family Portrait", Some(21)));
			assert_eq!(episodes[24], Episode::new(2, 1, 1627460, "The Old Wagon", Some(21)));
			assert_eq!(episodes[25], Episode::new(2, 2, 1627459, "The Kiss", Some(21)));
			assert_eq!(episodes[26], Episode::new(2, 3, 1627443, "Earthquake", Some(21)));
			assert_eq!(episodes[27], Episode::new(2, 4, 1627454, "Strangers on a Treadmill", Some(21)));
			assert_eq!(episodes[28], Episode::new(2, 5, 1627461, "Unplugged", Some(21)));
			assert_eq!(episodes[29], Episode::new(2, 6, 1627464, "Halloween", Some(21)));
			assert_eq!(episodes[30], Episode::new(2, 7, 1627458, "Chirp", Some(21)));
			assert_eq!(episodes[31], Episode::new(2, 8, 1627445, "Manny Get Your Gun", Some(20)));
			assert_eq!(episodes[32], Episode::new(2, 9, 1627462, "Mother Tucker", Some(21)));
			assert_eq!(episodes[33], Episode::new(2, 10, 1779206, "Dance Dance Revelation", Some(21)));
			assert_eq!(episodes[34], Episode::new(2, 11, 1627446, "Slow Down Your Neighbors", Some(21)));
			assert_eq!(episodes[35], Episode::new(2, 12, 1627444, "Our Children, Ourselves", Some(21)));
			assert_eq!(episodes[36], Episode::new(2, 13, 1627447, "Caught in the Act", Some(21)));
			assert_eq!(episodes[37], Episode::new(2, 14, 1627448, "Bixby\'s Back", Some(21)));
			assert_eq!(episodes[38], Episode::new(2, 15, 1627449, "Princess Party", Some(21)));
			assert_eq!(episodes[39], Episode::new(2, 16, 1836487, "Regrets Only", Some(21)));
			assert_eq!(episodes[40], Episode::new(2, 17, 1841231, "Two Monkeys and a Panda", Some(21)));
			assert_eq!(episodes[41], Episode::new(2, 18, 1627452, "Boys\' Night", Some(21)));
			assert_eq!(episodes[42], Episode::new(2, 19, 1627453, "The Musical Man", Some(21)));
			assert_eq!(episodes[43], Episode::new(2, 20, 1627455, "Someone to Watch Over Lily", Some(21)));
			assert_eq!(episodes[44], Episode::new(2, 21, 1627456, "Mother\'s Day", Some(21)));
			assert_eq!(episodes[45], Episode::new(2, 22, 1627457, "Good Cop Bad Dog", Some(21)));
			assert_eq!(episodes[46], Episode::new(2, 23, 1900347, "See You Next Fall", Some(21)));
			assert_eq!(episodes[47], Episode::new(2, 24, 1900348, "The One That Got Away", Some(21)));
			assert_eq!(episodes[48], Episode::new(3, 1, 2047147, "Dude Ranch", Some(21)));
			assert_eq!(episodes[49], Episode::new(3, 2, 2044356, "When Good Kids Go Bad", Some(21)));
			assert_eq!(episodes[50], Episode::new(3, 3, 1983675, "Phil on Wire", Some(21)));
			assert_eq!(episodes[51], Episode::new(3, 4, 2065470, "Door to Door", Some(21)));
			assert_eq!(episodes[52], Episode::new(3, 5, 2064470, "Hit and Run", Some(21)));
			assert_eq!(episodes[53], Episode::new(3, 6, 2081815, "Go Bullfrogs!", Some(21)));
			assert_eq!(episodes[54], Episode::new(3, 7, 2089248, "Treehouse", Some(22)));
			assert_eq!(episodes[55], Episode::new(3, 8, 2089249, "After the Fire", Some(22)));
			assert_eq!(episodes[56], Episode::new(3, 9, 2087341, "Punkin Chunkin", Some(21)));
			assert_eq!(episodes[57], Episode::new(3, 10, 2096098, "Express Christmas", Some(22)));
			assert_eq!(episodes[58], Episode::new(3, 11, 2105660, "Lifetime Supply", Some(22)));
			assert_eq!(episodes[59], Episode::new(3, 12, 2110811, "Egg Drop", Some(22)));
			assert_eq!(episodes[60], Episode::new(3, 13, 2181257, "Little Bo Bleep", Some(22)));
			assert_eq!(episodes[61], Episode::new(3, 14, 2189485, "Me? Jealous?", Some(21)));
			assert_eq!(episodes[62], Episode::new(3, 15, 2209159, "Aunt Mommy", Some(20)));
			assert_eq!(episodes[63], Episode::new(3, 16, 2237415, "Virgin Territory", Some(22)));
			assert_eq!(episodes[64], Episode::new(3, 17, 2244103, "Leap Day", Some(22)));
			assert_eq!(episodes[65], Episode::new(3, 18, 2239817, "Send Out the Clowns", Some(21)));
			assert_eq!(episodes[66], Episode::new(3, 19, 2305713, "Election Day", Some(21)));
			assert_eq!(episodes[67], Episode::new(3, 20, 2305715, "The Last Walt", Some(20)));
			assert_eq!(episodes[68], Episode::new(3, 21, 2247703, "Planes, Trains and Cars", Some(21)));
			assert_eq!(episodes[69], Episode::new(3, 22, 2342897, "Disneyland", Some(21)));
			assert_eq!(episodes[70], Episode::new(3, 23, 2303805, "Tableau Vivant", Some(20)));
			assert_eq!(episodes[71], Episode::new(3, 24, 2358075, "Baby on Board", Some(20)));
			assert_eq!(episodes[72], Episode::new(4, 1, 2397171, "Bringing Up Baby", Some(22)));
			assert_eq!(episodes[73], Episode::new(4, 2, 2360776, "Schooled", Some(22)));
			assert_eq!(episodes[74], Episode::new(4, 3, 2362724, "Snip", Some(22)));
			assert_eq!(episodes[75], Episode::new(4, 4, 2404438, "The Butler\'s Escape", Some(22)));
			assert_eq!(episodes[76], Episode::new(4, 5, 2393824, "Open House of Horrors", Some(21)));
			assert_eq!(episodes[77], Episode::new(4, 6, 2488642, "Yard Sale", Some(21)));
			assert_eq!(episodes[78], Episode::new(4, 7, 2488644, "Arrested", Some(22)));
			assert_eq!(episodes[79], Episode::new(4, 8, 2488652, "Mistery Date", Some(22)));
			assert_eq!(episodes[80], Episode::new(4, 9, 2459378, "When a Tree Falls", Some(22)));
			assert_eq!(episodes[81], Episode::new(4, 10, 2543110, "Diamond in the Rough", Some(22)));
			assert_eq!(episodes[82], Episode::new(4, 11, 2586840, "New Year\'s Eve", Some(22)));
			assert_eq!(episodes[83], Episode::new(4, 12, 2507294, "Party Crasher", Some(22)));
			assert_eq!(episodes[84], Episode::new(4, 13, 2517458, "Fulgencio", Some(22)));
			assert_eq!(episodes[85], Episode::new(4, 14, 2568884, "A Slight at the Opera", Some(22)));
			assert_eq!(episodes[86], Episode::new(4, 15, 2691640, "Heart Broken", Some(22)));
			assert_eq!(episodes[87], Episode::new(4, 16, 2708166, "Bad Hair Day", Some(22)));
			assert_eq!(episodes[88], Episode::new(4, 17, 2573094, "Best Men", Some(22)));
			assert_eq!(episodes[89], Episode::new(4, 18, 2586848, "The Wow Factor", Some(22)));
			assert_eq!(episodes[90], Episode::new(4, 19, 2799086, "The Future Dunphys", Some(22)));
			assert_eq!(episodes[91], Episode::new(4, 20, 2799078, "Flip Flop", Some(22)));
			assert_eq!(episodes[92], Episode::new(4, 21, 2716564, "Career Day", Some(22)));
			assert_eq!(episodes[93], Episode::new(4, 22, 2720510, "My Hero", Some(22)));
			assert_eq!(episodes[94], Episode::new(4, 23, 2814070, "Games People Play", Some(22)));
			assert_eq!(episodes[95], Episode::new(4, 24, 2770826, "Goodnight Gracie", Some(22)));
			assert_eq!(episodes[96], Episode::new(5, 1, 2911820, "Suddenly, Last Summer", Some(22)));
			assert_eq!(episodes[97], Episode::new(5, 2, 3116354, "First Days", Some(21)));
			assert_eq!(episodes[98], Episode::new(5, 3, 3172360, "Larry\'s Wife", Some(22)));
			assert_eq!(episodes[99], Episode::new(5, 4, 3183992, "Farm Strong", Some(21)));
			assert_eq!(episodes[100], Episode::new(5, 5, 3237940, "The Late Show", Some(21)));
			assert_eq!(episodes[101], Episode::new(5, 6, 3267504, "The Help", Some(22)));
			assert_eq!(episodes[102], Episode::new(5, 7, 3292104, "A Fair to Remember", Some(22)));
			assert_eq!(episodes[103], Episode::new(5, 8, 3293410, "ClosetCon \'13", Some(21)));
			assert_eq!(episodes[104], Episode::new(5, 9, 3266712, "The Big Game", Some(21)));
			assert_eq!(episodes[105], Episode::new(5, 10, 3335562, "The Old Man & the Tree", Some(21)));
			assert_eq!(episodes[106], Episode::new(5, 11, 3344996, "And One to Grow On", Some(21)));
			assert_eq!(episodes[107], Episode::new(5, 12, 3328416, "Under Pressure", Some(22)));
			assert_eq!(episodes[108], Episode::new(5, 13, 3410062, "Three Dinners", Some(21)));
			assert_eq!(episodes[109], Episode::new(5, 14, 3410066, "iSpy", Some(21)));
			assert_eq!(episodes[110], Episode::new(5, 15, 3508790, "The Feud", Some(21)));
			assert_eq!(episodes[111], Episode::new(5, 16, 3508794, "Spring-A-Ding-Fling", Some(21)));
			assert_eq!(episodes[112], Episode::new(5, 17, 3508792, "Other People\'s Children", Some(22)));
			assert_eq!(episodes[113], Episode::new(5, 18, 3508788, "Vegas", Some(21)));
			assert_eq!(episodes[114], Episode::new(5, 19, 3508796, "A Hard Jay\'s Night", Some(21)));
			assert_eq!(episodes[115], Episode::new(5, 20, 3675476, "Australia", Some(22)));
			assert_eq!(episodes[116], Episode::new(5, 21, 3684710, "Sleeper", Some(21)));
			assert_eq!(episodes[117], Episode::new(5, 22, 3684716, "Message Received", Some(22)));
			assert_eq!(episodes[118], Episode::new(5, 23, 3588986, "The Wedding, Part 1", Some(22)));
			assert_eq!(episodes[119], Episode::new(5, 24, 3687550, "The Wedding, Part 2", Some(21)));
			assert_eq!(episodes[120], Episode::new(6, 1, 3722370, "The Long Honeymoon", Some(21)));
			assert_eq!(episodes[121], Episode::new(6, 2, 4063326, "Don\'t Push", Some(22)));
			assert_eq!(episodes[122], Episode::new(6, 3, 4071558, "The Cold", Some(22)));
			assert_eq!(episodes[123], Episode::new(6, 4, 4085200, "Marco Polo", Some(22)));
			assert_eq!(episodes[124], Episode::new(6, 5, 4102996, "Won\'t You Be Our Neighbor", Some(22)));
			assert_eq!(episodes[125], Episode::new(6, 6, 4132862, "Halloween 3: AwesomeLand", Some(22)));
			assert_eq!(episodes[126], Episode::new(6, 7, 4166966, "Queer Eyes, Full Hearts", Some(21)));
			assert_eq!(episodes[127], Episode::new(6, 8, 4106068, "Three Turkeys", Some(21)));
			assert_eq!(episodes[128], Episode::new(6, 9, 4203910, "Strangers in the Night", Some(21)));
			assert_eq!(episodes[129], Episode::new(6, 10, 4241440, "Haley\'s 21st Birthday", Some(21)));
			assert_eq!(episodes[130], Episode::new(6, 11, 4303618, "The Day We Almost Died", Some(22)));
			assert_eq!(episodes[131], Episode::new(6, 12, 4303620, "The Big Guns", Some(21)));
			assert_eq!(episodes[132], Episode::new(6, 13, 4389738, "Rash Decisions", Some(22)));
			assert_eq!(episodes[133], Episode::new(6, 14, 4389740, "Valentine\'s Day 4: Twisted Sister", Some(22)));
			assert_eq!(episodes[134], Episode::new(6, 15, 4389742, "Fight or Flight", Some(21)));
			assert_eq!(episodes[135], Episode::new(6, 16, 4389744, "Connection Lost", Some(22)));
			assert_eq!(episodes[136], Episode::new(6, 17, 4454724, "Closet? You\'ll Love It!", Some(22)));
			assert_eq!(episodes[137], Episode::new(6, 18, 4464076, "Spring Break", Some(21)));
			assert_eq!(episodes[138], Episode::new(6, 19, 4464078, "Grill, Interrupted", Some(22)));
			assert_eq!(episodes[139], Episode::new(6, 20, 4454702, "Knock \'Em Down", Some(21)));
			assert_eq!(episodes[140], Episode::new(6, 21, 4464080, "Integrity", Some(22)));
			assert_eq!(episodes[141], Episode::new(6, 22, 4464088, "Patriot Games", Some(21)));
			assert_eq!(episodes[142], Episode::new(6, 23, 4464092, "Crying Out Loud", Some(22)));
			assert_eq!(episodes[143], Episode::new(6, 24, 4464094, "American Skyper", Some(21)));
			assert_eq!(episodes[144], Episode::new(7, 1, 4695160, "Summer Lovin\'", Some(22)));
			assert_eq!(episodes[145], Episode::new(7, 2, 4941946, "The Day Alex Left for College", Some(22)));
			assert_eq!(episodes[146], Episode::new(7, 3, 5056418, "The Closet Case", Some(22)));
			assert_eq!(episodes[147], Episode::new(7, 4, 5015956, "She Crazy", Some(22)));
			assert_eq!(episodes[148], Episode::new(7, 5, 5115000, "The Verdict", Some(22)));
			assert_eq!(episodes[149], Episode::new(7, 6, 5143558, "The More You Ignore Me", Some(21)));
			assert_eq!(episodes[150], Episode::new(7, 7, 5143564, "Phil\'s Sexy, Sexy House", Some(21)));
			assert_eq!(episodes[151], Episode::new(7, 8, 5197990, "Clean Out Your Junk Drawer", Some(21)));
			assert_eq!(episodes[152], Episode::new(7, 9, 5209922, "White Christmas", Some(21)));
			assert_eq!(episodes[153], Episode::new(7, 10, 5042714, "Playdates", Some(21)));
			assert_eq!(episodes[154], Episode::new(7, 11, 5042722, "Spread Your Wings", Some(21)));
			assert_eq!(episodes[155], Episode::new(7, 12, 5200458, "Clean for a Day", Some(21)));
			assert_eq!(episodes[156], Episode::new(7, 13, 5244906, "Thunk in the Trunk", Some(21)));
			assert_eq!(episodes[157], Episode::new(7, 14, 5259588, "The Storm", Some(21)));
			assert_eq!(episodes[158], Episode::new(7, 15, 5265080, "I Don\'t Know How She Does It", Some(21)));
			assert_eq!(episodes[159], Episode::new(7, 16, 5357382, "The Cover-Up", Some(21)));
			assert_eq!(episodes[160], Episode::new(7, 17, 5357386, "Express Yourself", Some(21)));
			assert_eq!(episodes[161], Episode::new(7, 18, 5357390, "The Party", Some(21)));
			assert_eq!(episodes[162], Episode::new(7, 19, 5357396, "Man Shouldn\'t Lie", Some(21)));
			assert_eq!(episodes[163], Episode::new(7, 20, 5357398, "Promposal", Some(21)));
			assert_eq!(episodes[164], Episode::new(7, 21, 5357400, "Crazy Train", Some(21)));
			assert_eq!(episodes[165], Episode::new(7, 22, 5357406, "Double Click", Some(21)));
			assert_eq!(episodes[166], Episode::new(8, 1, 5530750, "A Tale of Three Cities", Some(21)));
			assert_eq!(episodes[167], Episode::new(8, 2, 5357404, "A Stereotypical Day", Some(21)));
			assert_eq!(episodes[168], Episode::new(8, 3, 5042724, "Blindsided", Some(21)));
			assert_eq!(episodes[169], Episode::new(8, 4, 6014414, "Weathering Heights", Some(21)));
			assert_eq!(episodes[170], Episode::new(8, 5, 6074728, "Halloween 4: The Revenge of Rod Skyhook", Some(21)));
			assert_eq!(episodes[171], Episode::new(8, 6, 6074546, "Grab It", Some(21)));
			assert_eq!(episodes[172], Episode::new(8, 7, 6184702, "Thanksgiving Jamboree", Some(21)));
			assert_eq!(episodes[173], Episode::new(8, 8, 6189510, "The Alliance", Some(21)));
			assert_eq!(episodes[174], Episode::new(8, 9, 6254222, "Snow Ball", Some(21)));
			assert_eq!(episodes[175], Episode::new(8, 10, 6199718, "Ringmaster Keifth", Some(21)));
			assert_eq!(episodes[176], Episode::new(8, 11, 6300118, "Sarge & Pea", Some(21)));
			assert_eq!(episodes[177], Episode::new(8, 12, 6300120, "Do You Believe in Magic", Some(21)));
			assert_eq!(episodes[178], Episode::new(8, 13, 6264670, "Do It Yourself", Some(20)));
			assert_eq!(episodes[179], Episode::new(8, 14, 6320554, "Heavy Is the Head", Some(21)));
			assert_eq!(episodes[180], Episode::new(8, 15, 6412096, "Finding Fizbo", Some(21)));
			assert_eq!(episodes[181], Episode::new(8, 16, 6412100, "Basketball", Some(21)));
			assert_eq!(episodes[182], Episode::new(8, 17, 6421476, "Pig Moon Rising", Some(21)));
			assert_eq!(episodes[183], Episode::new(8, 18, 6421472, "Five Minutes", Some(21)));
			assert_eq!(episodes[184], Episode::new(8, 19, 6421474, "Frank\'s Wedding", Some(21)));
			assert_eq!(episodes[185], Episode::new(8, 20, 6421480, "All Things Being Equal", Some(21)));
			assert_eq!(episodes[186], Episode::new(8, 21, 6421484, "Alone Time", Some(21)));
			assert_eq!(episodes[187], Episode::new(8, 22, 5915114, "The Graduates", Some(21)));
			assert_eq!(episodes[188], Episode::new(9, 1, 6876608, "Lake Life", Some(21)));
			assert_eq!(episodes[189], Episode::new(9, 2, 7339924, "The Long Goodbye", Some(21)));
			assert_eq!(episodes[190], Episode::new(9, 3, 7339922, "Catch of the Day", Some(21)));
			assert_eq!(episodes[191], Episode::new(9, 4, 7459358, "Sex, Lies & Kickball", Some(21)));
			assert_eq!(episodes[192], Episode::new(9, 5, 7476576, "It\'s the Great Pumpkin, Phil Dunphy", Some(21)));
			assert_eq!(episodes[193], Episode::new(9, 6, 7514012, "Ten Years Later", Some(21)));
			assert_eq!(episodes[194], Episode::new(9, 7, 7569884, "Winner Winner Turkey Dinner", Some(21)));
			assert_eq!(episodes[195], Episode::new(9, 8, 7578286, "Brushes with Celebrity", Some(21)));
			assert_eq!(episodes[196], Episode::new(9, 9, 7656366, "Tough Love", Some(21)));
			assert_eq!(episodes[197], Episode::new(9, 10, 7671206, "No Small Feet", Some(21)));
			assert_eq!(episodes[198], Episode::new(9, 11, 7740568, "He Said, She Shed", Some(21)));
			assert_eq!(episodes[199], Episode::new(9, 12, 7768278, "Dear Beloved Family", Some(21)));
			assert_eq!(episodes[200], Episode::new(9, 13, 7768276, "In Your Head", Some(21)));
			assert_eq!(episodes[201], Episode::new(9, 14, 7781868, "Written in the Stars", Some(21)));
			assert_eq!(episodes[202], Episode::new(9, 15, 7822362, "Spanks for the Memories", Some(21)));
			assert_eq!(episodes[203], Episode::new(9, 16, 7822374, "Wine Weekend", Some(21)));
			assert_eq!(episodes[204], Episode::new(9, 17, 7822378, "Royal Visit", Some(21)));
			assert_eq!(episodes[205], Episode::new(9, 18, 7822380, "Daddy Issues", Some(21)));
			assert_eq!(episodes[206], Episode::new(9, 19, 7822384, "CHiPs and Salsa", Some(21)));
			assert_eq!(episodes[207], Episode::new(9, 20, 7822370, "Mother!", Some(21)));
			assert_eq!(episodes[208], Episode::new(9, 21, 7822366, "The Escape", Some(21)));
			assert_eq!(episodes[209], Episode::new(9, 22, 6876612, "Clash of Swords", Some(21)));
			assert_eq!(episodes[210], Episode::new(10, 1, 6876616, "I Love a Parade", Some(21)));
			assert_eq!(episodes[211], Episode::new(10, 2, 8428516, "Kiss and Tell", Some(21)));
			assert_eq!(episodes[212], Episode::new(10, 3, 8428520, "A Sketchy Area", Some(21)));
			assert_eq!(episodes[213], Episode::new(10, 4, 8428522, "Torn Between Two Lovers", Some(21)));
			assert_eq!(episodes[214], Episode::new(10, 5, 8428526, "Good Grief", Some(21)));
			assert_eq!(episodes[215], Episode::new(10, 6, 8428530, "On the Same Paige", Some(21)));
			assert_eq!(episodes[216], Episode::new(10, 7, 8428610, "Did the Chicken Cross the Road?", Some(21)));
			assert_eq!(episodes[217], Episode::new(10, 8, 8428532, "Kids These Days", Some(21)));
			assert_eq!(episodes[218], Episode::new(10, 9, 8428548, "Putting Down Roots", Some(21)));
			assert_eq!(episodes[219], Episode::new(10, 10, 8428534, "Stuck in a Moment", Some(21)));
			assert_eq!(episodes[220], Episode::new(10, 11, 8428554, "A Moving Day", Some(21)));
			assert_eq!(episodes[221], Episode::new(10, 12, 8428562, "Blasts from the Past", Some(21)));
			assert_eq!(episodes[222], Episode::new(10, 13, 8428572, "Whanex?", Some(21)));
			assert_eq!(episodes[223], Episode::new(10, 14, 8428576, "We Need to Talk About Lily", Some(21)));
			assert_eq!(episodes[224], Episode::new(10, 15, 8428590, "SuperShowerBabyBowl", Some(21)));
			assert_eq!(episodes[225], Episode::new(10, 16, 8428602, "Red Alert", Some(21)));
			assert_eq!(episodes[226], Episode::new(10, 17, 8428622, "The Wild", Some(21)));
			assert_eq!(episodes[227], Episode::new(10, 18, 8428626, "Stand by Your Man", Some(21)));
			assert_eq!(episodes[228], Episode::new(10, 19, 8428632, "Yes-Woman", Some(21)));
			assert_eq!(episodes[229], Episode::new(10, 20, 8428638, "Can\'t Elope", Some(21)));
			assert_eq!(episodes[230], Episode::new(10, 21, 8428648, "Commencement", Some(21)));
			assert_eq!(episodes[231], Episode::new(10, 22, 6876620, "A Year of Birthdays", Some(21)));
			assert_eq!(episodes[232], Episode::new(11, 1, 9728344, "New Kids on the Block", Some(21)));
			assert_eq!(episodes[233], Episode::new(11, 2, 10815674, "Snapped", Some(21)));
			assert_eq!(episodes[234], Episode::new(11, 3, 10851244, "Perfect Pairs", Some(21)));
			assert_eq!(episodes[235], Episode::new(11, 4, 10926374, "Pool Party", Some(21)));
			assert_eq!(episodes[236], Episode::new(11, 5, 11022174, "The Last Halloween", Some(21)));
			assert_eq!(episodes[237], Episode::new(11, 6, 10964158, "A Game of Chicken", Some(20)));
			assert_eq!(episodes[238], Episode::new(11, 7, 11224340, "The Last Thanksgiving", Some(21)));
			assert_eq!(episodes[239], Episode::new(11, 8, 11248096, "Tree\'s a Crowd", Some(21)));
			assert_eq!(episodes[240], Episode::new(11, 9, 11248100, "The Last Christmas", Some(20)));
			assert_eq!(episodes[241], Episode::new(11, 10, 11248102, "The Prescott", Some(21)));
			assert_eq!(episodes[242], Episode::new(11, 11, 11248106, "Legacy", Some(21)));
			assert_eq!(episodes[243], Episode::new(11, 12, 11248108, "Dead on a Rival", Some(21)));
			assert_eq!(episodes[244], Episode::new(11, 13, 11248110, "Paris", Some(21)));
			assert_eq!(episodes[245], Episode::new(11, 14, 11248112, "Spuds", Some(21)));
			assert_eq!(episodes[246], Episode::new(11, 15, 11248114, "Baby Steps", Some(21)));
			assert_eq!(episodes[247], Episode::new(11, 16, 11248116, "I\'m Going to Miss This", Some(21)));
			assert_eq!(episodes[248], Episode::new(11, 17, 11248118, "Finale Part 1", Some(43)));
			assert_eq!(episodes[249], Episode::new(11, 18, 10584958, "Finale Part 2", Some(20)));
			// }}}

			let mut episodes = episodes_by_show.remove(&944947).unwrap();
			episodes.sort_unstable();
			assert_eq!(episodes.len(), 73); // {{{
			assert_eq!(episodes[0], Episode::new(1, 1, 1480055, "Winter Is Coming", Some(62)));
			assert_eq!(episodes[1], Episode::new(1, 2, 1668746, "The Kingsroad", Some(56)));
			assert_eq!(episodes[2], Episode::new(1, 3, 1829962, "Lord Snow", Some(58)));
			assert_eq!(episodes[3], Episode::new(1, 4, 1829963, "Cripples, Bastards, and Broken Things", Some(56)));
			assert_eq!(episodes[4], Episode::new(1, 5, 1829964, "The Wolf and the Lion", Some(55)));
			assert_eq!(episodes[5], Episode::new(1, 6, 1837862, "A Golden Crown", Some(53)));
			assert_eq!(episodes[6], Episode::new(1, 7, 1837863, "You Win or You Die", Some(58)));
			assert_eq!(episodes[7], Episode::new(1, 8, 1837864, "The Pointy End", Some(59)));
			assert_eq!(episodes[8], Episode::new(1, 9, 1851398, "Baelor", Some(57)));
			assert_eq!(episodes[9], Episode::new(1, 10, 1851397, "Fire and Blood", Some(53)));
			assert_eq!(episodes[10], Episode::new(2, 1, 1971833, "The North Remembers", Some(53)));
			assert_eq!(episodes[11], Episode::new(2, 2, 2069318, "The Night Lands", Some(54)));
			assert_eq!(episodes[12], Episode::new(2, 3, 2070135, "What Is Dead May Never Die", Some(53)));
			assert_eq!(episodes[13], Episode::new(2, 4, 2069319, "Garden of Bones", Some(51)));
			assert_eq!(episodes[14], Episode::new(2, 5, 2074658, "The Ghost of Harrenhal", Some(55)));
			assert_eq!(episodes[15], Episode::new(2, 6, 2085238, "The Old Gods and the New", Some(54)));
			assert_eq!(episodes[16], Episode::new(2, 7, 2085239, "A Man Without Honor", Some(56)));
			assert_eq!(episodes[17], Episode::new(2, 8, 2085240, "The Prince of Winterfell", Some(54)));
			assert_eq!(episodes[18], Episode::new(2, 9, 2084342, "Blackwater", Some(55)));
			assert_eq!(episodes[19], Episode::new(2, 10, 2112510, "Valar Morghulis", Some(64)));
			assert_eq!(episodes[20], Episode::new(3, 1, 2178782, "Valar Dohaeris", Some(55)));
			assert_eq!(episodes[21], Episode::new(3, 2, 2178772, "Dark Wings, Dark Words", Some(56)));
			assert_eq!(episodes[22], Episode::new(3, 3, 2178802, "Walk of Punishment", Some(56)));
			assert_eq!(episodes[23], Episode::new(3, 4, 2178798, "And Now His Watch Is Ended", Some(53)));
			assert_eq!(episodes[24], Episode::new(3, 5, 2178788, "Kissed by Fire", Some(57)));
			assert_eq!(episodes[25], Episode::new(3, 6, 2178812, "The Climb", Some(53)));
			assert_eq!(episodes[26], Episode::new(3, 7, 2178814, "The Bear and the Maiden Fair", Some(58)));
			assert_eq!(episodes[27], Episode::new(3, 8, 2178806, "Second Sons", Some(56)));
			assert_eq!(episodes[28], Episode::new(3, 9, 2178784, "The Rains of Castamere", Some(51)));
			assert_eq!(episodes[29], Episode::new(3, 10, 2178796, "Mhysa", Some(63)));
			assert_eq!(episodes[30], Episode::new(4, 1, 2816136, "Two Swords", Some(58)));
			assert_eq!(episodes[31], Episode::new(4, 2, 2832378, "The Lion and the Rose", Some(52)));
			assert_eq!(episodes[32], Episode::new(4, 3, 2972426, "Breaker of Chains", Some(57)));
			assert_eq!(episodes[33], Episode::new(4, 4, 2972428, "Oathkeeper", Some(55)));
			assert_eq!(episodes[34], Episode::new(4, 5, 3060856, "First of His Name", Some(53)));
			assert_eq!(episodes[35], Episode::new(4, 6, 3060910, "The Laws of Gods and Men", Some(51)));
			assert_eq!(episodes[36], Episode::new(4, 7, 3060876, "Mockingbird", Some(51)));
			assert_eq!(episodes[37], Episode::new(4, 8, 3060782, "The Mountain and the Viper", Some(52)));
			assert_eq!(episodes[38], Episode::new(4, 9, 3060858, "The Watchers on the Wall", Some(51)));
			assert_eq!(episodes[39], Episode::new(4, 10, 3060860, "The Children", Some(65)));
			assert_eq!(episodes[40], Episode::new(5, 1, 3658012, "The Wars to Come", Some(53)));
			assert_eq!(episodes[41], Episode::new(5, 2, 3846626, "The House of Black and White", Some(56)));
			assert_eq!(episodes[42], Episode::new(5, 3, 3866836, "High Sparrow", Some(60)));
			assert_eq!(episodes[43], Episode::new(5, 4, 3866838, "Sons of the Harpy", Some(51)));
			assert_eq!(episodes[44], Episode::new(5, 5, 3866840, "Kill the Boy", Some(57)));
			assert_eq!(episodes[45], Episode::new(5, 6, 3866842, "Unbowed, Unbent, Unbroken", Some(54)));
			assert_eq!(episodes[46], Episode::new(5, 7, 3866846, "The Gift", Some(59)));
			assert_eq!(episodes[47], Episode::new(5, 8, 3866850, "Hardhome", Some(61)));
			assert_eq!(episodes[48], Episode::new(5, 9, 3866826, "The Dance of Dragons", Some(52)));
			assert_eq!(episodes[49], Episode::new(5, 10, 3866862, "Mother\'s Mercy", Some(60)));
			assert_eq!(episodes[50], Episode::new(6, 1, 3658014, "The Red Woman", Some(50)));
			assert_eq!(episodes[51], Episode::new(6, 2, 4077554, "Home", Some(54)));
			assert_eq!(episodes[52], Episode::new(6, 3, 4131606, "Oathbreaker", Some(52)));
			assert_eq!(episodes[53], Episode::new(6, 4, 4283016, "Book of the Stranger", Some(59)));
			assert_eq!(episodes[54], Episode::new(6, 5, 4283028, "The Door", Some(57)));
			assert_eq!(episodes[55], Episode::new(6, 6, 4283054, "Blood of My Blood", Some(52)));
			assert_eq!(episodes[56], Episode::new(6, 7, 4283060, "The Broken Man", Some(51)));
			assert_eq!(episodes[57], Episode::new(6, 8, 4283074, "No One", Some(59)));
			assert_eq!(episodes[58], Episode::new(6, 9, 4283088, "Battle of the Bastards", Some(60)));
			assert_eq!(episodes[59], Episode::new(6, 10, 4283094, "The Winds of Winter", Some(68)));
			assert_eq!(episodes[60], Episode::new(7, 1, 5654088, "Dragonstone", Some(59)));
			assert_eq!(episodes[61], Episode::new(7, 2, 5655178, "Stormborn", Some(59)));
			assert_eq!(episodes[62], Episode::new(7, 3, 5775840, "The Queen\'s Justice", Some(63)));
			assert_eq!(episodes[63], Episode::new(7, 4, 5775846, "The Spoils of War", Some(50)));
			assert_eq!(episodes[64], Episode::new(7, 5, 5775854, "Eastwatch", Some(59)));
			assert_eq!(episodes[65], Episode::new(7, 6, 5775864, "Beyond the Wall", Some(70)));
			assert_eq!(episodes[66], Episode::new(7, 7, 5775874, "The Dragon and the Wolf", Some(80)));
			assert_eq!(episodes[67], Episode::new(8, 1, 5924366, "Winterfell", Some(54)));
			assert_eq!(episodes[68], Episode::new(8, 2, 6027908, "A Knight of the Seven Kingdoms", Some(58)));
			assert_eq!(episodes[69], Episode::new(8, 3, 6027912, "The Long Night", Some(82)));
			assert_eq!(episodes[70], Episode::new(8, 4, 6027914, "The Last of the Starks", Some(78)));
			assert_eq!(episodes[71], Episode::new(8, 5, 6027916, "The Bells", Some(78)));
			assert_eq!(episodes[72], Episode::new(8, 6, 6027920, "The Iron Throne", Some(80)));
			// }}}

			let mut episodes = episodes_by_show.remove(&460681).unwrap();
			episodes.sort_unstable();
			assert_eq!(episodes.len(), 327); // {{{
			assert_eq!(episodes[0], Episode::new(1, 1, 713618, "Pilot", Some(44)));
			assert_eq!(episodes[1], Episode::new(1, 2, 713624, "Wendigo", Some(44)));
			assert_eq!(episodes[2], Episode::new(1, 3, 713613, "Dead in the Water", Some(44)));
			assert_eq!(episodes[3], Episode::new(1, 4, 713617, "Phantom Traveler", Some(43)));
			assert_eq!(episodes[4], Episode::new(1, 5, 713611, "Bloody Mary", Some(44)));
			assert_eq!(episodes[5], Episode::new(1, 6, 713622, "Skin", Some(42)));
			assert_eq!(episodes[6], Episode::new(1, 7, 713615, "Hook Man", Some(41)));
			assert_eq!(episodes[7], Episode::new(1, 8, 713612, "Bugs", Some(44)));
			assert_eq!(episodes[8], Episode::new(1, 9, 713623, "Home", Some(43)));
			assert_eq!(episodes[9], Episode::new(1, 10, 713610, "Asylum", Some(44)));
			assert_eq!(episodes[10], Episode::new(1, 11, 713620, "Scarecrow", Some(43)));
			assert_eq!(episodes[11], Episode::new(1, 12, 713614, "Faith", Some(44)));
			assert_eq!(episodes[12], Episode::new(1, 13, 713619, "Route 666", Some(50)));
			assert_eq!(episodes[13], Episode::new(1, 14, 713616, "Nightmare", Some(55)));
			assert_eq!(episodes[14], Episode::new(1, 15, 757750, "The Benders", Some(55)));
			assert_eq!(episodes[15], Episode::new(1, 16, 713621, "Shadow", Some(55)));
			assert_eq!(episodes[16], Episode::new(1, 17, 763064, "Hell House", Some(55)));
			assert_eq!(episodes[17], Episode::new(1, 18, 763065, "Something Wicked", Some(55)));
			assert_eq!(episodes[18], Episode::new(1, 19, 761316, "Provenance", Some(44)));
			assert_eq!(episodes[19], Episode::new(1, 20, 767486, "Dead Man\'s Blood", Some(55)));
			assert_eq!(episodes[20], Episode::new(1, 21, 772002, "Salvation", Some(60)));
			assert_eq!(episodes[21], Episode::new(1, 22, 775325, "Devil\'s Trap", Some(55)));
			assert_eq!(episodes[22], Episode::new(2, 1, 829063, "In My Time of Dying", Some(50)));
			assert_eq!(episodes[23], Episode::new(2, 2, 829062, "Everybody Loves a Clown", Some(55)));
			assert_eq!(episodes[24], Episode::new(2, 3, 835248, "Bloodlust", Some(50)));
			assert_eq!(episodes[25], Episode::new(2, 4, 837741, "Children Shouldn\'t Play with Dead Things", Some(42)));
			assert_eq!(episodes[26], Episode::new(2, 5, 851031, "Simon Said", Some(48)));
			assert_eq!(episodes[27], Episode::new(2, 6, 855641, "No Exit", Some(47)));
			assert_eq!(episodes[28], Episode::new(2, 7, 859555, "The Usual Suspects", Some(50)));
			assert_eq!(episodes[29], Episode::new(2, 8, 868625, "Crossroad Blues", Some(42)));
			assert_eq!(episodes[30], Episode::new(2, 9, 878590, "Croatoan", Some(47)));
			assert_eq!(episodes[31], Episode::new(2, 10, 883872, "Hunted", Some(49)));
			assert_eq!(episodes[32], Episode::new(2, 11, 890836, "Playthings", Some(49)));
			assert_eq!(episodes[33], Episode::new(2, 12, 942145, "Nightshifter", Some(47)));
			assert_eq!(episodes[34], Episode::new(2, 13, 921622, "Houses of the Holy", Some(50)));
			assert_eq!(episodes[35], Episode::new(2, 14, 940441, "Born Under a Bad Sign", Some(50)));
			assert_eq!(episodes[36], Episode::new(2, 15, 939444, "Tall Tales", Some(48)));
			assert_eq!(episodes[37], Episode::new(2, 16, 942786, "Roadkill", Some(40)));
			assert_eq!(episodes[38], Episode::new(2, 17, 959976, "Heart", Some(47)));
			assert_eq!(episodes[39], Episode::new(2, 18, 964436, "Hollywood Babylon", Some(50)));
			assert_eq!(episodes[40], Episode::new(2, 19, 1000536, "Folsom Prison Blues", Some(50)));
			assert_eq!(episodes[41], Episode::new(2, 20, 987715, "What Is and What Should Never Be", Some(50)));
			assert_eq!(episodes[42], Episode::new(2, 21, 987713, "All Hell Breaks Loose: Part 1", Some(50)));
			assert_eq!(episodes[43], Episode::new(2, 22, 987714, "All Hell Breaks Loose: Part 2", Some(55)));
			assert_eq!(episodes[44], Episode::new(3, 1, 1032129, "The Magnificent Seven", Some(49)));
			assert_eq!(episodes[45], Episode::new(3, 2, 1032134, "The Kids Are Alright", Some(43)));
			assert_eq!(episodes[46], Episode::new(3, 3, 1107710, "Bad Day at Black Rock", Some(50)));
			assert_eq!(episodes[47], Episode::new(3, 4, 1032135, "Sin City", Some(50)));
			assert_eq!(episodes[48], Episode::new(3, 5, 1032137, "Bedtime Stories", Some(50)));
			assert_eq!(episodes[49], Episode::new(3, 6, 1032138, "Red Sky at Morning", Some(50)));
			assert_eq!(episodes[50], Episode::new(3, 7, 1032139, "Fresh Blood", Some(50)));
			assert_eq!(episodes[51], Episode::new(3, 8, 1032140, "A Very Supernatural Christmas", Some(42)));
			assert_eq!(episodes[52], Episode::new(3, 9, 1032141, "Malleus Maleficarum", Some(40)));
			assert_eq!(episodes[53], Episode::new(3, 10, 1032130, "Dream a Little Dream of Me", Some(55)));
			assert_eq!(episodes[54], Episode::new(3, 11, 1032132, "Mystery Spot", Some(55)));
			assert_eq!(episodes[55], Episode::new(3, 12, 1032131, "Jus in Bello", Some(55)));
			assert_eq!(episodes[56], Episode::new(3, 13, 1032133, "Ghostfacers", Some(42)));
			assert_eq!(episodes[57], Episode::new(3, 14, 1206073, "Long Distance Call", Some(41)));
			assert_eq!(episodes[58], Episode::new(3, 15, 1222593, "Time Is on My Side", Some(50)));
			assert_eq!(episodes[59], Episode::new(3, 16, 1222594, "No Rest for the Wicked", Some(50)));
			assert_eq!(episodes[60], Episode::new(4, 1, 1222595, "Lazarus Rising", Some(55)));
			assert_eq!(episodes[61], Episode::new(4, 2, 1222606, "Are You There, God? It\'s Me, Dean Winchester", Some(50)));
			assert_eq!(episodes[62], Episode::new(4, 3, 1222611, "In the Beginning", Some(50)));
			assert_eq!(episodes[63], Episode::new(4, 4, 1222612, "Metamorphosis", Some(60)));
			assert_eq!(episodes[64], Episode::new(4, 5, 1222610, "Monster Movie", Some(55)));
			assert_eq!(episodes[65], Episode::new(4, 6, 1222613, "Yellow Fever", Some(43)));
			assert_eq!(episodes[66], Episode::new(4, 7, 1222614, "It\'s the Great Pumpkin, Sam Winchester", Some(42)));
			assert_eq!(episodes[67], Episode::new(4, 8, 1222615, "Wishful Thinking", Some(41)));
			assert_eq!(episodes[68], Episode::new(4, 9, 1222616, "I Know What You Did Last Summer", Some(43)));
			assert_eq!(episodes[69], Episode::new(4, 10, 1222596, "Heaven and Hell", Some(43)));
			assert_eq!(episodes[70], Episode::new(4, 11, 1222597, "Family Remains", Some(43)));
			assert_eq!(episodes[71], Episode::new(4, 12, 1222598, "Criss Angel Is a Douche Bag", Some(43)));
			assert_eq!(episodes[72], Episode::new(4, 13, 1222599, "After School Special", Some(55)));
			assert_eq!(episodes[73], Episode::new(4, 14, 1222600, "Sex and Violence", Some(43)));
			assert_eq!(episodes[74], Episode::new(4, 15, 1222601, "Death Takes a Holiday", Some(43)));
			assert_eq!(episodes[75], Episode::new(4, 16, 1222602, "On the Head of a Pin", Some(43)));
			assert_eq!(episodes[76], Episode::new(4, 17, 1222603, "It\'s a Terrible Life", Some(43)));
			assert_eq!(episodes[77], Episode::new(4, 18, 1222604, "The Monster at the End of This Book", Some(43)));
			assert_eq!(episodes[78], Episode::new(4, 19, 1222605, "Jump the Shark", Some(43)));
			assert_eq!(episodes[79], Episode::new(4, 20, 1222607, "The Rapture", Some(40)));
			assert_eq!(episodes[80], Episode::new(4, 21, 1222608, "When the Levee Breaks", Some(43)));
			assert_eq!(episodes[81], Episode::new(4, 22, 1222609, "Lucifer Rising", Some(43)));
			assert_eq!(episodes[82], Episode::new(5, 1, 1461906, "Sympathy for the Devil", Some(60)));
			assert_eq!(episodes[83], Episode::new(5, 2, 1464088, "Good God, Y\'All", Some(60)));
			assert_eq!(episodes[84], Episode::new(5, 3, 1464089, "Free to Be You and Me", Some(60)));
			assert_eq!(episodes[85], Episode::new(5, 4, 1488984, "The End", Some(60)));
			assert_eq!(episodes[86], Episode::new(5, 5, 1488983, "Fallen Idols", Some(60)));
			assert_eq!(episodes[87], Episode::new(5, 6, 1506849, "I Believe the Children Are Our Future", Some(42)));
			assert_eq!(episodes[88], Episode::new(5, 7, 1507550, "The Curious Case of Dean Winchester", Some(42)));
			assert_eq!(episodes[89], Episode::new(5, 8, 1511910, "Changing Channels", Some(60)));
			assert_eq!(episodes[90], Episode::new(5, 9, 1533530, "The Real Ghostbusters", Some(60)));
			assert_eq!(episodes[91], Episode::new(5, 10, 1532469, "Abandon All Hope", Some(60)));
			assert_eq!(episodes[92], Episode::new(5, 11, 1543420, "Sam, Interrupted", Some(60)));
			assert_eq!(episodes[93], Episode::new(5, 12, 1552579, "Swap Meat", Some(60)));
			assert_eq!(episodes[94], Episode::new(5, 13, 1541419, "The Song Remains the Same", Some(60)));
			assert_eq!(episodes[95], Episode::new(5, 14, 1588681, "My Bloody Valentine", Some(60)));
			assert_eq!(episodes[96], Episode::new(5, 15, 1588682, "Dead Men Don\'t Wear Plaid", Some(60)));
			assert_eq!(episodes[97], Episode::new(5, 16, 1588256, "Dark Side of the Moon", Some(60)));
			assert_eq!(episodes[98], Episode::new(5, 17, 1588255, "99 Problems", Some(60)));
			assert_eq!(episodes[99], Episode::new(5, 18, 1594854, "Point of No Return", Some(60)));
			assert_eq!(episodes[100], Episode::new(5, 19, 1598673, "Hammer of the Gods", Some(60)));
			assert_eq!(episodes[101], Episode::new(5, 20, 1599805, "The Devil You Know", Some(60)));
			assert_eq!(episodes[102], Episode::new(5, 21, 1626754, "Two Minutes to Midnight", Some(60)));
			assert_eq!(episodes[103], Episode::new(5, 22, 1626753, "Swan Song", Some(42)));
			assert_eq!(episodes[104], Episode::new(6, 1, 1627706, "Exile on Main St.", Some(42)));
			assert_eq!(episodes[105], Episode::new(6, 2, 1627717, "Two and a Half Men", Some(41)));
			assert_eq!(episodes[106], Episode::new(6, 3, 1627721, "The Third Man", Some(41)));
			assert_eq!(episodes[107], Episode::new(6, 4, 1627722, "Weekend at Bobby\'s", Some(40)));
			assert_eq!(episodes[108], Episode::new(6, 5, 1627724, "Live Free or Twihard", Some(42)));
			assert_eq!(episodes[109], Episode::new(6, 6, 1627723, "You Can\'t Handle the Truth", Some(60)));
			assert_eq!(episodes[110], Episode::new(6, 7, 1627725, "Family Matters", Some(41)));
			assert_eq!(episodes[111], Episode::new(6, 8, 1627726, "All Dogs Go to Heaven", Some(42)));
			assert_eq!(episodes[112], Episode::new(6, 9, 1627727, "Clap Your Hands If You Believe", Some(43)));
			assert_eq!(episodes[113], Episode::new(6, 10, 1627707, "Caged Heat", Some(60)));
			assert_eq!(episodes[114], Episode::new(6, 11, 1627708, "Appointment in Samarra", Some(42)));
			assert_eq!(episodes[115], Episode::new(6, 12, 1627709, "Like a Virgin", Some(60)));
			assert_eq!(episodes[116], Episode::new(6, 13, 1627710, "Unforgiven", Some(60)));
			assert_eq!(episodes[117], Episode::new(6, 14, 1627711, "Mannequin 3: The Reckoning", Some(60)));
			assert_eq!(episodes[118], Episode::new(6, 15, 1627712, "The French Mistake", Some(42)));
			assert_eq!(episodes[119], Episode::new(6, 16, 1627713, "And Then There Were None", Some(60)));
			assert_eq!(episodes[120], Episode::new(6, 17, 1627715, "My Heart Will Go On", Some(60)));
			assert_eq!(episodes[121], Episode::new(6, 18, 1627714, "Frontierland", Some(60)));
			assert_eq!(episodes[122], Episode::new(6, 19, 1627716, "Mommy Dearest", Some(60)));
			assert_eq!(episodes[123], Episode::new(6, 20, 1627718, "The Man Who Would Be King", Some(60)));
			assert_eq!(episodes[124], Episode::new(6, 21, 1627719, "Let It Bleed", Some(60)));
			assert_eq!(episodes[125], Episode::new(6, 22, 1864037, "The Man Who Knew Too Much", Some(43)));
			assert_eq!(episodes[126], Episode::new(7, 1, 1995102, "Meet the New Boss", Some(43)));
			assert_eq!(episodes[127], Episode::new(7, 2, 1995113, "Hello, Cruel World", Some(43)));
			assert_eq!(episodes[128], Episode::new(7, 3, 1995118, "The Girl Next Door", Some(42)));
			assert_eq!(episodes[129], Episode::new(7, 4, 1995119, "Defending Your Life", Some(43)));
			assert_eq!(episodes[130], Episode::new(7, 5, 1995120, "Shut Up, Dr. Phil", Some(42)));
			assert_eq!(episodes[131], Episode::new(7, 6, 1995121, "Slash Fiction", Some(43)));
			assert_eq!(episodes[132], Episode::new(7, 7, 1995122, "The Mentalists", Some(42)));
			assert_eq!(episodes[133], Episode::new(7, 8, 1995123, "Season Seven, Time for a Wedding!", Some(40)));
			assert_eq!(episodes[134], Episode::new(7, 9, 1995124, "How to Win Friends and Influence Monsters", Some(41)));
			assert_eq!(episodes[135], Episode::new(7, 10, 1995103, "Death\'s Door", Some(42)));
			assert_eq!(episodes[136], Episode::new(7, 11, 1995104, "Adventures in Babysitting", Some(43)));
			assert_eq!(episodes[137], Episode::new(7, 12, 1995105, "Time After Time", Some(43)));
			assert_eq!(episodes[138], Episode::new(7, 13, 1995106, "The Slice Girls", Some(42)));
			assert_eq!(episodes[139], Episode::new(7, 14, 1995107, "Plucky Pennywhistle\'s Magical Menagerie", Some(42)));
			assert_eq!(episodes[140], Episode::new(7, 15, 1995108, "Repo Man", Some(43)));
			assert_eq!(episodes[141], Episode::new(7, 16, 1995109, "Out with the Old", Some(43)));
			assert_eq!(episodes[142], Episode::new(7, 17, 1995110, "The Born-Again Identity", Some(43)));
			assert_eq!(episodes[143], Episode::new(7, 18, 1995111, "Party On, Garth", Some(43)));
			assert_eq!(episodes[144], Episode::new(7, 19, 1995112, "Of Grave Importance", Some(42)));
			assert_eq!(episodes[145], Episode::new(7, 20, 1995114, "The Girl with the Dungeons and Dragons Tattoo", Some(43)));
			assert_eq!(episodes[146], Episode::new(7, 21, 1995115, "Reading is Fundamental", Some(42)));
			assert_eq!(episodes[147], Episode::new(7, 22, 1995116, "There Will Be Blood", Some(43)));
			assert_eq!(episodes[148], Episode::new(7, 23, 1995117, "Survival of the Fittest", Some(42)));
			assert_eq!(episodes[149], Episode::new(8, 1, 2373078, "We Need to Talk About Kevin", Some(42)));
			assert_eq!(episodes[150], Episode::new(8, 2, 2404088, "What\'s Up, Tiger Mommy?", Some(42)));
			assert_eq!(episodes[151], Episode::new(8, 3, 2293862, "Heartache", Some(40)));
			assert_eq!(episodes[152], Episode::new(8, 4, 2323362, "Bitten", Some(42)));
			assert_eq!(episodes[153], Episode::new(8, 5, 2388382, "Blood Brother", Some(43)));
			assert_eq!(episodes[154], Episode::new(8, 6, 2372414, "Southern Comfort", Some(43)));
			assert_eq!(episodes[155], Episode::new(8, 7, 2425360, "A Little Slice of Kevin", Some(43)));
			assert_eq!(episodes[156], Episode::new(8, 8, 2426140, "Hunteri Heroici", Some(43)));
			assert_eq!(episodes[157], Episode::new(8, 9, 2426138, "Citizen Fang", Some(43)));
			assert_eq!(episodes[158], Episode::new(8, 10, 2425378, "Torn and Frayed", Some(43)));
			assert_eq!(episodes[159], Episode::new(8, 11, 2498912, "LARP and the Real Girl", Some(43)));
			assert_eq!(episodes[160], Episode::new(8, 12, 2506990, "As Time Goes By", Some(43)));
			assert_eq!(episodes[161], Episode::new(8, 13, 2578700, "Everybody Hates Hitler", Some(43)));
			assert_eq!(episodes[162], Episode::new(8, 14, 2503932, "Trial and Error", Some(43)));
			assert_eq!(episodes[163], Episode::new(8, 15, 2578704, "Man\'s Best Friend with Benefits", Some(43)));
			assert_eq!(episodes[164], Episode::new(8, 16, 2578708, "Remember the Titans", Some(42)));
			assert_eq!(episodes[165], Episode::new(8, 17, 2692382, "Goodbye Stranger", Some(43)));
			assert_eq!(episodes[166], Episode::new(8, 18, 2621384, "Freaks and Geeks", Some(42)));
			assert_eq!(episodes[167], Episode::new(8, 19, 2699592, "Taxi Driver", Some(43)));
			assert_eq!(episodes[168], Episode::new(8, 20, 2699596, "Pac-Man Fever", Some(43)));
			assert_eq!(episodes[169], Episode::new(8, 21, 2734858, "The Great Escapist", Some(43)));
			assert_eq!(episodes[170], Episode::new(8, 22, 2805138, "Clip Show", Some(43)));
			assert_eq!(episodes[171], Episode::new(8, 23, 2805140, "Sacrifice", Some(42)));
			assert_eq!(episodes[172], Episode::new(9, 1, 2702688, "I Think I\'m Gonna Like It Here", Some(42)));
			assert_eq!(episodes[173], Episode::new(9, 2, 3025548, "Devil May Care", Some(41)));
			assert_eq!(episodes[174], Episode::new(9, 3, 3097142, "I\'m No Angel", Some(42)));
			assert_eq!(episodes[175], Episode::new(9, 4, 3098422, "Slumber Party", Some(42)));
			assert_eq!(episodes[176], Episode::new(9, 5, 3112660, "Dog Dean Afternoon", Some(41)));
			assert_eq!(episodes[177], Episode::new(9, 6, 3112656, "Heaven Can\'t Wait", Some(42)));
			assert_eq!(episodes[178], Episode::new(9, 7, 3141550, "Bad Boys", Some(42)));
			assert_eq!(episodes[179], Episode::new(9, 8, 3158272, "Rock and a Hard Place", Some(42)));
			assert_eq!(episodes[180], Episode::new(9, 9, 3158274, "Holy Terror", Some(42)));
			assert_eq!(episodes[181], Episode::new(9, 10, 3158276, "Road Trip", Some(43)));
			assert_eq!(episodes[182], Episode::new(9, 11, 3158278, "First Born", Some(42)));
			assert_eq!(episodes[183], Episode::new(9, 12, 3158284, "Sharp Teeth", Some(42)));
			assert_eq!(episodes[184], Episode::new(9, 13, 3158282, "The Purge", Some(42)));
			assert_eq!(episodes[185], Episode::new(9, 14, 3182872, "Captives", Some(42)));
			assert_eq!(episodes[186], Episode::new(9, 15, 3182874, "#thinman", Some(42)));
			assert_eq!(episodes[187], Episode::new(9, 16, 3182880, "Blade Runners", Some(42)));
			assert_eq!(episodes[188], Episode::new(9, 17, 3182878, "Mother\'s Little Helper", Some(42)));
			assert_eq!(episodes[189], Episode::new(9, 18, 3182886, "Meta Fiction", Some(42)));
			assert_eq!(episodes[190], Episode::new(9, 19, 3182884, "Alex Annie Alexis Ann", Some(42)));
			assert_eq!(episodes[191], Episode::new(9, 20, 3182892, "Bloodlines", Some(42)));
			assert_eq!(episodes[192], Episode::new(9, 21, 3182890, "King of the Damned", Some(42)));
			assert_eq!(episodes[193], Episode::new(9, 22, 3182896, "Stairway to Heaven", Some(42)));
			assert_eq!(episodes[194], Episode::new(9, 23, 3182894, "Do You Believe in Miracles", Some(42)));
			assert_eq!(episodes[195], Episode::new(10, 1, 3532628, "Black", Some(43)));
			assert_eq!(episodes[196], Episode::new(10, 2, 3831300, "Reichenbach", Some(43)));
			assert_eq!(episodes[197], Episode::new(10, 3, 3831310, "Soul Survivor", Some(43)));
			assert_eq!(episodes[198], Episode::new(10, 4, 3845906, "Paper Moon", Some(41)));
			assert_eq!(episodes[199], Episode::new(10, 5, 3845910, "Fan Fiction", Some(43)));
			assert_eq!(episodes[200], Episode::new(10, 6, 3860118, "Ask Jeeves", Some(43)));
			assert_eq!(episodes[201], Episode::new(10, 7, 3860122, "Girls, Girls, Girls", Some(42)));
			assert_eq!(episodes[202], Episode::new(10, 8, 3997488, "Hibbing 911", Some(42)));
			assert_eq!(episodes[203], Episode::new(10, 9, 3997492, "The Things We Left Behind", Some(42)));
			assert_eq!(episodes[204], Episode::new(10, 10, 3997490, "The Hunter Games", Some(43)));
			assert_eq!(episodes[205], Episode::new(10, 11, 4274892, "There\'s No Place Like Home", Some(43)));
			assert_eq!(episodes[206], Episode::new(10, 12, 4344320, "About a Boy", Some(41)));
			assert_eq!(episodes[207], Episode::new(10, 13, 4344332, "Halt & Catch Fire", Some(42)));
			assert_eq!(episodes[208], Episode::new(10, 14, 4292452, "The Executioner\'s Song", Some(42)));
			assert_eq!(episodes[209], Episode::new(10, 15, 4333228, "The Things They Carried", Some(42)));
			assert_eq!(episodes[210], Episode::new(10, 16, 4427940, "Paint It Black", Some(42)));
			assert_eq!(episodes[211], Episode::new(10, 17, 4443976, "Inside Man", Some(42)));
			assert_eq!(episodes[212], Episode::new(10, 18, 4489472, "Book of the Damned", Some(43)));
			assert_eq!(episodes[213], Episode::new(10, 19, 4489474, "The Werther Project", Some(43)));
			assert_eq!(episodes[214], Episode::new(10, 20, 4489476, "Angel Heart", Some(42)));
			assert_eq!(episodes[215], Episode::new(10, 21, 4489482, "Dark Dynasty", Some(42)));
			assert_eq!(episodes[216], Episode::new(10, 22, 4489484, "The Prisoner", Some(42)));
			assert_eq!(episodes[217], Episode::new(10, 23, 4489486, "Brother\'s Keeper", Some(42)));
			assert_eq!(episodes[218], Episode::new(11, 1, 4346788, "Out of the Darkness, Into the Fire", Some(43)));
			assert_eq!(episodes[219], Episode::new(11, 2, 4857906, "Form and Void", Some(43)));
			assert_eq!(episodes[220], Episode::new(11, 3, 4857918, "The Bad Seed", Some(42)));
			assert_eq!(episodes[221], Episode::new(11, 4, 4995876, "Baby", Some(41)));
			assert_eq!(episodes[222], Episode::new(11, 5, 5056820, "Thin Lizzie", Some(42)));
			assert_eq!(episodes[223], Episode::new(11, 6, 5056822, "Our Little World", Some(42)));
			assert_eq!(episodes[224], Episode::new(11, 7, 5056828, "Plush", Some(41)));
			assert_eq!(episodes[225], Episode::new(11, 8, 5056832, "Just My Imagination", Some(42)));
			assert_eq!(episodes[226], Episode::new(11, 9, 5056836, "O Brother, Where Art Thou?", Some(42)));
			assert_eq!(episodes[227], Episode::new(11, 10, 5087242, "The Devil in the Details", Some(42)));
			assert_eq!(episodes[228], Episode::new(11, 11, 5239780, "Into the Mystic", Some(42)));
			assert_eq!(episodes[229], Episode::new(11, 12, 5265450, "Don\'t You Forget About Me", Some(42)));
			assert_eq!(episodes[230], Episode::new(11, 13, 5352054, "Love Hurts", Some(41)));
			assert_eq!(episodes[231], Episode::new(11, 14, 5390282, "The Vessel", Some(42)));
			assert_eq!(episodes[232], Episode::new(11, 15, 5432122, "Beyond the Mat", Some(42)));
			assert_eq!(episodes[233], Episode::new(11, 16, 5432150, "Safe House", Some(42)));
			assert_eq!(episodes[234], Episode::new(11, 17, 5432156, "Red Meat", Some(41)));
			assert_eq!(episodes[235], Episode::new(11, 18, 5539106, "Hell\'s Angel", Some(42)));
			assert_eq!(episodes[236], Episode::new(11, 19, 5538526, "The Chitters", Some(41)));
			assert_eq!(episodes[237], Episode::new(11, 20, 5547512, "Don\'t Call Me Shurley", Some(42)));
			assert_eq!(episodes[238], Episode::new(11, 21, 5547522, "All in the Family", Some(41)));
			assert_eq!(episodes[239], Episode::new(11, 22, 5547530, "We Happy Few", Some(41)));
			assert_eq!(episodes[240], Episode::new(11, 23, 5547538, "Alpha and Omega", Some(42)));
			assert_eq!(episodes[241], Episode::new(12, 1, 5540416, "Keep Calm and Carry On", Some(41)));
			assert_eq!(episodes[242], Episode::new(12, 2, 5669110, "Mamma Mia", Some(42)));
			assert_eq!(episodes[243], Episode::new(12, 3, 5669112, "The Foundry", Some(42)));
			assert_eq!(episodes[244], Episode::new(12, 4, 5669114, "American Nightmare", Some(42)));
			assert_eq!(episodes[245], Episode::new(12, 5, 5669118, "The One You\'ve Been Waiting For", Some(41)));
			assert_eq!(episodes[246], Episode::new(12, 6, 5669116, "Celebrating the Life of Asa Fox", Some(42)));
			assert_eq!(episodes[247], Episode::new(12, 7, 5669120, "Rock Never Dies", Some(42)));
			assert_eq!(episodes[248], Episode::new(12, 8, 5669122, "LOTUS", Some(41)));
			assert_eq!(episodes[249], Episode::new(12, 9, 5669126, "First Blood", Some(41)));
			assert_eq!(episodes[250], Episode::new(12, 10, 5669124, "Lily Sunder Has Some Regrets", Some(42)));
			assert_eq!(episodes[251], Episode::new(12, 11, 5669128, "Regarding Dean", Some(42)));
			assert_eq!(episodes[252], Episode::new(12, 12, 5669130, "Stuck in the Middle (With You)", Some(42)));
			assert_eq!(episodes[253], Episode::new(12, 13, 5669132, "Family Feud", Some(42)));
			assert_eq!(episodes[254], Episode::new(12, 14, 5669134, "The Raid", Some(41)));
			assert_eq!(episodes[255], Episode::new(12, 15, 5669138, "Somewhere Between Heaven and Hell", Some(41)));
			assert_eq!(episodes[256], Episode::new(12, 16, 5669136, "Ladies Drink Free", Some(42)));
			assert_eq!(episodes[257], Episode::new(12, 17, 5669140, "The British Invasion", Some(42)));
			assert_eq!(episodes[258], Episode::new(12, 18, 5669142, "The Memory Remains", Some(42)));
			assert_eq!(episodes[259], Episode::new(12, 19, 5669144, "The Future", Some(42)));
			assert_eq!(episodes[260], Episode::new(12, 20, 5669146, "Twigs & Twine & Tasha Banes", Some(42)));
			assert_eq!(episodes[261], Episode::new(12, 21, 5669148, "There\'s Something About Mary", Some(42)));
			assert_eq!(episodes[262], Episode::new(12, 22, 5669150, "Who We Are", Some(42)));
			assert_eq!(episodes[263], Episode::new(12, 23, 5669152, "All Along the Watchtower", Some(42)));
			assert_eq!(episodes[264], Episode::new(13, 1, 6403432, "Lost and Found", Some(41)));
			assert_eq!(episodes[265], Episode::new(13, 2, 6746126, "The Rising Son", Some(42)));
			assert_eq!(episodes[266], Episode::new(13, 3, 6746128, "Patience", Some(42)));
			assert_eq!(episodes[267], Episode::new(13, 4, 6746132, "The Big Empty", Some(42)));
			assert_eq!(episodes[268], Episode::new(13, 5, 6746134, "Advanced Thanatology", Some(41)));
			assert_eq!(episodes[269], Episode::new(13, 6, 6746140, "Tombstone", Some(41)));
			assert_eq!(episodes[270], Episode::new(13, 7, 6746146, "War of the Worlds", Some(42)));
			assert_eq!(episodes[271], Episode::new(13, 8, 6746148, "The Scorpion and the Frog", Some(41)));
			assert_eq!(episodes[272], Episode::new(13, 9, 6746160, "The Bad Place", Some(41)));
			assert_eq!(episodes[273], Episode::new(13, 10, 6746156, "Wayward Sisters", Some(42)));
			assert_eq!(episodes[274], Episode::new(13, 11, 6746164, "Breakdown", Some(41)));
			assert_eq!(episodes[275], Episode::new(13, 12, 6746170, "Various & Sundry Villains", Some(42)));
			assert_eq!(episodes[276], Episode::new(13, 13, 6746168, "Devil\'s Bargain", Some(41)));
			assert_eq!(episodes[277], Episode::new(13, 14, 6746174, "Good Intentions", Some(42)));
			assert_eq!(episodes[278], Episode::new(13, 15, 6746182, "A Most Holy Man", Some(42)));
			assert_eq!(episodes[279], Episode::new(13, 16, 6746190, "ScoobyNatural", Some(42)));
			assert_eq!(episodes[280], Episode::new(13, 17, 6746194, "The Thing", Some(42)));
			assert_eq!(episodes[281], Episode::new(13, 18, 6746200, "Bring \'em Back Alive", Some(42)));
			assert_eq!(episodes[282], Episode::new(13, 19, 6746208, "Funeralia", Some(42)));
			assert_eq!(episodes[283], Episode::new(13, 20, 6746210, "Unfinished Business", Some(42)));
			assert_eq!(episodes[284], Episode::new(13, 21, 6746216, "Beat the Devil", Some(42)));
			assert_eq!(episodes[285], Episode::new(13, 22, 6746226, "Exodus", Some(41)));
			assert_eq!(episodes[286], Episode::new(13, 23, 6746224, "Let the Good Times Roll", Some(42)));
			assert_eq!(episodes[287], Episode::new(14, 1, 8226756, "Stranger in a Strange Land", Some(41)));
			assert_eq!(episodes[288], Episode::new(14, 2, 8408494, "Gods and Monsters", Some(41)));
			assert_eq!(episodes[289], Episode::new(14, 3, 8408498, "The Scar", Some(42)));
			assert_eq!(episodes[290], Episode::new(14, 4, 8408504, "Mint Condition", Some(41)));
			assert_eq!(episodes[291], Episode::new(14, 5, 8408508, "Nightmare Logic", Some(42)));
			assert_eq!(episodes[292], Episode::new(14, 6, 8408506, "Optimism", Some(41)));
			assert_eq!(episodes[293], Episode::new(14, 7, 8408502, "Unhuman Nature", Some(41)));
			assert_eq!(episodes[294], Episode::new(14, 8, 8408500, "Byzantium", Some(42)));
			assert_eq!(episodes[295], Episode::new(14, 9, 8408512, "The Spear", Some(41)));
			assert_eq!(episodes[296], Episode::new(14, 10, 8408510, "Nihilism", Some(42)));
			assert_eq!(episodes[297], Episode::new(14, 11, 8962434, "Damaged Goods", Some(42)));
			assert_eq!(episodes[298], Episode::new(14, 12, 8962440, "Prophet and Loss", Some(42)));
			assert_eq!(episodes[299], Episode::new(14, 13, 8962446, "Lebanon", Some(42)));
			assert_eq!(episodes[300], Episode::new(14, 14, 9271138, "Ouroboros", Some(42)));
			assert_eq!(episodes[301], Episode::new(14, 15, 9271140, "Peace of Mind", Some(42)));
			assert_eq!(episodes[302], Episode::new(14, 16, 9271142, "Don\'t Go in the Woods", Some(41)));
			assert_eq!(episodes[303], Episode::new(14, 17, 9271146, "Game Night", Some(42)));
			assert_eq!(episodes[304], Episode::new(14, 18, 9271148, "Absence", Some(42)));
			assert_eq!(episodes[305], Episode::new(14, 19, 9271150, "Jack in the Box", Some(42)));
			assert_eq!(episodes[306], Episode::new(14, 20, 9271152, "Moriah", Some(41)));
			assert_eq!(episodes[307], Episode::new(15, 1, 9695548, "Back and to the Future", Some(41)));
			assert_eq!(episodes[308], Episode::new(15, 2, 10067710, "Raising Hell", Some(42)));
			assert_eq!(episodes[309], Episode::new(15, 3, 10250608, "The Rupture", Some(41)));
			assert_eq!(episodes[310], Episode::new(15, 4, 10067734, "Atomic Monsters", Some(42)));
			assert_eq!(episodes[311], Episode::new(15, 5, 10067742, "Proverbs 17:3", Some(42)));
			assert_eq!(episodes[312], Episode::new(15, 6, 10067744, "Golden Time", Some(42)));
			assert_eq!(episodes[313], Episode::new(15, 7, 10085356, "Last Call", Some(41)));
			assert_eq!(episodes[314], Episode::new(15, 8, 10717880, "Our Father, Who Aren\'t in Heaven", Some(42)));
			assert_eq!(episodes[315], Episode::new(15, 9, 10717886, "The Trap", Some(42)));
			assert_eq!(episodes[316], Episode::new(15, 10, 10717892, "The Heroes\' Journey", Some(42)));
			assert_eq!(episodes[317], Episode::new(15, 11, 10717964, "The Gamblers", Some(42)));
			assert_eq!(episodes[318], Episode::new(15, 12, 10717906, "Galaxy Brain", Some(42)));
			assert_eq!(episodes[319], Episode::new(15, 13, 10717912, "Destiny\'s Child", Some(42)));
			assert_eq!(episodes[320], Episode::new(15, 14, 10717914, "Last Holiday", Some(40)));
			assert_eq!(episodes[321], Episode::new(15, 15, 10717918, "Gimme Shelter", Some(41)));
			assert_eq!(episodes[322], Episode::new(15, 16, 10717922, "Drag Me Away (From You)", Some(42)));
			assert_eq!(episodes[323], Episode::new(15, 17, 10717928, "Unity", Some(42)));
			assert_eq!(episodes[324], Episode::new(15, 18, 10717930, "Despair", Some(41)));
			assert_eq!(episodes[325], Episode::new(15, 19, 10717932, "Inherit the Earth", Some(42)));
			assert_eq!(episodes[326], Episode::new(15, 20, 10073984, "Carry On", Some(42)));
			// }}}
		}
	} // }}}

	#[tokio::test]
	async fn shows_single() /* {{{ */ {
		let (_dir, mut client) = client().await;

		for i in 0..5 {
			println!(">>> {}", i);
			let mut show = client.get_show(903747).await.unwrap();
			show.episodes.sort_unstable();
			assert_eq!(show, Show::new(903747, "Breaking Bad", false, 2008, Some(2013), Some(49), &["Crime", "Drama", "Thriller"], &[ // {{{
				Episode::new(1, 1, 959621, "Pilot", Some(58)),
				Episode::new(1, 2, 1054724, "Cat\'s in the Bag...", Some(48)),
				Episode::new(1, 3, 1054725, "...And the Bag\'s in the River", Some(48)),
				Episode::new(1, 4, 1054726, "Cancer Man", Some(48)),
				Episode::new(1, 5, 1054727, "Gray Matter", Some(48)),
				Episode::new(1, 6, 1054728, "Crazy Handful of Nothin\'", Some(48)),
				Episode::new(1, 7, 1054729, "A No-Rough-Stuff-Type Deal", Some(48)),
				Episode::new(2, 1, 1232244, "Seven Thirty-Seven", Some(47)),
				Episode::new(2, 2, 1232249, "Grilled", Some(46)),
				Episode::new(2, 3, 1232250, "Bit by a Dead Bee", Some(47)),
				Episode::new(2, 4, 1232251, "Down", Some(47)),
				Episode::new(2, 5, 1232252, "Breakage", Some(47)),
				Episode::new(2, 6, 1232253, "Peekaboo", Some(47)),
				Episode::new(2, 7, 1232254, "Negro y Azul", Some(47)),
				Episode::new(2, 8, 1232255, "Better Call Saul", Some(47)),
				Episode::new(2, 9, 1232256, "4 Days Out", Some(47)),
				Episode::new(2, 10, 1232245, "Over", Some(47)),
				Episode::new(2, 11, 1232246, "Mandala", Some(47)),
				Episode::new(2, 12, 1232247, "Phoenix", Some(47)),
				Episode::new(2, 13, 1232248, "ABQ", Some(48)),
				Episode::new(3, 1, 1528116, "No Más", Some(47)),
				Episode::new(3, 2, 1615186, "Caballo sin Nombre", Some(47)),
				Episode::new(3, 3, 1615187, "I.F.T.", Some(47)),
				Episode::new(3, 4, 1615554, "Green Light", Some(47)),
				Episode::new(3, 5, 1615555, "Más", Some(47)),
				Episode::new(3, 6, 1615556, "Sunset", Some(47)),
				Episode::new(3, 7, 1615944, "One Minute", Some(47)),
				Episode::new(3, 8, 1615557, "I See You", Some(47)),
				Episode::new(3, 9, 1615558, "Kafkaesque", Some(47)),
				Episode::new(3, 10, 1615550, "Fly", Some(47)),
				Episode::new(3, 11, 1615551, "Abiquiu", Some(47)),
				Episode::new(3, 12, 1615552, "Half Measures", Some(47)),
				Episode::new(3, 13, 1615553, "Full Measure", Some(47)),
				Episode::new(4, 1, 1683084, "Box Cutter", Some(48)),
				Episode::new(4, 2, 1683089, "Thirty-Eight Snub", Some(60)),
				Episode::new(4, 3, 1683090, "Open House", Some(47)),
				Episode::new(4, 4, 1683091, "Bullet Points", Some(47)),
				Episode::new(4, 5, 1683092, "Shotgun", Some(47)),
				Episode::new(4, 6, 1683093, "Cornered", Some(47)),
				Episode::new(4, 7, 1683094, "Problem Dog", Some(47)),
				Episode::new(4, 8, 1683095, "Hermanos", Some(47)),
				Episode::new(4, 9, 1683096, "Bug", Some(47)),
				Episode::new(4, 10, 1683085, "Salud", Some(47)),
				Episode::new(4, 11, 1683086, "Crawl Space", Some(45)),
				Episode::new(4, 12, 1683087, "End Times", Some(47)),
				Episode::new(4, 13, 1683088, "Face Off", Some(51)),
				Episode::new(5, 1, 2081647, "Live Free or Die", Some(43)),
				Episode::new(5, 2, 2301457, "Madrigal", Some(48)),
				Episode::new(5, 3, 2301459, "Hazard Pay", Some(48)),
				Episode::new(5, 4, 2301461, "Fifty-One", Some(48)),
				Episode::new(5, 5, 2301463, "Dead Freight", Some(48)),
				Episode::new(5, 6, 2301465, "Buyout", Some(48)),
				Episode::new(5, 7, 2301467, "Say My Name", Some(48)),
				Episode::new(5, 8, 2301469, "Gliding Over All", Some(48)),
				Episode::new(5, 9, 2301471, "Blood Money", Some(48)),
				Episode::new(5, 10, 2301443, "Buried", Some(48)),
				Episode::new(5, 11, 2301445, "Confessions", Some(48)),
				Episode::new(5, 12, 2301447, "Rabid Dog", Some(48)),
				Episode::new(5, 13, 2301449, "To\'hajiilee", Some(47)),
				Episode::new(5, 14, 2301451, "Ozymandias", Some(48)),
				Episode::new(5, 15, 2301453, "Granite State", Some(55)),
				Episode::new(5, 16, 2301455, "Felina", Some(55))
			/* }}} */ ]));

			let mut show = client.get_show(417373).await.unwrap();
			show.episodes.sort_unstable();
			assert_eq!(show, Show::new(417373, "The Venture Bros.", false, 2003, Some(2018), Some(22), &["Action", "Adventure", "Animation"], &[ // {{{
				Episode::new(1, 0, 739360, "The Terrible Secret of Turtle Bay", Some(22)),
				Episode::new(1, 1, 739350, "Dia de los Dangerous", Some(22)),
				Episode::new(1, 2, 739349, "Careers in Science", Some(22)),
				Episode::new(1, 3, 739353, "Home Insecurity", Some(22)),
				Episode::new(1, 4, 739359, "The Incredible Mr. Brisby", Some(22)),
				Episode::new(1, 5, 739351, "Eeney, Meeney, Miney... Magic!", Some(22)),
				Episode::new(1, 6, 739352, "Ghosts of the Sargasso", Some(22)),
				Episode::new(1, 7, 739354, "Ice Station Impossible", Some(22)),
				Episode::new(1, 8, 739355, "Mid-Life Chrysalis", Some(22)),
				Episode::new(1, 9, 739348, "Are You There God, It\'s Me, Dean", Some(22)),
				Episode::new(1, 10, 739358, "Tag-Sale: You\'re It!", Some(22)),
				Episode::new(1, 11, 739356, "Past Tense", Some(22)),
				Episode::new(1, 12, 739361, "The Trial of the Monarch", Some(22)),
				Episode::new(1, 13, 739357, "Return to Spider-Skull Island", Some(22)),
				Episode::new(1, 14, 739347, "A Very Venture Christmas", Some(11)),
				Episode::new(2, 1, 822288, "Powerless in the Face of Death", Some(22)),
				Episode::new(2, 2, 822287, "Hate Floats", Some(22)),
				Episode::new(2, 3, 822286, "Assassinanny 911", Some(22)),
				Episode::new(2, 4, 825198, "Escape to the House of Mummies Part II", Some(22)),
				Episode::new(2, 5, 833403, "Twenty Years to Midnight", Some(22)),
				Episode::new(2, 6, 843050, "Victor. Echo. November.", Some(22)),
				Episode::new(2, 7, 839676, "Love-Bheits", Some(22)),
				Episode::new(2, 8, 845936, "Fallen Arches", Some(22)),
				Episode::new(2, 9, 847451, "Guess Who\'s Coming to State Dinner?", Some(22)),
				Episode::new(2, 10, 847452, "I Know Why the Caged Bird Kills", Some(22)),
				Episode::new(2, 11, 847454, "¡Viva los muertos!", Some(22)),
				Episode::new(2, 12, 847453, "Showdown at Cremation Creek: Part 1", Some(22)),
				Episode::new(2, 13, 878792, "Showdown at Cremation Creek: Part 2", Some(22)),
				Episode::new(3, 1, 1238620, "Shadowman 9: In the Cradle of Destiny", Some(24)),
				Episode::new(3, 2, 1238621, "The Doctor Is Sin", Some(23)),
				Episode::new(3, 3, 1238613, "The Invisible Hand of Fate", Some(23)),
				Episode::new(3, 4, 1238614, "Home Is Where the Hate Is", Some(23)),
				Episode::new(3, 5, 1238615, "The Buddy System", Some(23)),
				Episode::new(3, 6, 1238616, "Dr. Quymn, Medicine Woman", Some(23)),
				Episode::new(3, 7, 1238617, "What Goes Down Must Come Up", Some(23)),
				Episode::new(3, 8, 1238618, "Tears of a Sea Cow", Some(23)),
				Episode::new(3, 9, 1238619, "Now Museum - Now You Don\'t!", Some(23)),
				Episode::new(3, 10, 1238609, "The Lepidopterists", Some(21)),
				Episode::new(3, 11, 1238610, "ORB", Some(23)),
				Episode::new(3, 12, 1238611, "The Family That Slays Together, Stays Together: Part 1", Some(24)),
				Episode::new(3, 13, 1238612, "The Family That Slays Together, Stays Together: Part 2", Some(24)),
				Episode::new(4, 1, 1531880, "Blood of the Father, Heart of Steel", Some(24)),
				Episode::new(4, 2, 1539264, "Handsome Ransom", Some(23)),
				Episode::new(4, 3, 1541106, "Perchance to Dean", Some(23)),
				Episode::new(4, 4, 1560460, "Return to Malice", Some(23)),
				Episode::new(4, 5, 1550277, "The Revenge Society", Some(23)),
				Episode::new(4, 6, 1554061, "Self-Medication", Some(23)),
				Episode::new(4, 7, 1568069, "The Better Man", Some(23)),
				Episode::new(4, 8, 1573782, "Pinstripes and Poltergeists", Some(23)),
				Episode::new(4, 9, 1589919, "The Diving Bell vs. the Butterglider", Some(23)),
				Episode::new(4, 10, 1730658, "Pomp and Circuitry", Some(23)),
				Episode::new(4, 11, 1738335, "Every Which Way But Zeus", Some(23)),
				Episode::new(4, 12, 1738336, "Everybody Comes to Hank\'s", Some(22)),
				Episode::new(4, 13, 1742609, "Bright Lights, Dean City", Some(23)),
				Episode::new(4, 14, 1747772, "Assisted Suicide", Some(23)),
				Episode::new(4, 15, 1752777, "The Silent Partners", Some(23)),
				Episode::new(4, 16, 1772101, "Operation P.R.O.M.", Some(46)),
				Episode::new(4, 17, 2035364, "From the Ladle to the Grave: The Shallow Gravy Story", Some(12)),
				Episode::new(5, 0, 2423804, "A Very Venture Halloween", Some(60)),
				Episode::new(5, 1, 2951250, "What Color Is Your Cleansuit?", Some(46)),
				Episode::new(5, 2, 2951252, "Venture Libre", Some(23)),
				Episode::new(5, 3, 2959752, "Sphinx Rising", Some(23)),
				Episode::new(5, 4, 2973986, "Spanakopita", Some(23)),
				Episode::new(5, 5, 2991116, "O.S.I. Love You", Some(23)),
				Episode::new(5, 6, 3016486, "Momma\'s Boys", Some(23)),
				Episode::new(5, 7, 3024226, "Bot Seeks Bot", Some(23)),
				Episode::new(5, 8, 3040042, "The Devil\'s Grip", Some(23)),
				Episode::new(6, 0, 4334430, "All This and Gargantua-2", Some(48)),
				Episode::new(6, 1, 5024698, "Hostile Makeover", Some(23)),
				Episode::new(6, 2, 5218230, "Maybe No Go", Some(22)),
				Episode::new(6, 3, 5218240, "Faking Miracles", Some(22)),
				Episode::new(6, 4, 5218252, "Rapacity in Blue", Some(22)),
				Episode::new(6, 5, 5218278, "Tanks for Nuthin\'", Some(22)),
				Episode::new(6, 6, 5218290, "It Happening One Night", Some(22)),
				Episode::new(6, 7, 5321340, "A Party for Tarzan", Some(22)),
				Episode::new(6, 8, 5321386, "Red Means Stop", Some(22)),
				Episode::new(7, 1, 7525428, "The Venture Bros. and the Curse of the Haunted Problem", Some(22)),
				Episode::new(7, 2, 8781016, "The Rorqual Affair", Some(22)),
				Episode::new(7, 3, 8842110, "Arrears in Science", Some(23)),
				Episode::new(7, 4, 8842114, "The High Cost of Loathing", Some(22)),
				Episode::new(7, 5, 8842118, "The Inamorata Consequence", Some(22)),
				Episode::new(7, 6, 8972942, "The Bellicose Proxy", Some(22)),
				Episode::new(7, 7, 8980674, "The Unicorn in Captivity", Some(22)),
				Episode::new(7, 8, 8980676, "The Terminus Mandate", Some(22)),
				Episode::new(7, 9, 9035294, "The Forecast Manufacturer", Some(22)),
				Episode::new(7, 10, 9035298, "The Saphrax Protocol", Some(22))
			/* }}} */ ]));

			let mut show = client.get_show(460681).await.unwrap();
			show.episodes.sort_unstable();
			assert_eq!(show, Show::new(460681, "Supernatural", false, 2005, Some(2020), Some(44), &["Drama", "Fantasy", "Horror"], &[ // {{{
				Episode::new(1, 1, 713618, "Pilot", Some(44)),
				Episode::new(1, 2, 713624, "Wendigo", Some(44)),
				Episode::new(1, 3, 713613, "Dead in the Water", Some(44)),
				Episode::new(1, 4, 713617, "Phantom Traveler", Some(43)),
				Episode::new(1, 5, 713611, "Bloody Mary", Some(44)),
				Episode::new(1, 6, 713622, "Skin", Some(42)),
				Episode::new(1, 7, 713615, "Hook Man", Some(41)),
				Episode::new(1, 8, 713612, "Bugs", Some(44)),
				Episode::new(1, 9, 713623, "Home", Some(43)),
				Episode::new(1, 10, 713610, "Asylum", Some(44)),
				Episode::new(1, 11, 713620, "Scarecrow", Some(43)),
				Episode::new(1, 12, 713614, "Faith", Some(44)),
				Episode::new(1, 13, 713619, "Route 666", Some(50)),
				Episode::new(1, 14, 713616, "Nightmare", Some(55)),
				Episode::new(1, 15, 757750, "The Benders", Some(55)),
				Episode::new(1, 16, 713621, "Shadow", Some(55)),
				Episode::new(1, 17, 763064, "Hell House", Some(55)),
				Episode::new(1, 18, 763065, "Something Wicked", Some(55)),
				Episode::new(1, 19, 761316, "Provenance", Some(44)),
				Episode::new(1, 20, 767486, "Dead Man\'s Blood", Some(55)),
				Episode::new(1, 21, 772002, "Salvation", Some(60)),
				Episode::new(1, 22, 775325, "Devil\'s Trap", Some(55)),
				Episode::new(2, 1, 829063, "In My Time of Dying", Some(50)),
				Episode::new(2, 2, 829062, "Everybody Loves a Clown", Some(55)),
				Episode::new(2, 3, 835248, "Bloodlust", Some(50)),
				Episode::new(2, 4, 837741, "Children Shouldn\'t Play with Dead Things", Some(42)),
				Episode::new(2, 5, 851031, "Simon Said", Some(48)),
				Episode::new(2, 6, 855641, "No Exit", Some(47)),
				Episode::new(2, 7, 859555, "The Usual Suspects", Some(50)),
				Episode::new(2, 8, 868625, "Crossroad Blues", Some(42)),
				Episode::new(2, 9, 878590, "Croatoan", Some(47)),
				Episode::new(2, 10, 883872, "Hunted", Some(49)),
				Episode::new(2, 11, 890836, "Playthings", Some(49)),
				Episode::new(2, 12, 942145, "Nightshifter", Some(47)),
				Episode::new(2, 13, 921622, "Houses of the Holy", Some(50)),
				Episode::new(2, 14, 940441, "Born Under a Bad Sign", Some(50)),
				Episode::new(2, 15, 939444, "Tall Tales", Some(48)),
				Episode::new(2, 16, 942786, "Roadkill", Some(40)),
				Episode::new(2, 17, 959976, "Heart", Some(47)),
				Episode::new(2, 18, 964436, "Hollywood Babylon", Some(50)),
				Episode::new(2, 19, 1000536, "Folsom Prison Blues", Some(50)),
				Episode::new(2, 20, 987715, "What Is and What Should Never Be", Some(50)),
				Episode::new(2, 21, 987713, "All Hell Breaks Loose: Part 1", Some(50)),
				Episode::new(2, 22, 987714, "All Hell Breaks Loose: Part 2", Some(55)),
				Episode::new(3, 1, 1032129, "The Magnificent Seven", Some(49)),
				Episode::new(3, 2, 1032134, "The Kids Are Alright", Some(43)),
				Episode::new(3, 3, 1107710, "Bad Day at Black Rock", Some(50)),
				Episode::new(3, 4, 1032135, "Sin City", Some(50)),
				Episode::new(3, 5, 1032137, "Bedtime Stories", Some(50)),
				Episode::new(3, 6, 1032138, "Red Sky at Morning", Some(50)),
				Episode::new(3, 7, 1032139, "Fresh Blood", Some(50)),
				Episode::new(3, 8, 1032140, "A Very Supernatural Christmas", Some(42)),
				Episode::new(3, 9, 1032141, "Malleus Maleficarum", Some(40)),
				Episode::new(3, 10, 1032130, "Dream a Little Dream of Me", Some(55)),
				Episode::new(3, 11, 1032132, "Mystery Spot", Some(55)),
				Episode::new(3, 12, 1032131, "Jus in Bello", Some(55)),
				Episode::new(3, 13, 1032133, "Ghostfacers", Some(42)),
				Episode::new(3, 14, 1206073, "Long Distance Call", Some(41)),
				Episode::new(3, 15, 1222593, "Time Is on My Side", Some(50)),
				Episode::new(3, 16, 1222594, "No Rest for the Wicked", Some(50)),
				Episode::new(4, 1, 1222595, "Lazarus Rising", Some(55)),
				Episode::new(4, 2, 1222606, "Are You There, God? It\'s Me, Dean Winchester", Some(50)),
				Episode::new(4, 3, 1222611, "In the Beginning", Some(50)),
				Episode::new(4, 4, 1222612, "Metamorphosis", Some(60)),
				Episode::new(4, 5, 1222610, "Monster Movie", Some(55)),
				Episode::new(4, 6, 1222613, "Yellow Fever", Some(43)),
				Episode::new(4, 7, 1222614, "It\'s the Great Pumpkin, Sam Winchester", Some(42)),
				Episode::new(4, 8, 1222615, "Wishful Thinking", Some(41)),
				Episode::new(4, 9, 1222616, "I Know What You Did Last Summer", Some(43)),
				Episode::new(4, 10, 1222596, "Heaven and Hell", Some(43)),
				Episode::new(4, 11, 1222597, "Family Remains", Some(43)),
				Episode::new(4, 12, 1222598, "Criss Angel Is a Douche Bag", Some(43)),
				Episode::new(4, 13, 1222599, "After School Special", Some(55)),
				Episode::new(4, 14, 1222600, "Sex and Violence", Some(43)),
				Episode::new(4, 15, 1222601, "Death Takes a Holiday", Some(43)),
				Episode::new(4, 16, 1222602, "On the Head of a Pin", Some(43)),
				Episode::new(4, 17, 1222603, "It\'s a Terrible Life", Some(43)),
				Episode::new(4, 18, 1222604, "The Monster at the End of This Book", Some(43)),
				Episode::new(4, 19, 1222605, "Jump the Shark", Some(43)),
				Episode::new(4, 20, 1222607, "The Rapture", Some(40)),
				Episode::new(4, 21, 1222608, "When the Levee Breaks", Some(43)),
				Episode::new(4, 22, 1222609, "Lucifer Rising", Some(43)),
				Episode::new(5, 1, 1461906, "Sympathy for the Devil", Some(60)),
				Episode::new(5, 2, 1464088, "Good God, Y\'All", Some(60)),
				Episode::new(5, 3, 1464089, "Free to Be You and Me", Some(60)),
				Episode::new(5, 4, 1488984, "The End", Some(60)),
				Episode::new(5, 5, 1488983, "Fallen Idols", Some(60)),
				Episode::new(5, 6, 1506849, "I Believe the Children Are Our Future", Some(42)),
				Episode::new(5, 7, 1507550, "The Curious Case of Dean Winchester", Some(42)),
				Episode::new(5, 8, 1511910, "Changing Channels", Some(60)),
				Episode::new(5, 9, 1533530, "The Real Ghostbusters", Some(60)),
				Episode::new(5, 10, 1532469, "Abandon All Hope", Some(60)),
				Episode::new(5, 11, 1543420, "Sam, Interrupted", Some(60)),
				Episode::new(5, 12, 1552579, "Swap Meat", Some(60)),
				Episode::new(5, 13, 1541419, "The Song Remains the Same", Some(60)),
				Episode::new(5, 14, 1588681, "My Bloody Valentine", Some(60)),
				Episode::new(5, 15, 1588682, "Dead Men Don\'t Wear Plaid", Some(60)),
				Episode::new(5, 16, 1588256, "Dark Side of the Moon", Some(60)),
				Episode::new(5, 17, 1588255, "99 Problems", Some(60)),
				Episode::new(5, 18, 1594854, "Point of No Return", Some(60)),
				Episode::new(5, 19, 1598673, "Hammer of the Gods", Some(60)),
				Episode::new(5, 20, 1599805, "The Devil You Know", Some(60)),
				Episode::new(5, 21, 1626754, "Two Minutes to Midnight", Some(60)),
				Episode::new(5, 22, 1626753, "Swan Song", Some(42)),
				Episode::new(6, 1, 1627706, "Exile on Main St.", Some(42)),
				Episode::new(6, 2, 1627717, "Two and a Half Men", Some(41)),
				Episode::new(6, 3, 1627721, "The Third Man", Some(41)),
				Episode::new(6, 4, 1627722, "Weekend at Bobby\'s", Some(40)),
				Episode::new(6, 5, 1627724, "Live Free or Twihard", Some(42)),
				Episode::new(6, 6, 1627723, "You Can\'t Handle the Truth", Some(60)),
				Episode::new(6, 7, 1627725, "Family Matters", Some(41)),
				Episode::new(6, 8, 1627726, "All Dogs Go to Heaven", Some(42)),
				Episode::new(6, 9, 1627727, "Clap Your Hands If You Believe", Some(43)),
				Episode::new(6, 10, 1627707, "Caged Heat", Some(60)),
				Episode::new(6, 11, 1627708, "Appointment in Samarra", Some(42)),
				Episode::new(6, 12, 1627709, "Like a Virgin", Some(60)),
				Episode::new(6, 13, 1627710, "Unforgiven", Some(60)),
				Episode::new(6, 14, 1627711, "Mannequin 3: The Reckoning", Some(60)),
				Episode::new(6, 15, 1627712, "The French Mistake", Some(42)),
				Episode::new(6, 16, 1627713, "And Then There Were None", Some(60)),
				Episode::new(6, 17, 1627715, "My Heart Will Go On", Some(60)),
				Episode::new(6, 18, 1627714, "Frontierland", Some(60)),
				Episode::new(6, 19, 1627716, "Mommy Dearest", Some(60)),
				Episode::new(6, 20, 1627718, "The Man Who Would Be King", Some(60)),
				Episode::new(6, 21, 1627719, "Let It Bleed", Some(60)),
				Episode::new(6, 22, 1864037, "The Man Who Knew Too Much", Some(43)),
				Episode::new(7, 1, 1995102, "Meet the New Boss", Some(43)),
				Episode::new(7, 2, 1995113, "Hello, Cruel World", Some(43)),
				Episode::new(7, 3, 1995118, "The Girl Next Door", Some(42)),
				Episode::new(7, 4, 1995119, "Defending Your Life", Some(43)),
				Episode::new(7, 5, 1995120, "Shut Up, Dr. Phil", Some(42)),
				Episode::new(7, 6, 1995121, "Slash Fiction", Some(43)),
				Episode::new(7, 7, 1995122, "The Mentalists", Some(42)),
				Episode::new(7, 8, 1995123, "Season Seven, Time for a Wedding!", Some(40)),
				Episode::new(7, 9, 1995124, "How to Win Friends and Influence Monsters", Some(41)),
				Episode::new(7, 10, 1995103, "Death\'s Door", Some(42)),
				Episode::new(7, 11, 1995104, "Adventures in Babysitting", Some(43)),
				Episode::new(7, 12, 1995105, "Time After Time", Some(43)),
				Episode::new(7, 13, 1995106, "The Slice Girls", Some(42)),
				Episode::new(7, 14, 1995107, "Plucky Pennywhistle\'s Magical Menagerie", Some(42)),
				Episode::new(7, 15, 1995108, "Repo Man", Some(43)),
				Episode::new(7, 16, 1995109, "Out with the Old", Some(43)),
				Episode::new(7, 17, 1995110, "The Born-Again Identity", Some(43)),
				Episode::new(7, 18, 1995111, "Party On, Garth", Some(43)),
				Episode::new(7, 19, 1995112, "Of Grave Importance", Some(42)),
				Episode::new(7, 20, 1995114, "The Girl with the Dungeons and Dragons Tattoo", Some(43)),
				Episode::new(7, 21, 1995115, "Reading is Fundamental", Some(42)),
				Episode::new(7, 22, 1995116, "There Will Be Blood", Some(43)),
				Episode::new(7, 23, 1995117, "Survival of the Fittest", Some(42)),
				Episode::new(8, 1, 2373078, "We Need to Talk About Kevin", Some(42)),
				Episode::new(8, 2, 2404088, "What\'s Up, Tiger Mommy?", Some(42)),
				Episode::new(8, 3, 2293862, "Heartache", Some(40)),
				Episode::new(8, 4, 2323362, "Bitten", Some(42)),
				Episode::new(8, 5, 2388382, "Blood Brother", Some(43)),
				Episode::new(8, 6, 2372414, "Southern Comfort", Some(43)),
				Episode::new(8, 7, 2425360, "A Little Slice of Kevin", Some(43)),
				Episode::new(8, 8, 2426140, "Hunteri Heroici", Some(43)),
				Episode::new(8, 9, 2426138, "Citizen Fang", Some(43)),
				Episode::new(8, 10, 2425378, "Torn and Frayed", Some(43)),
				Episode::new(8, 11, 2498912, "LARP and the Real Girl", Some(43)),
				Episode::new(8, 12, 2506990, "As Time Goes By", Some(43)),
				Episode::new(8, 13, 2578700, "Everybody Hates Hitler", Some(43)),
				Episode::new(8, 14, 2503932, "Trial and Error", Some(43)),
				Episode::new(8, 15, 2578704, "Man\'s Best Friend with Benefits", Some(43)),
				Episode::new(8, 16, 2578708, "Remember the Titans", Some(42)),
				Episode::new(8, 17, 2692382, "Goodbye Stranger", Some(43)),
				Episode::new(8, 18, 2621384, "Freaks and Geeks", Some(42)),
				Episode::new(8, 19, 2699592, "Taxi Driver", Some(43)),
				Episode::new(8, 20, 2699596, "Pac-Man Fever", Some(43)),
				Episode::new(8, 21, 2734858, "The Great Escapist", Some(43)),
				Episode::new(8, 22, 2805138, "Clip Show", Some(43)),
				Episode::new(8, 23, 2805140, "Sacrifice", Some(42)),
				Episode::new(9, 1, 2702688, "I Think I\'m Gonna Like It Here", Some(42)),
				Episode::new(9, 2, 3025548, "Devil May Care", Some(41)),
				Episode::new(9, 3, 3097142, "I\'m No Angel", Some(42)),
				Episode::new(9, 4, 3098422, "Slumber Party", Some(42)),
				Episode::new(9, 5, 3112660, "Dog Dean Afternoon", Some(41)),
				Episode::new(9, 6, 3112656, "Heaven Can\'t Wait", Some(42)),
				Episode::new(9, 7, 3141550, "Bad Boys", Some(42)),
				Episode::new(9, 8, 3158272, "Rock and a Hard Place", Some(42)),
				Episode::new(9, 9, 3158274, "Holy Terror", Some(42)),
				Episode::new(9, 10, 3158276, "Road Trip", Some(43)),
				Episode::new(9, 11, 3158278, "First Born", Some(42)),
				Episode::new(9, 12, 3158284, "Sharp Teeth", Some(42)),
				Episode::new(9, 13, 3158282, "The Purge", Some(42)),
				Episode::new(9, 14, 3182872, "Captives", Some(42)),
				Episode::new(9, 15, 3182874, "#thinman", Some(42)),
				Episode::new(9, 16, 3182880, "Blade Runners", Some(42)),
				Episode::new(9, 17, 3182878, "Mother\'s Little Helper", Some(42)),
				Episode::new(9, 18, 3182886, "Meta Fiction", Some(42)),
				Episode::new(9, 19, 3182884, "Alex Annie Alexis Ann", Some(42)),
				Episode::new(9, 20, 3182892, "Bloodlines", Some(42)),
				Episode::new(9, 21, 3182890, "King of the Damned", Some(42)),
				Episode::new(9, 22, 3182896, "Stairway to Heaven", Some(42)),
				Episode::new(9, 23, 3182894, "Do You Believe in Miracles", Some(42)),
				Episode::new(10, 1, 3532628, "Black", Some(43)),
				Episode::new(10, 2, 3831300, "Reichenbach", Some(43)),
				Episode::new(10, 3, 3831310, "Soul Survivor", Some(43)),
				Episode::new(10, 4, 3845906, "Paper Moon", Some(41)),
				Episode::new(10, 5, 3845910, "Fan Fiction", Some(43)),
				Episode::new(10, 6, 3860118, "Ask Jeeves", Some(43)),
				Episode::new(10, 7, 3860122, "Girls, Girls, Girls", Some(42)),
				Episode::new(10, 8, 3997488, "Hibbing 911", Some(42)),
				Episode::new(10, 9, 3997492, "The Things We Left Behind", Some(42)),
				Episode::new(10, 10, 3997490, "The Hunter Games", Some(43)),
				Episode::new(10, 11, 4274892, "There\'s No Place Like Home", Some(43)),
				Episode::new(10, 12, 4344320, "About a Boy", Some(41)),
				Episode::new(10, 13, 4344332, "Halt & Catch Fire", Some(42)),
				Episode::new(10, 14, 4292452, "The Executioner\'s Song", Some(42)),
				Episode::new(10, 15, 4333228, "The Things They Carried", Some(42)),
				Episode::new(10, 16, 4427940, "Paint It Black", Some(42)),
				Episode::new(10, 17, 4443976, "Inside Man", Some(42)),
				Episode::new(10, 18, 4489472, "Book of the Damned", Some(43)),
				Episode::new(10, 19, 4489474, "The Werther Project", Some(43)),
				Episode::new(10, 20, 4489476, "Angel Heart", Some(42)),
				Episode::new(10, 21, 4489482, "Dark Dynasty", Some(42)),
				Episode::new(10, 22, 4489484, "The Prisoner", Some(42)),
				Episode::new(10, 23, 4489486, "Brother\'s Keeper", Some(42)),
				Episode::new(11, 1, 4346788, "Out of the Darkness, Into the Fire", Some(43)),
				Episode::new(11, 2, 4857906, "Form and Void", Some(43)),
				Episode::new(11, 3, 4857918, "The Bad Seed", Some(42)),
				Episode::new(11, 4, 4995876, "Baby", Some(41)),
				Episode::new(11, 5, 5056820, "Thin Lizzie", Some(42)),
				Episode::new(11, 6, 5056822, "Our Little World", Some(42)),
				Episode::new(11, 7, 5056828, "Plush", Some(41)),
				Episode::new(11, 8, 5056832, "Just My Imagination", Some(42)),
				Episode::new(11, 9, 5056836, "O Brother, Where Art Thou?", Some(42)),
				Episode::new(11, 10, 5087242, "The Devil in the Details", Some(42)),
				Episode::new(11, 11, 5239780, "Into the Mystic", Some(42)),
				Episode::new(11, 12, 5265450, "Don\'t You Forget About Me", Some(42)),
				Episode::new(11, 13, 5352054, "Love Hurts", Some(41)),
				Episode::new(11, 14, 5390282, "The Vessel", Some(42)),
				Episode::new(11, 15, 5432122, "Beyond the Mat", Some(42)),
				Episode::new(11, 16, 5432150, "Safe House", Some(42)),
				Episode::new(11, 17, 5432156, "Red Meat", Some(41)),
				Episode::new(11, 18, 5539106, "Hell\'s Angel", Some(42)),
				Episode::new(11, 19, 5538526, "The Chitters", Some(41)),
				Episode::new(11, 20, 5547512, "Don\'t Call Me Shurley", Some(42)),
				Episode::new(11, 21, 5547522, "All in the Family", Some(41)),
				Episode::new(11, 22, 5547530, "We Happy Few", Some(41)),
				Episode::new(11, 23, 5547538, "Alpha and Omega", Some(42)),
				Episode::new(12, 1, 5540416, "Keep Calm and Carry On", Some(41)),
				Episode::new(12, 2, 5669110, "Mamma Mia", Some(42)),
				Episode::new(12, 3, 5669112, "The Foundry", Some(42)),
				Episode::new(12, 4, 5669114, "American Nightmare", Some(42)),
				Episode::new(12, 5, 5669118, "The One You\'ve Been Waiting For", Some(41)),
				Episode::new(12, 6, 5669116, "Celebrating the Life of Asa Fox", Some(42)),
				Episode::new(12, 7, 5669120, "Rock Never Dies", Some(42)),
				Episode::new(12, 8, 5669122, "LOTUS", Some(41)),
				Episode::new(12, 9, 5669126, "First Blood", Some(41)),
				Episode::new(12, 10, 5669124, "Lily Sunder Has Some Regrets", Some(42)),
				Episode::new(12, 11, 5669128, "Regarding Dean", Some(42)),
				Episode::new(12, 12, 5669130, "Stuck in the Middle (With You)", Some(42)),
				Episode::new(12, 13, 5669132, "Family Feud", Some(42)),
				Episode::new(12, 14, 5669134, "The Raid", Some(41)),
				Episode::new(12, 15, 5669138, "Somewhere Between Heaven and Hell", Some(41)),
				Episode::new(12, 16, 5669136, "Ladies Drink Free", Some(42)),
				Episode::new(12, 17, 5669140, "The British Invasion", Some(42)),
				Episode::new(12, 18, 5669142, "The Memory Remains", Some(42)),
				Episode::new(12, 19, 5669144, "The Future", Some(42)),
				Episode::new(12, 20, 5669146, "Twigs & Twine & Tasha Banes", Some(42)),
				Episode::new(12, 21, 5669148, "There\'s Something About Mary", Some(42)),
				Episode::new(12, 22, 5669150, "Who We Are", Some(42)),
				Episode::new(12, 23, 5669152, "All Along the Watchtower", Some(42)),
				Episode::new(13, 1, 6403432, "Lost and Found", Some(41)),
				Episode::new(13, 2, 6746126, "The Rising Son", Some(42)),
				Episode::new(13, 3, 6746128, "Patience", Some(42)),
				Episode::new(13, 4, 6746132, "The Big Empty", Some(42)),
				Episode::new(13, 5, 6746134, "Advanced Thanatology", Some(41)),
				Episode::new(13, 6, 6746140, "Tombstone", Some(41)),
				Episode::new(13, 7, 6746146, "War of the Worlds", Some(42)),
				Episode::new(13, 8, 6746148, "The Scorpion and the Frog", Some(41)),
				Episode::new(13, 9, 6746160, "The Bad Place", Some(41)),
				Episode::new(13, 10, 6746156, "Wayward Sisters", Some(42)),
				Episode::new(13, 11, 6746164, "Breakdown", Some(41)),
				Episode::new(13, 12, 6746170, "Various & Sundry Villains", Some(42)),
				Episode::new(13, 13, 6746168, "Devil\'s Bargain", Some(41)),
				Episode::new(13, 14, 6746174, "Good Intentions", Some(42)),
				Episode::new(13, 15, 6746182, "A Most Holy Man", Some(42)),
				Episode::new(13, 16, 6746190, "ScoobyNatural", Some(42)),
				Episode::new(13, 17, 6746194, "The Thing", Some(42)),
				Episode::new(13, 18, 6746200, "Bring \'em Back Alive", Some(42)),
				Episode::new(13, 19, 6746208, "Funeralia", Some(42)),
				Episode::new(13, 20, 6746210, "Unfinished Business", Some(42)),
				Episode::new(13, 21, 6746216, "Beat the Devil", Some(42)),
				Episode::new(13, 22, 6746226, "Exodus", Some(41)),
				Episode::new(13, 23, 6746224, "Let the Good Times Roll", Some(42)),
				Episode::new(14, 1, 8226756, "Stranger in a Strange Land", Some(41)),
				Episode::new(14, 2, 8408494, "Gods and Monsters", Some(41)),
				Episode::new(14, 3, 8408498, "The Scar", Some(42)),
				Episode::new(14, 4, 8408504, "Mint Condition", Some(41)),
				Episode::new(14, 5, 8408508, "Nightmare Logic", Some(42)),
				Episode::new(14, 6, 8408506, "Optimism", Some(41)),
				Episode::new(14, 7, 8408502, "Unhuman Nature", Some(41)),
				Episode::new(14, 8, 8408500, "Byzantium", Some(42)),
				Episode::new(14, 9, 8408512, "The Spear", Some(41)),
				Episode::new(14, 10, 8408510, "Nihilism", Some(42)),
				Episode::new(14, 11, 8962434, "Damaged Goods", Some(42)),
				Episode::new(14, 12, 8962440, "Prophet and Loss", Some(42)),
				Episode::new(14, 13, 8962446, "Lebanon", Some(42)),
				Episode::new(14, 14, 9271138, "Ouroboros", Some(42)),
				Episode::new(14, 15, 9271140, "Peace of Mind", Some(42)),
				Episode::new(14, 16, 9271142, "Don\'t Go in the Woods", Some(41)),
				Episode::new(14, 17, 9271146, "Game Night", Some(42)),
				Episode::new(14, 18, 9271148, "Absence", Some(42)),
				Episode::new(14, 19, 9271150, "Jack in the Box", Some(42)),
				Episode::new(14, 20, 9271152, "Moriah", Some(41)),
				Episode::new(15, 1, 9695548, "Back and to the Future", Some(41)),
				Episode::new(15, 2, 10067710, "Raising Hell", Some(42)),
				Episode::new(15, 3, 10250608, "The Rupture", Some(41)),
				Episode::new(15, 4, 10067734, "Atomic Monsters", Some(42)),
				Episode::new(15, 5, 10067742, "Proverbs 17:3", Some(42)),
				Episode::new(15, 6, 10067744, "Golden Time", Some(42)),
				Episode::new(15, 7, 10085356, "Last Call", Some(41)),
				Episode::new(15, 8, 10717880, "Our Father, Who Aren\'t in Heaven", Some(42)),
				Episode::new(15, 9, 10717886, "The Trap", Some(42)),
				Episode::new(15, 10, 10717892, "The Heroes\' Journey", Some(42)),
				Episode::new(15, 11, 10717964, "The Gamblers", Some(42)),
				Episode::new(15, 12, 10717906, "Galaxy Brain", Some(42)),
				Episode::new(15, 13, 10717912, "Destiny\'s Child", Some(42)),
				Episode::new(15, 14, 10717914, "Last Holiday", Some(40)),
				Episode::new(15, 15, 10717918, "Gimme Shelter", Some(41)),
				Episode::new(15, 16, 10717922, "Drag Me Away (From You)", Some(42)),
				Episode::new(15, 17, 10717928, "Unity", Some(42)),
				Episode::new(15, 18, 10717930, "Despair", Some(41)),
				Episode::new(15, 19, 10717932, "Inherit the Earth", Some(42)),
				Episode::new(15, 20, 10073984, "Carry On", Some(42)),
			/* }}} */ ]));
		}
	} // }}}

	#[tokio::test]
	async fn shows_multi() /* {{{ */ {
		let (_dir, mut client) = client().await;

		for i in 0..5 {
			println!(">>> {}", i);
			let mut shows = client.get_shows_by_id(&[1758772, 898266, 1219024]).await.unwrap();
			assert_eq!(shows.len(), 3);
			let mut show = shows.remove(&1758772).unwrap();
			show.episodes.sort_unstable();
			assert_eq!(show, Show::new(1758772, "Switched at Birth", false, 2011, Some(2017), Some(60), &["Drama", "Family", "Romance"], &[ // {{{
				Episode::new(1, 1, 1953306, "This Is Not a Pipe", Some(46)),
				Episode::new(1, 2, 1953299, "American Gothic", Some(41)),
				Episode::new(1, 3, 1953304, "Portrait of My Father", Some(44)),
				Episode::new(1, 4, 1953300, "Dance Amongst Daggers", Some(44)),
				Episode::new(1, 5, 1953301, "Dogs Playing Poker", Some(45)),
				Episode::new(1, 6, 1953305, "The Persistence of Memory", Some(44)),
				Episode::new(1, 7, 1943479, "The Stag Hunt", Some(44)),
				Episode::new(1, 8, 1953302, "Pandora\'s Box", Some(45)),
				Episode::new(1, 9, 1953303, "Paradise Lost", Some(42)),
				Episode::new(1, 10, 1953298, "The Homecoming", Some(43)),
				Episode::new(1, 11, 2151279, "Starry Night", Some(41)),
				Episode::new(1, 12, 2091111, "The Tempest", Some(41)),
				Episode::new(1, 13, 2171455, "Self-Portrait with a Bandaged Ear", Some(42)),
				Episode::new(1, 14, 2171457, "Les Soeurs D\'Estrees", Some(42)),
				Episode::new(1, 15, 2171459, "Expulsion from the Garden of Eden", Some(41)),
				Episode::new(1, 16, 2171461, "Las Dos Fridas", Some(41)),
				Episode::new(1, 17, 2171463, "Protect Me from What I Want", Some(42)),
				Episode::new(1, 18, 2149991, "The Art of Painting", Some(41)),
				Episode::new(1, 19, 2247951, "Write a Lonely Soldier", Some(41)),
				Episode::new(1, 20, 2260511, "Game On", Some(42)),
				Episode::new(1, 21, 2280787, "The Sleep of Reason Produces Monsters", Some(42)),
				Episode::new(1, 22, 2264233, "Venus, Cupid, Folly, and Time", Some(43)),
				Episode::new(1, 23, 2380881, "This Is the Color of My Dreams", Some(42)),
				Episode::new(1, 24, 2200422, "The Intruder", Some(42)),
				Episode::new(1, 25, 2200424, "The Shock of Being Seen", Some(42)),
				Episode::new(1, 26, 2200426, "Tree of Forgiveness", Some(42)),
				Episode::new(1, 27, 2200428, "The Declaration of Independence", Some(41)),
				Episode::new(1, 28, 2200432, "We Are the Kraken of Our Own Sinking Ships", Some(42)),
				Episode::new(1, 29, 2200434, "The Trial", Some(42)),
				Episode::new(1, 30, 2200436, "Street Noises Invade the House", Some(42)),
				Episode::new(2, 0, 3097016, "Everyone Is Igual: Back for Season Two", None),
				Episode::new(2, 1, 2341888, "The Door to Freedom", Some(44)),
				Episode::new(2, 2, 2606760, "The Awakening Conscience", Some(44)),
				Episode::new(2, 3, 2621604, "Duel Between Two Women", Some(44)),
				Episode::new(2, 4, 2549168, "Dressing for the Charade", Some(41)),
				Episode::new(2, 5, 2650888, "The Acquired Inability to Escape", Some(41)),
				Episode::new(2, 6, 2662256, "Human/Need/Desire", Some(41)),
				Episode::new(2, 7, 2679928, "Drive in the Knife", Some(41)),
				Episode::new(2, 8, 2707026, "Tight Rope Walker", Some(44)),
				Episode::new(2, 9, 2721120, "Uprising", Some(44)),
				Episode::new(2, 10, 2711310, "Introducing the Miracle", Some(44)),
				Episode::new(2, 11, 2767620, "Mother and Child Divided", Some(44)),
				Episode::new(2, 12, 2990394, "Distorted House", Some(43)),
				Episode::new(2, 13, 2976830, "The Good Samaritan", Some(44)),
				Episode::new(2, 14, 2893496, "He Did What He Wanted", Some(44)),
				Episode::new(2, 15, 3007628, "Ecce Mono", Some(41)),
				Episode::new(2, 16, 2982932, "The Physical Impossibility of Death in the Mind of Someone Living", Some(44)),
				Episode::new(2, 17, 2958864, "Prudence, Avarice, Lust, Justice, Anger", Some(41)),
				Episode::new(2, 18, 2982938, "As the Shadows Deepen", Some(44)),
				Episode::new(2, 19, 2993418, "What Goes Up Must Come Down", Some(41)),
				Episode::new(2, 20, 3007632, "The Merrymakers", Some(41)),
				Episode::new(2, 21, 3043554, "Departure of Summer", Some(41)),
				Episode::new(3, 1, 3090846, "Drowning Girl", Some(42)),
				Episode::new(3, 2, 3315432, "Your Body Is a Battleground", Some(42)),
				Episode::new(3, 3, 3315446, "Fountain", Some(40)),
				Episode::new(3, 4, 3315456, "It Hurts to Wait with Love If Love Is Somewhere Else", Some(42)),
				Episode::new(3, 5, 3356216, "Have You Really the Courage?", Some(41)),
				Episode::new(3, 6, 3362000, "The Scream", Some(42)),
				Episode::new(3, 7, 3450960, "Memory Is Your Image of Perfection", Some(41)),
				Episode::new(3, 8, 3455710, "Dance Me to the End of Love", Some(41)),
				Episode::new(3, 9, 3455716, "The Past (Forgotten-Swallowed)", Some(42)),
				Episode::new(3, 10, 3527802, "The Ambush", Some(42)),
				Episode::new(3, 11, 3527804, "Love Seduces Innocence, Pleasure Entraps, and Remorse Follows", Some(40)),
				Episode::new(3, 12, 3589492, "Love Among the Ruins", Some(42)),
				Episode::new(3, 13, 3588558, "Like a Snowball Down a Mountain", Some(43)),
				Episode::new(3, 14, 3588556, "Oh, Future!", Some(43)),
				Episode::new(3, 15, 3588560, "And We Bring the Light", Some(43)),
				Episode::new(3, 16, 3727770, "The Image Disappears", Some(42)),
				Episode::new(3, 17, 3746404, "Girl with Death Mask (She Plays Alone)", Some(43)),
				Episode::new(3, 18, 3761286, "It Isn\'t What You Think", Some(43)),
				Episode::new(3, 19, 3790258, "You Will Not Escape", Some(43)),
				Episode::new(3, 20, 3854428, "The Girl on the Cliff", Some(43)),
				Episode::new(3, 21, 3827664, "And Life Begins Right Away", Some(43)),
				Episode::new(3, 22, 4229352, "Yuletide Fortune Tellers", Some(43)),
				Episode::new(4, 1, 3955204, "And It Cannot Be Changed", Some(43)),
				Episode::new(4, 2, 4162854, "Bracing the Waves", Some(42)),
				Episode::new(4, 3, 4191406, "I Lock the Door Upon Myself", Some(43)),
				Episode::new(4, 4, 4227422, "We Were So Close That Nothing Use to Stand Between Us", Some(43)),
				Episode::new(4, 5, 4227434, "At the First Clear Word", Some(43)),
				Episode::new(4, 6, 4227442, "Black and Gray", Some(43)),
				Episode::new(4, 7, 4273050, "Fog and Storm and Rain", Some(43)),
				Episode::new(4, 8, 4286688, "Art Like Love Is Dedication", Some(43)),
				Episode::new(4, 9, 4351374, "The Player\'s Choice", Some(43)),
				Episode::new(4, 10, 4371268, "There Is My Heart", Some(43)),
				Episode::new(4, 11, 4520968, "To Repel Ghosts", Some(43)),
				Episode::new(4, 12, 4608606, "How Does a Girl Like You Get to Be a Girl Like You", Some(43)),
				Episode::new(4, 13, 4644262, "Between Hope and Fear", Some(43)),
				Episode::new(4, 14, 4667606, "We Mourn, We Weep, We Love Again", Some(42)),
				Episode::new(4, 15, 4696414, "Instead of Damning the Darkness, It\'s Better to Light a Little Lantern", Some(43)),
				Episode::new(4, 16, 4715596, "Borrowing Your Enemy\'s Arrows", Some(43)),
				Episode::new(4, 17, 4732618, "To the Victor Belong the Spoils", Some(43)),
				Episode::new(4, 18, 4758690, "The Accommodations of Desire", Some(43)),
				Episode::new(4, 19, 4762322, "A Mad Tea Party", Some(43)),
				Episode::new(4, 20, 4841860, "And Always Searching for Beauty", Some(42)),
				Episode::new(5, 1, 5155250, "The Call", Some(42)),
				Episode::new(5, 2, 5360258, "This Has to Do with Me", Some(42)),
				Episode::new(5, 3, 5360266, "Surprise", Some(42)),
				Episode::new(5, 4, 5379234, "Relation of Lines and Colors", Some(42)),
				Episode::new(5, 5, 5512250, "Occupy Truth", Some(42)),
				Episode::new(5, 6, 5509956, "Four Ages in Life", Some(42)),
				Episode::new(5, 7, 5519592, "Memory (The Heart)", Some(42)),
				Episode::new(5, 8, 5541716, "Left in Charge", Some(42)),
				Episode::new(5, 9, 5541722, "The Wolf Is Waiting", Some(42)),
				Episode::new(5, 10, 6762660, "Long Live Love", Some(63))
			/* }}} */ ]));

			let mut show = shows.remove(&898266).unwrap();
			show.episodes.sort_unstable();
			assert_eq!(show, Show::new(898266, "The Big Bang Theory", false, 2007, Some(2019), Some(22), &["Comedy", "Romance"], &[ // {{{
				Episode::new(1, 0, 2176186, "Unaired Pilot", Some(23)),
				Episode::new(1, 1, 775431, "Pilot", Some(23)),
				Episode::new(1, 2, 1091289, "The Big Bran Hypothesis", Some(21)),
				Episode::new(1, 3, 1091290, "The Fuzzy Boots Corollary", Some(22)),
				Episode::new(1, 4, 1091291, "The Luminous Fish Effect", Some(21)),
				Episode::new(1, 5, 1091292, "The Hamburger Postulate", Some(20)),
				Episode::new(1, 6, 1091293, "The Middle Earth Paradigm", Some(21)),
				Episode::new(1, 7, 1136041, "The Dumpling Paradox", Some(21)),
				Episode::new(1, 8, 1127389, "The Grasshopper Experiment", Some(21)),
				Episode::new(1, 9, 1127390, "The Cooper-Hofstadter Polarization", Some(20)),
				Episode::new(1, 10, 1127384, "The Loobenfeld Decay", Some(21)),
				Episode::new(1, 11, 1127385, "The Pancake Batter Anomaly", Some(22)),
				Episode::new(1, 12, 1127386, "The Jerusalem Duality", Some(20)),
				Episode::new(1, 13, 1127387, "The Bat Jar Conjecture", Some(22)),
				Episode::new(1, 14, 1127905, "The Nerdvana Annihilation", Some(20)),
				Episode::new(1, 15, 1127906, "The Pork Chop Indeterminacy", Some(22)),
				Episode::new(1, 16, 1127907, "The Peanut Reaction", Some(20)),
				Episode::new(1, 17, 1127908, "The Tangerine Factor", Some(20)),
				Episode::new(2, 1, 1256599, "The Bad Fish Paradigm", Some(22)),
				Episode::new(2, 2, 1256030, "The Codpiece Topology", Some(21)),
				Episode::new(2, 3, 1256034, "The Barbarian Sublimation", Some(21)),
				Episode::new(2, 4, 1256035, "The Griffin Equivalency", Some(21)),
				Episode::new(2, 5, 1256036, "The Euclid Alternative", Some(20)),
				Episode::new(2, 6, 1256037, "The Cooper-Nowitzki Theorem", Some(21)),
				Episode::new(2, 7, 1256038, "The Panty Piñata Polarization", Some(21)),
				Episode::new(2, 8, 1256039, "The Lizard-Spock Expansion", Some(20)),
				Episode::new(2, 9, 1256040, "The White Asparagus Triangulation", Some(21)),
				Episode::new(2, 10, 1256020, "The Vartabedian Conundrum", Some(21)),
				Episode::new(2, 11, 1256021, "The Bath Item Gift Hypothesis", Some(21)),
				Episode::new(2, 12, 1256022, "The Killer Robot Instability", Some(21)),
				Episode::new(2, 13, 1256023, "The Friendship Algorithm", Some(21)),
				Episode::new(2, 14, 1256024, "The Financial Permeability", Some(21)),
				Episode::new(2, 15, 1256025, "The Maternal Capacitance", Some(21)),
				Episode::new(2, 16, 1256026, "The Cushion Saturation", Some(21)),
				Episode::new(2, 17, 1256027, "The Terminator Decoupling", Some(21)),
				Episode::new(2, 18, 1256028, "The Work Song Nanocluster", Some(20)),
				Episode::new(2, 19, 1256029, "The Dead Hooker Juxtaposition", Some(21)),
				Episode::new(2, 20, 1256031, "The Hofstadter Isotope", Some(20)),
				Episode::new(2, 21, 1256032, "The Vegas Renormalization", Some(21)),
				Episode::new(2, 22, 1256033, "The Classified Materials Turbulence", Some(19)),
				Episode::new(2, 23, 1426233, "The Monopolar Expedition", Some(21)),
				Episode::new(3, 1, 1487706, "The Electric Can Opener Fluctuation", Some(21)),
				Episode::new(3, 2, 1495238, "The Jiminy Conjecture", Some(21)),
				Episode::new(3, 3, 1494210, "The Gothowitz Deviation", Some(20)),
				Episode::new(3, 4, 1523181, "The Pirate Solution", Some(20)),
				Episode::new(3, 5, 1528188, "The Creepy Candy Coating Corollary", Some(20)),
				Episode::new(3, 6, 1523180, "The Cornhusker Vortex", Some(20)),
				Episode::new(3, 7, 1535334, "The Guitarist Amplification", Some(20)),
				Episode::new(3, 8, 1536922, "The Adhesive Duck Deficiency", Some(21)),
				Episode::new(3, 9, 1544819, "The Vengeance Formulation", Some(19)),
				Episode::new(3, 10, 1558529, "The Gorilla Experiment", Some(21)),
				Episode::new(3, 11, 1558530, "The Maternal Congruence", Some(20)),
				Episode::new(3, 12, 1567073, "The Psychic Vortex", Some(20)),
				Episode::new(3, 13, 1570312, "The Bozeman Reaction", Some(20)),
				Episode::new(3, 14, 1588684, "The Einstein Approximation", Some(20)),
				Episode::new(3, 15, 1589887, "The Large Hadron Collision", Some(21)),
				Episode::new(3, 16, 1592080, "The Excelsior Acquisition", Some(21)),
				Episode::new(3, 17, 1607974, "The Precious Fragmentation", Some(20)),
				Episode::new(3, 18, 1609715, "The Pants Alternative", Some(21)),
				Episode::new(3, 19, 1609410, "The Wheaton Recurrence", Some(21)),
				Episode::new(3, 20, 1628617, "The Spaghetti Catalyst", Some(20)),
				Episode::new(3, 21, 1648755, "The Plimpton Stimulation", Some(21)),
				Episode::new(3, 22, 1648756, "The Staircase Implementation", Some(20)),
				Episode::new(3, 23, 1627740, "The Lunar Excitation", Some(20)),
				Episode::new(4, 1, 1632224, "The Robotic Manipulation", Some(21)),
				Episode::new(4, 2, 1632235, "The Cruciferous Vegetable Amplification", Some(21)),
				Episode::new(4, 3, 1632240, "The Zazzy Substitution", Some(21)),
				Episode::new(4, 4, 1632241, "The Hot Troll Deviation", Some(20)),
				Episode::new(4, 5, 1632242, "The Desperation Emanation", Some(21)),
				Episode::new(4, 6, 1632243, "The Irish Pub Formulation", Some(25)),
				Episode::new(4, 7, 1632244, "The Apology Insufficiency", Some(21)),
				Episode::new(4, 8, 1632245, "The 21-Second Excitation", Some(19)),
				Episode::new(4, 9, 1632246, "The Boyfriend Complexity", Some(21)),
				Episode::new(4, 10, 1632225, "The Alien Parasite Hypothesis", Some(20)),
				Episode::new(4, 11, 1632226, "The Justice League Recombination", Some(20)),
				Episode::new(4, 12, 1632227, "The Bus Pants Utilization", Some(21)),
				Episode::new(4, 13, 1632228, "The Love Car Displacement", Some(21)),
				Episode::new(4, 14, 1632229, "The Thespian Catalyst", Some(21)),
				Episode::new(4, 15, 1632230, "The Benefactor Factor", Some(20)),
				Episode::new(4, 16, 1632231, "The Cohabitation Formulation", Some(21)),
				Episode::new(4, 17, 1632232, "The Toast Derivation", Some(22)),
				Episode::new(4, 18, 1632233, "The Prestidigitation Approximation", Some(21)),
				Episode::new(4, 19, 1632234, "The Zarnecki Incursion", Some(23)),
				Episode::new(4, 20, 1632236, "The Herb Garden Germination", Some(21)),
				Episode::new(4, 21, 1632237, "The Agreement Dissection", Some(21)),
				Episode::new(4, 22, 1632238, "The Wildebeest Implementation", Some(21)),
				Episode::new(4, 23, 1632239, "The Engagement Reaction", Some(20)),
				Episode::new(4, 24, 1918407, "The Roommate Transmogrification", Some(22)),
				Episode::new(5, 1, 1980710, "The Skank Reflex Analysis", Some(22)),
				Episode::new(5, 2, 2033019, "The Infestation Hypothesis", Some(20)),
				Episode::new(5, 3, 2057241, "The Pulled Groin Extrapolation", Some(20)),
				Episode::new(5, 4, 2058495, "The Wiggly Finger Catalyst", Some(20)),
				Episode::new(5, 5, 2063481, "The Russian Rocket Reaction", Some(22)),
				Episode::new(5, 6, 2079337, "The Rhinitis Revelation", Some(22)),
				Episode::new(5, 7, 2082016, "The Good Guy Fluctuation", Some(20)),
				Episode::new(5, 8, 2092310, "The Isolation Permutation", Some(21)),
				Episode::new(5, 9, 2100287, "The Ornithophobia Diffusion", Some(21)),
				Episode::new(5, 10, 2105978, "The Flaming Spittoon Acquisition", Some(21)),
				Episode::new(5, 11, 2119890, "The Speckerman Recurrence", Some(20)),
				Episode::new(5, 12, 2136314, "The Shiny Trinket Maneuver", Some(21)),
				Episode::new(5, 13, 2139151, "The Recombination Hypothesis", Some(21)),
				Episode::new(5, 14, 2181587, "The Beta Test Initiation", Some(21)),
				Episode::new(5, 15, 2192687, "The Friendship Contraction", Some(21)),
				Episode::new(5, 16, 2209665, "The Vacation Solution", Some(22)),
				Episode::new(5, 17, 2220955, "The Rothman Disintegration", Some(21)),
				Episode::new(5, 18, 2238153, "The Werewolf Transformation", Some(21)),
				Episode::new(5, 19, 2275203, "The Weekend Vortex", Some(20)),
				Episode::new(5, 20, 2275201, "The Transporter Malfunction", Some(21)),
				Episode::new(5, 21, 2300453, "The Hawking Excitation", Some(19)),
				Episode::new(5, 22, 2332311, "The Stag Convergence", Some(21)),
				Episode::new(5, 23, 2372043, "The Launch Acceleration", Some(21)),
				Episode::new(5, 24, 2379487, "The Countdown Reflection", Some(21)),
				Episode::new(6, 1, 2194640, "The Date Night Variable", Some(21)),
				Episode::new(6, 2, 2392630, "The Decoupling Fluctuation", Some(22)),
				Episode::new(6, 3, 2378184, "The Higgs Boson Observation", Some(19)),
				Episode::new(6, 4, 2411550, "The Re-Entry Minimization", Some(21)),
				Episode::new(6, 5, 2411582, "The Holographic Excitation", Some(21)),
				Episode::new(6, 6, 2450064, "The Extract Obliteration", Some(21)),
				Episode::new(6, 7, 2462410, "The Habitation Configuration", Some(21)),
				Episode::new(6, 8, 2502436, "The 43 Peculiarity", Some(21)),
				Episode::new(6, 9, 2505340, "The Parking Spot Escalation", Some(20)),
				Episode::new(6, 10, 2523744, "The Fish Guts Displacement", Some(22)),
				Episode::new(6, 11, 2559784, "The Santa Simulation", Some(21)),
				Episode::new(6, 12, 2571320, "The Egg Salad Equivalency", Some(21)),
				Episode::new(6, 13, 2582488, "The Bakersfield Expedition", Some(21)),
				Episode::new(6, 14, 2636834, "The Cooper/Kripke Inversion", Some(20)),
				Episode::new(6, 15, 2633894, "The Spoiler Alert Segmentation", Some(21)),
				Episode::new(6, 16, 2668846, "The Tangible Affection Proof", Some(21)),
				Episode::new(6, 17, 2697398, "The Monster Isolation", Some(20)),
				Episode::new(6, 18, 2708314, "The Contractual Obligation Implementation", Some(20)),
				Episode::new(6, 19, 2739286, "The Closet Reconfiguration", Some(20)),
				Episode::new(6, 20, 2784494, "The Tenure Turbulence", Some(20)),
				Episode::new(6, 21, 2833366, "The Closure Alternative", Some(20)),
				Episode::new(6, 22, 2833352, "The Proton Resurgence", Some(21)),
				Episode::new(6, 23, 2852720, "The Love Spell Potential", Some(21)),
				Episode::new(6, 24, 2891288, "The Bon Voyage Reaction", Some(20)),
				Episode::new(7, 1, 3111660, "The Hofstadter Insufficiency", Some(21)),
				Episode::new(7, 2, 2933998, "The Deception Verification", Some(21)),
				Episode::new(7, 3, 3218670, "The Scavenger Vortex", Some(21)),
				Episode::new(7, 4, 3221722, "The Raiders Minimization", Some(21)),
				Episode::new(7, 5, 3229392, "The Workplace Proximity", Some(19)),
				Episode::new(7, 6, 3229394, "The Romance Resonance", Some(21)),
				Episode::new(7, 7, 3229396, "The Proton Displacement", Some(20)),
				Episode::new(7, 8, 3303726, "The Itchy Brain Simulation", Some(21)),
				Episode::new(7, 9, 3309570, "The Thanksgiving Decoupling", Some(20)),
				Episode::new(7, 10, 3338358, "The Discovery Dissipation", Some(20)),
				Episode::new(7, 11, 3337724, "The Cooper Extraction", Some(21)),
				Episode::new(7, 12, 3337728, "The Hesitation Ramification", Some(21)),
				Episode::new(7, 13, 3337730, "The Occupation Recalibration", Some(21)),
				Episode::new(7, 14, 3453320, "The Convention Conundrum", Some(21)),
				Episode::new(7, 15, 3477310, "The Locomotive Manipulation", Some(20)),
				Episode::new(7, 16, 3507240, "The Table Polarization", Some(20)),
				Episode::new(7, 17, 3521962, "The Friendship Turbulence", Some(20)),
				Episode::new(7, 18, 3517894, "The Mommy Observation", Some(21)),
				Episode::new(7, 19, 3521964, "The Indecision Amalgamation", Some(19)),
				Episode::new(7, 20, 3554854, "The Relationship Diremption", Some(21)),
				Episode::new(7, 21, 3572170, "The Anything Can Happen Recurrence", Some(19)),
				Episode::new(7, 22, 3572176, "The Proton Transmogrification", Some(21)),
				Episode::new(7, 23, 3572182, "The Gorilla Dissolution", Some(19)),
				Episode::new(7, 24, 3572184, "The Status Quo Combustion", Some(21)),
				Episode::new(8, 1, 3596130, "The Locomotion Interruption", Some(21)),
				Episode::new(8, 2, 3807172, "The Junior Professor Solution", Some(20)),
				Episode::new(8, 3, 3811812, "The First Pitch Insufficiency", Some(19)),
				Episode::new(8, 4, 3811814, "The Hook-up Reverberation", Some(19)),
				Episode::new(8, 5, 3811816, "The Focus Attenuation", Some(19)),
				Episode::new(8, 6, 3823240, "The Expedition Approximation", Some(21)),
				Episode::new(8, 7, 3823244, "The Misinterpretation Agitation", Some(20)),
				Episode::new(8, 8, 3823242, "The Prom Equivalency", Some(20)),
				Episode::new(8, 9, 3823248, "The Septum Deviation", Some(20)),
				Episode::new(8, 10, 3823250, "The Champagne Reflection", Some(19)),
				Episode::new(8, 11, 3823252, "The Clean Room Infiltration", Some(19)),
				Episode::new(8, 12, 3823254, "The Space Probe Disintegration", Some(20)),
				Episode::new(8, 13, 3862708, "The Anxiety Optimization", Some(20)),
				Episode::new(8, 14, 3862712, "The Troll Manifestation", Some(21)),
				Episode::new(8, 15, 3862710, "The Comic Book Store Regeneration", Some(20)),
				Episode::new(8, 16, 3862714, "The Intimacy Acceleration", Some(20)),
				Episode::new(8, 17, 3862720, "The Colonization Application", Some(20)),
				Episode::new(8, 18, 3862722, "The Leftover Thermalization", Some(19)),
				Episode::new(8, 19, 3862724, "The Skywalker Incursion", Some(21)),
				Episode::new(8, 20, 3862726, "The Fortification Implementation", Some(20)),
				Episode::new(8, 21, 3862728, "The Communication Deterioration", Some(19)),
				Episode::new(8, 22, 3862732, "The Graduation Transmission", Some(19)),
				Episode::new(8, 23, 3862734, "The Maternal Combustion", Some(19)),
				Episode::new(8, 24, 3862738, "The Commitment Determination", Some(20)),
				Episode::new(9, 1, 3603346, "The Matrimonial Momentum", Some(20)),
				Episode::new(9, 2, 4938690, "The Separation Oscillation", Some(19)),
				Episode::new(9, 3, 4962326, "The Bachelor Party Corrosion", Some(20)),
				Episode::new(9, 4, 4995126, "The 2003 Approximation", Some(19)),
				Episode::new(9, 5, 5027380, "The Perspiration Implementation", Some(19)),
				Episode::new(9, 6, 5090816, "The Helium Insufficiency", Some(19)),
				Episode::new(9, 7, 5073156, "The Spock Resonance", Some(19)),
				Episode::new(9, 8, 5117928, "The Mystery Date Observation", Some(19)),
				Episode::new(9, 9, 5143500, "The Platonic Permutation", Some(21)),
				Episode::new(9, 10, 5143508, "The Earworm Reverberation", Some(21)),
				Episode::new(9, 11, 5143516, "The Opening Night Excitation", Some(21)),
				Episode::new(9, 12, 5273742, "The Sales Call Sublimation", Some(18)),
				Episode::new(9, 13, 5273740, "The Empathy Optimization", Some(19)),
				Episode::new(9, 14, 5295650, "The Meemaw Materialization", Some(19)),
				Episode::new(9, 15, 5341496, "The Valentino Submergence", Some(19)),
				Episode::new(9, 16, 5376678, "The Positive Negative Reaction", Some(19)),
				Episode::new(9, 17, 5436356, "The Celebration Experimentation", Some(19)),
				Episode::new(9, 18, 5497250, "The Application Deterioration", Some(19)),
				Episode::new(9, 19, 5522908, "The Solder Excursion Diversion", Some(20)),
				Episode::new(9, 20, 5586046, "The Big Bear Precipitation", Some(19)),
				Episode::new(9, 21, 5586050, "The Viewing Party Combustion", Some(18)),
				Episode::new(9, 22, 5597958, "The Fermentation Bifurcation", Some(20)),
				Episode::new(9, 23, 5597960, "The Line Substitution Solution", Some(18)),
				Episode::new(9, 24, 5562854, "The Convergence Convergence", Some(20)),
				Episode::new(10, 1, 3603372, "The Conjugal Conjecture", Some(21)),
				Episode::new(10, 2, 5973910, "The Military Miniaturization", Some(19)),
				Episode::new(10, 3, 5996928, "The Dependence Transcendence", Some(20)),
				Episode::new(10, 4, 6062772, "The Cohabitation Experimentation", Some(20)),
				Episode::new(10, 5, 6077690, "The Hot Tub Contamination", Some(20)),
				Episode::new(10, 6, 5825290, "The Fetal Kick Catalyst", Some(18)),
				Episode::new(10, 7, 6186420, "The Veracity Elasticity", Some(21)),
				Episode::new(10, 8, 6189462, "The Brain Bowl Incubation", Some(18)),
				Episode::new(10, 9, 6214960, "The Geology Elevation", Some(19)),
				Episode::new(10, 10, 6255718, "The Property Division Collision", Some(19)),
				Episode::new(10, 11, 6255770, "The Birthday Synchronicity", Some(20)),
				Episode::new(10, 12, 6320950, "The Holiday Summation", Some(21)),
				Episode::new(10, 13, 6337212, "The Romance Recalibration", Some(19)),
				Episode::new(10, 14, 6414316, "The Emotion Detection Automation", Some(19)),
				Episode::new(10, 15, 6438684, "The Locomotion Reverberation", Some(20)),
				Episode::new(10, 16, 6495864, "The Allowance Evaporation", Some(18)),
				Episode::new(10, 17, 6527026, "The Comic-Con Conundrum", Some(19)),
				Episode::new(10, 18, 6565510, "The Escape Hatch Identification", Some(20)),
				Episode::new(10, 19, 6565514, "The Collaboration Fluctuation", Some(18)),
				Episode::new(10, 20, 6565516, "The Recollection Dissipation", Some(19)),
				Episode::new(10, 21, 6565524, "The Separation Agitation", Some(20)),
				Episode::new(10, 22, 6565530, "The Cognition Regeneration", Some(20)),
				Episode::new(10, 23, 6706250, "The Gyroscopic Collapse", Some(19)),
				Episode::new(10, 24, 6706254, "The Long Distance Dissonance", Some(19)),
				Episode::new(11, 1, 6673774, "The Proposal Proposal", Some(21)),
				Episode::new(11, 2, 6674506, "The Retraction Reaction", Some(21)),
				Episode::new(11, 3, 6674504, "The Relaxation Integration", Some(20)),
				Episode::new(11, 4, 6674498, "The Explosion Implosion", Some(21)),
				Episode::new(11, 5, 6674496, "The Collaboration Contamination", Some(19)),
				Episode::new(11, 6, 6674494, "The Proton Regeneration", Some(21)),
				Episode::new(11, 7, 6674492, "The Geology Methodology", Some(19)),
				Episode::new(11, 8, 6674490, "The Tesla Recoil", Some(19)),
				Episode::new(11, 9, 6674486, "The Bitcoin Entanglement", Some(19)),
				Episode::new(11, 10, 6674480, "The Confidence Erosion", Some(21)),
				Episode::new(11, 11, 6674476, "The Celebration Reverberation", Some(19)),
				Episode::new(11, 12, 6674472, "The Matrimonial Metric", Some(20)),
				Episode::new(11, 13, 6674468, "The Solo Oscillation", Some(21)),
				Episode::new(11, 14, 6674466, "The Separation Triangulation", Some(19)),
				Episode::new(11, 15, 6674462, "The Novelization Correlation", Some(21)),
				Episode::new(11, 16, 6674448, "The Neonatal Nomenclature", Some(19)),
				Episode::new(11, 17, 6674452, "The Athenaeum Allocation", Some(20)),
				Episode::new(11, 18, 6674456, "The Gates Excitation", Some(20)),
				Episode::new(11, 19, 6674454, "The Tenant Disassociation", Some(21)),
				Episode::new(11, 20, 6674434, "The Reclusive Potential", Some(21)),
				Episode::new(11, 21, 6674432, "The Comet Polarization", Some(21)),
				Episode::new(11, 22, 6674428, "The Monetary Insufficiency", Some(19)),
				Episode::new(11, 23, 6674424, "The Sibling Realignment", Some(20)),
				Episode::new(11, 24, 6674430, "The Bow Tie Asymmetry", Some(22)),
				Episode::new(12, 1, 6673778, "The Conjugal Configuration", Some(20)),
				Episode::new(12, 2, 6674688, "The Wedding Gift Wormhole", Some(21)),
				Episode::new(12, 3, 6674692, "The Procreation Calculation", Some(20)),
				Episode::new(12, 4, 6674690, "The Tam Turbulence", Some(19)),
				Episode::new(12, 5, 6674694, "The Planetarium Collision", Some(19)),
				Episode::new(12, 6, 6674704, "The Imitation Perturbation", Some(19)),
				Episode::new(12, 7, 6674696, "The Grant Allocation Derivation", Some(19)),
				Episode::new(12, 8, 6674702, "The Consummation Deviation", Some(22)),
				Episode::new(12, 9, 6674698, "The Citation Negation", Some(20)),
				Episode::new(12, 10, 6674710, "The VCR Illumination", Some(20)),
				Episode::new(12, 11, 6674706, "The Paintball Scattering", Some(19)),
				Episode::new(12, 12, 6674708, "The Propagation Proposition", Some(20)),
				Episode::new(12, 13, 6674714, "The Confirmation Polarization", Some(20)),
				Episode::new(12, 14, 6674712, "The Meteorite Manifestation", Some(19)),
				Episode::new(12, 15, 6674716, "The Donation Oscillation", Some(21)),
				Episode::new(12, 16, 6674718, "The D & D Vortex", Some(20)),
				Episode::new(12, 17, 6674726, "The Conference Valuation", Some(19)),
				Episode::new(12, 18, 6674724, "The Laureate Accumulation", Some(21)),
				Episode::new(12, 19, 6674720, "The Inspiration Deprivation", Some(20)),
				Episode::new(12, 20, 6674728, "The Decision Reverberation", Some(19)),
				Episode::new(12, 21, 6674732, "The Plagiarism Schism", Some(19)),
				Episode::new(12, 22, 6674730, "The Maternal Conclusion", Some(20)),
				Episode::new(12, 23, 6674734, "The Change Constant", Some(19)),
				Episode::new(12, 24, 6674736, "The Stockholm Syndrome", Some(23))
			/* }}} */ ]));

			let mut show = shows.remove(&1219024).unwrap();
			show.episodes.sort_unstable();
			assert_eq!(show, Show::new(1219024, "Castle", false, 2009, Some(2016), Some(43), &["Comedy", "Crime", "Drama"], &[ // {{{
				Episode::new(1, 1, 1303973, "Flowers for Your Grave", Some(43)),
				Episode::new(1, 2, 1321564, "Nanny McDead", Some(43)),
				Episode::new(1, 3, 1327257, "Hedge Fund Homeboys", Some(43)),
				Episode::new(1, 4, 1318126, "Hell Hath No Fury", Some(43)),
				Episode::new(1, 5, 1334813, "A Chill Goes Through Her Veins", Some(43)),
				Episode::new(1, 6, 1361671, "Always Buy Retail", Some(43)),
				Episode::new(1, 7, 1345746, "Home Is Where the Heart Stops", Some(43)),
				Episode::new(1, 8, 1360964, "Ghosts", Some(43)),
				Episode::new(1, 9, 1375576, "Little Girl Lost", Some(60)),
				Episode::new(1, 10, 1383514, "A Death in the Family", Some(43)),
				Episode::new(2, 1, 1471460, "Deep in Death", Some(42)),
				Episode::new(2, 2, 1487354, "The Double Down", Some(43)),
				Episode::new(2, 3, 1476479, "Inventing the Girl", Some(43)),
				Episode::new(2, 4, 1485258, "Fool Me Once...", Some(60)),
				Episode::new(2, 5, 1493984, "When the Bough Breaks", Some(60)),
				Episode::new(2, 6, 1501858, "Vampire Weekend", Some(43)),
				Episode::new(2, 7, 1509365, "Famous Last Words", Some(60)),
				Episode::new(2, 8, 1515313, "Kill the Messenger", Some(43)),
				Episode::new(2, 9, 1522460, "Love Me Dead", Some(43)),
				Episode::new(2, 10, 1526786, "One Man\'s Treasure", Some(43)),
				Episode::new(2, 11, 1539352, "The Fifth Bullet", Some(43)),
				Episode::new(2, 12, 1546168, "A Rose for Everafter", Some(60)),
				Episode::new(2, 13, 1546169, "Sucker Punch", Some(60)),
				Episode::new(2, 14, 1562496, "The Third Man", Some(60)),
				Episode::new(2, 15, 1566998, "Suicide Squeeze", Some(60)),
				Episode::new(2, 16, 1575755, "The Mistress Always Spanks Twice", Some(60)),
				Episode::new(2, 17, 1587498, "Tick, Tick, Tick...", Some(60)),
				Episode::new(2, 18, 1587499, "Boom!", Some(43)),
				Episode::new(2, 19, 1599451, "Wrapped Up in Death", Some(60)),
				Episode::new(2, 20, 1605852, "The Late Shaft", Some(60)),
				Episode::new(2, 21, 1618570, "Den of Thieves", Some(60)),
				Episode::new(2, 22, 1624253, "Food to Die For", Some(60)),
				Episode::new(2, 23, 1628908, "Overkill", Some(60)),
				Episode::new(2, 24, 1636948, "A Deadly Game", Some(60)),
				Episode::new(3, 1, 1628159, "A Deadly Affair", Some(43)),
				Episode::new(3, 2, 1628174, "He\'s Dead, She\'s Dead", Some(43)),
				Episode::new(3, 3, 1628170, "Under the Gun", Some(43)),
				Episode::new(3, 4, 1628175, "Punked", Some(43)),
				Episode::new(3, 5, 1628176, "Anatomy of a Murder", Some(43)),
				Episode::new(3, 6, 1628178, "3XK", Some(43)),
				Episode::new(3, 7, 1628177, "Almost Famous", Some(43)),
				Episode::new(3, 8, 1628179, "Murder Most Fowl", Some(43)),
				Episode::new(3, 9, 1628180, "Close Encounters of the Murderous Kind", Some(43)),
				Episode::new(3, 10, 1628160, "Last Call", Some(43)),
				Episode::new(3, 11, 1628162, "Nikki Heat", Some(43)),
				Episode::new(3, 12, 1628161, "Poof, You\'re Dead", Some(43)),
				Episode::new(3, 13, 1628163, "Knockdown", Some(43)),
				Episode::new(3, 14, 1628164, "Lucky Stiff", Some(43)),
				Episode::new(3, 15, 1628165, "The Final Nail", Some(43)),
				Episode::new(3, 16, 1628166, "Setup", Some(43)),
				Episode::new(3, 17, 1628167, "Countdown", Some(43)),
				Episode::new(3, 18, 1628168, "One Life to Lose", Some(43)),
				Episode::new(3, 19, 1628169, "Law & Murder", Some(43)),
				Episode::new(3, 20, 1628171, "Slice of Death", Some(43)),
				Episode::new(3, 21, 1628172, "The Dead Pool", Some(43)),
				Episode::new(3, 22, 1628173, "To Love and Die in L.A.", Some(43)),
				Episode::new(3, 23, 1863574, "Pretty Dead", Some(43)),
				Episode::new(3, 24, 1863575, "Knockout", Some(43)),
				Episode::new(4, 1, 1979460, "Rise", Some(45)),
				Episode::new(4, 2, 1995597, "Heroes and Villains", Some(45)),
				Episode::new(4, 3, 2006355, "Head Case", Some(45)),
				Episode::new(4, 4, 2013361, "Kick the Ballistics", Some(44)),
				Episode::new(4, 5, 2020437, "Eye of the Beholder", Some(45)),
				Episode::new(4, 6, 2041717, "Demons", Some(45)),
				Episode::new(4, 7, 2069215, "Cops & Robbers", Some(45)),
				Episode::new(4, 8, 2090751, "Heartbreak Hotel", Some(45)),
				Episode::new(4, 9, 2104216, "Kill Shot", Some(45)),
				Episode::new(4, 10, 2108699, "Cuffed", Some(45)),
				Episode::new(4, 11, 2107921, "Til Death Do Us Part", Some(45)),
				Episode::new(4, 12, 2117068, "Dial M for Mayor", Some(45)),
				Episode::new(4, 13, 2122569, "An Embarrassment of Bitches", Some(45)),
				Episode::new(4, 14, 2122570, "The Blue Butterfly", Some(44)),
				Episode::new(4, 15, 2180817, "Pandora", Some(45)),
				Episode::new(4, 16, 2177999, "Linchpin", Some(45)),
				Episode::new(4, 17, 2196331, "Once Upon a Crime", Some(45)),
				Episode::new(4, 18, 2231927, "A Dance with Death", Some(45)),
				Episode::new(4, 19, 2262661, "47 Seconds", Some(45)),
				Episode::new(4, 20, 2262663, "The Limey", Some(45)),
				Episode::new(4, 21, 2336661, "Headhunters", Some(45)),
				Episode::new(4, 22, 2336663, "Undead Again", Some(45)),
				Episode::new(4, 23, 2342563, "Always", Some(45)),
				Episode::new(5, 1, 2396875, "After the Storm", Some(44)),
				Episode::new(5, 2, 2350522, "Cloudy with a Chance of Murder", Some(43)),
				Episode::new(5, 3, 2288866, "Secret\'s Safe with Me", Some(44)),
				Episode::new(5, 4, 2364464, "Murder He Wrote", Some(43)),
				Episode::new(5, 5, 2441912, "Probable Cause", Some(44)),
				Episode::new(5, 6, 2433380, "The Final Frontier", Some(44)),
				Episode::new(5, 7, 2360384, "Swan Song", Some(44)),
				Episode::new(5, 8, 2437444, "After Hours", Some(44)),
				Episode::new(5, 9, 2508758, "Secret Santa", Some(44)),
				Episode::new(5, 10, 2481820, "Significant Others", Some(41)),
				Episode::new(5, 11, 2620004, "Under the Influence", Some(44)),
				Episode::new(5, 12, 2523936, "Death Gone Crazy", Some(44)),
				Episode::new(5, 13, 2572704, "Recoil", Some(44)),
				Episode::new(5, 14, 2584542, "Reality Star Struck", Some(44)),
				Episode::new(5, 15, 2698144, "Target", Some(44)),
				Episode::new(5, 16, 2698182, "Hunt", Some(41)),
				Episode::new(5, 17, 2705774, "Scared to Death", Some(43)),
				Episode::new(5, 18, 2712738, "The Wild Rover", Some(44)),
				Episode::new(5, 19, 2727420, "The Lives of Others", Some(44)),
				Episode::new(5, 20, 2799398, "The Fast and the Furriest", Some(43)),
				Episode::new(5, 21, 2804932, "The Squab and the Quail", Some(42)),
				Episode::new(5, 22, 2788494, "Still", Some(41)),
				Episode::new(5, 23, 2815780, "The Human Factor", Some(43)),
				Episode::new(5, 24, 2883480, "Watershed", Some(44)),
				Episode::new(6, 1, 2911344, "Valkyrie", Some(43)),
				Episode::new(6, 2, 3147784, "Dreamworld", Some(44)),
				Episode::new(6, 3, 3172144, "Need to Know", Some(44)),
				Episode::new(6, 4, 3170326, "Number One Fan", Some(45)),
				Episode::new(6, 5, 3268978, "Time Will Tell", Some(45)),
				Episode::new(6, 6, 3252790, "Get a Clue", Some(45)),
				Episode::new(6, 7, 3255338, "Like Father, Like Daughter", Some(45)),
				Episode::new(6, 8, 3240206, "A Murder Is Forever", Some(43)),
				Episode::new(6, 9, 3258212, "Disciple", Some(45)),
				Episode::new(6, 10, 3279622, "The Good, the Bad & the Baby", Some(45)),
				Episode::new(6, 11, 3403936, "Under Fire", Some(43)),
				Episode::new(6, 12, 3410998, "Deep Cover", Some(44)),
				Episode::new(6, 13, 3329020, "Limelight", Some(43)),
				Episode::new(6, 14, 3432830, "Dressed to Kill", Some(43)),
				Episode::new(6, 15, 3453374, "Smells Like Teen Spirit", Some(43)),
				Episode::new(6, 16, 3510444, "Room 147", Some(42)),
				Episode::new(6, 17, 3485882, "In the Belly of the Beast", Some(42)),
				Episode::new(6, 18, 3534768, "The Way of the Ninja", Some(43)),
				Episode::new(6, 19, 3534830, "The Greater Good", Some(43)),
				Episode::new(6, 20, 3565816, "That \'70s Show", Some(44)),
				Episode::new(6, 21, 3580390, "Law & Boarder", Some(43)),
				Episode::new(6, 22, 3587866, "Veritas", Some(44)),
				Episode::new(6, 23, 3587872, "For Better or Worse", Some(43)),
				Episode::new(7, 1, 3720122, "Driven", Some(42)),
				Episode::new(7, 2, 3914558, "Montreal", Some(43)),
				Episode::new(7, 3, 3858678, "Clear & Present Danger", Some(43)),
				Episode::new(7, 4, 3858680, "Child\'s Play", Some(42)),
				Episode::new(7, 5, 4047082, "Meme Is Murder", Some(42)),
				Episode::new(7, 6, 4075550, "The Time of Our Lives", Some(43)),
				Episode::new(7, 7, 4075560, "Once Upon a Time in the West", Some(43)),
				Episode::new(7, 8, 4126128, "Kill Switch", Some(41)),
				Episode::new(7, 9, 4234882, "Last Action Hero", Some(43)),
				Episode::new(7, 10, 4121492, "Bad Santa", Some(43)),
				Episode::new(7, 11, 4279018, "Castle, P.I.", Some(43)),
				Episode::new(7, 12, 4242524, "Private Eye Caramba!", Some(43)),
				Episode::new(7, 13, 4393054, "I, Witness", Some(42)),
				Episode::new(7, 14, 4425334, "Resurrection", Some(42)),
				Episode::new(7, 15, 4425340, "Reckoning", Some(43)),
				Episode::new(7, 16, 4365028, "The Wrong Stuff", Some(43)),
				Episode::new(7, 17, 4452138, "Hong Kong Hustle", Some(43)),
				Episode::new(7, 18, 4442144, "At Close Range", Some(42)),
				Episode::new(7, 19, 4488190, "Habeas Corpse", Some(43)),
				Episode::new(7, 20, 4586238, "Sleeper", Some(41)),
				Episode::new(7, 21, 4544966, "In Plane Sight", Some(41)),
				Episode::new(7, 22, 4627742, "Dead from New York", Some(43)),
				Episode::new(7, 23, 4659186, "Hollander\'s Woods", Some(44)),
				Episode::new(8, 1, 4667522, "XY", Some(43)),
				Episode::new(8, 2, 4857928, "XX", Some(44)),
				Episode::new(8, 3, 4966312, "PhDead", Some(43)),
				Episode::new(8, 4, 4955212, "What Lies Beneath", Some(44)),
				Episode::new(8, 5, 5023896, "The Nose", Some(44)),
				Episode::new(8, 6, 5089670, "Cool Boys", Some(42)),
				Episode::new(8, 7, 5030300, "The Last Seduction", Some(42)),
				Episode::new(8, 8, 5089986, "Mr. & Mrs. Castle", Some(42)),
				Episode::new(8, 9, 5138378, "Tone Death", Some(42)),
				Episode::new(8, 10, 5198722, "Witness for the Prosecution", Some(42)),
				Episode::new(8, 11, 5223288, "Dead Red", Some(41)),
				Episode::new(8, 12, 5376586, "The Blame Game", Some(42)),
				Episode::new(8, 13, 5332056, "And Justice for All", Some(42)),
				Episode::new(8, 14, 5461824, "G.D.S.", Some(43)),
				Episode::new(8, 15, 5318930, "Fidelis Ad Mortem", Some(43)),
				Episode::new(8, 16, 5352752, "Heartbreaker", Some(43)),
				Episode::new(8, 17, 5584444, "Death Wish", Some(42)),
				Episode::new(8, 18, 5542624, "Backstabber", Some(43)),
				Episode::new(8, 19, 5523396, "Dead Again", Some(42)),
				Episode::new(8, 20, 5541138, "Much Ado About Murder", Some(42)),
				Episode::new(8, 21, 5594910, "Hell to Pay", Some(42)),
				Episode::new(8, 22, 5594914, "Crossfire", Some(42))
			/* }}} */ ]));
		}
	} // }}}

	#[tokio::test]
	async fn shows_by_name_and_year() /* {{{ */ {
		let (_dir, mut client) = client().await;
		for i in 0..5 {
			println!(">>> 1-{}", i);
			let show = client.get_show_by_name_and_year("The Big Bang Theory", 2007).await.unwrap().unwrap();
			assert_eq!(show.imdb_id, 898266);
			let show = client.get_show_by_name_and_year("WandaVision", 2021).await.unwrap().unwrap();
			assert_eq!(show.imdb_id, 9140560);
			let show = client.get_show_by_name_and_year("Castle", 2009).await.unwrap().unwrap();
			assert_eq!(show.imdb_id, 1219024);
			let show = client.get_show_by_name_and_year("Hawkeye", 2021).await.unwrap().unwrap();
			assert_eq!(show.imdb_id, 10160804);
			let show = client.get_show_by_name_and_year("Hawkeye", 2021).await.unwrap().unwrap();
			assert_eq!(show.imdb_id, 10160804);
		}
	} // }}}

	#[tokio::test]
	async fn shows_by_name_and_year_reopen() /* {{{ */ {
		let (dir, mut client) = client().await;
		for i in 0..5 {
			println!(">>> 1-{}", i);
			let show = client.get_show_by_name_and_year("The Big Bang Theory", 2007).await.unwrap().unwrap();
			assert_eq!(show.imdb_id, 898266);
			let show = client.get_show_by_name_and_year("WandaVision", 2021).await.unwrap().unwrap();
			assert_eq!(show.imdb_id, 9140560);
			let show = client.get_show_by_name_and_year("Castle", 2009).await.unwrap().unwrap();
			assert_eq!(show.imdb_id, 1219024);
			let show = client.get_show_by_name_and_year("Hawkeye", 2021).await.unwrap().unwrap();
			assert_eq!(show.imdb_id, 10160804);
			let show = client.get_show_by_name_and_year("Hawkeye", 2021).await.unwrap().unwrap();
			assert_eq!(show.imdb_id, 10160804);
		}
		drop(client);
		let mut client = Client::new(Duration::from_secs(86400), dir.path()).await.unwrap();
		for i in 0..5 {
			println!(">>> 2-{}", i);
			let show = client.get_show_by_name_and_year("The Big Bang Theory", 2007).await.unwrap().unwrap();
			assert_eq!(show.imdb_id, 898266);
			let show = client.get_show_by_name_and_year("WandaVision", 2021).await.unwrap().unwrap();
			assert_eq!(show.imdb_id, 9140560);
			let show = client.get_show_by_name_and_year("Castle", 2009).await.unwrap().unwrap();
			assert_eq!(show.imdb_id, 1219024);
			let show = client.get_show_by_name_and_year("Hawkeye", 2021).await.unwrap().unwrap();
			assert_eq!(show.imdb_id, 10160804);
			let show = client.get_show_by_name_and_year("Hawkeye", 2021).await.unwrap().unwrap();
			assert_eq!(show.imdb_id, 10160804);
		}
	} // }}}

	#[tokio::test]
	async fn movies_by_name_and_year() /* {{{ */ {
		let (_dir, mut client) = client().await;
		for i in 0..5 {
			println!(">>> {}", i);
			let movie = client.get_movie_by_name_and_year("Avengers: Endgame", 2019).await.unwrap().unwrap();
			assert_eq!(movie.imdb_id, 4154796);
			let movie = client.get_movie_by_name_and_year("Harry Potter and the Deathly Hallows: Part 1", 2010).await.unwrap().unwrap();
			assert_eq!(movie.imdb_id, 926084);
			let movie = client.get_movie_by_name_and_year("Four Christmases", 2008).await.unwrap().unwrap();
			assert_eq!(movie.imdb_id, 369436);
		}
	} // }}}
}

