use serde::Deserialize;
use serde::Serialize;

#[derive(Debug, Deserialize, Eq, PartialEq, Serialize)]
pub struct EpisodeLink {
	pub imdb_id: u64,
	pub series_imdb_id: u64,
	pub season: Option<u16>,
	pub episode: Option<u16>
}

// TODO:  Can we go this in a better way?  Maybe seperate Deserialize impls for CSV vs bincode?  Is there even any harm in doing this way, performance or otherwise?
impl From<EpisodeLinkFromImdb> for EpisodeLink {
	fn from(input: EpisodeLinkFromImdb) -> Self {
		Self{
			imdb_id: input.imdb_id,
			series_imdb_id: input.series_imdb_id,
			season: input.season,
			// There are a couple entries in tt0417331 with insane (>65535) episode numbers; we're going to explicitly ignore those here and let the overflow be truncated.
			episode: input.episode.map(|i| i as u16)
		}
	}
}

impl EpisodeLink {
	pub(crate) fn new(imdb_id: u64, series_imdb_id: u64, season: Option<u16>, episode: Option<u16>) -> Self {
		Self{
			imdb_id: imdb_id,
			series_imdb_id: series_imdb_id,
			season: season,
			episode: episode
		}
	}
}

#[derive(Debug, Deserialize, Eq, PartialEq)]
pub struct EpisodeLinkFromImdb {
	#[serde(rename = "tconst", deserialize_with = "crate::util::parse_imdb_id")]
	pub imdb_id: u64,
	#[serde(rename = "parentTconst", deserialize_with = "crate::util::parse_imdb_id")]
	pub series_imdb_id: u64,
	#[serde(rename = "seasonNumber", deserialize_with = "crate::util::parse_janky_tsv_option")]
	pub season: Option<u16>,
	#[serde(rename = "episodeNumber", deserialize_with = "crate::util::parse_janky_tsv_option_u32")]
	pub episode: Option<u32>
}

