use std::convert::TryFrom;

use crate::Error;
use crate::Title;
use crate::TitleType;

#[derive(Debug, Eq, PartialEq)]
pub struct Movie {
	pub imdb_id: u64,
	pub title: String,
	pub is_adult: bool,
	pub year: u16,
	pub runtime_minutes: Option<u16>,
	pub genres: Vec<String>
}

impl Movie {
	pub(crate) fn new(imdb_id: u64, title: &str, is_adult: bool, year: u16, runtime_minutes: Option<u16>, genres: &[&str]) -> Self {
		Self{
			imdb_id: imdb_id,
			title: title.to_string(),
			is_adult: is_adult,
			year: year,
			runtime_minutes: runtime_minutes,
			genres: genres.iter().map(|s| s.to_string()).collect()
		}
	}

	pub(crate) fn from_wrapped_title(input: Result<Title, Error>) -> Result<Self, Error> {
		Self::try_from(&input?)
	}
}

impl TryFrom<&Title> for Movie {
	type Error = Error;
	fn try_from(input: &Title) -> Result<Self, Error> {
		match input.title_type {
			TitleType::Movie => Ok(Self{
				imdb_id: input.imdb_id,
				title: input.primary_title.clone(),
				is_adult: input.is_adult,
				year: match input.start_year {
					Some(v) => v,
					None => match input.end_year {
						Some(v) => v,
						None => return Err(Error::YearMissing.into())
					}
				},
				//year: input.start_year,
				runtime_minutes: input.runtime_minutes,
				genres: input.genres.clone()
			}),
			_ => Err(Error::WrongMediaType(input.title_type.to_string(), "Movie".to_string()))
		}
	}
}

impl TryFrom<Title> for Movie {
	type Error = Error;
	fn try_from(input: Title) -> Result<Self, Error> {
		Self::try_from(&input)
	}
}

pub(crate) fn title_matches_movie_name_and_year(title: &Result<Title, Error>, name: &str, year: u16) -> bool {
	if let Ok(title) = title {
		if(title.title_type != TitleType::Movie) {
			return false;
		}
		if let Some(title_year) = title.start_year {
			if(title_year != year) {
				return false;
			}
		} else if let Some(title_year) = title.end_year {
			if(title_year != year) {
				return false;
			}
		} else {
			return false;
		}
		if(title.primary_title != name) {
			if(title.original_title != name) {
				return false;
			}
		}
		return true;
	}
	false
}

