use chrono::{DateTime, FixedOffset};
use serde::{de, Deserialize, Deserializer};
#[derive(Deserialize, Debug)]
pub struct AssetsSearchResult {
    #[serde(rename = "@odata.count")]
    pub odata_count: Option<i64>,
    #[serde(rename = "@search.facets")]
    pub search_facets: Option<SearchFacet>,
    pub value: Option<Vec<AssetsSearchValue>>,
}

#[derive(Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct AssetsSearchValue {
    pub id: i64,
    pub vault_id: i64,
    pub name: Option<String>,
    pub modified_by: String,
    #[serde(deserialize_with = "fixed_date_time_from_str")]
    pub modified: DateTime<FixedOffset>,
    pub version_id: i64,
    pub is_shared: Option<bool>,
    pub is_published: Option<bool>,
    pub conversions: Vec<AssetConversion>,
    pub metadata: Option<Vec<AssetMetadata>>,
    pub categories: Option<Vec<AssetCategory>>,
}

#[derive(Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct AssetCategory {
    pub id: i64,
    pub name: String,
}

#[derive(Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct AssetMetadata {
    pub definition_id: i64,
    pub value: String,
}
#[derive(Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct AssetConversion {
    pub format_id: i64,
    pub content_type: String,
    pub url: String,
    pub html: String,
}

#[derive(Deserialize, Debug)]
pub struct SearchFacet {
    #[serde(rename = "categoryId")]
    pub category_id: Option<Vec<SearchFacetCategory>>,
}

#[derive(Deserialize, Debug)]
pub struct SearchFacetCategory {
    pub value: i64,
    pub count: i64,
}

fn fixed_date_time_from_str<'de, D: Deserializer<'de>>(
    deserializer: D,
) -> Result<DateTime<FixedOffset>, D::Error> {
    let s: String = Deserialize::deserialize(deserializer)?;
    DateTime::parse_from_rfc3339(&s).map_err(de::Error::custom)
}
