use crate::authentication::Authentication;
use crate::error::Error;
use crate::Client;

type BoxedError = Box<dyn std::error::Error>;
const DELETE_ENDPOINT: &'static str = "/apiv2/vaultservice/delete/";

/// Calls `vaultservice/delete`.
///
/// Returns an `AuthenticationMissing` error
/// if the `Client` does not have any `Authentication` set.
///
/// ## Arguments
/// * `client` - The ImageVault `Client` to use.
/// * `vault_id` - The vault ID to delete.
///
/// ## Examples
///
/// ```
/// use std::path::Path;
/// use imagevault::{
///     service::vault_service,
///     Client,
///     authentication::ClientCredentialsAuthentication
/// };
///
/// # async fn test() -> Result<(), Box<dyn std::error::Error>> {
/// let authentication = ClientCredentialsAuthentication::default();
/// let client = Client::new(
///     "identity",
///     "secret",
///     "https://myimagevault.local"
///     )?
///     .with_authentication(authentication);
/// 
/// // Delete vault with ID 42
/// let delete_result = vault_service::delete(&client, 42).await?;
/// # Ok(())
/// # }
/// ```
pub async fn delete<T: Authentication>(
    client: &Client<T>,
    vault_id: i64,
) -> Result<(), BoxedError> {
    let auth_unwrapped = client
        .authentication
        .as_ref()
        .ok_or_else(|| Error::AuthenticationMissing)?;
    let auth_header = auth_unwrapped
        .lock()
        .unwrap()
        .authenticate(
            &client.client_identity,
            &client.client_secret,
            &client.base_url,
            &client.reqwest_client,
        )
        .await?;
    let full_url = client.base_url.join(DELETE_ENDPOINT)?;
    
    let response = client
        .reqwest_client
        .post(full_url)
        .bearer_auth(auth_header)
        .json(&vault_id)
        .send()
        .await?;

    if let Err(err) = response.error_for_status_ref() {
        return Err(Box::new(err));
    } else {
        return Ok(());
    }
}