#![allow(non_camel_case_types)]

use criterion::*;

type fb = imagescript::framebuffer;
fn read(p: &str) -> lodepng::Bitmap<lodepng::RGBA> { return lodepng::decode32_file(p).unwrap(); }

fn imagers_rotate90(img: &mut image::DynamicImage) { img.rotate90(); }
fn imagers_rotate180(img: &mut image::DynamicImage) { img.rotate180(); }
fn imagers_rotate270(img: &mut image::DynamicImage) { img.rotate270(); }
fn imagers_flip_vertical(img: &mut image::DynamicImage) { img.flipv(); }
fn imagers_flip_horizontal(img: &mut image::DynamicImage) { img.fliph(); }
fn imagescript_rotate90(fb: &mut fb) { imagescript::ops::rotate::rotate90(fb); }
fn imagescript_init(width: usize, height: usize) -> fb { fb::new(width, height) }
fn imagescript_rotate180(fb: &mut fb) { imagescript::ops::rotate::rotate180(fb); }
fn imagescript_rotate270(fb: &mut fb) { imagescript::ops::rotate::rotate270(fb); }
fn imagescript_flip_vertical(fb: &mut fb) { imagescript::ops::flip::vertical(fb); }
fn imagescript_fill(fb: &mut fb) { imagescript::ops::fill::color(fb, 1, 2, 3, 4); }
fn imagescript_flip_horizontal(fb: &mut fb) { imagescript::ops::flip::horizontal(fb); }
fn imagers_blur_gaussian(img: &mut image::DynamicImage, sigma: f32) { img.blur(sigma); }
fn imagescript_overlay(fb: &mut fb, fg: &fb) { imagescript::ops::overlay::blend(fb, fg, 0, 0); }
fn imagescript_replace(fb: &mut fb, fg: &fb) { imagescript::ops::overlay::replace(fb, fg, 0, 0); }
fn imagescript_blur_gaussian(fb: &mut fb, sigma: f32) { imagescript::ops::blur::gaussian(fb, sigma); }
fn imagers_replace(img: &mut image::DynamicImage, fg: &image::DynamicImage) { image::imageops::replace(img, fg, 0, 0); }
fn imagers_overlay(img: &mut image::DynamicImage, fg: &image::DynamicImage) { image::imageops::overlay(img, fg, 0, 0); }
fn imagescript_resize_nearest(fb: &mut fb, width: usize, height: usize) { imagescript::ops::resize::nearest(fb, width, height); }
fn imagers_init(width: usize, height: usize) -> image::DynamicImage { image::DynamicImage::new_rgba8(width as u32, height as u32) }
fn imagers_fill(img: &mut image::DynamicImage) { let c = image::Rgba::<u8>([1, 2, 3, 4]); img.as_mut_rgba8().unwrap().pixels_mut().for_each(|p| *p = c); }
fn imagers_resize_nearest(img: &mut image::DynamicImage, width: usize, height: usize) { img.resize(width as u32, height as u32, image::imageops::FilterType::Nearest); }


fn bench(c: &mut Criterion) {
  let milk = read("tests/images/milk.png");
  let fb2 = unsafe { imagescript::framebuffer::from_ptr(milk.width, milk.height, milk.buffer.as_ptr() as _).clone() };
  let mut fb = unsafe { imagescript::framebuffer::from_ptr(milk.width, milk.height, milk.buffer.as_ptr() as _).clone() };
  let image2 = image::DynamicImage::ImageRgba8(image::RgbaImage::from_vec(fb.width as u32, fb.height as u32, fb.slice().to_owned()).unwrap());
  let mut image = image::DynamicImage::ImageRgba8(image::RgbaImage::from_vec(fb.width as u32, fb.height as u32, fb.slice().to_owned()).unwrap());

  c.benchmark_group("ops::fill")
    .bench_function("image-rs::color", |b| b.iter(|| imagers_fill(&mut image)))
    .bench_function("imagescript::color", |b| b.iter(|| imagescript_fill(&mut fb)));

  c.benchmark_group("init")
    .bench_function("image-rs::new()", |b| b.iter(|| imagers_init(milk.width, milk.height)))
    .bench_function("imagescript::new()", |b| b.iter(|| imagescript_init(milk.width, milk.height)));

  c.benchmark_group("ops::blur")
    .measurement_time(std::time::Duration::from_secs(10))
    .bench_function("image-rs::gaussian(2)", |b| b.iter(|| imagers_blur_gaussian(&mut image, 2.0)))
    .bench_function("imagescript::gaussian(2)", |b| b.iter(|| imagescript_blur_gaussian(&mut fb, 2.0)));

  c.benchmark_group("ops::flip")
    .measurement_time(std::time::Duration::from_secs(10))
    .bench_function("image-rs::flip_vertical", |b| b.iter(|| imagers_flip_vertical(&mut image)))
    .bench_function("image-rs::flip_horizontal", |b| b.iter(|| imagers_flip_horizontal(&mut image)))
    .bench_function("imagescript::flip_vertical", |b| b.iter(|| imagescript_flip_vertical(&mut fb)))
    .bench_function("imagescript::flip_horizontal", |b| b.iter(|| imagescript_flip_horizontal(&mut fb)));

  c.benchmark_group("ops::overlay")
    .measurement_time(std::time::Duration::from_secs(10))
    .bench_function("image-rs::blend", |b| b.iter(|| imagers_overlay(&mut image, &image2)))
    .bench_function("imagescript::blend", |b| b.iter(|| imagescript_overlay(&mut fb, &fb2)))
    .bench_function("image-rs::replace", |b| b.iter(|| imagers_replace(&mut image, &image2)))
    .bench_function("imagescript::replace", |b| b.iter(|| imagescript_replace(&mut fb, &fb2)));

  c.benchmark_group("ops::resize")
    .measurement_time(std::time::Duration::from_secs(10))
    .bench_function("image-rs::nearest(256x256)", |b| b.iter(|| imagers_resize_nearest(&mut image, 256, 256)))
    .bench_function("image-rs::nearest(1024x1024)", |b| b.iter(|| imagers_resize_nearest(&mut image, 1024, 1024)))
    .bench_function("imagescript::nearest(256x256)", |b| b.iter(|| imagescript_resize_nearest(&mut fb, 256, 256)))
    .bench_function("imagescript::nearest(1024x1024)", |b| b.iter(|| imagescript_resize_nearest(&mut fb, 1024, 1024)));

  c.benchmark_group("ops::rotate")
    .measurement_time(std::time::Duration::from_secs(10))
    .bench_function("image-rs::rotate(90)", |b| b.iter(|| imagers_rotate90(&mut image)))
    .bench_function("image-rs::rotate(180)", |b| b.iter(|| imagers_rotate180(&mut image)))
    .bench_function("image-rs::rotate(270)", |b| b.iter(|| imagers_rotate270(&mut image)))
    .bench_function("imagescript::rotate(90)", |b| b.iter(|| imagescript_rotate90(&mut fb)))
    .bench_function("imagescript::rotate(180)", |b| b.iter(|| imagescript_rotate180(&mut fb)))
    .bench_function("imagescript::rotate(270)", |b| b.iter(|| imagescript_rotate270(&mut fb)));
}

criterion_group!(benches, bench); criterion_main!(benches);