use std::{iter, path::Path, process, string::FromUtf8Error};

use clap::Parser;
use futures::future::join_all;
use hyper::{
  body::Body,
  client::{connect::HttpConnector, Client},
  http::{header, method::Method, uri::InvalidUri, StatusCode, Uri},
  Request,
};
use hyper_tls::HttpsConnector;
use serde::Deserialize;
use thiserror::Error;

#[derive(Debug, Parser)]
#[clap(author = "Ravern Koh <ravernkoh@gmail.com>")]
#[clap(version = "0.1.0")]
#[clap(about = "Generate gitignore files with ease")]
enum Command {
  /// Generate gitignore for given environments
  Generate {
    /// Environments to generate gitignore for
    environments: Vec<String>,
  },

  /// Search through available platforms
  Search {
    /// Keyword to use when searching
    query: String,
  },
}

#[derive(Debug, Error)]
enum Error {
  #[error("{}", _0)]
  InvalidUri(#[from] InvalidUri),
  #[error("{}", _0)]
  Utf8(#[from] FromUtf8Error),
  #[error("{}", _0)]
  Hyper(#[from] hyper::Error),
  #[error("{}", _0)]
  HyperHttp(#[from] hyper::http::Error),
  #[error("failed to fetch gitignore for {}: {}", _0, _1)]
  Http(String, String),
  #[error("failed to parse search data: {}", _0)]
  Deserialize(#[from] serde_json::Error),
}

#[tokio::main]
async fn main() {
  let connector = HttpsConnector::new();
  let client = Client::builder().build(connector);

  if let Err(error) = run(&client).await {
    eprintln!("Error: {}", error);
    process::exit(1);
  }
}

async fn run(
  client: &Client<HttpsConnector<HttpConnector>>,
) -> Result<(), Error> {
  match Command::parse() {
    Command::Generate { environments } => {
      run_generate(client, environments).await
    }
    Command::Search { query } => run_search(client, query).await,
  }
}

async fn run_generate(
  client: &Client<HttpsConnector<HttpConnector>>,
  environments: Vec<String>,
) -> Result<(), Error> {
  let gitignores = environments
    .into_iter()
    .map(|environment| fetch_gitignore(&client, environment));

  let gitignores = join_all(gitignores)
    .await
    .into_iter()
    .collect::<Result<Vec<(String, String)>, Error>>()?;

  println!("##################################");
  println!("### Generated by igno v0.1.0 ###");
  println!("##################################");

  for (environment, content) in gitignores {
    let padding = iter::repeat("#")
      .take(environment.len())
      .collect::<String>();
    println!("");
    println!("####{}####", &padding);
    println!("### {} ###", environment);
    println!("####{}####", &padding);
    println!("");
    println!("{}", content);
  }

  println!("###############");
  println!("### The end ###");
  println!("###############");

  Ok(())
}

#[derive(Debug, Deserialize)]
struct SearchData {
  tree: Vec<Object>,
}

#[derive(Debug, Deserialize)]
struct Object {
  path: String,
}

fn get_gitignore_name(path: &str) -> &str {
  &Path::new(path).file_stem().unwrap().to_str().unwrap()
}

async fn run_search(
  client: &Client<HttpsConnector<HttpConnector>>,
  query: String,
) -> Result<(), Error> {
  let uri: Uri = "https://api.github.com/repos/toptal/gitignore/git/trees/master?recursive=true".parse()?;

  let req = Request::builder()
    .method(Method::GET)
    .uri(uri)
    .header(header::USER_AGENT, "Ignore/0.1")
    .body(Body::empty())?;

  let resp = client.request(req).await?;

  let bytes = hyper::body::to_bytes(resp.into_body()).await?;
  let search_data: SearchData = serde_json::from_slice(&bytes)?;

  search_data.tree.into_iter().for_each(|object| {
    if object.path == ".gitignore" || !object.path.ends_with(".gitignore") {
      return;
    }

    let name = get_gitignore_name(&object.path);
    if name.to_uppercase().contains(&query.to_uppercase()) {
      println!("{}", name);
    }
  });

  Ok(())
}

async fn fetch_gitignore(
  client: &Client<HttpsConnector<HttpConnector>>,
  environment: String,
) -> Result<(String, String), Error> {
  let uri = format!(
    "https://raw.githubusercontent.com/toptal/gitignore/master/templates/{}.gitignore",
    &environment
  )
  .parse()?;

  let resp = client.get(uri).await?;
  let status = resp.status();

  let bytes = hyper::body::to_bytes(resp.into_body()).await?;
  let content = String::from_utf8(bytes.to_vec())?;

  if status == StatusCode::OK {
    Ok((environment, content))
  } else {
    Err(Error::Http(environment, content))
  }
}
