#![feature(array_chunks)]
use std::fs::File;
use std::io::prelude::*;

#[derive(Debug, Clone, PartialEq)]
pub struct IDXFile {
    pub data: Data,
}
impl IDXFile {
    pub fn from_file(path: String) -> Result<IDXFile, ParseError> {
        let mut file = File::open(path).unwrap();
        let mut bytes = Vec::new();
        file.read_to_end(&mut bytes).unwrap();

        if bytes[0..2] != [0, 0] {
            return Err(ParseError::FirstTwoBytesNotZero(bytes[0], bytes[1]));
        };

        let data_type = match bytes[2] {
            0x08 => DataType::UnsignedByte,
            0x09 => DataType::SignedByte,
            0x0b => DataType::Short,
            0x0c => DataType::Int,
            0x0d => DataType::Float,
            0x0e => DataType::Double,
            _ => return Err(ParseError::InvalidDataType(bytes[2])),
        };

        let dimentions = if bytes[3] != 0 {
            bytes[3]
        } else {
            return Err(ParseError::ZeroDim);
        };

        let bytes = bytes.split_at(4).1.to_vec();
        let data = match data_type {
            DataType::UnsignedByte => Data::UnsignedByte(
                bytes
                    .chunks(dimentions as usize)
                    .into_iter()
                    .map(|c| c.to_vec())
                    .collect(),
            ),
            DataType::SignedByte => Data::SignedByte(
                bytes
                    .into_iter()
                    .map(|b| b as i8)
                    .collect::<Vec<i8>>()
                    .chunks(dimentions as usize)
                    .into_iter()
                    .map(|c| c.to_vec())
                    .collect(),
            ),
            DataType::Short => Data::Short(
                bytes
                    .array_chunks::<2>()
                    .into_iter()
                    .map(|c| i16::from_le_bytes(*c))
                    .collect::<Vec<i16>>()
                    .chunks(dimentions as usize)
                    .into_iter()
                    .map(|c| c.to_vec())
                    .collect(),
            ),
            DataType::Int => Data::Int(
                bytes
                    .array_chunks::<4>()
                    .into_iter()
                    .map(|c| i32::from_le_bytes(*c))
                    .collect::<Vec<i32>>()
                    .chunks(dimentions as usize)
                    .into_iter()
                    .map(|c| c.to_vec())
                    .collect(),
            ),
            DataType::Float => Data::Float(
                bytes
                    .array_chunks::<4>()
                    .into_iter()
                    .map(|c| f32::from_le_bytes(*c))
                    .collect::<Vec<f32>>()
                    .chunks(dimentions as usize)
                    .into_iter()
                    .map(|c| c.to_vec())
                    .collect(),
            ),
            DataType::Double => Data::Double(
                bytes
                    .array_chunks::<8>()
                    .into_iter()
                    .map(|c| f64::from_le_bytes(*c))
                    .collect::<Vec<f64>>()
                    .chunks(dimentions as usize)
                    .into_iter()
                    .map(|c| c.to_vec())
                    .collect(),
            ),
        };

        Ok(IDXFile { data })
    }
}

#[derive(Debug, Copy, Clone, PartialEq)]
enum DataType {
    UnsignedByte,
    SignedByte,
    Short,
    Int,
    Float,
    Double,
}
#[derive(Debug, Clone, PartialEq)]
pub enum Data {
    UnsignedByte(Vec<Vec<u8>>),
    SignedByte(Vec<Vec<i8>>),
    Short(Vec<Vec<i16>>),
    Int(Vec<Vec<i32>>),
    Float(Vec<Vec<f32>>),
    Double(Vec<Vec<f64>>),
}

#[derive(Debug, Copy, Clone, PartialEq)]
pub enum ParseError {
    FirstTwoBytesNotZero(u8, u8),
    ZeroDimByte(u8),
    InvalidDataType(u8),
    ZeroDim,
}

#[cfg(test)]
mod test {
    use super::*;
    #[test]
    pub fn test_bag_magic_number() {
        let file = "./tests/bad_magic_number_test".into();
        let result = IDXFile::from_file(file).err().unwrap();
        assert_eq!(result, ParseError::FirstTwoBytesNotZero(0x01, 0x00))
    }
    #[test]
    pub fn test_bad_dim() {
        let file = "./tests/bad_dim_test".into();
        let result = IDXFile::from_file(file).err().unwrap();
        assert_eq!(result, ParseError::ZeroDim)
    }
    #[test]
    pub fn test_u8_dim3() {
        let file = "./tests/u8_dim3_test".into();
        let idx_file = IDXFile::from_file(file).unwrap();
        assert_eq!(
            idx_file.data,
            Data::UnsignedByte(vec![
                vec![0xc2, 0xa7, 0xc2],
                vec![0xa7, 0xc3, 0xbf],
                vec![0xc3, 0xbf, 0x00],
                vec![0x00, 0xc2, 0xa7],
                vec![0xc2, 0xa7, 0x0a]
            ])
        )
    }
    #[test]
    pub fn test_i8_dim3() {
        let file = "./tests/i8_dim3_test".into();
        let idx_file = IDXFile::from_file(file).unwrap();
        assert_eq!(
            idx_file.data,
            Data::SignedByte(
                vec![
                    vec![0xc2 as u8, 0xa7, 0xc2],
                    vec![0xa7 as u8, 0xc3, 0xbf],
                    vec![0xc3 as u8, 0xbf, 0x00],
                    vec![0x00 as u8, 0xc2, 0xa7],
                    vec![0xc2 as u8, 0xa7, 0x0a]
                ]
                .iter()
                .map(|v| v.iter().map(|u| i8::from_le_bytes([*u])).collect())
                .collect()
            )
        )
    }
}
