//! # IdGenerator
//!
//! A powerful unique id generator.
//!
//! Inspired by [yitter/IdGenerator](https://github.com/yitter/IdGenerator).
//!
//! Support for one instance with one generator (should be given a unique worker id).
//! Also support multiple generators with different worker id in one instance, using HashMap or Vector.

// #[macro_use]
// extern crate lazy_static;

pub mod idgen;

pub use idgen::id_generator::DefaultIdGenerator;
pub use idgen::id_generator_options::IdGeneratorOptions;
pub use idgen::id_helper::{IdHelper, IdMapHelper, IdVecHelper};

#[cfg(test)]
mod tests {
    use crate::{IdGeneratorOptions, IdHelper, IdMapHelper, IdVecHelper};
    use chrono::Utc;
    use std::collections::HashSet;

    #[test]
    fn single_thread_test() {
        let mut times = 5000000;
        IdHelper::init();
        let mut options = IdGeneratorOptions::new(24);
        options.seq_bit_len = 10;
        IdHelper::set_id_generator(options);
        let start = Utc::now().timestamp_millis();
        let mut new_id: i64 = 0;
        while times > 0 {
            new_id = IdHelper::next_id();
            times -= 1;
        }
        let end = Utc::now().timestamp_millis();
        println!("Last one {}-ID: {}", 50000 - times, new_id);
        println!("Single thread time: {} ms", end - start);
    }

    #[test]
    fn single_thread_check() {
        let mut set: HashSet<i64> = HashSet::new();
        let mut times = 500000;
        IdHelper::init();
        let options = IdGeneratorOptions::new(23);
        IdHelper::set_id_generator(options);
        let start = Utc::now().timestamp_millis();
        while times > 0 {
            let new_id = IdHelper::next_id();
            if !set.contains(&new_id) {
                set.insert(new_id);
            } else {
                panic!("Check fails! Same id!");
            }
            times -= 1;
        }
        let end = Utc::now().timestamp_millis();
        println!("Single thread check time: {} ms", end - start);
    }

    #[test]
    fn single_thread_test_map() {
        let mut times = 5000000;
        let worker_id: u32 = 36;
        IdMapHelper::init(vec![worker_id]);
        let mut options = IdGeneratorOptions::new(worker_id);
        options.seq_bit_len = 10;
        IdMapHelper::set_id_generator(options);
        let start = Utc::now().timestamp_millis();
        let mut new_id: i64 = 0;
        while times > 0 {
            new_id = IdMapHelper::next_id(worker_id);
            times -= 1;
        }
        let end = Utc::now().timestamp_millis();
        println!("Last one {}-ID: {}", 500000 - times, new_id);
        println!("Single thread time: {} ms", end - start);
    }

    #[test]
    fn single_thread_check_map() {
        let mut set: HashSet<i64> = HashSet::new();
        let mut times = 500000;
        let worker_id: u32 = 38;
        IdMapHelper::init(vec![worker_id, worker_id + 1]);
        let options = IdGeneratorOptions::new(worker_id);
        IdMapHelper::set_id_generator(options);
        let start = Utc::now().timestamp_millis();
        while times > 0 {
            let new_id = IdMapHelper::next_id(worker_id);
            if !set.contains(&new_id) {
                set.insert(new_id);
            } else {
                panic!("Check fails! Same id!");
            }
            times -= 1;
        }
        let end = Utc::now().timestamp_millis();
        println!("Single thread check time: {} ms", end - start);
        let options = IdGeneratorOptions::new(worker_id + 1);
        IdMapHelper::set_id_generator(options);
        times = 500000;
        let start = Utc::now().timestamp_millis();
        while times > 0 {
            let new_id = IdMapHelper::next_id(worker_id + 1);
            if !set.contains(&new_id) {
                set.insert(new_id);
            } else {
                panic!("Check fails! Same id!");
            }
            times -= 1;
        }
        let end = Utc::now().timestamp_millis();
        println!("Single thread check time: {} ms", end - start);
    }

    #[test]
    fn single_thread_test_vec() {
        let mut times = 5000000;
        let worker_id: u32 = 1;
        IdVecHelper::init(worker_id, 2, 2);
        let mut options = IdGeneratorOptions::new(worker_id + 2);
        options.seq_bit_len = 10;
        IdVecHelper::set_id_generator(options);
        let start = Utc::now().timestamp_millis();
        let mut new_id: i64 = 0;
        while times > 0 {
            new_id = IdVecHelper::next_id(worker_id + 2);
            times -= 1;
        }
        let end = Utc::now().timestamp_millis();
        println!("Last one {}-ID: {}", 500000 - times, new_id);
        println!("Single thread time: {} ms", end - start);
    }

    #[test]
    fn single_thread_check_vec() {
        let mut set: HashSet<i64> = HashSet::new();
        let mut times = 500000;
        let worker_id: u32 = 10;
        IdVecHelper::init(worker_id, 2, 3);
        let options = IdGeneratorOptions::new(worker_id);
        IdVecHelper::set_id_generator(options);
        let start = Utc::now().timestamp_millis();
        while times > 0 {
            let new_id = IdVecHelper::next_id(worker_id);
            if !set.contains(&new_id) {
                set.insert(new_id);
            } else {
                panic!("Check fails! Same id!");
            }
            times -= 1;
        }
        let end = Utc::now().timestamp_millis();
        println!("Single thread check time: {} ms", end - start);
        times = 500000;
        let options = IdGeneratorOptions::new(worker_id + 4);
        IdVecHelper::set_id_generator(options);
        let start = Utc::now().timestamp_millis();
        while times > 0 {
            let new_id = IdVecHelper::next_id(worker_id + 4);
            if !set.contains(&new_id) {
                set.insert(new_id);
            } else {
                panic!("Check fails! Same id!");
            }
            times -= 1;
        }
        let end = Utc::now().timestamp_millis();
        println!("Single thread check time: {} ms", end - start);
    }
}
