// SPDX-License-Identifier: MIT
// Copyright (C) 2018-present iced project and contributors

use crate::iced_constants::IcedConstants;
use crate::iced_error::IcedError;
use core::convert::TryFrom;
use core::iter::{ExactSizeIterator, FusedIterator, Iterator};
use core::{fmt, mem};

// GENERATOR-BEGIN: CodeSize
// ⚠️This was generated by GENERATOR!🦹‍♂️
/// The code size (16/32/64) that was used when an instruction was decoded
#[derive(Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash)]
pub enum CodeSize {
	/// Unknown size
	Unknown = 0,
	/// 16-bit code
	Code16 = 1,
	/// 32-bit code
	Code32 = 2,
	/// 64-bit code
	Code64 = 3,
}
#[rustfmt::skip]
static GEN_DEBUG_CODE_SIZE: [&str; 4] = [
	"Unknown",
	"Code16",
	"Code32",
	"Code64",
];
impl fmt::Debug for CodeSize {
	#[inline]
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{}", GEN_DEBUG_CODE_SIZE[*self as usize])
	}
}
impl Default for CodeSize {
	#[must_use]
	#[inline]
	fn default() -> Self {
		CodeSize::Unknown
	}
}
#[allow(non_camel_case_types)]
#[allow(dead_code)]
pub(crate) type CodeSizeUnderlyingType = u8;
#[rustfmt::skip]
impl CodeSize {
	/// Iterates over all `CodeSize` enum values
	#[inline]
	pub fn values() -> impl Iterator<Item = CodeSize> + DoubleEndedIterator + ExactSizeIterator + FusedIterator {
		// SAFETY: all values 0-max are valid enum values
		(0..IcedConstants::CODE_SIZE_ENUM_COUNT).map(|x| unsafe { mem::transmute::<u8, CodeSize>(x as u8) })
	}
}
#[test]
#[rustfmt::skip]
fn test_codesize_values() {
	let mut iter = CodeSize::values();
	assert_eq!(iter.size_hint(), (IcedConstants::CODE_SIZE_ENUM_COUNT, Some(IcedConstants::CODE_SIZE_ENUM_COUNT)));
	assert_eq!(iter.len(), IcedConstants::CODE_SIZE_ENUM_COUNT);
	assert!(iter.next().is_some());
	assert_eq!(iter.size_hint(), (IcedConstants::CODE_SIZE_ENUM_COUNT - 1, Some(IcedConstants::CODE_SIZE_ENUM_COUNT - 1)));
	assert_eq!(iter.len(), IcedConstants::CODE_SIZE_ENUM_COUNT - 1);

	let values: Vec<CodeSize> = CodeSize::values().collect();
	assert_eq!(values.len(), IcedConstants::CODE_SIZE_ENUM_COUNT);
	for (i, value) in values.into_iter().enumerate() {
		assert_eq!(i, value as usize);
	}

	let values1: Vec<CodeSize> = CodeSize::values().collect();
	let mut values2: Vec<CodeSize> = CodeSize::values().rev().collect();
	values2.reverse();
	assert_eq!(values1, values2);
}
#[rustfmt::skip]
impl TryFrom<usize> for CodeSize {
	type Error = IcedError;
	#[inline]
	fn try_from(value: usize) -> Result<Self, Self::Error> {
		if value < IcedConstants::CODE_SIZE_ENUM_COUNT {
			// SAFETY: all values 0-max are valid enum values
			Ok(unsafe { mem::transmute(value as u8) })
		} else {
			Err(IcedError::new("Invalid CodeSize value"))
		}
	}
}
#[test]
#[rustfmt::skip]
fn test_codesize_try_from_usize() {
	for value in CodeSize::values() {
		let converted = <CodeSize as TryFrom<usize>>::try_from(value as usize).unwrap();
		assert_eq!(converted, value);
	}
	assert!(<CodeSize as TryFrom<usize>>::try_from(IcedConstants::CODE_SIZE_ENUM_COUNT).is_err());
	assert!(<CodeSize as TryFrom<usize>>::try_from(core::usize::MAX).is_err());
}
// GENERATOR-END: CodeSize

// GENERATOR-BEGIN: RoundingControl
// ⚠️This was generated by GENERATOR!🦹‍♂️
/// Rounding control
#[derive(Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash)]
pub enum RoundingControl {
	/// No rounding mode
	None = 0,
	/// Round to nearest (even)
	RoundToNearest = 1,
	/// Round down (toward -inf)
	RoundDown = 2,
	/// Round up (toward +inf)
	RoundUp = 3,
	/// Round toward zero (truncate)
	RoundTowardZero = 4,
}
#[rustfmt::skip]
static GEN_DEBUG_ROUNDING_CONTROL: [&str; 5] = [
	"None",
	"RoundToNearest",
	"RoundDown",
	"RoundUp",
	"RoundTowardZero",
];
impl fmt::Debug for RoundingControl {
	#[inline]
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{}", GEN_DEBUG_ROUNDING_CONTROL[*self as usize])
	}
}
impl Default for RoundingControl {
	#[must_use]
	#[inline]
	fn default() -> Self {
		RoundingControl::None
	}
}
#[allow(non_camel_case_types)]
#[allow(dead_code)]
pub(crate) type RoundingControlUnderlyingType = u8;
#[rustfmt::skip]
impl RoundingControl {
	/// Iterates over all `RoundingControl` enum values
	#[inline]
	pub fn values() -> impl Iterator<Item = RoundingControl> + DoubleEndedIterator + ExactSizeIterator + FusedIterator {
		// SAFETY: all values 0-max are valid enum values
		(0..IcedConstants::ROUNDING_CONTROL_ENUM_COUNT).map(|x| unsafe { mem::transmute::<u8, RoundingControl>(x as u8) })
	}
}
#[test]
#[rustfmt::skip]
fn test_roundingcontrol_values() {
	let mut iter = RoundingControl::values();
	assert_eq!(iter.size_hint(), (IcedConstants::ROUNDING_CONTROL_ENUM_COUNT, Some(IcedConstants::ROUNDING_CONTROL_ENUM_COUNT)));
	assert_eq!(iter.len(), IcedConstants::ROUNDING_CONTROL_ENUM_COUNT);
	assert!(iter.next().is_some());
	assert_eq!(iter.size_hint(), (IcedConstants::ROUNDING_CONTROL_ENUM_COUNT - 1, Some(IcedConstants::ROUNDING_CONTROL_ENUM_COUNT - 1)));
	assert_eq!(iter.len(), IcedConstants::ROUNDING_CONTROL_ENUM_COUNT - 1);

	let values: Vec<RoundingControl> = RoundingControl::values().collect();
	assert_eq!(values.len(), IcedConstants::ROUNDING_CONTROL_ENUM_COUNT);
	for (i, value) in values.into_iter().enumerate() {
		assert_eq!(i, value as usize);
	}

	let values1: Vec<RoundingControl> = RoundingControl::values().collect();
	let mut values2: Vec<RoundingControl> = RoundingControl::values().rev().collect();
	values2.reverse();
	assert_eq!(values1, values2);
}
#[rustfmt::skip]
impl TryFrom<usize> for RoundingControl {
	type Error = IcedError;
	#[inline]
	fn try_from(value: usize) -> Result<Self, Self::Error> {
		if value < IcedConstants::ROUNDING_CONTROL_ENUM_COUNT {
			// SAFETY: all values 0-max are valid enum values
			Ok(unsafe { mem::transmute(value as u8) })
		} else {
			Err(IcedError::new("Invalid RoundingControl value"))
		}
	}
}
#[test]
#[rustfmt::skip]
fn test_roundingcontrol_try_from_usize() {
	for value in RoundingControl::values() {
		let converted = <RoundingControl as TryFrom<usize>>::try_from(value as usize).unwrap();
		assert_eq!(converted, value);
	}
	assert!(<RoundingControl as TryFrom<usize>>::try_from(IcedConstants::ROUNDING_CONTROL_ENUM_COUNT).is_err());
	assert!(<RoundingControl as TryFrom<usize>>::try_from(core::usize::MAX).is_err());
}
// GENERATOR-END: RoundingControl

// GENERATOR-BEGIN: OpKind
// ⚠️This was generated by GENERATOR!🦹‍♂️
/// Instruction operand kind
#[derive(Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash)]
#[allow(non_camel_case_types)]
pub enum OpKind {
	/// A register ([`Register`]).
	///
	/// This operand kind uses [`Instruction::op0_register()`], [`Instruction::op1_register()`], [`Instruction::op2_register()`], [`Instruction::op3_register()`] or [`Instruction::op4_register()`] depending on operand number. See also [`Instruction::op_register()`].
	///
	/// [`Register`]: enum.Register.html
	/// [`Instruction::op0_register()`]: struct.Instruction.html#method.op0_register
	/// [`Instruction::op1_register()`]: struct.Instruction.html#method.op1_register
	/// [`Instruction::op2_register()`]: struct.Instruction.html#method.op2_register
	/// [`Instruction::op3_register()`]: struct.Instruction.html#method.op3_register
	/// [`Instruction::op4_register()`]: struct.Instruction.html#method.op4_register
	/// [`Instruction::op_register()`]: struct.Instruction.html#method.op_register
	Register = 0,
	/// Near 16-bit branch. This operand kind uses [`Instruction::near_branch16()`]
	///
	/// [`Instruction::near_branch16()`]: struct.Instruction.html#method.near_branch16
	NearBranch16 = 1,
	/// Near 32-bit branch. This operand kind uses [`Instruction::near_branch32()`]
	///
	/// [`Instruction::near_branch32()`]: struct.Instruction.html#method.near_branch32
	NearBranch32 = 2,
	/// Near 64-bit branch. This operand kind uses [`Instruction::near_branch64()`]
	///
	/// [`Instruction::near_branch64()`]: struct.Instruction.html#method.near_branch64
	NearBranch64 = 3,
	/// Far 16-bit branch. This operand kind uses [`Instruction::far_branch16()`] and [`Instruction::far_branch_selector()`]
	///
	/// [`Instruction::far_branch16()`]: struct.Instruction.html#method.far_branch16
	/// [`Instruction::far_branch_selector()`]: struct.Instruction.html#method.far_branch_selector
	FarBranch16 = 4,
	/// Far 32-bit branch. This operand kind uses [`Instruction::far_branch32()`] and [`Instruction::far_branch_selector()`]
	///
	/// [`Instruction::far_branch32()`]: struct.Instruction.html#method.far_branch32
	/// [`Instruction::far_branch_selector()`]: struct.Instruction.html#method.far_branch_selector
	FarBranch32 = 5,
	/// 8-bit constant. This operand kind uses [`Instruction::immediate8()`]
	///
	/// [`Instruction::immediate8()`]: struct.Instruction.html#method.immediate8
	Immediate8 = 6,
	/// 8-bit constant used by the `ENTER`, `EXTRQ`, `INSERTQ` instructions. This operand kind uses [`Instruction::immediate8_2nd()`]
	///
	/// [`Instruction::immediate8_2nd()`]: struct.Instruction.html#method.immediate8_2nd
	Immediate8_2nd = 7,
	/// 16-bit constant. This operand kind uses [`Instruction::immediate16()`]
	///
	/// [`Instruction::immediate16()`]: struct.Instruction.html#method.immediate16
	Immediate16 = 8,
	/// 32-bit constant. This operand kind uses [`Instruction::immediate32()`]
	///
	/// [`Instruction::immediate32()`]: struct.Instruction.html#method.immediate32
	Immediate32 = 9,
	/// 64-bit constant. This operand kind uses [`Instruction::immediate64()`]
	///
	/// [`Instruction::immediate64()`]: struct.Instruction.html#method.immediate64
	Immediate64 = 10,
	/// An 8-bit value sign extended to 16 bits. This operand kind uses [`Instruction::immediate8to16()`]
	///
	/// [`Instruction::immediate8to16()`]: struct.Instruction.html#method.immediate8to16
	Immediate8to16 = 11,
	/// An 8-bit value sign extended to 32 bits. This operand kind uses [`Instruction::immediate8to32()`]
	///
	/// [`Instruction::immediate8to32()`]: struct.Instruction.html#method.immediate8to32
	Immediate8to32 = 12,
	/// An 8-bit value sign extended to 64 bits. This operand kind uses [`Instruction::immediate8to64()`]
	///
	/// [`Instruction::immediate8to64()`]: struct.Instruction.html#method.immediate8to64
	Immediate8to64 = 13,
	/// A 32-bit value sign extended to 64 bits. This operand kind uses [`Instruction::immediate32to64()`]
	///
	/// [`Instruction::immediate32to64()`]: struct.Instruction.html#method.immediate32to64
	Immediate32to64 = 14,
	/// `seg:[SI]`. This operand kind uses [`Instruction::memory_size()`], [`Instruction::memory_segment()`], [`Instruction::segment_prefix()`]
	///
	/// [`Instruction::memory_size()`]: struct.Instruction.html#method.memory_size
	/// [`Instruction::memory_segment()`]: struct.Instruction.html#method.memory_segment
	/// [`Instruction::segment_prefix()`]: struct.Instruction.html#method.segment_prefix
	MemorySegSI = 15,
	/// `seg:[ESI]`. This operand kind uses [`Instruction::memory_size()`], [`Instruction::memory_segment()`], [`Instruction::segment_prefix()`]
	///
	/// [`Instruction::memory_size()`]: struct.Instruction.html#method.memory_size
	/// [`Instruction::memory_segment()`]: struct.Instruction.html#method.memory_segment
	/// [`Instruction::segment_prefix()`]: struct.Instruction.html#method.segment_prefix
	MemorySegESI = 16,
	/// `seg:[RSI]`. This operand kind uses [`Instruction::memory_size()`], [`Instruction::memory_segment()`], [`Instruction::segment_prefix()`]
	///
	/// [`Instruction::memory_size()`]: struct.Instruction.html#method.memory_size
	/// [`Instruction::memory_segment()`]: struct.Instruction.html#method.memory_segment
	/// [`Instruction::segment_prefix()`]: struct.Instruction.html#method.segment_prefix
	MemorySegRSI = 17,
	/// `seg:[DI]`. This operand kind uses [`Instruction::memory_size()`], [`Instruction::memory_segment()`], [`Instruction::segment_prefix()`]
	///
	/// [`Instruction::memory_size()`]: struct.Instruction.html#method.memory_size
	/// [`Instruction::memory_segment()`]: struct.Instruction.html#method.memory_segment
	/// [`Instruction::segment_prefix()`]: struct.Instruction.html#method.segment_prefix
	MemorySegDI = 18,
	/// `seg:[EDI]`. This operand kind uses [`Instruction::memory_size()`], [`Instruction::memory_segment()`], [`Instruction::segment_prefix()`]
	///
	/// [`Instruction::memory_size()`]: struct.Instruction.html#method.memory_size
	/// [`Instruction::memory_segment()`]: struct.Instruction.html#method.memory_segment
	/// [`Instruction::segment_prefix()`]: struct.Instruction.html#method.segment_prefix
	MemorySegEDI = 19,
	/// `seg:[RDI]`. This operand kind uses [`Instruction::memory_size()`], [`Instruction::memory_segment()`], [`Instruction::segment_prefix()`]
	///
	/// [`Instruction::memory_size()`]: struct.Instruction.html#method.memory_size
	/// [`Instruction::memory_segment()`]: struct.Instruction.html#method.memory_segment
	/// [`Instruction::segment_prefix()`]: struct.Instruction.html#method.segment_prefix
	MemorySegRDI = 20,
	/// `ES:[DI]`. This operand kind uses [`Instruction::memory_size()`]
	///
	/// [`Instruction::memory_size()`]: struct.Instruction.html#method.memory_size
	MemoryESDI = 21,
	/// `ES:[EDI]`. This operand kind uses [`Instruction::memory_size()`]
	///
	/// [`Instruction::memory_size()`]: struct.Instruction.html#method.memory_size
	MemoryESEDI = 22,
	/// `ES:[RDI]`. This operand kind uses [`Instruction::memory_size()`]
	///
	/// [`Instruction::memory_size()`]: struct.Instruction.html#method.memory_size
	MemoryESRDI = 23,
	/// DEPRECATED. Use [`Memory`]
	///
	/// [`Memory`]: enum.OpKind.html#variant.Memory
	#[deprecated(since = "1.11.0", note = "Don't use it!")]
	Memory64 = 24,
	/// Memory operand.
	///
	/// This operand kind uses [`Instruction::memory_displ_size()`], [`Instruction::memory_size()`], [`Instruction::memory_index_scale()`], [`Instruction::memory_displacement64()`], [`Instruction::memory_base()`], [`Instruction::memory_index()`], [`Instruction::memory_segment()`], [`Instruction::segment_prefix()`]
	///
	/// [`Instruction::memory_displ_size()`]: struct.Instruction.html#method.memory_displ_size
	/// [`Instruction::memory_size()`]: struct.Instruction.html#method.memory_size
	/// [`Instruction::memory_index_scale()`]: struct.Instruction.html#method.memory_index_scale
	/// [`Instruction::memory_displacement64()`]: struct.Instruction.html#method.memory_displacement64
	/// [`Instruction::memory_base()`]: struct.Instruction.html#method.memory_base
	/// [`Instruction::memory_index()`]: struct.Instruction.html#method.memory_index
	/// [`Instruction::memory_segment()`]: struct.Instruction.html#method.memory_segment
	/// [`Instruction::segment_prefix()`]: struct.Instruction.html#method.segment_prefix
	Memory = 25,
}
#[rustfmt::skip]
static GEN_DEBUG_OP_KIND: [&str; 26] = [
	"Register",
	"NearBranch16",
	"NearBranch32",
	"NearBranch64",
	"FarBranch16",
	"FarBranch32",
	"Immediate8",
	"Immediate8_2nd",
	"Immediate16",
	"Immediate32",
	"Immediate64",
	"Immediate8to16",
	"Immediate8to32",
	"Immediate8to64",
	"Immediate32to64",
	"MemorySegSI",
	"MemorySegESI",
	"MemorySegRSI",
	"MemorySegDI",
	"MemorySegEDI",
	"MemorySegRDI",
	"MemoryESDI",
	"MemoryESEDI",
	"MemoryESRDI",
	"Memory64",
	"Memory",
];
impl fmt::Debug for OpKind {
	#[inline]
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{}", GEN_DEBUG_OP_KIND[*self as usize])
	}
}
impl Default for OpKind {
	#[must_use]
	#[inline]
	fn default() -> Self {
		OpKind::Register
	}
}
#[allow(non_camel_case_types)]
#[allow(dead_code)]
pub(crate) type OpKindUnderlyingType = u8;
#[rustfmt::skip]
impl OpKind {
	/// Iterates over all `OpKind` enum values
	#[inline]
	pub fn values() -> impl Iterator<Item = OpKind> + DoubleEndedIterator + ExactSizeIterator + FusedIterator {
		// SAFETY: all values 0-max are valid enum values
		(0..IcedConstants::OP_KIND_ENUM_COUNT).map(|x| unsafe { mem::transmute::<u8, OpKind>(x as u8) })
	}
}
#[test]
#[rustfmt::skip]
fn test_opkind_values() {
	let mut iter = OpKind::values();
	assert_eq!(iter.size_hint(), (IcedConstants::OP_KIND_ENUM_COUNT, Some(IcedConstants::OP_KIND_ENUM_COUNT)));
	assert_eq!(iter.len(), IcedConstants::OP_KIND_ENUM_COUNT);
	assert!(iter.next().is_some());
	assert_eq!(iter.size_hint(), (IcedConstants::OP_KIND_ENUM_COUNT - 1, Some(IcedConstants::OP_KIND_ENUM_COUNT - 1)));
	assert_eq!(iter.len(), IcedConstants::OP_KIND_ENUM_COUNT - 1);

	let values: Vec<OpKind> = OpKind::values().collect();
	assert_eq!(values.len(), IcedConstants::OP_KIND_ENUM_COUNT);
	for (i, value) in values.into_iter().enumerate() {
		assert_eq!(i, value as usize);
	}

	let values1: Vec<OpKind> = OpKind::values().collect();
	let mut values2: Vec<OpKind> = OpKind::values().rev().collect();
	values2.reverse();
	assert_eq!(values1, values2);
}
#[rustfmt::skip]
impl TryFrom<usize> for OpKind {
	type Error = IcedError;
	#[inline]
	fn try_from(value: usize) -> Result<Self, Self::Error> {
		if value < IcedConstants::OP_KIND_ENUM_COUNT {
			// SAFETY: all values 0-max are valid enum values
			Ok(unsafe { mem::transmute(value as u8) })
		} else {
			Err(IcedError::new("Invalid OpKind value"))
		}
	}
}
#[test]
#[rustfmt::skip]
fn test_opkind_try_from_usize() {
	for value in OpKind::values() {
		let converted = <OpKind as TryFrom<usize>>::try_from(value as usize).unwrap();
		assert_eq!(converted, value);
	}
	assert!(<OpKind as TryFrom<usize>>::try_from(IcedConstants::OP_KIND_ENUM_COUNT).is_err());
	assert!(<OpKind as TryFrom<usize>>::try_from(core::usize::MAX).is_err());
}
// GENERATOR-END: OpKind

// GENERATOR-BEGIN: VectorLength
// ⚠️This was generated by GENERATOR!🦹‍♂️
#[derive(Copy, Clone, Eq, PartialEq)]
#[cfg(any(feature = "decoder", feature = "encoder"))]
#[repr(u32)]
#[allow(dead_code)]
pub(crate) enum VectorLength {
	L128,
	L256,
	L512,
	Unknown,
}
#[cfg(any(feature = "decoder", feature = "encoder"))]
#[rustfmt::skip]
static GEN_DEBUG_VECTOR_LENGTH: [&str; 4] = [
	"L128",
	"L256",
	"L512",
	"Unknown",
];
#[cfg(any(feature = "decoder", feature = "encoder"))]
impl fmt::Debug for VectorLength {
	#[inline]
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{}", GEN_DEBUG_VECTOR_LENGTH[*self as usize])
	}
}
#[cfg(any(feature = "decoder", feature = "encoder"))]
impl Default for VectorLength {
	#[must_use]
	#[inline]
	fn default() -> Self {
		VectorLength::L128
	}
}
// GENERATOR-END: VectorLength

// GENERATOR-BEGIN: MandatoryPrefixByte
// ⚠️This was generated by GENERATOR!🦹‍♂️
#[derive(Copy, Clone, Eq, PartialEq)]
#[cfg(feature = "encoder")]
#[allow(dead_code)]
pub(crate) enum MandatoryPrefixByte {
	None,
	P66,
	PF3,
	PF2,
}
#[cfg(feature = "encoder")]
#[rustfmt::skip]
static GEN_DEBUG_MANDATORY_PREFIX_BYTE: [&str; 4] = [
	"None",
	"P66",
	"PF3",
	"PF2",
];
#[cfg(feature = "encoder")]
impl fmt::Debug for MandatoryPrefixByte {
	#[inline]
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{}", GEN_DEBUG_MANDATORY_PREFIX_BYTE[*self as usize])
	}
}
#[cfg(feature = "encoder")]
impl Default for MandatoryPrefixByte {
	#[must_use]
	#[inline]
	fn default() -> Self {
		MandatoryPrefixByte::None
	}
}
// GENERATOR-END: MandatoryPrefixByte

// GENERATOR-BEGIN: EncodingKind
// ⚠️This was generated by GENERATOR!🦹‍♂️
/// Instruction encoding
#[derive(Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash)]
#[cfg_attr(not(feature = "exhaustive_enums"), non_exhaustive)]
#[cfg(any(feature = "decoder", feature = "encoder", feature = "instr_info"))]
pub enum EncodingKind {
	/// Legacy encoding
	Legacy = 0,
	/// VEX encoding
	VEX = 1,
	/// EVEX encoding
	EVEX = 2,
	/// XOP encoding
	XOP = 3,
	/// 3DNow! encoding
	D3NOW = 4,
}
#[cfg(any(feature = "decoder", feature = "encoder", feature = "instr_info"))]
#[rustfmt::skip]
static GEN_DEBUG_ENCODING_KIND: [&str; 5] = [
	"Legacy",
	"VEX",
	"EVEX",
	"XOP",
	"D3NOW",
];
#[cfg(any(feature = "decoder", feature = "encoder", feature = "instr_info"))]
impl fmt::Debug for EncodingKind {
	#[inline]
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{}", GEN_DEBUG_ENCODING_KIND[*self as usize])
	}
}
#[cfg(any(feature = "decoder", feature = "encoder", feature = "instr_info"))]
impl Default for EncodingKind {
	#[must_use]
	#[inline]
	fn default() -> Self {
		EncodingKind::Legacy
	}
}
#[cfg(any(feature = "decoder", feature = "encoder", feature = "instr_info"))]
#[allow(non_camel_case_types)]
#[allow(dead_code)]
pub(crate) type EncodingKindUnderlyingType = u8;
#[cfg(any(feature = "decoder", feature = "encoder", feature = "instr_info"))]
#[rustfmt::skip]
impl EncodingKind {
	/// Iterates over all `EncodingKind` enum values
	#[inline]
	pub fn values() -> impl Iterator<Item = EncodingKind> + DoubleEndedIterator + ExactSizeIterator + FusedIterator {
		// SAFETY: all values 0-max are valid enum values
		(0..IcedConstants::ENCODING_KIND_ENUM_COUNT).map(|x| unsafe { mem::transmute::<u8, EncodingKind>(x as u8) })
	}
}
#[test]
#[cfg(any(feature = "decoder", feature = "encoder", feature = "instr_info"))]
#[rustfmt::skip]
fn test_encodingkind_values() {
	let mut iter = EncodingKind::values();
	assert_eq!(iter.size_hint(), (IcedConstants::ENCODING_KIND_ENUM_COUNT, Some(IcedConstants::ENCODING_KIND_ENUM_COUNT)));
	assert_eq!(iter.len(), IcedConstants::ENCODING_KIND_ENUM_COUNT);
	assert!(iter.next().is_some());
	assert_eq!(iter.size_hint(), (IcedConstants::ENCODING_KIND_ENUM_COUNT - 1, Some(IcedConstants::ENCODING_KIND_ENUM_COUNT - 1)));
	assert_eq!(iter.len(), IcedConstants::ENCODING_KIND_ENUM_COUNT - 1);

	let values: Vec<EncodingKind> = EncodingKind::values().collect();
	assert_eq!(values.len(), IcedConstants::ENCODING_KIND_ENUM_COUNT);
	for (i, value) in values.into_iter().enumerate() {
		assert_eq!(i, value as usize);
	}

	let values1: Vec<EncodingKind> = EncodingKind::values().collect();
	let mut values2: Vec<EncodingKind> = EncodingKind::values().rev().collect();
	values2.reverse();
	assert_eq!(values1, values2);
}
#[cfg(any(feature = "decoder", feature = "encoder", feature = "instr_info"))]
#[rustfmt::skip]
impl TryFrom<usize> for EncodingKind {
	type Error = IcedError;
	#[inline]
	fn try_from(value: usize) -> Result<Self, Self::Error> {
		if value < IcedConstants::ENCODING_KIND_ENUM_COUNT {
			// SAFETY: all values 0-max are valid enum values
			Ok(unsafe { mem::transmute(value as u8) })
		} else {
			Err(IcedError::new("Invalid EncodingKind value"))
		}
	}
}
#[cfg(any(feature = "decoder", feature = "encoder", feature = "instr_info"))]
#[test]
#[rustfmt::skip]
fn test_encodingkind_try_from_usize() {
	for value in EncodingKind::values() {
		let converted = <EncodingKind as TryFrom<usize>>::try_from(value as usize).unwrap();
		assert_eq!(converted, value);
	}
	assert!(<EncodingKind as TryFrom<usize>>::try_from(IcedConstants::ENCODING_KIND_ENUM_COUNT).is_err());
	assert!(<EncodingKind as TryFrom<usize>>::try_from(core::usize::MAX).is_err());
}
// GENERATOR-END: EncodingKind

// GENERATOR-BEGIN: TupleType
// ⚠️This was generated by GENERATOR!🦹‍♂️
/// Tuple type (EVEX) which can be used to get the disp8 scale factor `N`
#[derive(Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash)]
#[cfg_attr(not(feature = "exhaustive_enums"), non_exhaustive)]
#[cfg(any(feature = "decoder", feature = "encoder"))]
#[allow(non_camel_case_types)]
pub enum TupleType {
	/// `N = 1`
	N1 = 0,
	/// `N = 2`
	N2 = 1,
	/// `N = 4`
	N4 = 2,
	/// `N = 8`
	N8 = 3,
	/// `N = 16`
	N16 = 4,
	/// `N = 32`
	N32 = 5,
	/// `N = 64`
	N64 = 6,
	/// `N = b ? 4 : 8`
	N8b4 = 7,
	/// `N = b ? 4 : 16`
	N16b4 = 8,
	/// `N = b ? 4 : 32`
	N32b4 = 9,
	/// `N = b ? 4 : 64`
	N64b4 = 10,
	/// `N = b ? 8 : 16`
	N16b8 = 11,
	/// `N = b ? 8 : 32`
	N32b8 = 12,
	/// `N = b ? 8 : 64`
	N64b8 = 13,
	/// `N = b ? 2 : 4`
	N4b2 = 14,
	/// `N = b ? 2 : 8`
	N8b2 = 15,
	/// `N = b ? 2 : 16`
	N16b2 = 16,
	/// `N = b ? 2 : 32`
	N32b2 = 17,
	/// `N = b ? 2 : 64`
	N64b2 = 18,
}
#[cfg(any(feature = "decoder", feature = "encoder"))]
#[rustfmt::skip]
static GEN_DEBUG_TUPLE_TYPE: [&str; 19] = [
	"N1",
	"N2",
	"N4",
	"N8",
	"N16",
	"N32",
	"N64",
	"N8b4",
	"N16b4",
	"N32b4",
	"N64b4",
	"N16b8",
	"N32b8",
	"N64b8",
	"N4b2",
	"N8b2",
	"N16b2",
	"N32b2",
	"N64b2",
];
#[cfg(any(feature = "decoder", feature = "encoder"))]
impl fmt::Debug for TupleType {
	#[inline]
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{}", GEN_DEBUG_TUPLE_TYPE[*self as usize])
	}
}
#[cfg(any(feature = "decoder", feature = "encoder"))]
impl Default for TupleType {
	#[must_use]
	#[inline]
	fn default() -> Self {
		TupleType::N1
	}
}
#[cfg(any(feature = "decoder", feature = "encoder"))]
#[allow(non_camel_case_types)]
#[allow(dead_code)]
pub(crate) type TupleTypeUnderlyingType = u8;
#[cfg(any(feature = "decoder", feature = "encoder"))]
#[rustfmt::skip]
impl TupleType {
	/// Iterates over all `TupleType` enum values
	#[inline]
	pub fn values() -> impl Iterator<Item = TupleType> + DoubleEndedIterator + ExactSizeIterator + FusedIterator {
		// SAFETY: all values 0-max are valid enum values
		(0..IcedConstants::TUPLE_TYPE_ENUM_COUNT).map(|x| unsafe { mem::transmute::<u8, TupleType>(x as u8) })
	}
}
#[test]
#[cfg(any(feature = "decoder", feature = "encoder"))]
#[rustfmt::skip]
fn test_tupletype_values() {
	let mut iter = TupleType::values();
	assert_eq!(iter.size_hint(), (IcedConstants::TUPLE_TYPE_ENUM_COUNT, Some(IcedConstants::TUPLE_TYPE_ENUM_COUNT)));
	assert_eq!(iter.len(), IcedConstants::TUPLE_TYPE_ENUM_COUNT);
	assert!(iter.next().is_some());
	assert_eq!(iter.size_hint(), (IcedConstants::TUPLE_TYPE_ENUM_COUNT - 1, Some(IcedConstants::TUPLE_TYPE_ENUM_COUNT - 1)));
	assert_eq!(iter.len(), IcedConstants::TUPLE_TYPE_ENUM_COUNT - 1);

	let values: Vec<TupleType> = TupleType::values().collect();
	assert_eq!(values.len(), IcedConstants::TUPLE_TYPE_ENUM_COUNT);
	for (i, value) in values.into_iter().enumerate() {
		assert_eq!(i, value as usize);
	}

	let values1: Vec<TupleType> = TupleType::values().collect();
	let mut values2: Vec<TupleType> = TupleType::values().rev().collect();
	values2.reverse();
	assert_eq!(values1, values2);
}
#[cfg(any(feature = "decoder", feature = "encoder"))]
#[rustfmt::skip]
impl TryFrom<usize> for TupleType {
	type Error = IcedError;
	#[inline]
	fn try_from(value: usize) -> Result<Self, Self::Error> {
		if value < IcedConstants::TUPLE_TYPE_ENUM_COUNT {
			// SAFETY: all values 0-max are valid enum values
			Ok(unsafe { mem::transmute(value as u8) })
		} else {
			Err(IcedError::new("Invalid TupleType value"))
		}
	}
}
#[cfg(any(feature = "decoder", feature = "encoder"))]
#[test]
#[rustfmt::skip]
fn test_tupletype_try_from_usize() {
	for value in TupleType::values() {
		let converted = <TupleType as TryFrom<usize>>::try_from(value as usize).unwrap();
		assert_eq!(converted, value);
	}
	assert!(<TupleType as TryFrom<usize>>::try_from(IcedConstants::TUPLE_TYPE_ENUM_COUNT).is_err());
	assert!(<TupleType as TryFrom<usize>>::try_from(core::usize::MAX).is_err());
}
// GENERATOR-END: TupleType

// GENERATOR-BEGIN: FlowControl
// ⚠️This was generated by GENERATOR!🦹‍♂️
/// Control flow
#[derive(Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash)]
#[cfg(feature = "instr_info")]
pub enum FlowControl {
	/// The next instruction that will be executed is the next instruction in the instruction stream
	Next = 0,
	/// It's an unconditional branch instruction: `JMP NEAR`, `JMP FAR`
	UnconditionalBranch = 1,
	/// It's an unconditional indirect branch: `JMP NEAR reg`, `JMP NEAR [mem]`, `JMP FAR [mem]`
	IndirectBranch = 2,
	/// It's a conditional branch instruction: `Jcc SHORT`, `Jcc NEAR`, `LOOP`, `LOOPcc`, `JRCXZ`
	ConditionalBranch = 3,
	/// It's a return instruction: `RET NEAR`, `RET FAR`, `IRET`, `SYSRET`, `SYSEXIT`, `RSM`, `SKINIT`, `RDM`, `UIRET`
	Return = 4,
	/// It's a call instruction: `CALL NEAR`, `CALL FAR`, `SYSCALL`, `SYSENTER`, `VMLAUNCH`, `VMRESUME`, `VMCALL`, `VMMCALL`, `VMGEXIT`, `VMRUN`, `TDCALL`, `SEAMCALL`, `SEAMRET`
	Call = 5,
	/// It's an indirect call instruction: `CALL NEAR reg`, `CALL NEAR [mem]`, `CALL FAR [mem]`
	IndirectCall = 6,
	/// It's an interrupt instruction: `INT n`, `INT3`, `INT1`, `INTO`, `SMINT`, `DMINT`
	Interrupt = 7,
	/// It's `XBEGIN`
	XbeginXabortXend = 8,
	/// It's an invalid instruction, eg. [`Code::INVALID`], `UD0`, `UD1`, `UD2`
	///
	/// [`Code::INVALID`]: enum.Code.html#variant.INVALID
	Exception = 9,
}
#[cfg(feature = "instr_info")]
#[rustfmt::skip]
static GEN_DEBUG_FLOW_CONTROL: [&str; 10] = [
	"Next",
	"UnconditionalBranch",
	"IndirectBranch",
	"ConditionalBranch",
	"Return",
	"Call",
	"IndirectCall",
	"Interrupt",
	"XbeginXabortXend",
	"Exception",
];
#[cfg(feature = "instr_info")]
impl fmt::Debug for FlowControl {
	#[inline]
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{}", GEN_DEBUG_FLOW_CONTROL[*self as usize])
	}
}
#[cfg(feature = "instr_info")]
impl Default for FlowControl {
	#[must_use]
	#[inline]
	fn default() -> Self {
		FlowControl::Next
	}
}
#[cfg(feature = "instr_info")]
#[allow(non_camel_case_types)]
#[allow(dead_code)]
pub(crate) type FlowControlUnderlyingType = u8;
#[cfg(feature = "instr_info")]
#[rustfmt::skip]
impl FlowControl {
	/// Iterates over all `FlowControl` enum values
	#[inline]
	pub fn values() -> impl Iterator<Item = FlowControl> + DoubleEndedIterator + ExactSizeIterator + FusedIterator {
		// SAFETY: all values 0-max are valid enum values
		(0..IcedConstants::FLOW_CONTROL_ENUM_COUNT).map(|x| unsafe { mem::transmute::<u8, FlowControl>(x as u8) })
	}
}
#[test]
#[cfg(feature = "instr_info")]
#[rustfmt::skip]
fn test_flowcontrol_values() {
	let mut iter = FlowControl::values();
	assert_eq!(iter.size_hint(), (IcedConstants::FLOW_CONTROL_ENUM_COUNT, Some(IcedConstants::FLOW_CONTROL_ENUM_COUNT)));
	assert_eq!(iter.len(), IcedConstants::FLOW_CONTROL_ENUM_COUNT);
	assert!(iter.next().is_some());
	assert_eq!(iter.size_hint(), (IcedConstants::FLOW_CONTROL_ENUM_COUNT - 1, Some(IcedConstants::FLOW_CONTROL_ENUM_COUNT - 1)));
	assert_eq!(iter.len(), IcedConstants::FLOW_CONTROL_ENUM_COUNT - 1);

	let values: Vec<FlowControl> = FlowControl::values().collect();
	assert_eq!(values.len(), IcedConstants::FLOW_CONTROL_ENUM_COUNT);
	for (i, value) in values.into_iter().enumerate() {
		assert_eq!(i, value as usize);
	}

	let values1: Vec<FlowControl> = FlowControl::values().collect();
	let mut values2: Vec<FlowControl> = FlowControl::values().rev().collect();
	values2.reverse();
	assert_eq!(values1, values2);
}
#[cfg(feature = "instr_info")]
#[rustfmt::skip]
impl TryFrom<usize> for FlowControl {
	type Error = IcedError;
	#[inline]
	fn try_from(value: usize) -> Result<Self, Self::Error> {
		if value < IcedConstants::FLOW_CONTROL_ENUM_COUNT {
			// SAFETY: all values 0-max are valid enum values
			Ok(unsafe { mem::transmute(value as u8) })
		} else {
			Err(IcedError::new("Invalid FlowControl value"))
		}
	}
}
#[cfg(feature = "instr_info")]
#[test]
#[rustfmt::skip]
fn test_flowcontrol_try_from_usize() {
	for value in FlowControl::values() {
		let converted = <FlowControl as TryFrom<usize>>::try_from(value as usize).unwrap();
		assert_eq!(converted, value);
	}
	assert!(<FlowControl as TryFrom<usize>>::try_from(IcedConstants::FLOW_CONTROL_ENUM_COUNT).is_err());
	assert!(<FlowControl as TryFrom<usize>>::try_from(core::usize::MAX).is_err());
}
// GENERATOR-END: FlowControl

// GENERATOR-BEGIN: OpCodeOperandKind
// ⚠️This was generated by GENERATOR!🦹‍♂️
/// Operand kind
#[derive(Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash)]
#[cfg_attr(not(feature = "exhaustive_enums"), non_exhaustive)]
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[allow(non_camel_case_types)]
pub enum OpCodeOperandKind {
	/// No operand
	None = 0,
	/// Far branch 16-bit offset, 16-bit segment/selector
	farbr2_2 = 1,
	/// Far branch 32-bit offset, 16-bit segment/selector
	farbr4_2 = 2,
	/// Memory offset without a modrm byte (eg. `MOV AL,[offset]`)
	mem_offs = 3,
	/// Memory (modrm)
	mem = 4,
	/// Memory (modrm), MPX:
	///
	/// 16/32-bit mode: must be 32-bit addressing
	///
	/// 64-bit mode: 64-bit addressing is forced and must not be RIP relative
	mem_mpx = 5,
	/// Memory (modrm), MPX:
	///
	/// 16/32-bit mode: must be 32-bit addressing
	///
	/// 64-bit mode: 64-bit addressing is forced and must not be RIP relative
	mem_mib = 6,
	/// Memory (modrm), vsib32, `XMM` registers
	mem_vsib32x = 7,
	/// Memory (modrm), vsib64, `XMM` registers
	mem_vsib64x = 8,
	/// Memory (modrm), vsib32, `YMM` registers
	mem_vsib32y = 9,
	/// Memory (modrm), vsib64, `YMM` registers
	mem_vsib64y = 10,
	/// Memory (modrm), vsib32, `ZMM` registers
	mem_vsib32z = 11,
	/// Memory (modrm), vsib64, `ZMM` registers
	mem_vsib64z = 12,
	/// 8-bit GPR or memory
	r8_or_mem = 13,
	/// 16-bit GPR or memory
	r16_or_mem = 14,
	/// 32-bit GPR or memory
	r32_or_mem = 15,
	/// 32-bit GPR or memory, MPX: 16/32-bit mode: must be 32-bit addressing, 64-bit mode: 64-bit addressing is forced
	r32_or_mem_mpx = 16,
	/// 64-bit GPR or memory
	r64_or_mem = 17,
	/// 64-bit GPR or memory, MPX: 16/32-bit mode: must be 32-bit addressing, 64-bit mode: 64-bit addressing is forced
	r64_or_mem_mpx = 18,
	/// `MM` register or memory
	mm_or_mem = 19,
	/// `XMM` register or memory
	xmm_or_mem = 20,
	/// `YMM` register or memory
	ymm_or_mem = 21,
	/// `ZMM` register or memory
	zmm_or_mem = 22,
	/// `BND` register or memory, MPX: 16/32-bit mode: must be 32-bit addressing, 64-bit mode: 64-bit addressing is forced
	bnd_or_mem_mpx = 23,
	/// `K` register or memory
	k_or_mem = 24,
	/// 8-bit GPR encoded in the `reg` field of the modrm byte
	r8_reg = 25,
	/// 8-bit GPR encoded in the low 3 bits of the opcode
	r8_opcode = 26,
	/// 16-bit GPR encoded in the `reg` field of the modrm byte
	r16_reg = 27,
	/// 16-bit GPR encoded in the `reg` field of the modrm byte. This is a memory operand and it uses the address size prefix (`67h`) not the operand size prefix (`66h`).
	r16_reg_mem = 28,
	/// 16-bit GPR encoded in the `mod + r/m` fields of the modrm byte
	r16_rm = 29,
	/// 16-bit GPR encoded in the low 3 bits of the opcode
	r16_opcode = 30,
	/// 32-bit GPR encoded in the `reg` field of the modrm byte
	r32_reg = 31,
	/// 32-bit GPR encoded in the `reg` field of the modrm byte. This is a memory operand and it uses the address size prefix (`67h`) not the operand size prefix (`66h`).
	r32_reg_mem = 32,
	/// 32-bit GPR encoded in the `mod + r/m` fields of the modrm byte
	r32_rm = 33,
	/// 32-bit GPR encoded in the low 3 bits of the opcode
	r32_opcode = 34,
	/// 32-bit GPR encoded in the the `V'vvvv` field (VEX/EVEX/XOP)
	r32_vvvv = 35,
	/// 64-bit GPR encoded in the `reg` field of the modrm byte
	r64_reg = 36,
	/// 64-bit GPR encoded in the `reg` field of the modrm byte. This is a memory operand and it uses the address size prefix (`67h`) not the operand size prefix (`66h`).
	r64_reg_mem = 37,
	/// 64-bit GPR encoded in the `mod + r/m` fields of the modrm byte
	r64_rm = 38,
	/// 64-bit GPR encoded in the low 3 bits of the opcode
	r64_opcode = 39,
	/// 64-bit GPR encoded in the the `V'vvvv` field (VEX/EVEX/XOP)
	r64_vvvv = 40,
	/// Segment register encoded in the `reg` field of the modrm byte
	seg_reg = 41,
	/// `K` register encoded in the `reg` field of the modrm byte
	k_reg = 42,
	/// `K` register (+1) encoded in the `reg` field of the modrm byte
	kp1_reg = 43,
	/// `K` register encoded in the `mod + r/m` fields of the modrm byte
	k_rm = 44,
	/// `K` register encoded in the the `V'vvvv` field (VEX/EVEX/XOP)
	k_vvvv = 45,
	/// `MM` register encoded in the `reg` field of the modrm byte
	mm_reg = 46,
	/// `MM` register encoded in the `mod + r/m` fields of the modrm byte
	mm_rm = 47,
	/// `XMM` register encoded in the `reg` field of the modrm byte
	xmm_reg = 48,
	/// `XMM` register encoded in the `mod + r/m` fields of the modrm byte
	xmm_rm = 49,
	/// `XMM` register encoded in the the `V'vvvv` field (VEX/EVEX/XOP)
	xmm_vvvv = 50,
	/// `XMM` register (+3) encoded in the the `V'vvvv` field (VEX/EVEX/XOP)
	xmmp3_vvvv = 51,
	/// `XMM` register encoded in the the high 4 bits of the last 8-bit immediate (VEX/XOP only so only `XMM0`-`XMM15`)
	xmm_is4 = 52,
	/// `XMM` register encoded in the the high 4 bits of the last 8-bit immediate (VEX/XOP only so only `XMM0`-`XMM15`)
	xmm_is5 = 53,
	/// `YMM` register encoded in the `reg` field of the modrm byte
	ymm_reg = 54,
	/// `YMM` register encoded in the `mod + r/m` fields of the modrm byte
	ymm_rm = 55,
	/// `YMM` register encoded in the the `V'vvvv` field (VEX/EVEX/XOP)
	ymm_vvvv = 56,
	/// `YMM` register encoded in the the high 4 bits of the last 8-bit immediate (VEX/XOP only so only `YMM0`-`YMM15`)
	ymm_is4 = 57,
	/// `YMM` register encoded in the the high 4 bits of the last 8-bit immediate (VEX/XOP only so only `YMM0`-`YMM15`)
	ymm_is5 = 58,
	/// `ZMM` register encoded in the `reg` field of the modrm byte
	zmm_reg = 59,
	/// `ZMM` register encoded in the `mod + r/m` fields of the modrm byte
	zmm_rm = 60,
	/// `ZMM` register encoded in the the `V'vvvv` field (VEX/EVEX/XOP)
	zmm_vvvv = 61,
	/// `ZMM` register (+3) encoded in the the `V'vvvv` field (VEX/EVEX/XOP)
	zmmp3_vvvv = 62,
	/// `CR` register encoded in the `reg` field of the modrm byte
	cr_reg = 63,
	/// `DR` register encoded in the `reg` field of the modrm byte
	dr_reg = 64,
	/// `TR` register encoded in the `reg` field of the modrm byte
	tr_reg = 65,
	/// `BND` register encoded in the `reg` field of the modrm byte
	bnd_reg = 66,
	/// `ES` register
	es = 67,
	/// `CS` register
	cs = 68,
	/// `SS` register
	ss = 69,
	/// `DS` register
	ds = 70,
	/// `FS` register
	fs = 71,
	/// `GS` register
	gs = 72,
	/// `AL` register
	al = 73,
	/// `CL` register
	cl = 74,
	/// `AX` register
	ax = 75,
	/// `DX` register
	dx = 76,
	/// `EAX` register
	eax = 77,
	/// `RAX` register
	rax = 78,
	/// `ST(0)` register
	st0 = 79,
	/// `ST(i)` register encoded in the low 3 bits of the opcode
	sti_opcode = 80,
	/// 4-bit immediate (m2z field, low 4 bits of the /is5 immediate, eg. `VPERMIL2PS`)
	imm4_m2z = 81,
	/// 8-bit immediate
	imm8 = 82,
	/// Constant 1 (8-bit immediate)
	imm8_const_1 = 83,
	/// 8-bit immediate sign extended to 16 bits
	imm8sex16 = 84,
	/// 8-bit immediate sign extended to 32 bits
	imm8sex32 = 85,
	/// 8-bit immediate sign extended to 64 bits
	imm8sex64 = 86,
	/// 16-bit immediate
	imm16 = 87,
	/// 32-bit immediate
	imm32 = 88,
	/// 32-bit immediate sign extended to 64 bits
	imm32sex64 = 89,
	/// 64-bit immediate
	imm64 = 90,
	/// `seg:[rSI]` memory operand (string instructions)
	seg_rSI = 91,
	/// `es:[rDI]` memory operand (string instructions)
	es_rDI = 92,
	/// `seg:[rDI]` memory operand (`(V)MASKMOVQ` instructions)
	seg_rDI = 93,
	/// `seg:[rBX+al]` memory operand (`XLATB` instruction)
	seg_rBX_al = 94,
	/// 16-bit branch, 1-byte signed relative offset
	br16_1 = 95,
	/// 32-bit branch, 1-byte signed relative offset
	br32_1 = 96,
	/// 64-bit branch, 1-byte signed relative offset
	br64_1 = 97,
	/// 16-bit branch, 2-byte signed relative offset
	br16_2 = 98,
	/// 32-bit branch, 4-byte signed relative offset
	br32_4 = 99,
	/// 64-bit branch, 4-byte signed relative offset
	br64_4 = 100,
	/// `XBEGIN`, 2-byte signed relative offset
	xbegin_2 = 101,
	/// `XBEGIN`, 4-byte signed relative offset
	xbegin_4 = 102,
	/// 2-byte branch offset (`JMPE` instruction)
	brdisp_2 = 103,
	/// 4-byte branch offset (`JMPE` instruction)
	brdisp_4 = 104,
	/// Memory (modrm) and the sib byte must be present
	sibmem = 105,
	/// `TMM` register encoded in the `reg` field of the modrm byte
	tmm_reg = 106,
	/// `TMM` register encoded in the `mod + r/m` fields of the modrm byte
	tmm_rm = 107,
	/// `TMM` register encoded in the the `V'vvvv` field (VEX/EVEX/XOP)
	tmm_vvvv = 108,
}
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[rustfmt::skip]
static GEN_DEBUG_OP_CODE_OPERAND_KIND: [&str; 109] = [
	"None",
	"farbr2_2",
	"farbr4_2",
	"mem_offs",
	"mem",
	"mem_mpx",
	"mem_mib",
	"mem_vsib32x",
	"mem_vsib64x",
	"mem_vsib32y",
	"mem_vsib64y",
	"mem_vsib32z",
	"mem_vsib64z",
	"r8_or_mem",
	"r16_or_mem",
	"r32_or_mem",
	"r32_or_mem_mpx",
	"r64_or_mem",
	"r64_or_mem_mpx",
	"mm_or_mem",
	"xmm_or_mem",
	"ymm_or_mem",
	"zmm_or_mem",
	"bnd_or_mem_mpx",
	"k_or_mem",
	"r8_reg",
	"r8_opcode",
	"r16_reg",
	"r16_reg_mem",
	"r16_rm",
	"r16_opcode",
	"r32_reg",
	"r32_reg_mem",
	"r32_rm",
	"r32_opcode",
	"r32_vvvv",
	"r64_reg",
	"r64_reg_mem",
	"r64_rm",
	"r64_opcode",
	"r64_vvvv",
	"seg_reg",
	"k_reg",
	"kp1_reg",
	"k_rm",
	"k_vvvv",
	"mm_reg",
	"mm_rm",
	"xmm_reg",
	"xmm_rm",
	"xmm_vvvv",
	"xmmp3_vvvv",
	"xmm_is4",
	"xmm_is5",
	"ymm_reg",
	"ymm_rm",
	"ymm_vvvv",
	"ymm_is4",
	"ymm_is5",
	"zmm_reg",
	"zmm_rm",
	"zmm_vvvv",
	"zmmp3_vvvv",
	"cr_reg",
	"dr_reg",
	"tr_reg",
	"bnd_reg",
	"es",
	"cs",
	"ss",
	"ds",
	"fs",
	"gs",
	"al",
	"cl",
	"ax",
	"dx",
	"eax",
	"rax",
	"st0",
	"sti_opcode",
	"imm4_m2z",
	"imm8",
	"imm8_const_1",
	"imm8sex16",
	"imm8sex32",
	"imm8sex64",
	"imm16",
	"imm32",
	"imm32sex64",
	"imm64",
	"seg_rSI",
	"es_rDI",
	"seg_rDI",
	"seg_rBX_al",
	"br16_1",
	"br32_1",
	"br64_1",
	"br16_2",
	"br32_4",
	"br64_4",
	"xbegin_2",
	"xbegin_4",
	"brdisp_2",
	"brdisp_4",
	"sibmem",
	"tmm_reg",
	"tmm_rm",
	"tmm_vvvv",
];
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
impl fmt::Debug for OpCodeOperandKind {
	#[inline]
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{}", GEN_DEBUG_OP_CODE_OPERAND_KIND[*self as usize])
	}
}
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
impl Default for OpCodeOperandKind {
	#[must_use]
	#[inline]
	fn default() -> Self {
		OpCodeOperandKind::None
	}
}
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[allow(non_camel_case_types)]
#[allow(dead_code)]
pub(crate) type OpCodeOperandKindUnderlyingType = u8;
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[rustfmt::skip]
impl OpCodeOperandKind {
	/// Iterates over all `OpCodeOperandKind` enum values
	#[inline]
	pub fn values() -> impl Iterator<Item = OpCodeOperandKind> + DoubleEndedIterator + ExactSizeIterator + FusedIterator {
		// SAFETY: all values 0-max are valid enum values
		(0..IcedConstants::OP_CODE_OPERAND_KIND_ENUM_COUNT).map(|x| unsafe { mem::transmute::<u8, OpCodeOperandKind>(x as u8) })
	}
}
#[test]
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[rustfmt::skip]
fn test_opcodeoperandkind_values() {
	let mut iter = OpCodeOperandKind::values();
	assert_eq!(iter.size_hint(), (IcedConstants::OP_CODE_OPERAND_KIND_ENUM_COUNT, Some(IcedConstants::OP_CODE_OPERAND_KIND_ENUM_COUNT)));
	assert_eq!(iter.len(), IcedConstants::OP_CODE_OPERAND_KIND_ENUM_COUNT);
	assert!(iter.next().is_some());
	assert_eq!(iter.size_hint(), (IcedConstants::OP_CODE_OPERAND_KIND_ENUM_COUNT - 1, Some(IcedConstants::OP_CODE_OPERAND_KIND_ENUM_COUNT - 1)));
	assert_eq!(iter.len(), IcedConstants::OP_CODE_OPERAND_KIND_ENUM_COUNT - 1);

	let values: Vec<OpCodeOperandKind> = OpCodeOperandKind::values().collect();
	assert_eq!(values.len(), IcedConstants::OP_CODE_OPERAND_KIND_ENUM_COUNT);
	for (i, value) in values.into_iter().enumerate() {
		assert_eq!(i, value as usize);
	}

	let values1: Vec<OpCodeOperandKind> = OpCodeOperandKind::values().collect();
	let mut values2: Vec<OpCodeOperandKind> = OpCodeOperandKind::values().rev().collect();
	values2.reverse();
	assert_eq!(values1, values2);
}
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[rustfmt::skip]
impl TryFrom<usize> for OpCodeOperandKind {
	type Error = IcedError;
	#[inline]
	fn try_from(value: usize) -> Result<Self, Self::Error> {
		if value < IcedConstants::OP_CODE_OPERAND_KIND_ENUM_COUNT {
			// SAFETY: all values 0-max are valid enum values
			Ok(unsafe { mem::transmute(value as u8) })
		} else {
			Err(IcedError::new("Invalid OpCodeOperandKind value"))
		}
	}
}
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[test]
#[rustfmt::skip]
fn test_opcodeoperandkind_try_from_usize() {
	for value in OpCodeOperandKind::values() {
		let converted = <OpCodeOperandKind as TryFrom<usize>>::try_from(value as usize).unwrap();
		assert_eq!(converted, value);
	}
	assert!(<OpCodeOperandKind as TryFrom<usize>>::try_from(IcedConstants::OP_CODE_OPERAND_KIND_ENUM_COUNT).is_err());
	assert!(<OpCodeOperandKind as TryFrom<usize>>::try_from(core::usize::MAX).is_err());
}
// GENERATOR-END: OpCodeOperandKind

// GENERATOR-BEGIN: CpuidFeature
// ⚠️This was generated by GENERATOR!🦹‍♂️
/// `CPUID` feature flags
#[derive(Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash)]
#[cfg_attr(not(feature = "exhaustive_enums"), non_exhaustive)]
#[cfg(feature = "instr_info")]
#[allow(non_camel_case_types)]
pub enum CpuidFeature {
	/// 8086 or later
	INTEL8086 = 0,
	/// 8086 only
	INTEL8086_ONLY = 1,
	/// 80186 or later
	INTEL186 = 2,
	/// 80286 or later
	INTEL286 = 3,
	/// 80286 only
	INTEL286_ONLY = 4,
	/// 80386 or later
	INTEL386 = 5,
	/// 80386 only
	INTEL386_ONLY = 6,
	/// 80386 A0-B0 stepping only (`XBTS`, `IBTS` instructions)
	INTEL386_A0_ONLY = 7,
	/// Intel486 or later
	INTEL486 = 8,
	/// Intel486 A stepping only (`CMPXCHG`)
	INTEL486_A_ONLY = 9,
	/// UMOV (80386 and Intel486)
	UMOV = 10,
	/// IA-64
	IA64 = 11,
	/// CPUID.80000001H:EDX.LM\[bit 29\]
	X64 = 12,
	/// CPUID.(EAX=07H, ECX=0H):EBX.ADX\[bit 19\]
	ADX = 13,
	/// CPUID.01H:ECX.AES\[bit 25\]
	AES = 14,
	/// CPUID.01H:ECX.AVX\[bit 28\]
	AVX = 15,
	/// CPUID.(EAX=07H, ECX=0H):EBX.AVX2\[bit 5\]
	AVX2 = 16,
	/// CPUID.(EAX=07H, ECX=0H):EDX.AVX512_4FMAPS\[bit 3\]
	AVX512_4FMAPS = 17,
	/// CPUID.(EAX=07H, ECX=0H):EDX.AVX512_4VNNIW\[bit 2\]
	AVX512_4VNNIW = 18,
	/// CPUID.(EAX=07H, ECX=1H):EAX.AVX512_BF16\[bit 5\]
	AVX512_BF16 = 19,
	/// CPUID.(EAX=07H, ECX=0H):ECX.AVX512_BITALG\[bit 12\]
	AVX512_BITALG = 20,
	/// CPUID.(EAX=07H, ECX=0H):EBX.AVX512_IFMA\[bit 21\]
	AVX512_IFMA = 21,
	/// CPUID.(EAX=07H, ECX=0H):ECX.AVX512_VBMI\[bit 1\]
	AVX512_VBMI = 22,
	/// CPUID.(EAX=07H, ECX=0H):ECX.AVX512_VBMI2\[bit 6\]
	AVX512_VBMI2 = 23,
	/// CPUID.(EAX=07H, ECX=0H):ECX.AVX512_VNNI\[bit 11\]
	AVX512_VNNI = 24,
	/// CPUID.(EAX=07H, ECX=0H):EDX.AVX512_VP2INTERSECT\[bit 08\]
	AVX512_VP2INTERSECT = 25,
	/// CPUID.(EAX=07H, ECX=0H):ECX.AVX512_VPOPCNTDQ\[bit 14\]
	AVX512_VPOPCNTDQ = 26,
	/// CPUID.(EAX=07H, ECX=0H):EBX.AVX512BW\[bit 30\]
	AVX512BW = 27,
	/// CPUID.(EAX=07H, ECX=0H):EBX.AVX512CD\[bit 28\]
	AVX512CD = 28,
	/// CPUID.(EAX=07H, ECX=0H):EBX.AVX512DQ\[bit 17\]
	AVX512DQ = 29,
	/// CPUID.(EAX=07H, ECX=0H):EBX.AVX512ER\[bit 27\]
	AVX512ER = 30,
	/// CPUID.(EAX=07H, ECX=0H):EBX.AVX512F\[bit 16\]
	AVX512F = 31,
	/// CPUID.(EAX=07H, ECX=0H):EBX.AVX512PF\[bit 26\]
	AVX512PF = 32,
	/// CPUID.(EAX=07H, ECX=0H):EBX.AVX512VL\[bit 31\]
	AVX512VL = 33,
	/// CPUID.(EAX=07H, ECX=0H):EBX.BMI1\[bit 3\]
	BMI1 = 34,
	/// CPUID.(EAX=07H, ECX=0H):EBX.BMI2\[bit 8\]
	BMI2 = 35,
	/// CPUID.(EAX=07H, ECX=0H):EDX.CET_IBT\[bit 20\]
	CET_IBT = 36,
	/// CPUID.(EAX=07H, ECX=0H):ECX.CET_SS\[bit 7\]
	CET_SS = 37,
	/// `CL1INVMB` instruction (Intel SCC = Single-Chip Computer)
	CL1INVMB = 38,
	/// CPUID.(EAX=07H, ECX=0H):ECX.CLDEMOTE\[bit 25\]
	CLDEMOTE = 39,
	/// CPUID.(EAX=07H, ECX=0H):EBX.CLFLUSHOPT\[bit 23\]
	CLFLUSHOPT = 40,
	/// CPUID.01H:EDX.CLFSH\[bit 19\]
	CLFSH = 41,
	/// CPUID.(EAX=07H, ECX=0H):EBX.CLWB\[bit 24\]
	CLWB = 42,
	/// CPUID.80000008H:EBX.CLZERO\[bit 0\]
	CLZERO = 43,
	/// CPUID.01H:EDX.CMOV\[bit 15\]
	CMOV = 44,
	/// CPUID.01H:ECX.CMPXCHG16B\[bit 13\]
	CMPXCHG16B = 45,
	/// `RFLAGS.ID` can be toggled
	CPUID = 46,
	/// CPUID.01H:EDX.CX8\[bit 8\]
	CX8 = 47,
	/// CPUID.80000001H:EDX.3DNOW\[bit 31\]
	D3NOW = 48,
	/// CPUID.80000001H:EDX.3DNOWEXT\[bit 30\]
	D3NOWEXT = 49,
	/// CPUID.(EAX=12H, ECX=0H):EAX.OSS\[bit 5\]
	OSS = 50,
	/// CPUID.(EAX=07H, ECX=0H):ECX.ENQCMD\[bit 29\]
	ENQCMD = 51,
	/// CPUID.01H:ECX.F16C\[bit 29\]
	F16C = 52,
	/// CPUID.01H:ECX.FMA\[bit 12\]
	FMA = 53,
	/// CPUID.80000001H:ECX.FMA4\[bit 16\]
	FMA4 = 54,
	/// 8087 or later (CPUID.01H:EDX.FPU\[bit 0\])
	FPU = 55,
	/// 80287 or later
	FPU287 = 56,
	/// 80287XL only
	FPU287XL_ONLY = 57,
	/// 80387 or later
	FPU387 = 58,
	/// 80387SL only
	FPU387SL_ONLY = 59,
	/// CPUID.(EAX=07H, ECX=0H):EBX.FSGSBASE\[bit 0\]
	FSGSBASE = 60,
	/// CPUID.01H:EDX.FXSR\[bit 24\]
	FXSR = 61,
	/// Cyrix (AMD Geode GX/LX) 3DNow! instructions
	CYRIX_D3NOW = 62,
	/// CPUID.(EAX=07H, ECX=0H):ECX.GFNI\[bit 8\]
	GFNI = 63,
	/// CPUID.(EAX=07H, ECX=0H):EBX.HLE\[bit 4\]
	HLE = 64,
	/// [`HLE`] or [`RTM`]
	///
	/// [`HLE`]: enum.CpuidFeature.html#variant.HLE
	/// [`RTM`]: enum.CpuidFeature.html#variant.RTM
	HLE_or_RTM = 65,
	/// IA32_VMX_EPT_VPID_CAP\[bit 20\]
	INVEPT = 66,
	/// CPUID.(EAX=07H, ECX=0H):EBX.INVPCID\[bit 10\]
	INVPCID = 67,
	/// IA32_VMX_EPT_VPID_CAP\[bit 32\]
	INVVPID = 68,
	/// CPUID.80000001H:ECX.LWP\[bit 15\]
	LWP = 69,
	/// CPUID.80000001H:ECX.LZCNT\[bit 5\]
	LZCNT = 70,
	/// CPUID.80000008H:EBX.MCOMMIT\[bit 8\]
	MCOMMIT = 71,
	/// CPUID.01H:EDX.MMX\[bit 23\]
	MMX = 72,
	/// CPUID.01H:ECX.MONITOR\[bit 3\]
	MONITOR = 73,
	/// CPUID.80000001H:ECX.MONITORX\[bit 29\]
	MONITORX = 74,
	/// CPUID.01H:ECX.MOVBE\[bit 22\]
	MOVBE = 75,
	/// CPUID.(EAX=07H, ECX=0H):ECX.MOVDIR64B\[bit 28\]
	MOVDIR64B = 76,
	/// CPUID.(EAX=07H, ECX=0H):ECX.MOVDIRI\[bit 27\]
	MOVDIRI = 77,
	/// CPUID.(EAX=07H, ECX=0H):EBX.MPX\[bit 14\]
	MPX = 78,
	/// CPUID.01H:EDX.MSR\[bit 5\]
	MSR = 79,
	/// Multi-byte nops (`0F1F /0`): CPUID.01H.EAX\[Bits 11:8\] = 0110B or 1111B
	MULTIBYTENOP = 80,
	/// CPUID.0C0000000H:EAX >= 0C0000001H AND CPUID.0C0000001H:EDX.ACE\[Bits 7:6\] = 11B (\[6\] = exists, \[7\] = enabled)
	PADLOCK_ACE = 81,
	/// CPUID.0C0000000H:EAX >= 0C0000001H AND CPUID.0C0000001H:EDX.PHE\[Bits 11:10\] = 11B (\[10\] = exists, \[11\] = enabled)
	PADLOCK_PHE = 82,
	/// CPUID.0C0000000H:EAX >= 0C0000001H AND CPUID.0C0000001H:EDX.PMM\[Bits 13:12\] = 11B (\[12\] = exists, \[13\] = enabled)
	PADLOCK_PMM = 83,
	/// CPUID.0C0000000H:EAX >= 0C0000001H AND CPUID.0C0000001H:EDX.RNG\[Bits 3:2\] = 11B (\[2\] = exists, \[3\] = enabled)
	PADLOCK_RNG = 84,
	/// `PAUSE` instruction (Pentium 4 or later)
	PAUSE = 85,
	/// CPUID.01H:ECX.PCLMULQDQ\[bit 1\]
	PCLMULQDQ = 86,
	/// CPUID.(EAX=07H, ECX=0H):EBX.PCOMMIT\[bit 22\]
	PCOMMIT = 87,
	/// CPUID.(EAX=07H, ECX=0H):EDX.PCONFIG\[bit 18\]
	PCONFIG = 88,
	/// CPUID.(EAX=07H, ECX=0H):ECX.PKU\[bit 3\]
	PKU = 89,
	/// CPUID.01H:ECX.POPCNT\[bit 23\]
	POPCNT = 90,
	/// CPUID.80000001H:ECX.PREFETCHW\[bit 8\]
	PREFETCHW = 91,
	/// CPUID.(EAX=07H, ECX=0H):ECX.PREFETCHWT1\[bit 0\]
	PREFETCHWT1 = 92,
	/// CPUID.(EAX=14H, ECX=0H):EBX.PTWRITE\[bit 4\]
	PTWRITE = 93,
	/// CPUID.(EAX=07H, ECX=0H):ECX.RDPID\[bit 22\]
	RDPID = 94,
	/// `RDPMC` instruction (Pentium MMX or later, or Pentium Pro or later)
	RDPMC = 95,
	/// CPUID.80000008H:EBX.RDPRU\[bit 4\]
	RDPRU = 96,
	/// CPUID.01H:ECX.RDRAND\[bit 30\]
	RDRAND = 97,
	/// CPUID.(EAX=07H, ECX=0H):EBX.RDSEED\[bit 18\]
	RDSEED = 98,
	/// CPUID.80000001H:EDX.RDTSCP\[bit 27\]
	RDTSCP = 99,
	/// CPUID.(EAX=07H, ECX=0H):EBX.RTM\[bit 11\]
	RTM = 100,
	/// CPUID.01H:EDX.SEP\[bit 11\]
	SEP = 101,
	/// CPUID.(EAX=12H, ECX=0H):EAX.SGX1\[bit 0\]
	SGX1 = 102,
	/// CPUID.(EAX=07H, ECX=0H):EBX.SHA\[bit 29\]
	SHA = 103,
	/// CPUID.80000001H:ECX.SKINIT\[bit 12\]
	SKINIT = 104,
	/// [`SKINIT`] or [`SVM`]
	///
	/// [`SKINIT`]: enum.CpuidFeature.html#variant.SKINIT
	/// [`SVM`]: enum.CpuidFeature.html#variant.SVM
	SKINIT_or_SVM = 105,
	/// CPUID.(EAX=07H, ECX=0H):EBX.SMAP\[bit 20\]
	SMAP = 106,
	/// CPUID.01H:ECX.SMX\[bit 6\]
	SMX = 107,
	/// CPUID.01H:EDX.SSE\[bit 25\]
	SSE = 108,
	/// CPUID.01H:EDX.SSE2\[bit 26\]
	SSE2 = 109,
	/// CPUID.01H:ECX.SSE3\[bit 0\]
	SSE3 = 110,
	/// CPUID.01H:ECX.SSE4_1\[bit 19\]
	SSE4_1 = 111,
	/// CPUID.01H:ECX.SSE4_2\[bit 20\]
	SSE4_2 = 112,
	/// CPUID.80000001H:ECX.SSE4A\[bit 6\]
	SSE4A = 113,
	/// CPUID.01H:ECX.SSSE3\[bit 9\]
	SSSE3 = 114,
	/// CPUID.80000001H:ECX.SVM\[bit 2\]
	SVM = 115,
	/// CPUID.8000001FH:EAX.SEV-ES\[bit 3\]
	SEV_ES = 116,
	/// CPUID.80000001H:EDX.SYSCALL\[bit 11\]
	SYSCALL = 117,
	/// CPUID.80000001H:ECX.TBM\[bit 21\]
	TBM = 118,
	/// CPUID.01H:EDX.TSC\[bit 4\]
	TSC = 119,
	/// CPUID.(EAX=07H, ECX=0H):ECX.VAES\[bit 9\]
	VAES = 120,
	/// CPUID.01H:ECX.VMX\[bit 5\]
	VMX = 121,
	/// CPUID.(EAX=07H, ECX=0H):ECX.VPCLMULQDQ\[bit 10\]
	VPCLMULQDQ = 122,
	/// CPUID.(EAX=07H, ECX=0H):ECX.WAITPKG\[bit 5\]
	WAITPKG = 123,
	/// CPUID.(EAX=80000008H, ECX=0H):EBX.WBNOINVD\[bit 9\]
	WBNOINVD = 124,
	/// CPUID.80000001H:ECX.XOP\[bit 11\]
	XOP = 125,
	/// CPUID.01H:ECX.XSAVE\[bit 26\]
	XSAVE = 126,
	/// CPUID.(EAX=0DH, ECX=1H):EAX.XSAVEC\[bit 1\]
	XSAVEC = 127,
	/// CPUID.(EAX=0DH, ECX=1H):EAX.XSAVEOPT\[bit 0\]
	XSAVEOPT = 128,
	/// CPUID.(EAX=0DH, ECX=1H):EAX.XSAVES\[bit 3\]
	XSAVES = 129,
	/// CPUID.8000001FH:EAX.SEV-SNP\[bit 4\]
	SEV_SNP = 130,
	/// CPUID.(EAX=07H, ECX=0H):EDX.SERIALIZE\[bit 14\]
	SERIALIZE = 131,
	/// CPUID.(EAX=07H, ECX=0H):EDX.TSXLDTRK\[bit 16\]
	TSXLDTRK = 132,
	/// CPUID.80000008H:EBX.INVLPGB\[bit 3\]
	INVLPGB = 133,
	/// CPUID.(EAX=07H, ECX=0H):EDX.AMX-BF16\[bit 22\]
	AMX_BF16 = 134,
	/// CPUID.(EAX=07H, ECX=0H):EDX.AMX-TILE\[bit 24\]
	AMX_TILE = 135,
	/// CPUID.(EAX=07H, ECX=0H):EDX.AMX-INT8\[bit 25\]
	AMX_INT8 = 136,
	/// Cyrix FPU instructions (Cyrix, AMD Geode GX/LX)
	CYRIX_FPU = 137,
	/// Cyrix SMM instructions: `SVDC`, `RSDC`, `SVLDT`, `RSLDT`, `SVTS`, `RSTS` (Cyrix, AMD Geode GX/LX)
	CYRIX_SMM = 138,
	/// Cyrix `SMINT 0F38` (6x86MX and later, AMD Geode GX/LX)
	CYRIX_SMINT = 139,
	/// Cyrix `SMINT 0F7E` (6x86 or earlier)
	CYRIX_SMINT_0F7E = 140,
	/// Cyrix SMM instructions: `RDSHR`, `WRSHR` (6x86MX, M II, Cyrix III)
	CYRIX_SHR = 141,
	/// Cyrix DDI instructions: `BB0_Reset`, `BB1_Reset`, `CPU_READ`, `CPU_WRITE` (MediaGX, GXm, GXLV, GX1)
	CYRIX_DDI = 142,
	/// Cyrix AND CPUID.80000001H:EDX.EMMI\[bit 24\]
	CYRIX_EMMI = 143,
	/// Cyrix DMI instructions: `DMINT`, `RDM` (AMD Geode GX/LX)
	CYRIX_DMI = 144,
	/// CPUID.0C0000000H:EAX >= 0C0000001H AND CPUID.0C0000001H:EDX.AIS\[Bits 1:0\] = 11B (\[0\] = exists, \[1\] = enabled)
	CENTAUR_AIS = 145,
	/// MOV to/from TR (80386, Intel486, Cyrix, Geode)
	MOV_TR = 146,
	/// `RSM` instruction (some 386s, some 486s, Pentium and later)
	SMM = 147,
	/// CPUID.(EAX=??H, ECX=?H):???.????\[bit ??\]
	TDX = 148,
	/// CPUID.(EAX=07H, ECX=0H):ECX.KL\[bit 23\]
	KL = 149,
	/// CPUID.19H:EBX.AESKLE\[bit 0\]
	AESKLE = 150,
	/// CPUID.19H:EBX.WIDE_KL\[bit 2\]
	WIDE_KL = 151,
	/// CPUID.(EAX=07H, ECX=0H):EDX.UINTR\[bit 5\]
	UINTR = 152,
	/// CPUID.(EAX=07H, ECX=01H):EAX.HRESET\[bit 22\]
	HRESET = 153,
	/// CPUID.(EAX=07H, ECX=01H):EAX.AVX-VNNI\[bit 4\]
	AVX_VNNI = 154,
	/// CPUID.0C0000000H:EAX >= 0C0000001H AND CPUID.0C0000001H:EDX.GMI\[Bits 5:4\] = 11B (\[4\] = exists, \[5\] = enabled)
	PADLOCK_GMI = 155,
	/// CPUID.(EAX=07H, ECX=01H):EAX.FRED\[bit 17\]
	FRED = 156,
	/// CPUID.(EAX=07H, ECX=01H):EAX.LKGS\[bit 18\]
	LKGS = 157,
	/// CPUID.(EAX=07H, ECX=0H):EDX.AVX512-FP16\[bit 23\]
	AVX512_FP16 = 158,
	/// Undocumented Intel `RDUDBG` and `WRUDBG` instructions
	UDBG = 159,
}
#[cfg(feature = "instr_info")]
#[rustfmt::skip]
static GEN_DEBUG_CPUID_FEATURE: [&str; 160] = [
	"INTEL8086",
	"INTEL8086_ONLY",
	"INTEL186",
	"INTEL286",
	"INTEL286_ONLY",
	"INTEL386",
	"INTEL386_ONLY",
	"INTEL386_A0_ONLY",
	"INTEL486",
	"INTEL486_A_ONLY",
	"UMOV",
	"IA64",
	"X64",
	"ADX",
	"AES",
	"AVX",
	"AVX2",
	"AVX512_4FMAPS",
	"AVX512_4VNNIW",
	"AVX512_BF16",
	"AVX512_BITALG",
	"AVX512_IFMA",
	"AVX512_VBMI",
	"AVX512_VBMI2",
	"AVX512_VNNI",
	"AVX512_VP2INTERSECT",
	"AVX512_VPOPCNTDQ",
	"AVX512BW",
	"AVX512CD",
	"AVX512DQ",
	"AVX512ER",
	"AVX512F",
	"AVX512PF",
	"AVX512VL",
	"BMI1",
	"BMI2",
	"CET_IBT",
	"CET_SS",
	"CL1INVMB",
	"CLDEMOTE",
	"CLFLUSHOPT",
	"CLFSH",
	"CLWB",
	"CLZERO",
	"CMOV",
	"CMPXCHG16B",
	"CPUID",
	"CX8",
	"D3NOW",
	"D3NOWEXT",
	"OSS",
	"ENQCMD",
	"F16C",
	"FMA",
	"FMA4",
	"FPU",
	"FPU287",
	"FPU287XL_ONLY",
	"FPU387",
	"FPU387SL_ONLY",
	"FSGSBASE",
	"FXSR",
	"CYRIX_D3NOW",
	"GFNI",
	"HLE",
	"HLE_or_RTM",
	"INVEPT",
	"INVPCID",
	"INVVPID",
	"LWP",
	"LZCNT",
	"MCOMMIT",
	"MMX",
	"MONITOR",
	"MONITORX",
	"MOVBE",
	"MOVDIR64B",
	"MOVDIRI",
	"MPX",
	"MSR",
	"MULTIBYTENOP",
	"PADLOCK_ACE",
	"PADLOCK_PHE",
	"PADLOCK_PMM",
	"PADLOCK_RNG",
	"PAUSE",
	"PCLMULQDQ",
	"PCOMMIT",
	"PCONFIG",
	"PKU",
	"POPCNT",
	"PREFETCHW",
	"PREFETCHWT1",
	"PTWRITE",
	"RDPID",
	"RDPMC",
	"RDPRU",
	"RDRAND",
	"RDSEED",
	"RDTSCP",
	"RTM",
	"SEP",
	"SGX1",
	"SHA",
	"SKINIT",
	"SKINIT_or_SVM",
	"SMAP",
	"SMX",
	"SSE",
	"SSE2",
	"SSE3",
	"SSE4_1",
	"SSE4_2",
	"SSE4A",
	"SSSE3",
	"SVM",
	"SEV_ES",
	"SYSCALL",
	"TBM",
	"TSC",
	"VAES",
	"VMX",
	"VPCLMULQDQ",
	"WAITPKG",
	"WBNOINVD",
	"XOP",
	"XSAVE",
	"XSAVEC",
	"XSAVEOPT",
	"XSAVES",
	"SEV_SNP",
	"SERIALIZE",
	"TSXLDTRK",
	"INVLPGB",
	"AMX_BF16",
	"AMX_TILE",
	"AMX_INT8",
	"CYRIX_FPU",
	"CYRIX_SMM",
	"CYRIX_SMINT",
	"CYRIX_SMINT_0F7E",
	"CYRIX_SHR",
	"CYRIX_DDI",
	"CYRIX_EMMI",
	"CYRIX_DMI",
	"CENTAUR_AIS",
	"MOV_TR",
	"SMM",
	"TDX",
	"KL",
	"AESKLE",
	"WIDE_KL",
	"UINTR",
	"HRESET",
	"AVX_VNNI",
	"PADLOCK_GMI",
	"FRED",
	"LKGS",
	"AVX512_FP16",
	"UDBG",
];
#[cfg(feature = "instr_info")]
impl fmt::Debug for CpuidFeature {
	#[inline]
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{}", GEN_DEBUG_CPUID_FEATURE[*self as usize])
	}
}
#[cfg(feature = "instr_info")]
impl Default for CpuidFeature {
	#[must_use]
	#[inline]
	fn default() -> Self {
		CpuidFeature::INTEL8086
	}
}
#[cfg(feature = "instr_info")]
#[allow(non_camel_case_types)]
#[allow(dead_code)]
pub(crate) type CpuidFeatureUnderlyingType = u8;
#[cfg(feature = "instr_info")]
#[rustfmt::skip]
impl CpuidFeature {
	/// Iterates over all `CpuidFeature` enum values
	#[inline]
	pub fn values() -> impl Iterator<Item = CpuidFeature> + DoubleEndedIterator + ExactSizeIterator + FusedIterator {
		// SAFETY: all values 0-max are valid enum values
		(0..IcedConstants::CPUID_FEATURE_ENUM_COUNT).map(|x| unsafe { mem::transmute::<u8, CpuidFeature>(x as u8) })
	}
}
#[test]
#[cfg(feature = "instr_info")]
#[rustfmt::skip]
fn test_cpuidfeature_values() {
	let mut iter = CpuidFeature::values();
	assert_eq!(iter.size_hint(), (IcedConstants::CPUID_FEATURE_ENUM_COUNT, Some(IcedConstants::CPUID_FEATURE_ENUM_COUNT)));
	assert_eq!(iter.len(), IcedConstants::CPUID_FEATURE_ENUM_COUNT);
	assert!(iter.next().is_some());
	assert_eq!(iter.size_hint(), (IcedConstants::CPUID_FEATURE_ENUM_COUNT - 1, Some(IcedConstants::CPUID_FEATURE_ENUM_COUNT - 1)));
	assert_eq!(iter.len(), IcedConstants::CPUID_FEATURE_ENUM_COUNT - 1);

	let values: Vec<CpuidFeature> = CpuidFeature::values().collect();
	assert_eq!(values.len(), IcedConstants::CPUID_FEATURE_ENUM_COUNT);
	for (i, value) in values.into_iter().enumerate() {
		assert_eq!(i, value as usize);
	}

	let values1: Vec<CpuidFeature> = CpuidFeature::values().collect();
	let mut values2: Vec<CpuidFeature> = CpuidFeature::values().rev().collect();
	values2.reverse();
	assert_eq!(values1, values2);
}
#[cfg(feature = "instr_info")]
#[rustfmt::skip]
impl TryFrom<usize> for CpuidFeature {
	type Error = IcedError;
	#[inline]
	fn try_from(value: usize) -> Result<Self, Self::Error> {
		if value < IcedConstants::CPUID_FEATURE_ENUM_COUNT {
			// SAFETY: all values 0-max are valid enum values
			Ok(unsafe { mem::transmute(value as u8) })
		} else {
			Err(IcedError::new("Invalid CpuidFeature value"))
		}
	}
}
#[cfg(feature = "instr_info")]
#[test]
#[rustfmt::skip]
fn test_cpuidfeature_try_from_usize() {
	for value in CpuidFeature::values() {
		let converted = <CpuidFeature as TryFrom<usize>>::try_from(value as usize).unwrap();
		assert_eq!(converted, value);
	}
	assert!(<CpuidFeature as TryFrom<usize>>::try_from(IcedConstants::CPUID_FEATURE_ENUM_COUNT).is_err());
	assert!(<CpuidFeature as TryFrom<usize>>::try_from(core::usize::MAX).is_err());
}
// GENERATOR-END: CpuidFeature

// GENERATOR-BEGIN: RflagsBits
// ⚠️This was generated by GENERATOR!🦹‍♂️
/// `RFLAGS` bits, FPU condition code bits and misc bits (`UIF`) supported by the instruction info code
#[allow(missing_copy_implementations)]
#[allow(missing_debug_implementations)]
#[cfg(feature = "instr_info")]
pub struct RflagsBits;
#[cfg(feature = "instr_info")]
impl RflagsBits {
	/// No bit is set
	pub const NONE: u32 = 0x0000_0000;
	/// `RFLAGS.OF`
	pub const OF: u32 = 0x0000_0001;
	/// `RFLAGS.SF`
	pub const SF: u32 = 0x0000_0002;
	/// `RFLAGS.ZF`
	pub const ZF: u32 = 0x0000_0004;
	/// `RFLAGS.AF`
	pub const AF: u32 = 0x0000_0008;
	/// `RFLAGS.CF`
	pub const CF: u32 = 0x0000_0010;
	/// `RFLAGS.PF`
	pub const PF: u32 = 0x0000_0020;
	/// `RFLAGS.DF`
	pub const DF: u32 = 0x0000_0040;
	/// `RFLAGS.IF`
	pub const IF: u32 = 0x0000_0080;
	/// `RFLAGS.AC`
	pub const AC: u32 = 0x0000_0100;
	/// `UIF`
	pub const UIF: u32 = 0x0000_0200;
	/// FPU status word bit `C0`
	pub const C0: u32 = 0x0000_0400;
	/// FPU status word bit `C1`
	pub const C1: u32 = 0x0000_0800;
	/// FPU status word bit `C2`
	pub const C2: u32 = 0x0000_1000;
	/// FPU status word bit `C3`
	pub const C3: u32 = 0x0000_2000;
}
// GENERATOR-END: RflagsBits

// GENERATOR-BEGIN: OpAccess
// ⚠️This was generated by GENERATOR!🦹‍♂️
/// Operand, register and memory access
#[derive(Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash)]
#[cfg(feature = "instr_info")]
pub enum OpAccess {
	/// Nothing is read and nothing is written
	None = 0,
	/// The value is read
	Read = 1,
	/// The value is sometimes read and sometimes not
	CondRead = 2,
	/// The value is completely overwritten
	Write = 3,
	/// Conditional write, sometimes it's written and sometimes it's not modified
	CondWrite = 4,
	/// The value is read and written
	ReadWrite = 5,
	/// The value is read and sometimes written
	ReadCondWrite = 6,
	/// The memory operand doesn't refer to memory (eg. `LEA` instruction) or it's an instruction that doesn't read the data to a register or doesn't write to the memory location, it just prefetches/invalidates it, eg. `INVLPG`, `PREFETCHNTA`, `VGATHERPF0DPS`, etc. Some of those instructions still check if the code can access the memory location.
	NoMemAccess = 7,
}
#[cfg(feature = "instr_info")]
#[rustfmt::skip]
static GEN_DEBUG_OP_ACCESS: [&str; 8] = [
	"None",
	"Read",
	"CondRead",
	"Write",
	"CondWrite",
	"ReadWrite",
	"ReadCondWrite",
	"NoMemAccess",
];
#[cfg(feature = "instr_info")]
impl fmt::Debug for OpAccess {
	#[inline]
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{}", GEN_DEBUG_OP_ACCESS[*self as usize])
	}
}
#[cfg(feature = "instr_info")]
impl Default for OpAccess {
	#[must_use]
	#[inline]
	fn default() -> Self {
		OpAccess::None
	}
}
#[cfg(feature = "instr_info")]
#[allow(non_camel_case_types)]
#[allow(dead_code)]
pub(crate) type OpAccessUnderlyingType = u8;
#[cfg(feature = "instr_info")]
#[rustfmt::skip]
impl OpAccess {
	/// Iterates over all `OpAccess` enum values
	#[inline]
	pub fn values() -> impl Iterator<Item = OpAccess> + DoubleEndedIterator + ExactSizeIterator + FusedIterator {
		// SAFETY: all values 0-max are valid enum values
		(0..IcedConstants::OP_ACCESS_ENUM_COUNT).map(|x| unsafe { mem::transmute::<u8, OpAccess>(x as u8) })
	}
}
#[test]
#[cfg(feature = "instr_info")]
#[rustfmt::skip]
fn test_opaccess_values() {
	let mut iter = OpAccess::values();
	assert_eq!(iter.size_hint(), (IcedConstants::OP_ACCESS_ENUM_COUNT, Some(IcedConstants::OP_ACCESS_ENUM_COUNT)));
	assert_eq!(iter.len(), IcedConstants::OP_ACCESS_ENUM_COUNT);
	assert!(iter.next().is_some());
	assert_eq!(iter.size_hint(), (IcedConstants::OP_ACCESS_ENUM_COUNT - 1, Some(IcedConstants::OP_ACCESS_ENUM_COUNT - 1)));
	assert_eq!(iter.len(), IcedConstants::OP_ACCESS_ENUM_COUNT - 1);

	let values: Vec<OpAccess> = OpAccess::values().collect();
	assert_eq!(values.len(), IcedConstants::OP_ACCESS_ENUM_COUNT);
	for (i, value) in values.into_iter().enumerate() {
		assert_eq!(i, value as usize);
	}

	let values1: Vec<OpAccess> = OpAccess::values().collect();
	let mut values2: Vec<OpAccess> = OpAccess::values().rev().collect();
	values2.reverse();
	assert_eq!(values1, values2);
}
#[cfg(feature = "instr_info")]
#[rustfmt::skip]
impl TryFrom<usize> for OpAccess {
	type Error = IcedError;
	#[inline]
	fn try_from(value: usize) -> Result<Self, Self::Error> {
		if value < IcedConstants::OP_ACCESS_ENUM_COUNT {
			// SAFETY: all values 0-max are valid enum values
			Ok(unsafe { mem::transmute(value as u8) })
		} else {
			Err(IcedError::new("Invalid OpAccess value"))
		}
	}
}
#[cfg(feature = "instr_info")]
#[test]
#[rustfmt::skip]
fn test_opaccess_try_from_usize() {
	for value in OpAccess::values() {
		let converted = <OpAccess as TryFrom<usize>>::try_from(value as usize).unwrap();
		assert_eq!(converted, value);
	}
	assert!(<OpAccess as TryFrom<usize>>::try_from(IcedConstants::OP_ACCESS_ENUM_COUNT).is_err());
	assert!(<OpAccess as TryFrom<usize>>::try_from(core::usize::MAX).is_err());
}
// GENERATOR-END: OpAccess

// GENERATOR-BEGIN: ConditionCode
// ⚠️This was generated by GENERATOR!🦹‍♂️
/// Instruction condition code (used by `Jcc`, `SETcc`, `CMOVcc`, `LOOPcc`)
#[derive(Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash)]
#[cfg(feature = "instr_info")]
#[allow(non_camel_case_types)]
pub enum ConditionCode {
	/// The instruction doesn't have a condition code
	None = 0,
	/// Overflow (`OF=1`)
	o = 1,
	/// Not overflow (`OF=0`)
	no = 2,
	/// Below (unsigned) (`CF=1`)
	b = 3,
	/// Above or equal (unsigned) (`CF=0`)
	ae = 4,
	/// Equal / zero (`ZF=1`)
	e = 5,
	/// Not equal / zero (`ZF=0`)
	ne = 6,
	/// Below or equal (unsigned) (`CF=1 or ZF=1`)
	be = 7,
	/// Above (unsigned) (`CF=0 and ZF=0`)
	a = 8,
	/// Signed (`SF=1`)
	s = 9,
	/// Not signed (`SF=0`)
	ns = 10,
	/// Parity (`PF=1`)
	p = 11,
	/// Not parity (`PF=0`)
	np = 12,
	/// Less (signed) (`SF!=OF`)
	l = 13,
	/// Greater than or equal (signed) (`SF=OF`)
	ge = 14,
	/// Less than or equal (signed) (`ZF=1 or SF!=OF`)
	le = 15,
	/// Greater (signed) (`ZF=0 and SF=OF`)
	g = 16,
}
#[cfg(feature = "instr_info")]
#[rustfmt::skip]
static GEN_DEBUG_CONDITION_CODE: [&str; 17] = [
	"None",
	"o",
	"no",
	"b",
	"ae",
	"e",
	"ne",
	"be",
	"a",
	"s",
	"ns",
	"p",
	"np",
	"l",
	"ge",
	"le",
	"g",
];
#[cfg(feature = "instr_info")]
impl fmt::Debug for ConditionCode {
	#[inline]
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{}", GEN_DEBUG_CONDITION_CODE[*self as usize])
	}
}
#[cfg(feature = "instr_info")]
impl Default for ConditionCode {
	#[must_use]
	#[inline]
	fn default() -> Self {
		ConditionCode::None
	}
}
#[cfg(feature = "instr_info")]
#[allow(non_camel_case_types)]
#[allow(dead_code)]
pub(crate) type ConditionCodeUnderlyingType = u8;
#[cfg(feature = "instr_info")]
#[rustfmt::skip]
impl ConditionCode {
	/// Iterates over all `ConditionCode` enum values
	#[inline]
	pub fn values() -> impl Iterator<Item = ConditionCode> + DoubleEndedIterator + ExactSizeIterator + FusedIterator {
		// SAFETY: all values 0-max are valid enum values
		(0..IcedConstants::CONDITION_CODE_ENUM_COUNT).map(|x| unsafe { mem::transmute::<u8, ConditionCode>(x as u8) })
	}
}
#[test]
#[cfg(feature = "instr_info")]
#[rustfmt::skip]
fn test_conditioncode_values() {
	let mut iter = ConditionCode::values();
	assert_eq!(iter.size_hint(), (IcedConstants::CONDITION_CODE_ENUM_COUNT, Some(IcedConstants::CONDITION_CODE_ENUM_COUNT)));
	assert_eq!(iter.len(), IcedConstants::CONDITION_CODE_ENUM_COUNT);
	assert!(iter.next().is_some());
	assert_eq!(iter.size_hint(), (IcedConstants::CONDITION_CODE_ENUM_COUNT - 1, Some(IcedConstants::CONDITION_CODE_ENUM_COUNT - 1)));
	assert_eq!(iter.len(), IcedConstants::CONDITION_CODE_ENUM_COUNT - 1);

	let values: Vec<ConditionCode> = ConditionCode::values().collect();
	assert_eq!(values.len(), IcedConstants::CONDITION_CODE_ENUM_COUNT);
	for (i, value) in values.into_iter().enumerate() {
		assert_eq!(i, value as usize);
	}

	let values1: Vec<ConditionCode> = ConditionCode::values().collect();
	let mut values2: Vec<ConditionCode> = ConditionCode::values().rev().collect();
	values2.reverse();
	assert_eq!(values1, values2);
}
#[cfg(feature = "instr_info")]
#[rustfmt::skip]
impl TryFrom<usize> for ConditionCode {
	type Error = IcedError;
	#[inline]
	fn try_from(value: usize) -> Result<Self, Self::Error> {
		if value < IcedConstants::CONDITION_CODE_ENUM_COUNT {
			// SAFETY: all values 0-max are valid enum values
			Ok(unsafe { mem::transmute(value as u8) })
		} else {
			Err(IcedError::new("Invalid ConditionCode value"))
		}
	}
}
#[cfg(feature = "instr_info")]
#[test]
#[rustfmt::skip]
fn test_conditioncode_try_from_usize() {
	for value in ConditionCode::values() {
		let converted = <ConditionCode as TryFrom<usize>>::try_from(value as usize).unwrap();
		assert_eq!(converted, value);
	}
	assert!(<ConditionCode as TryFrom<usize>>::try_from(IcedConstants::CONDITION_CODE_ENUM_COUNT).is_err());
	assert!(<ConditionCode as TryFrom<usize>>::try_from(core::usize::MAX).is_err());
}
// GENERATOR-END: ConditionCode

// GENERATOR-BEGIN: MandatoryPrefix
// ⚠️This was generated by GENERATOR!🦹‍♂️
/// Mandatory prefix
#[derive(Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash)]
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
pub enum MandatoryPrefix {
	/// No mandatory prefix (legacy and 3DNow! tables only)
	None = 0,
	/// Empty mandatory prefix (no `66`, `F3` or `F2` prefix)
	PNP = 1,
	/// `66` prefix
	P66 = 2,
	/// `F3` prefix
	PF3 = 3,
	/// `F2` prefix
	PF2 = 4,
}
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[rustfmt::skip]
static GEN_DEBUG_MANDATORY_PREFIX: [&str; 5] = [
	"None",
	"PNP",
	"P66",
	"PF3",
	"PF2",
];
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
impl fmt::Debug for MandatoryPrefix {
	#[inline]
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{}", GEN_DEBUG_MANDATORY_PREFIX[*self as usize])
	}
}
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
impl Default for MandatoryPrefix {
	#[must_use]
	#[inline]
	fn default() -> Self {
		MandatoryPrefix::None
	}
}
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[allow(non_camel_case_types)]
#[allow(dead_code)]
pub(crate) type MandatoryPrefixUnderlyingType = u8;
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[rustfmt::skip]
impl MandatoryPrefix {
	/// Iterates over all `MandatoryPrefix` enum values
	#[inline]
	pub fn values() -> impl Iterator<Item = MandatoryPrefix> + DoubleEndedIterator + ExactSizeIterator + FusedIterator {
		// SAFETY: all values 0-max are valid enum values
		(0..IcedConstants::MANDATORY_PREFIX_ENUM_COUNT).map(|x| unsafe { mem::transmute::<u8, MandatoryPrefix>(x as u8) })
	}
}
#[test]
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[rustfmt::skip]
fn test_mandatoryprefix_values() {
	let mut iter = MandatoryPrefix::values();
	assert_eq!(iter.size_hint(), (IcedConstants::MANDATORY_PREFIX_ENUM_COUNT, Some(IcedConstants::MANDATORY_PREFIX_ENUM_COUNT)));
	assert_eq!(iter.len(), IcedConstants::MANDATORY_PREFIX_ENUM_COUNT);
	assert!(iter.next().is_some());
	assert_eq!(iter.size_hint(), (IcedConstants::MANDATORY_PREFIX_ENUM_COUNT - 1, Some(IcedConstants::MANDATORY_PREFIX_ENUM_COUNT - 1)));
	assert_eq!(iter.len(), IcedConstants::MANDATORY_PREFIX_ENUM_COUNT - 1);

	let values: Vec<MandatoryPrefix> = MandatoryPrefix::values().collect();
	assert_eq!(values.len(), IcedConstants::MANDATORY_PREFIX_ENUM_COUNT);
	for (i, value) in values.into_iter().enumerate() {
		assert_eq!(i, value as usize);
	}

	let values1: Vec<MandatoryPrefix> = MandatoryPrefix::values().collect();
	let mut values2: Vec<MandatoryPrefix> = MandatoryPrefix::values().rev().collect();
	values2.reverse();
	assert_eq!(values1, values2);
}
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[rustfmt::skip]
impl TryFrom<usize> for MandatoryPrefix {
	type Error = IcedError;
	#[inline]
	fn try_from(value: usize) -> Result<Self, Self::Error> {
		if value < IcedConstants::MANDATORY_PREFIX_ENUM_COUNT {
			// SAFETY: all values 0-max are valid enum values
			Ok(unsafe { mem::transmute(value as u8) })
		} else {
			Err(IcedError::new("Invalid MandatoryPrefix value"))
		}
	}
}
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[test]
#[rustfmt::skip]
fn test_mandatoryprefix_try_from_usize() {
	for value in MandatoryPrefix::values() {
		let converted = <MandatoryPrefix as TryFrom<usize>>::try_from(value as usize).unwrap();
		assert_eq!(converted, value);
	}
	assert!(<MandatoryPrefix as TryFrom<usize>>::try_from(IcedConstants::MANDATORY_PREFIX_ENUM_COUNT).is_err());
	assert!(<MandatoryPrefix as TryFrom<usize>>::try_from(core::usize::MAX).is_err());
}
// GENERATOR-END: MandatoryPrefix

// GENERATOR-BEGIN: OpCodeTableKind
// ⚠️This was generated by GENERATOR!🦹‍♂️
/// Opcode table
#[derive(Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash)]
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[cfg_attr(not(feature = "exhaustive_enums"), non_exhaustive)]
pub enum OpCodeTableKind {
	/// Legacy/`MAP0` table
	Normal = 0,
	/// `0F`/`MAP1` table (legacy, VEX, EVEX)
	T0F = 1,
	/// `0F38`/`MAP2` table (legacy, VEX, EVEX)
	T0F38 = 2,
	/// `0F3A`/`MAP3` table (legacy, VEX, EVEX)
	T0F3A = 3,
	/// `MAP5` table (EVEX)
	MAP5 = 4,
	/// `MAP6` table (EVEX)
	MAP6 = 5,
	/// `MAP8` table (XOP)
	MAP8 = 6,
	/// `MAP9` table (XOP)
	MAP9 = 7,
	/// `MAP10` table (XOP)
	MAP10 = 8,
}
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[rustfmt::skip]
static GEN_DEBUG_OP_CODE_TABLE_KIND: [&str; 9] = [
	"Normal",
	"T0F",
	"T0F38",
	"T0F3A",
	"MAP5",
	"MAP6",
	"MAP8",
	"MAP9",
	"MAP10",
];
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
impl fmt::Debug for OpCodeTableKind {
	#[inline]
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		write!(f, "{}", GEN_DEBUG_OP_CODE_TABLE_KIND[*self as usize])
	}
}
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
impl Default for OpCodeTableKind {
	#[must_use]
	#[inline]
	fn default() -> Self {
		OpCodeTableKind::Normal
	}
}
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[allow(non_camel_case_types)]
#[allow(dead_code)]
pub(crate) type OpCodeTableKindUnderlyingType = u8;
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[rustfmt::skip]
impl OpCodeTableKind {
	/// Iterates over all `OpCodeTableKind` enum values
	#[inline]
	pub fn values() -> impl Iterator<Item = OpCodeTableKind> + DoubleEndedIterator + ExactSizeIterator + FusedIterator {
		// SAFETY: all values 0-max are valid enum values
		(0..IcedConstants::OP_CODE_TABLE_KIND_ENUM_COUNT).map(|x| unsafe { mem::transmute::<u8, OpCodeTableKind>(x as u8) })
	}
}
#[test]
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[rustfmt::skip]
fn test_opcodetablekind_values() {
	let mut iter = OpCodeTableKind::values();
	assert_eq!(iter.size_hint(), (IcedConstants::OP_CODE_TABLE_KIND_ENUM_COUNT, Some(IcedConstants::OP_CODE_TABLE_KIND_ENUM_COUNT)));
	assert_eq!(iter.len(), IcedConstants::OP_CODE_TABLE_KIND_ENUM_COUNT);
	assert!(iter.next().is_some());
	assert_eq!(iter.size_hint(), (IcedConstants::OP_CODE_TABLE_KIND_ENUM_COUNT - 1, Some(IcedConstants::OP_CODE_TABLE_KIND_ENUM_COUNT - 1)));
	assert_eq!(iter.len(), IcedConstants::OP_CODE_TABLE_KIND_ENUM_COUNT - 1);

	let values: Vec<OpCodeTableKind> = OpCodeTableKind::values().collect();
	assert_eq!(values.len(), IcedConstants::OP_CODE_TABLE_KIND_ENUM_COUNT);
	for (i, value) in values.into_iter().enumerate() {
		assert_eq!(i, value as usize);
	}

	let values1: Vec<OpCodeTableKind> = OpCodeTableKind::values().collect();
	let mut values2: Vec<OpCodeTableKind> = OpCodeTableKind::values().rev().collect();
	values2.reverse();
	assert_eq!(values1, values2);
}
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[rustfmt::skip]
impl TryFrom<usize> for OpCodeTableKind {
	type Error = IcedError;
	#[inline]
	fn try_from(value: usize) -> Result<Self, Self::Error> {
		if value < IcedConstants::OP_CODE_TABLE_KIND_ENUM_COUNT {
			// SAFETY: all values 0-max are valid enum values
			Ok(unsafe { mem::transmute(value as u8) })
		} else {
			Err(IcedError::new("Invalid OpCodeTableKind value"))
		}
	}
}
#[cfg(all(feature = "encoder", feature = "op_code_info"))]
#[test]
#[rustfmt::skip]
fn test_opcodetablekind_try_from_usize() {
	for value in OpCodeTableKind::values() {
		let converted = <OpCodeTableKind as TryFrom<usize>>::try_from(value as usize).unwrap();
		assert_eq!(converted, value);
	}
	assert!(<OpCodeTableKind as TryFrom<usize>>::try_from(IcedConstants::OP_CODE_TABLE_KIND_ENUM_COUNT).is_err());
	assert!(<OpCodeTableKind as TryFrom<usize>>::try_from(core::usize::MAX).is_err());
}
// GENERATOR-END: OpCodeTableKind

#[derive(Debug, Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Hash)]
pub(crate) enum InstrScale {
	Scale1 = 0,
	Scale2 = 1,
	Scale4 = 2,
	Scale8 = 3,
}
impl Default for InstrScale {
	fn default() -> Self {
		InstrScale::Scale1
	}
}
