// SPDX-License-Identifier: MIT
// Copyright (C) 2018-present iced project and contributors

use crate::block_enc::tests::*;
use core::u32;

const BITNESS: u32 = 16;
const ORIG_RIP: u64 = 0x8000;
const NEW_RIP: u64 = 0xF000;

#[test]
fn jcc_short_fwd() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x70, 0x3E,// jo short 0040h
		/*0002*/ 0xB0, 0x00,// mov al,0
		/*0004*/ 0x71, 0x3A,// jno short 0040h
		/*0006*/ 0xB0, 0x01,// mov al,1
		/*0008*/ 0x72, 0x36,// jb short 0040h
		/*000A*/ 0xB0, 0x02,// mov al,2
		/*000C*/ 0x73, 0x32,// jae short 0040h
		/*000E*/ 0xB0, 0x03,// mov al,3
		/*0010*/ 0x74, 0x2E,// je short 0040h
		/*0012*/ 0xB0, 0x04,// mov al,4
		/*0014*/ 0x75, 0x2A,// jne short 0040h
		/*0016*/ 0xB0, 0x05,// mov al,5
		/*0018*/ 0x76, 0x26,// jbe short 0040h
		/*001A*/ 0xB0, 0x06,// mov al,6
		/*001C*/ 0x77, 0x22,// ja short 0040h
		/*001E*/ 0xB0, 0x07,// mov al,7
		/*0020*/ 0x78, 0x1E,// js short 0040h
		/*0022*/ 0xB0, 0x08,// mov al,8
		/*0024*/ 0x79, 0x1A,// jns short 0040h
		/*0026*/ 0xB0, 0x09,// mov al,9
		/*0028*/ 0x7A, 0x16,// jp short 0040h
		/*002A*/ 0xB0, 0x0A,// mov al,0Ah
		/*002C*/ 0x7B, 0x12,// jnp short 0040h
		/*002E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0030*/ 0x7C, 0x0E,// jl short 0040h
		/*0032*/ 0xB0, 0x0C,// mov al,0Ch
		/*0034*/ 0x7D, 0x0A,// jge short 0040h
		/*0036*/ 0xB0, 0x0D,// mov al,0Dh
		/*0038*/ 0x7E, 0x06,// jle short 0040h
		/*003A*/ 0xB0, 0x0E,// mov al,0Eh
		/*003C*/ 0x7F, 0x02,// jg short 0040h
		/*003E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0040*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x70, 0x3E,// jo short 0040h
		/*0002*/ 0xB0, 0x00,// mov al,0
		/*0004*/ 0x71, 0x3A,// jno short 0040h
		/*0006*/ 0xB0, 0x01,// mov al,1
		/*0008*/ 0x72, 0x36,// jb short 0040h
		/*000A*/ 0xB0, 0x02,// mov al,2
		/*000C*/ 0x73, 0x32,// jae short 0040h
		/*000E*/ 0xB0, 0x03,// mov al,3
		/*0010*/ 0x74, 0x2E,// je short 0040h
		/*0012*/ 0xB0, 0x04,// mov al,4
		/*0014*/ 0x75, 0x2A,// jne short 0040h
		/*0016*/ 0xB0, 0x05,// mov al,5
		/*0018*/ 0x76, 0x26,// jbe short 0040h
		/*001A*/ 0xB0, 0x06,// mov al,6
		/*001C*/ 0x77, 0x22,// ja short 0040h
		/*001E*/ 0xB0, 0x07,// mov al,7
		/*0020*/ 0x78, 0x1E,// js short 0040h
		/*0022*/ 0xB0, 0x08,// mov al,8
		/*0024*/ 0x79, 0x1A,// jns short 0040h
		/*0026*/ 0xB0, 0x09,// mov al,9
		/*0028*/ 0x7A, 0x16,// jp short 0040h
		/*002A*/ 0xB0, 0x0A,// mov al,0Ah
		/*002C*/ 0x7B, 0x12,// jnp short 0040h
		/*002E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0030*/ 0x7C, 0x0E,// jl short 0040h
		/*0032*/ 0xB0, 0x0C,// mov al,0Ch
		/*0034*/ 0x7D, 0x0A,// jge short 0040h
		/*0036*/ 0xB0, 0x0D,// mov al,0Dh
		/*0038*/ 0x7E, 0x06,// jle short 0040h
		/*003A*/ 0xB0, 0x0E,// mov al,0Eh
		/*003C*/ 0x7F, 0x02,// jg short 0040h
		/*003E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0040*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0002,
		0x0004,
		0x0006,
		0x0008,
		0x000A,
		0x000C,
		0x000E,
		0x0010,
		0x0012,
		0x0014,
		0x0016,
		0x0018,
		0x001A,
		0x001C,
		0x001E,
		0x0020,
		0x0022,
		0x0024,
		0x0026,
		0x0028,
		0x002A,
		0x002C,
		0x002E,
		0x0030,
		0x0032,
		0x0034,
		0x0036,
		0x0038,
		0x003A,
		0x003C,
		0x003E,
		0x0040,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_short_fwd_os() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x66, 0x70, 0x4D,// jo short 00000050h
		/*0003*/ 0xB0, 0x00,// mov al,0
		/*0005*/ 0x66, 0x71, 0x48,// jno short 00000050h
		/*0008*/ 0xB0, 0x01,// mov al,1
		/*000A*/ 0x66, 0x72, 0x43,// jb short 00000050h
		/*000D*/ 0xB0, 0x02,// mov al,2
		/*000F*/ 0x66, 0x73, 0x3E,// jae short 00000050h
		/*0012*/ 0xB0, 0x03,// mov al,3
		/*0014*/ 0x66, 0x74, 0x39,// je short 00000050h
		/*0017*/ 0xB0, 0x04,// mov al,4
		/*0019*/ 0x66, 0x75, 0x34,// jne short 00000050h
		/*001C*/ 0xB0, 0x05,// mov al,5
		/*001E*/ 0x66, 0x76, 0x2F,// jbe short 00000050h
		/*0021*/ 0xB0, 0x06,// mov al,6
		/*0023*/ 0x66, 0x77, 0x2A,// ja short 00000050h
		/*0026*/ 0xB0, 0x07,// mov al,7
		/*0028*/ 0x66, 0x78, 0x25,// js short 00000050h
		/*002B*/ 0xB0, 0x08,// mov al,8
		/*002D*/ 0x66, 0x79, 0x20,// jns short 00000050h
		/*0030*/ 0xB0, 0x09,// mov al,9
		/*0032*/ 0x66, 0x7A, 0x1B,// jp short 00000050h
		/*0035*/ 0xB0, 0x0A,// mov al,0Ah
		/*0037*/ 0x66, 0x7B, 0x16,// jnp short 00000050h
		/*003A*/ 0xB0, 0x0B,// mov al,0Bh
		/*003C*/ 0x66, 0x7C, 0x11,// jl short 00000050h
		/*003F*/ 0xB0, 0x0C,// mov al,0Ch
		/*0041*/ 0x66, 0x7D, 0x0C,// jge short 00000050h
		/*0044*/ 0xB0, 0x0D,// mov al,0Dh
		/*0046*/ 0x66, 0x7E, 0x07,// jle short 00000050h
		/*0049*/ 0xB0, 0x0E,// mov al,0Eh
		/*004B*/ 0x66, 0x7F, 0x02,// jg short 00000050h
		/*004E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0050*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x66, 0x70, 0x4D,// jo short 00000050h
		/*0003*/ 0xB0, 0x00,// mov al,0
		/*0005*/ 0x66, 0x71, 0x48,// jno short 00000050h
		/*0008*/ 0xB0, 0x01,// mov al,1
		/*000A*/ 0x66, 0x72, 0x43,// jb short 00000050h
		/*000D*/ 0xB0, 0x02,// mov al,2
		/*000F*/ 0x66, 0x73, 0x3E,// jae short 00000050h
		/*0012*/ 0xB0, 0x03,// mov al,3
		/*0014*/ 0x66, 0x74, 0x39,// je short 00000050h
		/*0017*/ 0xB0, 0x04,// mov al,4
		/*0019*/ 0x66, 0x75, 0x34,// jne short 00000050h
		/*001C*/ 0xB0, 0x05,// mov al,5
		/*001E*/ 0x66, 0x76, 0x2F,// jbe short 00000050h
		/*0021*/ 0xB0, 0x06,// mov al,6
		/*0023*/ 0x66, 0x77, 0x2A,// ja short 00000050h
		/*0026*/ 0xB0, 0x07,// mov al,7
		/*0028*/ 0x66, 0x78, 0x25,// js short 00000050h
		/*002B*/ 0xB0, 0x08,// mov al,8
		/*002D*/ 0x66, 0x79, 0x20,// jns short 00000050h
		/*0030*/ 0xB0, 0x09,// mov al,9
		/*0032*/ 0x66, 0x7A, 0x1B,// jp short 00000050h
		/*0035*/ 0xB0, 0x0A,// mov al,0Ah
		/*0037*/ 0x66, 0x7B, 0x16,// jnp short 00000050h
		/*003A*/ 0xB0, 0x0B,// mov al,0Bh
		/*003C*/ 0x66, 0x7C, 0x11,// jl short 00000050h
		/*003F*/ 0xB0, 0x0C,// mov al,0Ch
		/*0041*/ 0x66, 0x7D, 0x0C,// jge short 00000050h
		/*0044*/ 0xB0, 0x0D,// mov al,0Dh
		/*0046*/ 0x66, 0x7E, 0x07,// jle short 00000050h
		/*0049*/ 0xB0, 0x0E,// mov al,0Eh
		/*004B*/ 0x66, 0x7F, 0x02,// jg short 00000050h
		/*004E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0050*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0003,
		0x0005,
		0x0008,
		0x000A,
		0x000D,
		0x000F,
		0x0012,
		0x0014,
		0x0017,
		0x0019,
		0x001C,
		0x001E,
		0x0021,
		0x0023,
		0x0026,
		0x0028,
		0x002B,
		0x002D,
		0x0030,
		0x0032,
		0x0035,
		0x0037,
		0x003A,
		0x003C,
		0x003F,
		0x0041,
		0x0044,
		0x0046,
		0x0049,
		0x004B,
		0x004E,
		0x0050,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_short_bwd() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0xB0, 0x10,// mov al,10h
		/*0002*/ 0x70, 0xFC,// jo short 0
		/*0004*/ 0xB0, 0x00,// mov al,0
		/*0006*/ 0x71, 0xF8,// jno short 0
		/*0008*/ 0xB0, 0x01,// mov al,1
		/*000A*/ 0x72, 0xF4,// jb short 0
		/*000C*/ 0xB0, 0x02,// mov al,2
		/*000E*/ 0x73, 0xF0,// jae short 0
		/*0010*/ 0xB0, 0x03,// mov al,3
		/*0012*/ 0x74, 0xEC,// je short 0
		/*0014*/ 0xB0, 0x04,// mov al,4
		/*0016*/ 0x75, 0xE8,// jne short 0
		/*0018*/ 0xB0, 0x05,// mov al,5
		/*001A*/ 0x76, 0xE4,// jbe short 0
		/*001C*/ 0xB0, 0x06,// mov al,6
		/*001E*/ 0x77, 0xE0,// ja short 0
		/*0020*/ 0xB0, 0x07,// mov al,7
		/*0022*/ 0x78, 0xDC,// js short 0
		/*0024*/ 0xB0, 0x08,// mov al,8
		/*0026*/ 0x79, 0xD8,// jns short 0
		/*0028*/ 0xB0, 0x09,// mov al,9
		/*002A*/ 0x7A, 0xD4,// jp short 0
		/*002C*/ 0xB0, 0x0A,// mov al,0Ah
		/*002E*/ 0x7B, 0xD0,// jnp short 0
		/*0030*/ 0xB0, 0x0B,// mov al,0Bh
		/*0032*/ 0x7C, 0xCC,// jl short 0
		/*0034*/ 0xB0, 0x0C,// mov al,0Ch
		/*0036*/ 0x7D, 0xC8,// jge short 0
		/*0038*/ 0xB0, 0x0D,// mov al,0Dh
		/*003A*/ 0x7E, 0xC4,// jle short 0
		/*003C*/ 0xB0, 0x0E,// mov al,0Eh
		/*003E*/ 0x7F, 0xC0,// jg short 0
		/*0040*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0xB0, 0x10,// mov al,10h
		/*0002*/ 0x70, 0xFC,// jo short 0
		/*0004*/ 0xB0, 0x00,// mov al,0
		/*0006*/ 0x71, 0xF8,// jno short 0
		/*0008*/ 0xB0, 0x01,// mov al,1
		/*000A*/ 0x72, 0xF4,// jb short 0
		/*000C*/ 0xB0, 0x02,// mov al,2
		/*000E*/ 0x73, 0xF0,// jae short 0
		/*0010*/ 0xB0, 0x03,// mov al,3
		/*0012*/ 0x74, 0xEC,// je short 0
		/*0014*/ 0xB0, 0x04,// mov al,4
		/*0016*/ 0x75, 0xE8,// jne short 0
		/*0018*/ 0xB0, 0x05,// mov al,5
		/*001A*/ 0x76, 0xE4,// jbe short 0
		/*001C*/ 0xB0, 0x06,// mov al,6
		/*001E*/ 0x77, 0xE0,// ja short 0
		/*0020*/ 0xB0, 0x07,// mov al,7
		/*0022*/ 0x78, 0xDC,// js short 0
		/*0024*/ 0xB0, 0x08,// mov al,8
		/*0026*/ 0x79, 0xD8,// jns short 0
		/*0028*/ 0xB0, 0x09,// mov al,9
		/*002A*/ 0x7A, 0xD4,// jp short 0
		/*002C*/ 0xB0, 0x0A,// mov al,0Ah
		/*002E*/ 0x7B, 0xD0,// jnp short 0
		/*0030*/ 0xB0, 0x0B,// mov al,0Bh
		/*0032*/ 0x7C, 0xCC,// jl short 0
		/*0034*/ 0xB0, 0x0C,// mov al,0Ch
		/*0036*/ 0x7D, 0xC8,// jge short 0
		/*0038*/ 0xB0, 0x0D,// mov al,0Dh
		/*003A*/ 0x7E, 0xC4,// jle short 0
		/*003C*/ 0xB0, 0x0E,// mov al,0Eh
		/*003E*/ 0x7F, 0xC0,// jg short 0
		/*0040*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0002,
		0x0004,
		0x0006,
		0x0008,
		0x000A,
		0x000C,
		0x000E,
		0x0010,
		0x0012,
		0x0014,
		0x0016,
		0x0018,
		0x001A,
		0x001C,
		0x001E,
		0x0020,
		0x0022,
		0x0024,
		0x0026,
		0x0028,
		0x002A,
		0x002C,
		0x002E,
		0x0030,
		0x0032,
		0x0034,
		0x0036,
		0x0038,
		0x003A,
		0x003C,
		0x003E,
		0x0040,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_short_bwd_os() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0xB0, 0x10,// mov al,10h
		/*0002*/ 0x66, 0x70, 0xFB,// jo short 0
		/*0005*/ 0xB0, 0x00,// mov al,0
		/*0007*/ 0x66, 0x71, 0xF6,// jno short 0
		/*000A*/ 0xB0, 0x01,// mov al,1
		/*000C*/ 0x66, 0x72, 0xF1,// jb short 0
		/*000F*/ 0xB0, 0x02,// mov al,2
		/*0011*/ 0x66, 0x73, 0xEC,// jae short 0
		/*0014*/ 0xB0, 0x03,// mov al,3
		/*0016*/ 0x66, 0x74, 0xE7,// je short 0
		/*0019*/ 0xB0, 0x04,// mov al,4
		/*001B*/ 0x66, 0x75, 0xE2,// jne short 0
		/*001E*/ 0xB0, 0x05,// mov al,5
		/*0020*/ 0x66, 0x76, 0xDD,// jbe short 0
		/*0023*/ 0xB0, 0x06,// mov al,6
		/*0025*/ 0x66, 0x77, 0xD8,// ja short 0
		/*0028*/ 0xB0, 0x07,// mov al,7
		/*002A*/ 0x66, 0x78, 0xD3,// js short 0
		/*002D*/ 0xB0, 0x08,// mov al,8
		/*002F*/ 0x66, 0x79, 0xCE,// jns short 0
		/*0032*/ 0xB0, 0x09,// mov al,9
		/*0034*/ 0x66, 0x7A, 0xC9,// jp short 0
		/*0037*/ 0xB0, 0x0A,// mov al,0Ah
		/*0039*/ 0x66, 0x7B, 0xC4,// jnp short 0
		/*003C*/ 0xB0, 0x0B,// mov al,0Bh
		/*003E*/ 0x66, 0x7C, 0xBF,// jl short 0
		/*0041*/ 0xB0, 0x0C,// mov al,0Ch
		/*0043*/ 0x66, 0x7D, 0xBA,// jge short 0
		/*0046*/ 0xB0, 0x0D,// mov al,0Dh
		/*0048*/ 0x66, 0x7E, 0xB5,// jle short 0
		/*004B*/ 0xB0, 0x0E,// mov al,0Eh
		/*004D*/ 0x66, 0x7F, 0xB0,// jg short 0
		/*0050*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0xB0, 0x10,// mov al,10h
		/*0002*/ 0x66, 0x70, 0xFB,// jo short 0
		/*0005*/ 0xB0, 0x00,// mov al,0
		/*0007*/ 0x66, 0x71, 0xF6,// jno short 0
		/*000A*/ 0xB0, 0x01,// mov al,1
		/*000C*/ 0x66, 0x72, 0xF1,// jb short 0
		/*000F*/ 0xB0, 0x02,// mov al,2
		/*0011*/ 0x66, 0x73, 0xEC,// jae short 0
		/*0014*/ 0xB0, 0x03,// mov al,3
		/*0016*/ 0x66, 0x74, 0xE7,// je short 0
		/*0019*/ 0xB0, 0x04,// mov al,4
		/*001B*/ 0x66, 0x75, 0xE2,// jne short 0
		/*001E*/ 0xB0, 0x05,// mov al,5
		/*0020*/ 0x66, 0x76, 0xDD,// jbe short 0
		/*0023*/ 0xB0, 0x06,// mov al,6
		/*0025*/ 0x66, 0x77, 0xD8,// ja short 0
		/*0028*/ 0xB0, 0x07,// mov al,7
		/*002A*/ 0x66, 0x78, 0xD3,// js short 0
		/*002D*/ 0xB0, 0x08,// mov al,8
		/*002F*/ 0x66, 0x79, 0xCE,// jns short 0
		/*0032*/ 0xB0, 0x09,// mov al,9
		/*0034*/ 0x66, 0x7A, 0xC9,// jp short 0
		/*0037*/ 0xB0, 0x0A,// mov al,0Ah
		/*0039*/ 0x66, 0x7B, 0xC4,// jnp short 0
		/*003C*/ 0xB0, 0x0B,// mov al,0Bh
		/*003E*/ 0x66, 0x7C, 0xBF,// jl short 0
		/*0041*/ 0xB0, 0x0C,// mov al,0Ch
		/*0043*/ 0x66, 0x7D, 0xBA,// jge short 0
		/*0046*/ 0xB0, 0x0D,// mov al,0Dh
		/*0048*/ 0x66, 0x7E, 0xB5,// jle short 0
		/*004B*/ 0xB0, 0x0E,// mov al,0Eh
		/*004D*/ 0x66, 0x7F, 0xB0,// jg short 0
		/*0050*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0002,
		0x0005,
		0x0007,
		0x000A,
		0x000C,
		0x000F,
		0x0011,
		0x0014,
		0x0016,
		0x0019,
		0x001B,
		0x001E,
		0x0020,
		0x0023,
		0x0025,
		0x0028,
		0x002A,
		0x002D,
		0x002F,
		0x0032,
		0x0034,
		0x0037,
		0x0039,
		0x003C,
		0x003E,
		0x0041,
		0x0043,
		0x0046,
		0x0048,
		0x004B,
		0x004D,
		0x0050,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_short_other_short() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x70, 0x3E,// jo short 0040h
		/*0002*/ 0xB0, 0x00,// mov al,0
		/*0004*/ 0x71, 0x3A,// jno short 0040h
		/*0006*/ 0xB0, 0x01,// mov al,1
		/*0008*/ 0x72, 0x36,// jb short 0040h
		/*000A*/ 0xB0, 0x02,// mov al,2
		/*000C*/ 0x73, 0x32,// jae short 0040h
		/*000E*/ 0xB0, 0x03,// mov al,3
		/*0010*/ 0x74, 0x2E,// je short 0040h
		/*0012*/ 0xB0, 0x04,// mov al,4
		/*0014*/ 0x75, 0x2A,// jne short 0040h
		/*0016*/ 0xB0, 0x05,// mov al,5
		/*0018*/ 0x76, 0x26,// jbe short 0040h
		/*001A*/ 0xB0, 0x06,// mov al,6
		/*001C*/ 0x77, 0x22,// ja short 0040h
		/*001E*/ 0xB0, 0x07,// mov al,7
		/*0020*/ 0x78, 0x1E,// js short 0040h
		/*0022*/ 0xB0, 0x08,// mov al,8
		/*0024*/ 0x79, 0x1A,// jns short 0040h
		/*0026*/ 0xB0, 0x09,// mov al,9
		/*0028*/ 0x7A, 0x16,// jp short 0040h
		/*002A*/ 0xB0, 0x0A,// mov al,0Ah
		/*002C*/ 0x7B, 0x12,// jnp short 0040h
		/*002E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0030*/ 0x7C, 0x0E,// jl short 0040h
		/*0032*/ 0xB0, 0x0C,// mov al,0Ch
		/*0034*/ 0x7D, 0x0A,// jge short 0040h
		/*0036*/ 0xB0, 0x0D,// mov al,0Dh
		/*0038*/ 0x7E, 0x06,// jle short 0040h
		/*003A*/ 0xB0, 0x0E,// mov al,0Eh
		/*003C*/ 0x7F, 0x02,// jg short 0040h
		/*003E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x70, 0x3F,// jo short 0041h
		/*0002*/ 0xB0, 0x00,// mov al,0
		/*0004*/ 0x71, 0x3B,// jno short 0041h
		/*0006*/ 0xB0, 0x01,// mov al,1
		/*0008*/ 0x72, 0x37,// jb short 0041h
		/*000A*/ 0xB0, 0x02,// mov al,2
		/*000C*/ 0x73, 0x33,// jae short 0041h
		/*000E*/ 0xB0, 0x03,// mov al,3
		/*0010*/ 0x74, 0x2F,// je short 0041h
		/*0012*/ 0xB0, 0x04,// mov al,4
		/*0014*/ 0x75, 0x2B,// jne short 0041h
		/*0016*/ 0xB0, 0x05,// mov al,5
		/*0018*/ 0x76, 0x27,// jbe short 0041h
		/*001A*/ 0xB0, 0x06,// mov al,6
		/*001C*/ 0x77, 0x23,// ja short 0041h
		/*001E*/ 0xB0, 0x07,// mov al,7
		/*0020*/ 0x78, 0x1F,// js short 0041h
		/*0022*/ 0xB0, 0x08,// mov al,8
		/*0024*/ 0x79, 0x1B,// jns short 0041h
		/*0026*/ 0xB0, 0x09,// mov al,9
		/*0028*/ 0x7A, 0x17,// jp short 0041h
		/*002A*/ 0xB0, 0x0A,// mov al,0Ah
		/*002C*/ 0x7B, 0x13,// jnp short 0041h
		/*002E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0030*/ 0x7C, 0x0F,// jl short 0041h
		/*0032*/ 0xB0, 0x0C,// mov al,0Ch
		/*0034*/ 0x7D, 0x0B,// jge short 0041h
		/*0036*/ 0xB0, 0x0D,// mov al,0Dh
		/*0038*/ 0x7E, 0x07,// jle short 0041h
		/*003A*/ 0xB0, 0x0E,// mov al,0Eh
		/*003C*/ 0x7F, 0x03,// jg short 0041h
		/*003E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0002,
		0x0004,
		0x0006,
		0x0008,
		0x000A,
		0x000C,
		0x000E,
		0x0010,
		0x0012,
		0x0014,
		0x0016,
		0x0018,
		0x001A,
		0x001C,
		0x001E,
		0x0020,
		0x0022,
		0x0024,
		0x0026,
		0x0028,
		0x002A,
		0x002C,
		0x002E,
		0x0030,
		0x0032,
		0x0034,
		0x0036,
		0x0038,
		0x003A,
		0x003C,
		0x003E,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		ORIG_RIP - 1,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_short_other_near() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x70, 0x3E,// jo short 0040h
		/*0002*/ 0xB0, 0x00,// mov al,0
		/*0004*/ 0x71, 0x3A,// jno short 0040h
		/*0006*/ 0xB0, 0x01,// mov al,1
		/*0008*/ 0x72, 0x36,// jb short 0040h
		/*000A*/ 0xB0, 0x02,// mov al,2
		/*000C*/ 0x73, 0x32,// jae short 0040h
		/*000E*/ 0xB0, 0x03,// mov al,3
		/*0010*/ 0x74, 0x2E,// je short 0040h
		/*0012*/ 0xB0, 0x04,// mov al,4
		/*0014*/ 0x75, 0x2A,// jne short 0040h
		/*0016*/ 0xB0, 0x05,// mov al,5
		/*0018*/ 0x76, 0x26,// jbe short 0040h
		/*001A*/ 0xB0, 0x06,// mov al,6
		/*001C*/ 0x77, 0x22,// ja short 0040h
		/*001E*/ 0xB0, 0x07,// mov al,7
		/*0020*/ 0x78, 0x1E,// js short 0040h
		/*0022*/ 0xB0, 0x08,// mov al,8
		/*0024*/ 0x79, 0x1A,// jns short 0040h
		/*0026*/ 0xB0, 0x09,// mov al,9
		/*0028*/ 0x7A, 0x16,// jp short 0040h
		/*002A*/ 0xB0, 0x0A,// mov al,0Ah
		/*002C*/ 0x7B, 0x12,// jnp short 0040h
		/*002E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0030*/ 0x7C, 0x0E,// jl short 0040h
		/*0032*/ 0xB0, 0x0C,// mov al,0Ch
		/*0034*/ 0x7D, 0x0A,// jge short 0040h
		/*0036*/ 0xB0, 0x0D,// mov al,0Dh
		/*0038*/ 0x7E, 0x06,// jle short 0040h
		/*003A*/ 0xB0, 0x0E,// mov al,0Eh
		/*003C*/ 0x7F, 0x02,// jg short 0040h
		/*003E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x0F, 0x80, 0x3C, 0xF0,// jo near ptr 0F040h
		/*0004*/ 0xB0, 0x00,// mov al,0
		/*0006*/ 0x0F, 0x81, 0x36, 0xF0,// jno near ptr 0F040h
		/*000A*/ 0xB0, 0x01,// mov al,1
		/*000C*/ 0x0F, 0x82, 0x30, 0xF0,// jb near ptr 0F040h
		/*0010*/ 0xB0, 0x02,// mov al,2
		/*0012*/ 0x0F, 0x83, 0x2A, 0xF0,// jae near ptr 0F040h
		/*0016*/ 0xB0, 0x03,// mov al,3
		/*0018*/ 0x0F, 0x84, 0x24, 0xF0,// je near ptr 0F040h
		/*001C*/ 0xB0, 0x04,// mov al,4
		/*001E*/ 0x0F, 0x85, 0x1E, 0xF0,// jne near ptr 0F040h
		/*0022*/ 0xB0, 0x05,// mov al,5
		/*0024*/ 0x0F, 0x86, 0x18, 0xF0,// jbe near ptr 0F040h
		/*0028*/ 0xB0, 0x06,// mov al,6
		/*002A*/ 0x0F, 0x87, 0x12, 0xF0,// ja near ptr 0F040h
		/*002E*/ 0xB0, 0x07,// mov al,7
		/*0030*/ 0x0F, 0x88, 0x0C, 0xF0,// js near ptr 0F040h
		/*0034*/ 0xB0, 0x08,// mov al,8
		/*0036*/ 0x0F, 0x89, 0x06, 0xF0,// jns near ptr 0F040h
		/*003A*/ 0xB0, 0x09,// mov al,9
		/*003C*/ 0x0F, 0x8A, 0x00, 0xF0,// jp near ptr 0F040h
		/*0040*/ 0xB0, 0x0A,// mov al,0Ah
		/*0042*/ 0x0F, 0x8B, 0xFA, 0xEF,// jnp near ptr 0F040h
		/*0046*/ 0xB0, 0x0B,// mov al,0Bh
		/*0048*/ 0x0F, 0x8C, 0xF4, 0xEF,// jl near ptr 0F040h
		/*004C*/ 0xB0, 0x0C,// mov al,0Ch
		/*004E*/ 0x0F, 0x8D, 0xEE, 0xEF,// jge near ptr 0F040h
		/*0052*/ 0xB0, 0x0D,// mov al,0Dh
		/*0054*/ 0x0F, 0x8E, 0xE8, 0xEF,// jle near ptr 0F040h
		/*0058*/ 0xB0, 0x0E,// mov al,0Eh
		/*005A*/ 0x0F, 0x8F, 0xE2, 0xEF,// jg near ptr 0F040h
		/*005E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0004,
		0x0006,
		0x000A,
		0x000C,
		0x0010,
		0x0012,
		0x0016,
		0x0018,
		0x001C,
		0x001E,
		0x0022,
		0x0024,
		0x0028,
		0x002A,
		0x002E,
		0x0030,
		0x0034,
		0x0036,
		0x003A,
		0x003C,
		0x0040,
		0x0042,
		0x0046,
		0x0048,
		0x004C,
		0x004E,
		0x0052,
		0x0054,
		0x0058,
		0x005A,
		0x005E,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		ORIG_RIP + 0x1000,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_short_other_short_os() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x66, 0x70, 0x4D,// jo short 00008050h
		/*0003*/ 0xB0, 0x00,// mov al,0
		/*0005*/ 0x66, 0x71, 0x49,// jno short 00008051h
		/*0008*/ 0xB0, 0x01,// mov al,1
		/*000A*/ 0x66, 0x72, 0x45,// jb short 00008052h
		/*000D*/ 0xB0, 0x02,// mov al,2
		/*000F*/ 0x66, 0x73, 0x41,// jae short 00008053h
		/*0012*/ 0xB0, 0x03,// mov al,3
		/*0014*/ 0x66, 0x74, 0x3D,// je short 00008054h
		/*0017*/ 0xB0, 0x04,// mov al,4
		/*0019*/ 0x66, 0x75, 0x39,// jne short 00008055h
		/*001C*/ 0xB0, 0x05,// mov al,5
		/*001E*/ 0x66, 0x76, 0x35,// jbe short 00008056h
		/*0021*/ 0xB0, 0x06,// mov al,6
		/*0023*/ 0x66, 0x77, 0x31,// ja short 00008057h
		/*0026*/ 0xB0, 0x07,// mov al,7
		/*0028*/ 0x66, 0x78, 0x2D,// js short 00008058h
		/*002B*/ 0xB0, 0x08,// mov al,8
		/*002D*/ 0x66, 0x79, 0x29,// jns short 00008059h
		/*0030*/ 0xB0, 0x09,// mov al,9
		/*0032*/ 0x66, 0x7A, 0x25,// jp short 0000805Ah
		/*0035*/ 0xB0, 0x0A,// mov al,0Ah
		/*0037*/ 0x66, 0x7B, 0x21,// jnp short 0000805Bh
		/*003A*/ 0xB0, 0x0B,// mov al,0Bh
		/*003C*/ 0x66, 0x7C, 0x1D,// jl short 0000805Ch
		/*003F*/ 0xB0, 0x0C,// mov al,0Ch
		/*0041*/ 0x66, 0x7D, 0x19,// jge short 0000805Dh
		/*0044*/ 0xB0, 0x0D,// mov al,0Dh
		/*0046*/ 0x66, 0x7E, 0x15,// jle short 0000805Eh
		/*0049*/ 0xB0, 0x0E,// mov al,0Eh
		/*004B*/ 0x66, 0x7F, 0x11,// jg short 0000805Fh
		/*004E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x66, 0x70, 0x4E,// jo short 00008050h
		/*0003*/ 0xB0, 0x00,// mov al,0
		/*0005*/ 0x66, 0x71, 0x4A,// jno short 00008051h
		/*0008*/ 0xB0, 0x01,// mov al,1
		/*000A*/ 0x66, 0x72, 0x46,// jb short 00008052h
		/*000D*/ 0xB0, 0x02,// mov al,2
		/*000F*/ 0x66, 0x73, 0x42,// jae short 00008053h
		/*0012*/ 0xB0, 0x03,// mov al,3
		/*0014*/ 0x66, 0x74, 0x3E,// je short 00008054h
		/*0017*/ 0xB0, 0x04,// mov al,4
		/*0019*/ 0x66, 0x75, 0x3A,// jne short 00008055h
		/*001C*/ 0xB0, 0x05,// mov al,5
		/*001E*/ 0x66, 0x76, 0x36,// jbe short 00008056h
		/*0021*/ 0xB0, 0x06,// mov al,6
		/*0023*/ 0x66, 0x77, 0x32,// ja short 00008057h
		/*0026*/ 0xB0, 0x07,// mov al,7
		/*0028*/ 0x66, 0x78, 0x2E,// js short 00008058h
		/*002B*/ 0xB0, 0x08,// mov al,8
		/*002D*/ 0x66, 0x79, 0x2A,// jns short 00008059h
		/*0030*/ 0xB0, 0x09,// mov al,9
		/*0032*/ 0x66, 0x7A, 0x26,// jp short 0000805Ah
		/*0035*/ 0xB0, 0x0A,// mov al,0Ah
		/*0037*/ 0x66, 0x7B, 0x22,// jnp short 0000805Bh
		/*003A*/ 0xB0, 0x0B,// mov al,0Bh
		/*003C*/ 0x66, 0x7C, 0x1E,// jl short 0000805Ch
		/*003F*/ 0xB0, 0x0C,// mov al,0Ch
		/*0041*/ 0x66, 0x7D, 0x1A,// jge short 0000805Dh
		/*0044*/ 0xB0, 0x0D,// mov al,0Dh
		/*0046*/ 0x66, 0x7E, 0x16,// jle short 0000805Eh
		/*0049*/ 0xB0, 0x0E,// mov al,0Eh
		/*004B*/ 0x66, 0x7F, 0x12,// jg short 0000805Fh
		/*004E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0003,
		0x0005,
		0x0008,
		0x000A,
		0x000D,
		0x000F,
		0x0012,
		0x0014,
		0x0017,
		0x0019,
		0x001C,
		0x001E,
		0x0021,
		0x0023,
		0x0026,
		0x0028,
		0x002B,
		0x002D,
		0x0030,
		0x0032,
		0x0035,
		0x0037,
		0x003A,
		0x003C,
		0x003F,
		0x0041,
		0x0044,
		0x0046,
		0x0049,
		0x004B,
		0x004E,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		ORIG_RIP - 1,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_short_other_near_os() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x66, 0x70, 0x4D,// jo short 00008050h
		/*0003*/ 0xB0, 0x00,// mov al,0
		/*0005*/ 0x66, 0x71, 0x49,// jno short 00008051h
		/*0008*/ 0xB0, 0x01,// mov al,1
		/*000A*/ 0x66, 0x72, 0x45,// jb short 00008052h
		/*000D*/ 0xB0, 0x02,// mov al,2
		/*000F*/ 0x66, 0x73, 0x41,// jae short 00008053h
		/*0012*/ 0xB0, 0x03,// mov al,3
		/*0014*/ 0x66, 0x74, 0x3D,// je short 00008054h
		/*0017*/ 0xB0, 0x04,// mov al,4
		/*0019*/ 0x66, 0x75, 0x39,// jne short 00008055h
		/*001C*/ 0xB0, 0x05,// mov al,5
		/*001E*/ 0x66, 0x76, 0x35,// jbe short 00008056h
		/*0021*/ 0xB0, 0x06,// mov al,6
		/*0023*/ 0x66, 0x77, 0x31,// ja short 00008057h
		/*0026*/ 0xB0, 0x07,// mov al,7
		/*0028*/ 0x66, 0x78, 0x2D,// js short 00008058h
		/*002B*/ 0xB0, 0x08,// mov al,8
		/*002D*/ 0x66, 0x79, 0x29,// jns short 00008059h
		/*0030*/ 0xB0, 0x09,// mov al,9
		/*0032*/ 0x66, 0x7A, 0x25,// jp short 0000805Ah
		/*0035*/ 0xB0, 0x0A,// mov al,0Ah
		/*0037*/ 0x66, 0x7B, 0x21,// jnp short 0000805Bh
		/*003A*/ 0xB0, 0x0B,// mov al,0Bh
		/*003C*/ 0x66, 0x7C, 0x1D,// jl short 0000805Ch
		/*003F*/ 0xB0, 0x0C,// mov al,0Ch
		/*0041*/ 0x66, 0x7D, 0x19,// jge short 0000805Dh
		/*0044*/ 0xB0, 0x0D,// mov al,0Dh
		/*0046*/ 0x66, 0x7E, 0x15,// jle short 0000805Eh
		/*0049*/ 0xB0, 0x0E,// mov al,0Eh
		/*004B*/ 0x66, 0x7F, 0x11,// jg short 0000805Fh
		/*004E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x66, 0x0F, 0x80, 0x49, 0xF0, 0xFF, 0xFF,// jo near ptr 00008050h
		/*0007*/ 0xB0, 0x00,// mov al,0
		/*0009*/ 0x66, 0x0F, 0x81, 0x41, 0xF0, 0xFF, 0xFF,// jno near ptr 00008051h
		/*0010*/ 0xB0, 0x01,// mov al,1
		/*0012*/ 0x66, 0x0F, 0x82, 0x39, 0xF0, 0xFF, 0xFF,// jb near ptr 00008052h
		/*0019*/ 0xB0, 0x02,// mov al,2
		/*001B*/ 0x66, 0x0F, 0x83, 0x31, 0xF0, 0xFF, 0xFF,// jae near ptr 00008053h
		/*0022*/ 0xB0, 0x03,// mov al,3
		/*0024*/ 0x66, 0x0F, 0x84, 0x29, 0xF0, 0xFF, 0xFF,// je near ptr 00008054h
		/*002B*/ 0xB0, 0x04,// mov al,4
		/*002D*/ 0x66, 0x0F, 0x85, 0x21, 0xF0, 0xFF, 0xFF,// jne near ptr 00008055h
		/*0034*/ 0xB0, 0x05,// mov al,5
		/*0036*/ 0x66, 0x0F, 0x86, 0x19, 0xF0, 0xFF, 0xFF,// jbe near ptr 00008056h
		/*003D*/ 0xB0, 0x06,// mov al,6
		/*003F*/ 0x66, 0x0F, 0x87, 0x11, 0xF0, 0xFF, 0xFF,// ja near ptr 00008057h
		/*0046*/ 0xB0, 0x07,// mov al,7
		/*0048*/ 0x66, 0x0F, 0x88, 0x09, 0xF0, 0xFF, 0xFF,// js near ptr 00008058h
		/*004F*/ 0xB0, 0x08,// mov al,8
		/*0051*/ 0x66, 0x0F, 0x89, 0x01, 0xF0, 0xFF, 0xFF,// jns near ptr 00008059h
		/*0058*/ 0xB0, 0x09,// mov al,9
		/*005A*/ 0x66, 0x0F, 0x8A, 0xF9, 0xEF, 0xFF, 0xFF,// jp near ptr 0000805Ah
		/*0061*/ 0xB0, 0x0A,// mov al,0Ah
		/*0063*/ 0x66, 0x0F, 0x8B, 0xF1, 0xEF, 0xFF, 0xFF,// jnp near ptr 0000805Bh
		/*006A*/ 0xB0, 0x0B,// mov al,0Bh
		/*006C*/ 0x66, 0x0F, 0x8C, 0xE9, 0xEF, 0xFF, 0xFF,// jl near ptr 0000805Ch
		/*0073*/ 0xB0, 0x0C,// mov al,0Ch
		/*0075*/ 0x66, 0x0F, 0x8D, 0xE1, 0xEF, 0xFF, 0xFF,// jge near ptr 0000805Dh
		/*007C*/ 0xB0, 0x0D,// mov al,0Dh
		/*007E*/ 0x66, 0x0F, 0x8E, 0xD9, 0xEF, 0xFF, 0xFF,// jle near ptr 0000805Eh
		/*0085*/ 0xB0, 0x0E,// mov al,0Eh
		/*0087*/ 0x66, 0x0F, 0x8F, 0xD1, 0xEF, 0xFF, 0xFF,// jg near ptr 0000805Fh
		/*008E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0007,
		0x0009,
		0x0010,
		0x0012,
		0x0019,
		0x001B,
		0x0022,
		0x0024,
		0x002B,
		0x002D,
		0x0034,
		0x0036,
		0x003D,
		0x003F,
		0x0046,
		0x0048,
		0x004F,
		0x0051,
		0x0058,
		0x005A,
		0x0061,
		0x0063,
		0x006A,
		0x006C,
		0x0073,
		0x0075,
		0x007C,
		0x007E,
		0x0085,
		0x0087,
		0x008E,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		ORIG_RIP + 0x1000,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_near_fwd_short() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x0F, 0x80, 0x5C, 0x00,// jo near ptr 8060h
		/*0004*/ 0xB0, 0x00,// mov al,0
		/*0006*/ 0x0F, 0x81, 0x56, 0x00,// jno near ptr 8060h
		/*000A*/ 0xB0, 0x01,// mov al,1
		/*000C*/ 0x0F, 0x82, 0x50, 0x00,// jb near ptr 8060h
		/*0010*/ 0xB0, 0x02,// mov al,2
		/*0012*/ 0x0F, 0x83, 0x4A, 0x00,// jae near ptr 8060h
		/*0016*/ 0xB0, 0x03,// mov al,3
		/*0018*/ 0x0F, 0x84, 0x44, 0x00,// je near ptr 8060h
		/*001C*/ 0xB0, 0x04,// mov al,4
		/*001E*/ 0x0F, 0x85, 0x3E, 0x00,// jne near ptr 8060h
		/*0022*/ 0xB0, 0x05,// mov al,5
		/*0024*/ 0x0F, 0x86, 0x38, 0x00,// jbe near ptr 8060h
		/*0028*/ 0xB0, 0x06,// mov al,6
		/*002A*/ 0x0F, 0x87, 0x32, 0x00,// ja near ptr 8060h
		/*002E*/ 0xB0, 0x07,// mov al,7
		/*0030*/ 0x0F, 0x88, 0x2C, 0x00,// js near ptr 8060h
		/*0034*/ 0xB0, 0x08,// mov al,8
		/*0036*/ 0x0F, 0x89, 0x26, 0x00,// jns near ptr 8060h
		/*003A*/ 0xB0, 0x09,// mov al,9
		/*003C*/ 0x0F, 0x8A, 0x20, 0x00,// jp near ptr 8060h
		/*0040*/ 0xB0, 0x0A,// mov al,0Ah
		/*0042*/ 0x0F, 0x8B, 0x1A, 0x00,// jnp near ptr 8060h
		/*0046*/ 0xB0, 0x0B,// mov al,0Bh
		/*0048*/ 0x0F, 0x8C, 0x14, 0x00,// jl near ptr 8060h
		/*004C*/ 0xB0, 0x0C,// mov al,0Ch
		/*004E*/ 0x0F, 0x8D, 0x0E, 0x00,// jge near ptr 8060h
		/*0052*/ 0xB0, 0x0D,// mov al,0Dh
		/*0054*/ 0x0F, 0x8E, 0x08, 0x00,// jle near ptr 8060h
		/*0058*/ 0xB0, 0x0E,// mov al,0Eh
		/*005A*/ 0x0F, 0x8F, 0x02, 0x00,// jg near ptr 8060h
		/*005E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0060*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x70, 0x3E,// jo short 0F040h
		/*0002*/ 0xB0, 0x00,// mov al,0
		/*0004*/ 0x71, 0x3A,// jno short 0F040h
		/*0006*/ 0xB0, 0x01,// mov al,1
		/*0008*/ 0x72, 0x36,// jb short 0F040h
		/*000A*/ 0xB0, 0x02,// mov al,2
		/*000C*/ 0x73, 0x32,// jae short 0F040h
		/*000E*/ 0xB0, 0x03,// mov al,3
		/*0010*/ 0x74, 0x2E,// je short 0F040h
		/*0012*/ 0xB0, 0x04,// mov al,4
		/*0014*/ 0x75, 0x2A,// jne short 0F040h
		/*0016*/ 0xB0, 0x05,// mov al,5
		/*0018*/ 0x76, 0x26,// jbe short 0F040h
		/*001A*/ 0xB0, 0x06,// mov al,6
		/*001C*/ 0x77, 0x22,// ja short 0F040h
		/*001E*/ 0xB0, 0x07,// mov al,7
		/*0020*/ 0x78, 0x1E,// js short 0F040h
		/*0022*/ 0xB0, 0x08,// mov al,8
		/*0024*/ 0x79, 0x1A,// jns short 0F040h
		/*0026*/ 0xB0, 0x09,// mov al,9
		/*0028*/ 0x7A, 0x16,// jp short 0F040h
		/*002A*/ 0xB0, 0x0A,// mov al,0Ah
		/*002C*/ 0x7B, 0x12,// jnp short 0F040h
		/*002E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0030*/ 0x7C, 0x0E,// jl short 0F040h
		/*0032*/ 0xB0, 0x0C,// mov al,0Ch
		/*0034*/ 0x7D, 0x0A,// jge short 0F040h
		/*0036*/ 0xB0, 0x0D,// mov al,0Dh
		/*0038*/ 0x7E, 0x06,// jle short 0F040h
		/*003A*/ 0xB0, 0x0E,// mov al,0Eh
		/*003C*/ 0x7F, 0x02,// jg short 0F040h
		/*003E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0040*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0002,
		0x0004,
		0x0006,
		0x0008,
		0x000A,
		0x000C,
		0x000E,
		0x0010,
		0x0012,
		0x0014,
		0x0016,
		0x0018,
		0x001A,
		0x001C,
		0x001E,
		0x0020,
		0x0022,
		0x0024,
		0x0026,
		0x0028,
		0x002A,
		0x002C,
		0x002E,
		0x0030,
		0x0032,
		0x0034,
		0x0036,
		0x0038,
		0x003A,
		0x003C,
		0x003E,
		0x0040,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_near_fwd_near() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x0F, 0x80, 0x02, 0x01,// jo near ptr 8106h
		/*0004*/ 0xB0, 0x00,// mov al,0
		/*0006*/ 0x0F, 0x81, 0xFC, 0x00,// jno near ptr 8106h
		/*000A*/ 0xB0, 0x01,// mov al,1
		/*000C*/ 0x0F, 0x82, 0xF6, 0x00,// jb near ptr 8106h
		/*0010*/ 0xB0, 0x02,// mov al,2
		/*0012*/ 0x0F, 0x83, 0xF0, 0x00,// jae near ptr 8106h
		/*0016*/ 0xB0, 0x03,// mov al,3
		/*0018*/ 0x0F, 0x84, 0xEA, 0x00,// je near ptr 8106h
		/*001C*/ 0xB0, 0x04,// mov al,4
		/*001E*/ 0x0F, 0x85, 0xE4, 0x00,// jne near ptr 8106h
		/*0022*/ 0xB0, 0x05,// mov al,5
		/*0024*/ 0x0F, 0x86, 0xDE, 0x00,// jbe near ptr 8106h
		/*0028*/ 0xB0, 0x06,// mov al,6
		/*002A*/ 0x0F, 0x87, 0xD8, 0x00,// ja near ptr 8106h
		/*002E*/ 0xB0, 0x07,// mov al,7
		/*0030*/ 0x0F, 0x88, 0xD2, 0x00,// js near ptr 8106h
		/*0034*/ 0xB0, 0x08,// mov al,8
		/*0036*/ 0x0F, 0x89, 0xCC, 0x00,// jns near ptr 8106h
		/*003A*/ 0xB0, 0x09,// mov al,9
		/*003C*/ 0x0F, 0x8A, 0xC6, 0x00,// jp near ptr 8106h
		/*0040*/ 0xB0, 0x0A,// mov al,0Ah
		/*0042*/ 0x0F, 0x8B, 0xC0, 0x00,// jnp near ptr 8106h
		/*0046*/ 0xB0, 0x0B,// mov al,0Bh
		/*0048*/ 0x0F, 0x8C, 0xBA, 0x00,// jl near ptr 8106h
		/*004C*/ 0xB0, 0x0C,// mov al,0Ch
		/*004E*/ 0x0F, 0x8D, 0xB4, 0x00,// jge near ptr 8106h
		/*0052*/ 0xB0, 0x0D,// mov al,0Dh
		/*0054*/ 0x0F, 0x8E, 0xAE, 0x00,// jle near ptr 8106h
		/*0058*/ 0xB0, 0x0E,// mov al,0Eh
		/*005A*/ 0x0F, 0x8F, 0xA8, 0x00,// jg near ptr 8106h
		/*005E*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*006C*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*007A*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0088*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0096*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00A4*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00B2*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00C0*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00CE*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00DC*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00EA*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00F8*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0106*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x0F, 0x80, 0x02, 0x01,// jo near ptr 0F106h
		/*0004*/ 0xB0, 0x00,// mov al,0
		/*0006*/ 0x0F, 0x81, 0xFC, 0x00,// jno near ptr 0F106h
		/*000A*/ 0xB0, 0x01,// mov al,1
		/*000C*/ 0x0F, 0x82, 0xF6, 0x00,// jb near ptr 0F106h
		/*0010*/ 0xB0, 0x02,// mov al,2
		/*0012*/ 0x0F, 0x83, 0xF0, 0x00,// jae near ptr 0F106h
		/*0016*/ 0xB0, 0x03,// mov al,3
		/*0018*/ 0x0F, 0x84, 0xEA, 0x00,// je near ptr 0F106h
		/*001C*/ 0xB0, 0x04,// mov al,4
		/*001E*/ 0x0F, 0x85, 0xE4, 0x00,// jne near ptr 0F106h
		/*0022*/ 0xB0, 0x05,// mov al,5
		/*0024*/ 0x0F, 0x86, 0xDE, 0x00,// jbe near ptr 0F106h
		/*0028*/ 0xB0, 0x06,// mov al,6
		/*002A*/ 0x0F, 0x87, 0xD8, 0x00,// ja near ptr 0F106h
		/*002E*/ 0xB0, 0x07,// mov al,7
		/*0030*/ 0x0F, 0x88, 0xD2, 0x00,// js near ptr 0F106h
		/*0034*/ 0xB0, 0x08,// mov al,8
		/*0036*/ 0x0F, 0x89, 0xCC, 0x00,// jns near ptr 0F106h
		/*003A*/ 0xB0, 0x09,// mov al,9
		/*003C*/ 0x0F, 0x8A, 0xC6, 0x00,// jp near ptr 0F106h
		/*0040*/ 0xB0, 0x0A,// mov al,0Ah
		/*0042*/ 0x0F, 0x8B, 0xC0, 0x00,// jnp near ptr 0F106h
		/*0046*/ 0xB0, 0x0B,// mov al,0Bh
		/*0048*/ 0x0F, 0x8C, 0xBA, 0x00,// jl near ptr 0F106h
		/*004C*/ 0xB0, 0x0C,// mov al,0Ch
		/*004E*/ 0x0F, 0x8D, 0xB4, 0x00,// jge near ptr 0F106h
		/*0052*/ 0xB0, 0x0D,// mov al,0Dh
		/*0054*/ 0x0F, 0x8E, 0xAE, 0x00,// jle near ptr 0F106h
		/*0058*/ 0xB0, 0x0E,// mov al,0Eh
		/*005A*/ 0x0F, 0x8F, 0xA8, 0x00,// jg near ptr 0F106h
		/*005E*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*006C*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*007A*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0088*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0096*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00A4*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00B2*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00C0*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00CE*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00DC*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00EA*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00F8*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0106*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0004,
		0x0006,
		0x000A,
		0x000C,
		0x0010,
		0x0012,
		0x0016,
		0x0018,
		0x001C,
		0x001E,
		0x0022,
		0x0024,
		0x0028,
		0x002A,
		0x002E,
		0x0030,
		0x0034,
		0x0036,
		0x003A,
		0x003C,
		0x0040,
		0x0042,
		0x0046,
		0x0048,
		0x004C,
		0x004E,
		0x0052,
		0x0054,
		0x0058,
		0x005A,
		0x005E,
		0x006C,
		0x007A,
		0x0088,
		0x0096,
		0x00A4,
		0x00B2,
		0x00C0,
		0x00CE,
		0x00DC,
		0x00EA,
		0x00F8,
		0x0106,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_near_bwd_short() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x0F, 0x80, 0xFC, 0xFF,// jo near ptr 8000h
		/*0004*/ 0xB0, 0x00,// mov al,0
		/*0006*/ 0x0F, 0x81, 0xF6, 0xFF,// jno near ptr 8000h
		/*000A*/ 0xB0, 0x01,// mov al,1
		/*000C*/ 0x0F, 0x82, 0xF0, 0xFF,// jb near ptr 8000h
		/*0010*/ 0xB0, 0x02,// mov al,2
		/*0012*/ 0x0F, 0x83, 0xEA, 0xFF,// jae near ptr 8000h
		/*0016*/ 0xB0, 0x03,// mov al,3
		/*0018*/ 0x0F, 0x84, 0xE4, 0xFF,// je near ptr 8000h
		/*001C*/ 0xB0, 0x04,// mov al,4
		/*001E*/ 0x0F, 0x85, 0xDE, 0xFF,// jne near ptr 8000h
		/*0022*/ 0xB0, 0x05,// mov al,5
		/*0024*/ 0x0F, 0x86, 0xD8, 0xFF,// jbe near ptr 8000h
		/*0028*/ 0xB0, 0x06,// mov al,6
		/*002A*/ 0x0F, 0x87, 0xD2, 0xFF,// ja near ptr 8000h
		/*002E*/ 0xB0, 0x07,// mov al,7
		/*0030*/ 0x0F, 0x88, 0xCC, 0xFF,// js near ptr 8000h
		/*0034*/ 0xB0, 0x08,// mov al,8
		/*0036*/ 0x0F, 0x89, 0xC6, 0xFF,// jns near ptr 8000h
		/*003A*/ 0xB0, 0x09,// mov al,9
		/*003C*/ 0x0F, 0x8A, 0xC0, 0xFF,// jp near ptr 8000h
		/*0040*/ 0xB0, 0x0A,// mov al,0Ah
		/*0042*/ 0x0F, 0x8B, 0xBA, 0xFF,// jnp near ptr 8000h
		/*0046*/ 0xB0, 0x0B,// mov al,0Bh
		/*0048*/ 0x0F, 0x8C, 0xB4, 0xFF,// jl near ptr 8000h
		/*004C*/ 0xB0, 0x0C,// mov al,0Ch
		/*004E*/ 0x0F, 0x8D, 0xAE, 0xFF,// jge near ptr 8000h
		/*0052*/ 0xB0, 0x0D,// mov al,0Dh
		/*0054*/ 0x0F, 0x8E, 0xA8, 0xFF,// jle near ptr 8000h
		/*0058*/ 0xB0, 0x0E,// mov al,0Eh
		/*005A*/ 0x0F, 0x8F, 0xA2, 0xFF,// jg near ptr 8000h
		/*005E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x70, 0xFE,// jo short 0F000h
		/*0002*/ 0xB0, 0x00,// mov al,0
		/*0004*/ 0x71, 0xFA,// jno short 0F000h
		/*0006*/ 0xB0, 0x01,// mov al,1
		/*0008*/ 0x72, 0xF6,// jb short 0F000h
		/*000A*/ 0xB0, 0x02,// mov al,2
		/*000C*/ 0x73, 0xF2,// jae short 0F000h
		/*000E*/ 0xB0, 0x03,// mov al,3
		/*0010*/ 0x74, 0xEE,// je short 0F000h
		/*0012*/ 0xB0, 0x04,// mov al,4
		/*0014*/ 0x75, 0xEA,// jne short 0F000h
		/*0016*/ 0xB0, 0x05,// mov al,5
		/*0018*/ 0x76, 0xE6,// jbe short 0F000h
		/*001A*/ 0xB0, 0x06,// mov al,6
		/*001C*/ 0x77, 0xE2,// ja short 0F000h
		/*001E*/ 0xB0, 0x07,// mov al,7
		/*0020*/ 0x78, 0xDE,// js short 0F000h
		/*0022*/ 0xB0, 0x08,// mov al,8
		/*0024*/ 0x79, 0xDA,// jns short 0F000h
		/*0026*/ 0xB0, 0x09,// mov al,9
		/*0028*/ 0x7A, 0xD6,// jp short 0F000h
		/*002A*/ 0xB0, 0x0A,// mov al,0Ah
		/*002C*/ 0x7B, 0xD2,// jnp short 0F000h
		/*002E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0030*/ 0x7C, 0xCE,// jl short 0F000h
		/*0032*/ 0xB0, 0x0C,// mov al,0Ch
		/*0034*/ 0x7D, 0xCA,// jge short 0F000h
		/*0036*/ 0xB0, 0x0D,// mov al,0Dh
		/*0038*/ 0x7E, 0xC6,// jle short 0F000h
		/*003A*/ 0xB0, 0x0E,// mov al,0Eh
		/*003C*/ 0x7F, 0xC2,// jg short 0F000h
		/*003E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0002,
		0x0004,
		0x0006,
		0x0008,
		0x000A,
		0x000C,
		0x000E,
		0x0010,
		0x0012,
		0x0014,
		0x0016,
		0x0018,
		0x001A,
		0x001C,
		0x001E,
		0x0020,
		0x0022,
		0x0024,
		0x0026,
		0x0028,
		0x002A,
		0x002C,
		0x002E,
		0x0030,
		0x0032,
		0x0034,
		0x0036,
		0x0038,
		0x003A,
		0x003C,
		0x003E,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_near_bwd_near() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*000E*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*001C*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*002A*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0038*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0046*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0054*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0062*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0070*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*007E*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*008C*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*009A*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00A8*/ 0x0F, 0x80, 0x54, 0xFF,// jo near ptr 8000h
		/*00AC*/ 0xB0, 0x00,// mov al,0
		/*00AE*/ 0x0F, 0x81, 0x4E, 0xFF,// jno near ptr 8000h
		/*00B2*/ 0xB0, 0x01,// mov al,1
		/*00B4*/ 0x0F, 0x82, 0x48, 0xFF,// jb near ptr 8000h
		/*00B8*/ 0xB0, 0x02,// mov al,2
		/*00BA*/ 0x0F, 0x83, 0x42, 0xFF,// jae near ptr 8000h
		/*00BE*/ 0xB0, 0x03,// mov al,3
		/*00C0*/ 0x0F, 0x84, 0x3C, 0xFF,// je near ptr 8000h
		/*00C4*/ 0xB0, 0x04,// mov al,4
		/*00C6*/ 0x0F, 0x85, 0x36, 0xFF,// jne near ptr 8000h
		/*00CA*/ 0xB0, 0x05,// mov al,5
		/*00CC*/ 0x0F, 0x86, 0x30, 0xFF,// jbe near ptr 8000h
		/*00D0*/ 0xB0, 0x06,// mov al,6
		/*00D2*/ 0x0F, 0x87, 0x2A, 0xFF,// ja near ptr 8000h
		/*00D6*/ 0xB0, 0x07,// mov al,7
		/*00D8*/ 0x0F, 0x88, 0x24, 0xFF,// js near ptr 8000h
		/*00DC*/ 0xB0, 0x08,// mov al,8
		/*00DE*/ 0x0F, 0x89, 0x1E, 0xFF,// jns near ptr 8000h
		/*00E2*/ 0xB0, 0x09,// mov al,9
		/*00E4*/ 0x0F, 0x8A, 0x18, 0xFF,// jp near ptr 8000h
		/*00E8*/ 0xB0, 0x0A,// mov al,0Ah
		/*00EA*/ 0x0F, 0x8B, 0x12, 0xFF,// jnp near ptr 8000h
		/*00EE*/ 0xB0, 0x0B,// mov al,0Bh
		/*00F0*/ 0x0F, 0x8C, 0x0C, 0xFF,// jl near ptr 8000h
		/*00F4*/ 0xB0, 0x0C,// mov al,0Ch
		/*00F6*/ 0x0F, 0x8D, 0x06, 0xFF,// jge near ptr 8000h
		/*00FA*/ 0xB0, 0x0D,// mov al,0Dh
		/*00FC*/ 0x0F, 0x8E, 0x00, 0xFF,// jle near ptr 8000h
		/*0100*/ 0xB0, 0x0E,// mov al,0Eh
		/*0102*/ 0x0F, 0x8F, 0xFA, 0xFE,// jg near ptr 8000h
		/*0106*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*000E*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*001C*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*002A*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0038*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0046*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0054*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0062*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0070*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*007E*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*008C*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*009A*/ 0x64, 0x66, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00A8*/ 0x0F, 0x80, 0x54, 0xFF,// jo near ptr 0F000h
		/*00AC*/ 0xB0, 0x00,// mov al,0
		/*00AE*/ 0x0F, 0x81, 0x4E, 0xFF,// jno near ptr 0F000h
		/*00B2*/ 0xB0, 0x01,// mov al,1
		/*00B4*/ 0x0F, 0x82, 0x48, 0xFF,// jb near ptr 0F000h
		/*00B8*/ 0xB0, 0x02,// mov al,2
		/*00BA*/ 0x0F, 0x83, 0x42, 0xFF,// jae near ptr 0F000h
		/*00BE*/ 0xB0, 0x03,// mov al,3
		/*00C0*/ 0x0F, 0x84, 0x3C, 0xFF,// je near ptr 0F000h
		/*00C4*/ 0xB0, 0x04,// mov al,4
		/*00C6*/ 0x0F, 0x85, 0x36, 0xFF,// jne near ptr 0F000h
		/*00CA*/ 0xB0, 0x05,// mov al,5
		/*00CC*/ 0x0F, 0x86, 0x30, 0xFF,// jbe near ptr 0F000h
		/*00D0*/ 0xB0, 0x06,// mov al,6
		/*00D2*/ 0x0F, 0x87, 0x2A, 0xFF,// ja near ptr 0F000h
		/*00D6*/ 0xB0, 0x07,// mov al,7
		/*00D8*/ 0x0F, 0x88, 0x24, 0xFF,// js near ptr 0F000h
		/*00DC*/ 0xB0, 0x08,// mov al,8
		/*00DE*/ 0x0F, 0x89, 0x1E, 0xFF,// jns near ptr 0F000h
		/*00E2*/ 0xB0, 0x09,// mov al,9
		/*00E4*/ 0x0F, 0x8A, 0x18, 0xFF,// jp near ptr 0F000h
		/*00E8*/ 0xB0, 0x0A,// mov al,0Ah
		/*00EA*/ 0x0F, 0x8B, 0x12, 0xFF,// jnp near ptr 0F000h
		/*00EE*/ 0xB0, 0x0B,// mov al,0Bh
		/*00F0*/ 0x0F, 0x8C, 0x0C, 0xFF,// jl near ptr 0F000h
		/*00F4*/ 0xB0, 0x0C,// mov al,0Ch
		/*00F6*/ 0x0F, 0x8D, 0x06, 0xFF,// jge near ptr 0F000h
		/*00FA*/ 0xB0, 0x0D,// mov al,0Dh
		/*00FC*/ 0x0F, 0x8E, 0x00, 0xFF,// jle near ptr 0F000h
		/*0100*/ 0xB0, 0x0E,// mov al,0Eh
		/*0102*/ 0x0F, 0x8F, 0xFA, 0xFE,// jg near ptr 0F000h
		/*0106*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x000E,
		0x001C,
		0x002A,
		0x0038,
		0x0046,
		0x0054,
		0x0062,
		0x0070,
		0x007E,
		0x008C,
		0x009A,
		0x00A8,
		0x00AC,
		0x00AE,
		0x00B2,
		0x00B4,
		0x00B8,
		0x00BA,
		0x00BE,
		0x00C0,
		0x00C4,
		0x00C6,
		0x00CA,
		0x00CC,
		0x00D0,
		0x00D2,
		0x00D6,
		0x00D8,
		0x00DC,
		0x00DE,
		0x00E2,
		0x00E4,
		0x00E8,
		0x00EA,
		0x00EE,
		0x00F0,
		0x00F4,
		0x00F6,
		0x00FA,
		0x00FC,
		0x0100,
		0x0102,
		0x0106,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_near_other_short() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x0F, 0x80, 0x5C, 0x00,// jo near ptr 8060h
		/*0004*/ 0xB0, 0x00,// mov al,0
		/*0006*/ 0x0F, 0x81, 0x57, 0x00,// jno near ptr 8061h
		/*000A*/ 0xB0, 0x01,// mov al,1
		/*000C*/ 0x0F, 0x82, 0x52, 0x00,// jb near ptr 8062h
		/*0010*/ 0xB0, 0x02,// mov al,2
		/*0012*/ 0x0F, 0x83, 0x4D, 0x00,// jae near ptr 8063h
		/*0016*/ 0xB0, 0x03,// mov al,3
		/*0018*/ 0x0F, 0x84, 0x48, 0x00,// je near ptr 8064h
		/*001C*/ 0xB0, 0x04,// mov al,4
		/*001E*/ 0x0F, 0x85, 0x43, 0x00,// jne near ptr 8065h
		/*0022*/ 0xB0, 0x05,// mov al,5
		/*0024*/ 0x0F, 0x86, 0x3E, 0x00,// jbe near ptr 8066h
		/*0028*/ 0xB0, 0x06,// mov al,6
		/*002A*/ 0x0F, 0x87, 0x39, 0x00,// ja near ptr 8067h
		/*002E*/ 0xB0, 0x07,// mov al,7
		/*0030*/ 0x0F, 0x88, 0x34, 0x00,// js near ptr 8068h
		/*0034*/ 0xB0, 0x08,// mov al,8
		/*0036*/ 0x0F, 0x89, 0x2F, 0x00,// jns near ptr 8069h
		/*003A*/ 0xB0, 0x09,// mov al,9
		/*003C*/ 0x0F, 0x8A, 0x2A, 0x00,// jp near ptr 806Ah
		/*0040*/ 0xB0, 0x0A,// mov al,0Ah
		/*0042*/ 0x0F, 0x8B, 0x25, 0x00,// jnp near ptr 806Bh
		/*0046*/ 0xB0, 0x0B,// mov al,0Bh
		/*0048*/ 0x0F, 0x8C, 0x20, 0x00,// jl near ptr 806Ch
		/*004C*/ 0xB0, 0x0C,// mov al,0Ch
		/*004E*/ 0x0F, 0x8D, 0x1B, 0x00,// jge near ptr 806Dh
		/*0052*/ 0xB0, 0x0D,// mov al,0Dh
		/*0054*/ 0x0F, 0x8E, 0x16, 0x00,// jle near ptr 806Eh
		/*0058*/ 0xB0, 0x0E,// mov al,0Eh
		/*005A*/ 0x0F, 0x8F, 0x11, 0x00,// jg near ptr 806Fh
		/*005E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x70, 0x5F,// jo short 8060h
		/*0002*/ 0xB0, 0x00,// mov al,0
		/*0004*/ 0x71, 0x5C,// jno short 8061h
		/*0006*/ 0xB0, 0x01,// mov al,1
		/*0008*/ 0x72, 0x59,// jb short 8062h
		/*000A*/ 0xB0, 0x02,// mov al,2
		/*000C*/ 0x73, 0x56,// jae short 8063h
		/*000E*/ 0xB0, 0x03,// mov al,3
		/*0010*/ 0x74, 0x53,// je short 8064h
		/*0012*/ 0xB0, 0x04,// mov al,4
		/*0014*/ 0x75, 0x50,// jne short 8065h
		/*0016*/ 0xB0, 0x05,// mov al,5
		/*0018*/ 0x76, 0x4D,// jbe short 8066h
		/*001A*/ 0xB0, 0x06,// mov al,6
		/*001C*/ 0x77, 0x4A,// ja short 8067h
		/*001E*/ 0xB0, 0x07,// mov al,7
		/*0020*/ 0x78, 0x47,// js short 8068h
		/*0022*/ 0xB0, 0x08,// mov al,8
		/*0024*/ 0x79, 0x44,// jns short 8069h
		/*0026*/ 0xB0, 0x09,// mov al,9
		/*0028*/ 0x7A, 0x41,// jp short 806Ah
		/*002A*/ 0xB0, 0x0A,// mov al,0Ah
		/*002C*/ 0x7B, 0x3E,// jnp short 806Bh
		/*002E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0030*/ 0x7C, 0x3B,// jl short 806Ch
		/*0032*/ 0xB0, 0x0C,// mov al,0Ch
		/*0034*/ 0x7D, 0x38,// jge short 806Dh
		/*0036*/ 0xB0, 0x0D,// mov al,0Dh
		/*0038*/ 0x7E, 0x35,// jle short 806Eh
		/*003A*/ 0xB0, 0x0E,// mov al,0Eh
		/*003C*/ 0x7F, 0x32,// jg short 806Fh
		/*003E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0002,
		0x0004,
		0x0006,
		0x0008,
		0x000A,
		0x000C,
		0x000E,
		0x0010,
		0x0012,
		0x0014,
		0x0016,
		0x0018,
		0x001A,
		0x001C,
		0x001E,
		0x0020,
		0x0022,
		0x0024,
		0x0026,
		0x0028,
		0x002A,
		0x002C,
		0x002E,
		0x0030,
		0x0032,
		0x0034,
		0x0036,
		0x0038,
		0x003A,
		0x003C,
		0x003E,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		ORIG_RIP - 1,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_near_other_near() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x0F, 0x80, 0x5C, 0x00,// jo near ptr 8060h
		/*0004*/ 0xB0, 0x00,// mov al,0
		/*0006*/ 0x0F, 0x81, 0x57, 0x00,// jno near ptr 8061h
		/*000A*/ 0xB0, 0x01,// mov al,1
		/*000C*/ 0x0F, 0x82, 0x52, 0x00,// jb near ptr 8062h
		/*0010*/ 0xB0, 0x02,// mov al,2
		/*0012*/ 0x0F, 0x83, 0x4D, 0x00,// jae near ptr 8063h
		/*0016*/ 0xB0, 0x03,// mov al,3
		/*0018*/ 0x0F, 0x84, 0x48, 0x00,// je near ptr 8064h
		/*001C*/ 0xB0, 0x04,// mov al,4
		/*001E*/ 0x0F, 0x85, 0x43, 0x00,// jne near ptr 8065h
		/*0022*/ 0xB0, 0x05,// mov al,5
		/*0024*/ 0x0F, 0x86, 0x3E, 0x00,// jbe near ptr 8066h
		/*0028*/ 0xB0, 0x06,// mov al,6
		/*002A*/ 0x0F, 0x87, 0x39, 0x00,// ja near ptr 8067h
		/*002E*/ 0xB0, 0x07,// mov al,7
		/*0030*/ 0x0F, 0x88, 0x34, 0x00,// js near ptr 8068h
		/*0034*/ 0xB0, 0x08,// mov al,8
		/*0036*/ 0x0F, 0x89, 0x2F, 0x00,// jns near ptr 8069h
		/*003A*/ 0xB0, 0x09,// mov al,9
		/*003C*/ 0x0F, 0x8A, 0x2A, 0x00,// jp near ptr 806Ah
		/*0040*/ 0xB0, 0x0A,// mov al,0Ah
		/*0042*/ 0x0F, 0x8B, 0x25, 0x00,// jnp near ptr 806Bh
		/*0046*/ 0xB0, 0x0B,// mov al,0Bh
		/*0048*/ 0x0F, 0x8C, 0x20, 0x00,// jl near ptr 806Ch
		/*004C*/ 0xB0, 0x0C,// mov al,0Ch
		/*004E*/ 0x0F, 0x8D, 0x1B, 0x00,// jge near ptr 806Dh
		/*0052*/ 0xB0, 0x0D,// mov al,0Dh
		/*0054*/ 0x0F, 0x8E, 0x16, 0x00,// jle near ptr 806Eh
		/*0058*/ 0xB0, 0x0E,// mov al,0Eh
		/*005A*/ 0x0F, 0x8F, 0x11, 0x00,// jg near ptr 806Fh
		/*005E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x0F, 0x80, 0x5C, 0xF0,// jo near ptr 8060h
		/*0004*/ 0xB0, 0x00,// mov al,0
		/*0006*/ 0x0F, 0x81, 0x57, 0xF0,// jno near ptr 8061h
		/*000A*/ 0xB0, 0x01,// mov al,1
		/*000C*/ 0x0F, 0x82, 0x52, 0xF0,// jb near ptr 8062h
		/*0010*/ 0xB0, 0x02,// mov al,2
		/*0012*/ 0x0F, 0x83, 0x4D, 0xF0,// jae near ptr 8063h
		/*0016*/ 0xB0, 0x03,// mov al,3
		/*0018*/ 0x0F, 0x84, 0x48, 0xF0,// je near ptr 8064h
		/*001C*/ 0xB0, 0x04,// mov al,4
		/*001E*/ 0x0F, 0x85, 0x43, 0xF0,// jne near ptr 8065h
		/*0022*/ 0xB0, 0x05,// mov al,5
		/*0024*/ 0x0F, 0x86, 0x3E, 0xF0,// jbe near ptr 8066h
		/*0028*/ 0xB0, 0x06,// mov al,6
		/*002A*/ 0x0F, 0x87, 0x39, 0xF0,// ja near ptr 8067h
		/*002E*/ 0xB0, 0x07,// mov al,7
		/*0030*/ 0x0F, 0x88, 0x34, 0xF0,// js near ptr 8068h
		/*0034*/ 0xB0, 0x08,// mov al,8
		/*0036*/ 0x0F, 0x89, 0x2F, 0xF0,// jns near ptr 8069h
		/*003A*/ 0xB0, 0x09,// mov al,9
		/*003C*/ 0x0F, 0x8A, 0x2A, 0xF0,// jp near ptr 806Ah
		/*0040*/ 0xB0, 0x0A,// mov al,0Ah
		/*0042*/ 0x0F, 0x8B, 0x25, 0xF0,// jnp near ptr 806Bh
		/*0046*/ 0xB0, 0x0B,// mov al,0Bh
		/*0048*/ 0x0F, 0x8C, 0x20, 0xF0,// jl near ptr 806Ch
		/*004C*/ 0xB0, 0x0C,// mov al,0Ch
		/*004E*/ 0x0F, 0x8D, 0x1B, 0xF0,// jge near ptr 806Dh
		/*0052*/ 0xB0, 0x0D,// mov al,0Dh
		/*0054*/ 0x0F, 0x8E, 0x16, 0xF0,// jle near ptr 806Eh
		/*0058*/ 0xB0, 0x0E,// mov al,0Eh
		/*005A*/ 0x0F, 0x8F, 0x11, 0xF0,// jg near ptr 806Fh
		/*005E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0004,
		0x0006,
		0x000A,
		0x000C,
		0x0010,
		0x0012,
		0x0016,
		0x0018,
		0x001C,
		0x001E,
		0x0022,
		0x0024,
		0x0028,
		0x002A,
		0x002E,
		0x0030,
		0x0034,
		0x0036,
		0x003A,
		0x003C,
		0x0040,
		0x0042,
		0x0046,
		0x0048,
		0x004C,
		0x004E,
		0x0052,
		0x0054,
		0x0058,
		0x005A,
		0x005E,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		ORIG_RIP + 0x1000,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_near_fwd_short_no_opt() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x0F, 0x80, 0x5C, 0x00,// jo near ptr 8060h
		/*0004*/ 0xB0, 0x00,// mov al,0
		/*0006*/ 0x0F, 0x81, 0x56, 0x00,// jno near ptr 8060h
		/*000A*/ 0xB0, 0x01,// mov al,1
		/*000C*/ 0x0F, 0x82, 0x50, 0x00,// jb near ptr 8060h
		/*0010*/ 0xB0, 0x02,// mov al,2
		/*0012*/ 0x0F, 0x83, 0x4A, 0x00,// jae near ptr 8060h
		/*0016*/ 0xB0, 0x03,// mov al,3
		/*0018*/ 0x0F, 0x84, 0x44, 0x00,// je near ptr 8060h
		/*001C*/ 0xB0, 0x04,// mov al,4
		/*001E*/ 0x0F, 0x85, 0x3E, 0x00,// jne near ptr 8060h
		/*0022*/ 0xB0, 0x05,// mov al,5
		/*0024*/ 0x0F, 0x86, 0x38, 0x00,// jbe near ptr 8060h
		/*0028*/ 0xB0, 0x06,// mov al,6
		/*002A*/ 0x0F, 0x87, 0x32, 0x00,// ja near ptr 8060h
		/*002E*/ 0xB0, 0x07,// mov al,7
		/*0030*/ 0x0F, 0x88, 0x2C, 0x00,// js near ptr 8060h
		/*0034*/ 0xB0, 0x08,// mov al,8
		/*0036*/ 0x0F, 0x89, 0x26, 0x00,// jns near ptr 8060h
		/*003A*/ 0xB0, 0x09,// mov al,9
		/*003C*/ 0x0F, 0x8A, 0x20, 0x00,// jp near ptr 8060h
		/*0040*/ 0xB0, 0x0A,// mov al,0Ah
		/*0042*/ 0x0F, 0x8B, 0x1A, 0x00,// jnp near ptr 8060h
		/*0046*/ 0xB0, 0x0B,// mov al,0Bh
		/*0048*/ 0x0F, 0x8C, 0x14, 0x00,// jl near ptr 8060h
		/*004C*/ 0xB0, 0x0C,// mov al,0Ch
		/*004E*/ 0x0F, 0x8D, 0x0E, 0x00,// jge near ptr 8060h
		/*0052*/ 0xB0, 0x0D,// mov al,0Dh
		/*0054*/ 0x0F, 0x8E, 0x08, 0x00,// jle near ptr 8060h
		/*0058*/ 0xB0, 0x0E,// mov al,0Eh
		/*005A*/ 0x0F, 0x8F, 0x02, 0x00,// jg near ptr 8060h
		/*005E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0060*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x0F, 0x80, 0x5C, 0x00,// jo near ptr 8060h
		/*0004*/ 0xB0, 0x00,// mov al,0
		/*0006*/ 0x0F, 0x81, 0x56, 0x00,// jno near ptr 8060h
		/*000A*/ 0xB0, 0x01,// mov al,1
		/*000C*/ 0x0F, 0x82, 0x50, 0x00,// jb near ptr 8060h
		/*0010*/ 0xB0, 0x02,// mov al,2
		/*0012*/ 0x0F, 0x83, 0x4A, 0x00,// jae near ptr 8060h
		/*0016*/ 0xB0, 0x03,// mov al,3
		/*0018*/ 0x0F, 0x84, 0x44, 0x00,// je near ptr 8060h
		/*001C*/ 0xB0, 0x04,// mov al,4
		/*001E*/ 0x0F, 0x85, 0x3E, 0x00,// jne near ptr 8060h
		/*0022*/ 0xB0, 0x05,// mov al,5
		/*0024*/ 0x0F, 0x86, 0x38, 0x00,// jbe near ptr 8060h
		/*0028*/ 0xB0, 0x06,// mov al,6
		/*002A*/ 0x0F, 0x87, 0x32, 0x00,// ja near ptr 8060h
		/*002E*/ 0xB0, 0x07,// mov al,7
		/*0030*/ 0x0F, 0x88, 0x2C, 0x00,// js near ptr 8060h
		/*0034*/ 0xB0, 0x08,// mov al,8
		/*0036*/ 0x0F, 0x89, 0x26, 0x00,// jns near ptr 8060h
		/*003A*/ 0xB0, 0x09,// mov al,9
		/*003C*/ 0x0F, 0x8A, 0x20, 0x00,// jp near ptr 8060h
		/*0040*/ 0xB0, 0x0A,// mov al,0Ah
		/*0042*/ 0x0F, 0x8B, 0x1A, 0x00,// jnp near ptr 8060h
		/*0046*/ 0xB0, 0x0B,// mov al,0Bh
		/*0048*/ 0x0F, 0x8C, 0x14, 0x00,// jl near ptr 8060h
		/*004C*/ 0xB0, 0x0C,// mov al,0Ch
		/*004E*/ 0x0F, 0x8D, 0x0E, 0x00,// jge near ptr 8060h
		/*0052*/ 0xB0, 0x0D,// mov al,0Dh
		/*0054*/ 0x0F, 0x8E, 0x08, 0x00,// jle near ptr 8060h
		/*0058*/ 0xB0, 0x0E,// mov al,0Eh
		/*005A*/ 0x0F, 0x8F, 0x02, 0x00,// jg near ptr 8060h
		/*005E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0060*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0004,
		0x0006,
		0x000A,
		0x000C,
		0x0010,
		0x0012,
		0x0016,
		0x0018,
		0x001C,
		0x001E,
		0x0022,
		0x0024,
		0x0028,
		0x002A,
		0x002E,
		0x0030,
		0x0034,
		0x0036,
		0x003A,
		0x003C,
		0x0040,
		0x0042,
		0x0046,
		0x0048,
		0x004C,
		0x004E,
		0x0052,
		0x0054,
		0x0058,
		0x005A,
		0x005E,
		0x0060,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::DONT_FIX_BRANCHES;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}
