// SPDX-License-Identifier: MIT
// Copyright (C) 2018-present iced project and contributors

use crate::iced_constants::IcedConstants;
use crate::iced_error::IcedError;
#[cfg(feature = "instr_info")]
use crate::info::enums::*;
use crate::instruction_internal;
use crate::*;
#[cfg(any(feature = "gas", feature = "intel", feature = "masm", feature = "nasm", feature = "fast_fmt"))]
use core::fmt;
use core::hash::{Hash, Hasher};
use core::iter::{ExactSizeIterator, FusedIterator};
#[cfg(feature = "encoder")]
use core::ptr;
use core::{mem, slice, u16, u32, u64};
use static_assertions::{const_assert, const_assert_eq};

// GENERATOR-BEGIN: MemoryFlags
// ⚠️This was generated by GENERATOR!🦹‍♂️
pub(crate) struct MemoryFlags;
#[allow(dead_code)]
impl MemoryFlags {
	pub(crate) const SCALE_MASK: u32 = 0x0000_0003;
	pub(crate) const DISPL_SIZE_SHIFT: u32 = 0x0000_0002;
	pub(crate) const DISPL_SIZE_MASK: u32 = 0x0000_0007;
	pub(crate) const SEGMENT_PREFIX_SHIFT: u32 = 0x0000_0005;
	pub(crate) const SEGMENT_PREFIX_MASK: u32 = 0x0000_0007;
	pub(crate) const BROADCAST: u32 = 0x0000_8000;
}
// GENERATOR-END: MemoryFlags

// GENERATOR-BEGIN: OpKindFlags
// ⚠️This was generated by GENERATOR!🦹‍♂️
pub(crate) struct OpKindFlags;
#[allow(dead_code)]
impl OpKindFlags {
	pub(crate) const OP_KIND_BITS: u32 = 0x0000_0005;
	pub(crate) const OP_KIND_MASK: u32 = 0x0000_001F;
	pub(crate) const OP1_KIND_SHIFT: u32 = 0x0000_0005;
	pub(crate) const OP2_KIND_SHIFT: u32 = 0x0000_000A;
	pub(crate) const OP3_KIND_SHIFT: u32 = 0x0000_000F;
	pub(crate) const DATA_LENGTH_MASK: u32 = 0x0000_000F;
	pub(crate) const DATA_LENGTH_SHIFT: u32 = 0x0000_0014;
	pub(crate) const CODE_SIZE_MASK: u32 = 0x0000_0003;
	pub(crate) const CODE_SIZE_SHIFT: u32 = 0x0000_001E;
	pub(crate) const EQUALS_IGNORE_MASK: u32 = 0xC000_0000;
}
// GENERATOR-END: OpKindFlags

// GENERATOR-BEGIN: CodeFlags
// ⚠️This was generated by GENERATOR!🦹‍♂️
pub(crate) struct CodeFlags;
#[allow(dead_code)]
impl CodeFlags {
	pub(crate) const CODE_BITS: u32 = 0x0000_000D;
	pub(crate) const CODE_MASK: u32 = 0x0000_1FFF;
	pub(crate) const ROUNDING_CONTROL_MASK: u32 = 0x0000_0007;
	pub(crate) const ROUNDING_CONTROL_SHIFT: u32 = 0x0000_000D;
	pub(crate) const OP_MASK_MASK: u32 = 0x0000_0007;
	pub(crate) const OP_MASK_SHIFT: u32 = 0x0000_0010;
	pub(crate) const INSTR_LENGTH_MASK: u32 = 0x0000_000F;
	pub(crate) const INSTR_LENGTH_SHIFT: u32 = 0x0000_0013;
	pub(crate) const SUPPRESS_ALL_EXCEPTIONS: u32 = 0x0200_0000;
	pub(crate) const ZEROING_MASKING: u32 = 0x0400_0000;
	pub(crate) const XACQUIRE_PREFIX: u32 = 0x0800_0000;
	pub(crate) const XRELEASE_PREFIX: u32 = 0x1000_0000;
	pub(crate) const REPE_PREFIX: u32 = 0x2000_0000;
	pub(crate) const REPNE_PREFIX: u32 = 0x4000_0000;
	pub(crate) const LOCK_PREFIX: u32 = 0x8000_0000;
	pub(crate) const EQUALS_IGNORE_MASK: u32 = 0x0078_0000;
}
// GENERATOR-END: CodeFlags

/// A 16/32/64-bit x86 instruction. Created by [`Decoder`] or by `Instruction::with*()` methods.
///
/// [`Decoder`]: struct.Decoder.html
#[derive(Debug, Default, Copy, Clone)]
pub struct Instruction {
	pub(crate) next_rip: u64,
	pub(crate) code_flags: u32,    // CodeFlags
	pub(crate) op_kind_flags: u32, // OpKindFlags
	// If it's a 64-bit immediate/offset/target, the high 32 bits is in mem_displ
	pub(crate) immediate: u32,
	// This is the high 32 bits if it's a 64-bit immediate/offset/target
	pub(crate) mem_displ: u32,
	pub(crate) mem_displ_hi: u32,
	pub(crate) memory_flags: u16, // MemoryFlags
	pub(crate) mem_base_reg: u8,  // Register
	pub(crate) mem_index_reg: u8, // Register
	pub(crate) reg0: u8,          // Register
	pub(crate) reg1: u8,          // Register
	pub(crate) reg2: u8,          // Register
	pub(crate) reg3: u8,          // Register
	#[allow(dead_code)]
	res: [u8; 4],
}
#[cfg(test)]
#[allow(dead_code)]
pub(crate) const INSTRUCTION_TOTAL_SIZE: usize = 40;

#[allow(clippy::len_without_is_empty)]
impl Instruction {
	/// Creates an empty `Instruction` (all fields are cleared). See also the `with_*()` constructor methods.
	#[must_use]
	#[inline]
	pub fn new() -> Self {
		Instruction::default()
	}

	/// Checks if two instructions are equal, comparing all bits, not ignoring anything. `==` ignores some fields.
	#[must_use]
	#[allow(trivial_casts)]
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn eq_all_bits(&self, other: &Self) -> bool {
		// The compiler generated better code than if we compare all fields one at a time.
		// SAFETY: see `slice::from_raw_parts()`
		unsafe {
			let a: *const u8 = self as *const Self as *const u8;
			let b: *const u8 = other as *const Self as *const u8;
			let sa = slice::from_raw_parts(a, mem::size_of::<Self>());
			let sb = slice::from_raw_parts(b, mem::size_of::<Self>());
			sa == sb
		}
	}

	/// Gets the 16-bit IP of the instruction
	#[must_use]
	#[inline]
	pub fn ip16(&self) -> u16 {
		(self.next_rip as u16).wrapping_sub(self.len() as u16)
	}

	/// Sets the 16-bit IP of the instruction
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_ip16(&mut self, new_value: u16) {
		self.next_rip = (new_value as u64).wrapping_add(self.len() as u64);
	}

	/// Gets the 32-bit IP of the instruction
	#[must_use]
	#[inline]
	pub fn ip32(&self) -> u32 {
		(self.next_rip as u32).wrapping_sub(self.len() as u32)
	}

	/// Sets the 32-bit IP of the instruction
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_ip32(&mut self, new_value: u32) {
		self.next_rip = (new_value as u64).wrapping_add(self.len() as u64);
	}

	/// Gets the 64-bit IP of the instruction
	#[must_use]
	#[inline]
	pub fn ip(&self) -> u64 {
		self.next_rip.wrapping_sub(self.len() as u64)
	}

	/// Sets the 64-bit IP of the instruction
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_ip(&mut self, new_value: u64) {
		self.next_rip = new_value.wrapping_add(self.len() as u64);
	}

	/// Gets the 16-bit IP of the next instruction
	#[must_use]
	#[inline]
	pub fn next_ip16(&self) -> u16 {
		self.next_rip as u16
	}

	/// Sets the 16-bit IP of the next instruction
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_next_ip16(&mut self, new_value: u16) {
		self.next_rip = new_value as u64;
	}

	/// Gets the 32-bit IP of the next instruction
	#[must_use]
	#[inline]
	pub fn next_ip32(&self) -> u32 {
		self.next_rip as u32
	}

	/// Sets the 32-bit IP of the next instruction
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_next_ip32(&mut self, new_value: u32) {
		self.next_rip = new_value as u64;
	}

	/// Gets the 64-bit IP of the next instruction
	#[must_use]
	#[inline]
	pub fn next_ip(&self) -> u64 {
		self.next_rip
	}

	/// Sets the 64-bit IP of the next instruction
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_next_ip(&mut self, new_value: u64) {
		self.next_rip = new_value;
	}

	/// Gets the code size when the instruction was decoded. This value is informational and can
	/// be used by a formatter.
	#[must_use]
	#[inline]
	pub fn code_size(&self) -> CodeSize {
		unsafe { mem::transmute(((self.op_kind_flags >> OpKindFlags::CODE_SIZE_SHIFT) & OpKindFlags::CODE_SIZE_MASK) as u8) }
	}

	/// Sets the code size when the instruction was decoded. This value is informational and can
	/// be used by a formatter.
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_code_size(&mut self, new_value: CodeSize) {
		self.op_kind_flags = (self.op_kind_flags & !(OpKindFlags::CODE_SIZE_MASK << OpKindFlags::CODE_SIZE_SHIFT))
			| (((new_value as u32) & OpKindFlags::CODE_SIZE_MASK) << OpKindFlags::CODE_SIZE_SHIFT);
	}

	/// Checks if it's an invalid instruction ([`code()`] == [`Code::INVALID`])
	///
	/// [`code()`]: #method.code
	/// [`Code::INVALID`]: enum.Code.html#variant.INVALID
	#[must_use]
	#[inline]
	pub fn is_invalid(&self) -> bool {
		const_assert_eq!(Code::INVALID as u32, 0);
		(self.code_flags & CodeFlags::CODE_MASK) == 0
	}

	/// Gets the instruction code, see also [`mnemonic()`]
	///
	/// [`mnemonic()`]: #method.mnemonic
	#[must_use]
	#[inline]
	pub fn code(&self) -> Code {
		unsafe { mem::transmute((self.code_flags & CodeFlags::CODE_MASK) as u16) }
	}

	/// Sets the instruction code
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_code(&mut self, new_value: Code) {
		self.code_flags = (self.code_flags & !CodeFlags::CODE_MASK) | new_value as u32;
	}

	/// Gets the mnemonic, see also [`code()`]
	///
	/// [`code()`]: #method.code
	#[must_use]
	#[inline]
	pub fn mnemonic(&self) -> Mnemonic {
		self.code().mnemonic()
	}

	/// Gets the operand count. An instruction can have 0-5 operands.
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // add [rax],ebx
	/// let bytes = b"\x01\x18";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	/// let instr = decoder.decode();
	///
	/// assert_eq!(instr.op_count(), 2);
	/// ```
	#[must_use]
	#[inline]
	pub fn op_count(&self) -> u32 {
		// SAFETY: All Code values are valid indexes
		unsafe { *instruction_op_counts::OP_COUNT.get_unchecked((self.code_flags & CodeFlags::CODE_MASK) as usize) as u32 }
	}

	/// Gets the length of the instruction, 0-15 bytes. This is just informational. If you modify the instruction
	/// or create a new one, this method could return the wrong value.
	#[must_use]
	#[inline]
	pub fn len(&self) -> usize {
		((self.code_flags >> CodeFlags::INSTR_LENGTH_SHIFT) & CodeFlags::INSTR_LENGTH_MASK) as usize
	}

	/// Sets the length of the instruction, 0-15 bytes. This is just informational. If you modify the instruction
	/// or create a new one, this method could return the wrong value.
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_len(&mut self, new_value: usize) {
		self.code_flags = (self.code_flags & !(CodeFlags::INSTR_LENGTH_MASK << CodeFlags::INSTR_LENGTH_SHIFT))
			| (((new_value as u32) & CodeFlags::INSTR_LENGTH_MASK) << CodeFlags::INSTR_LENGTH_SHIFT);
	}

	/// `true` if the instruction has the `XACQUIRE` prefix (`F2`)
	#[must_use]
	#[inline]
	pub fn has_xacquire_prefix(&self) -> bool {
		(self.code_flags & CodeFlags::XACQUIRE_PREFIX) != 0
	}

	/// `true` if the instruction has the `XACQUIRE` prefix (`F2`)
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_has_xacquire_prefix(&mut self, new_value: bool) {
		if new_value {
			self.code_flags |= CodeFlags::XACQUIRE_PREFIX;
		} else {
			self.code_flags &= !CodeFlags::XACQUIRE_PREFIX;
		}
	}

	/// `true` if the instruction has the `XRELEASE` prefix (`F3`)
	#[must_use]
	#[inline]
	pub fn has_xrelease_prefix(&self) -> bool {
		(self.code_flags & CodeFlags::XRELEASE_PREFIX) != 0
	}

	/// `true` if the instruction has the `XRELEASE` prefix (`F3`)
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_has_xrelease_prefix(&mut self, new_value: bool) {
		if new_value {
			self.code_flags |= CodeFlags::XRELEASE_PREFIX;
		} else {
			self.code_flags &= !CodeFlags::XRELEASE_PREFIX;
		}
	}

	/// `true` if the instruction has the `REPE` or `REP` prefix (`F3`)
	#[must_use]
	#[inline]
	pub fn has_rep_prefix(&self) -> bool {
		(self.code_flags & CodeFlags::REPE_PREFIX) != 0
	}

	/// `true` if the instruction has the `REPE` or `REP` prefix (`F3`)
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_has_rep_prefix(&mut self, new_value: bool) {
		if new_value {
			self.code_flags |= CodeFlags::REPE_PREFIX;
		} else {
			self.code_flags &= !CodeFlags::REPE_PREFIX;
		}
	}

	/// `true` if the instruction has the `REPE` or `REP` prefix (`F3`)
	#[must_use]
	#[inline]
	pub fn has_repe_prefix(&self) -> bool {
		(self.code_flags & CodeFlags::REPE_PREFIX) != 0
	}

	/// `true` if the instruction has the `REPE` or `REP` prefix (`F3`)
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_has_repe_prefix(&mut self, new_value: bool) {
		if new_value {
			self.code_flags |= CodeFlags::REPE_PREFIX;
		} else {
			self.code_flags &= !CodeFlags::REPE_PREFIX;
		}
	}

	/// `true` if the instruction has the `REPNE` prefix (`F2`)
	#[must_use]
	#[inline]
	pub fn has_repne_prefix(&self) -> bool {
		(self.code_flags & CodeFlags::REPNE_PREFIX) != 0
	}

	/// `true` if the instruction has the `REPNE` prefix (`F2`)
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_has_repne_prefix(&mut self, new_value: bool) {
		if new_value {
			self.code_flags |= CodeFlags::REPNE_PREFIX;
		} else {
			self.code_flags &= !CodeFlags::REPNE_PREFIX;
		}
	}

	/// `true` if the instruction has the `LOCK` prefix (`F0`)
	#[must_use]
	#[inline]
	pub fn has_lock_prefix(&self) -> bool {
		(self.code_flags & CodeFlags::LOCK_PREFIX) != 0
	}

	/// `true` if the instruction has the `LOCK` prefix (`F0`)
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_has_lock_prefix(&mut self, new_value: bool) {
		if new_value {
			self.code_flags |= CodeFlags::LOCK_PREFIX;
		} else {
			self.code_flags &= !CodeFlags::LOCK_PREFIX;
		}
	}

	/// Gets operand #0's kind if the operand exists (see [`op_count()`] and [`try_op_kind()`])
	///
	/// [`op_count()`]: #method.op_count
	/// [`try_op_kind()`]: #method.try_op_kind
	#[must_use]
	#[inline]
	pub fn op0_kind(&self) -> OpKind {
		unsafe { mem::transmute((self.op_kind_flags & OpKindFlags::OP_KIND_MASK) as u8) }
	}

	/// Sets operand #0's kind if the operand exists (see [`op_count()`] and [`try_set_op_kind()`])
	///
	/// [`op_count()`]: #method.op_count
	/// [`try_set_op_kind()`]: #method.try_set_op_kind
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_op0_kind(&mut self, new_value: OpKind) {
		self.op_kind_flags = (self.op_kind_flags & !OpKindFlags::OP_KIND_MASK) | ((new_value as u32) & OpKindFlags::OP_KIND_MASK);
	}

	/// Gets operand #1's kind if the operand exists (see [`op_count()`] and [`try_op_kind()`])
	///
	/// [`op_count()`]: #method.op_count
	/// [`try_op_kind()`]: #method.try_op_kind
	#[must_use]
	#[inline]
	pub fn op1_kind(&self) -> OpKind {
		unsafe { mem::transmute(((self.op_kind_flags >> OpKindFlags::OP1_KIND_SHIFT) & OpKindFlags::OP_KIND_MASK) as u8) }
	}

	/// Sets operand #1's kind if the operand exists (see [`op_count()`] and [`try_set_op_kind()`])
	///
	/// [`op_count()`]: #method.op_count
	/// [`try_set_op_kind()`]: #method.try_set_op_kind
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_op1_kind(&mut self, new_value: OpKind) {
		self.op_kind_flags = (self.op_kind_flags & !(OpKindFlags::OP_KIND_MASK << OpKindFlags::OP1_KIND_SHIFT))
			| (((new_value as u32) & OpKindFlags::OP_KIND_MASK) << OpKindFlags::OP1_KIND_SHIFT);
	}

	/// Gets operand #2's kind if the operand exists (see [`op_count()`] and [`try_op_kind()`])
	///
	/// [`op_count()`]: #method.op_count
	/// [`try_op_kind()`]: #method.try_op_kind
	#[must_use]
	#[inline]
	pub fn op2_kind(&self) -> OpKind {
		unsafe { mem::transmute(((self.op_kind_flags >> OpKindFlags::OP2_KIND_SHIFT) & OpKindFlags::OP_KIND_MASK) as u8) }
	}

	/// Sets operand #2's kind if the operand exists (see [`op_count()`] and [`try_set_op_kind()`])
	///
	/// [`op_count()`]: #method.op_count
	/// [`try_set_op_kind()`]: #method.try_set_op_kind
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_op2_kind(&mut self, new_value: OpKind) {
		self.op_kind_flags = (self.op_kind_flags & !(OpKindFlags::OP_KIND_MASK << OpKindFlags::OP2_KIND_SHIFT))
			| (((new_value as u32) & OpKindFlags::OP_KIND_MASK) << OpKindFlags::OP2_KIND_SHIFT);
	}

	/// Gets operand #3's kind if the operand exists (see [`op_count()`] and [`try_op_kind()`])
	///
	/// [`op_count()`]: #method.op_count
	/// [`try_op_kind()`]: #method.try_op_kind
	#[must_use]
	#[inline]
	pub fn op3_kind(&self) -> OpKind {
		unsafe { mem::transmute(((self.op_kind_flags >> OpKindFlags::OP3_KIND_SHIFT) & OpKindFlags::OP_KIND_MASK) as u8) }
	}

	/// Sets operand #3's kind if the operand exists (see [`op_count()`] and [`try_set_op_kind()`])
	///
	/// [`op_count()`]: #method.op_count
	/// [`try_set_op_kind()`]: #method.try_set_op_kind
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[inline]
	pub fn set_op3_kind(&mut self, new_value: OpKind) {
		self.op_kind_flags = (self.op_kind_flags & !(OpKindFlags::OP_KIND_MASK << OpKindFlags::OP3_KIND_SHIFT))
			| (((new_value as u32) & OpKindFlags::OP_KIND_MASK) << OpKindFlags::OP3_KIND_SHIFT);
	}

	/// Gets operand #4's kind if the operand exists (see [`op_count()`] and [`try_op_kind()`])
	///
	/// [`op_count()`]: #method.op_count
	/// [`try_op_kind()`]: #method.try_op_kind
	#[allow(clippy::unused_self)]
	#[must_use]
	#[inline]
	pub fn op4_kind(&self) -> OpKind {
		OpKind::Immediate8
	}

	/// Sets operand #4's kind if the operand exists (see [`op_count()`] and [`try_set_op_kind()`])
	///
	/// [`op_count()`]: #method.op_count
	/// [`try_set_op_kind()`]: #method.try_set_op_kind
	///
	/// # Panics
	///
	/// Panics if `new_value` is invalid.
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_set_op4_kind() instead")]
	#[allow(clippy::unwrap_used)]
	#[inline]
	pub fn set_op4_kind(&mut self, new_value: OpKind) {
		self.try_set_op4_kind(new_value).unwrap()
	}

	/// Sets operand #4's kind if the operand exists (see [`op_count()`] and [`try_set_op_kind()`])
	///
	/// [`op_count()`]: #method.op_count
	/// [`try_set_op_kind()`]: #method.try_set_op_kind
	///
	/// # Errors
	///
	/// Fails if `new_value` is invalid.
	///
	/// # Arguments
	///
	/// * `new_value`: new value
	#[allow(clippy::unused_self)]
	#[inline]
	pub fn try_set_op4_kind(&mut self, new_value: OpKind) -> Result<(), IcedError> {
		if new_value != OpKind::Immediate8 {
			Err(IcedError::new("Invalid opkind"))
		} else {
			Ok(())
		}
	}

	/// Gets all op kinds ([`op_count()`] values)
	///
	/// [`op_count()`]: #method.op_count
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // add [rax],ebx
	/// let bytes = b"\x01\x18";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	/// let instr = decoder.decode();
	///
	/// for (i, op_kind) in instr.op_kinds().enumerate() {
	///     println!("op kind #{} = {:?}", i, op_kind);
	/// }
	/// ```
	#[inline]
	pub fn op_kinds(&self) -> impl Iterator<Item = OpKind> + ExactSizeIterator + FusedIterator {
		OpKindIterator::new(self)
	}

	/// Gets an operand's kind if it exists (see [`op_count()`])
	///
	/// [`op_count()`]: #method.op_count
	///
	/// # Panics
	///
	/// Panics if `operand` is invalid
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // add [rax],ebx
	/// let bytes = b"\x01\x18";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	/// let instr = decoder.decode();
	///
	/// assert_eq!(instr.op_count(), 2);
	/// assert_eq!(instr.op_kind(0), OpKind::Memory);
	/// assert_eq!(instr.memory_base(), Register::RAX);
	/// assert_eq!(instr.memory_index(), Register::None);
	/// assert_eq!(instr.op_kind(1), OpKind::Register);
	/// assert_eq!(instr.op_register(1), Register::EBX);
	/// ```
	#[must_use]
	#[inline]
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_op_kind() instead")]
	#[allow(clippy::unwrap_used)]
	pub fn op_kind(&self, operand: u32) -> OpKind {
		self.try_op_kind(operand).unwrap()
	}

	/// Gets an operand's kind if it exists (see [`op_count()`])
	///
	/// [`op_count()`]: #method.op_count
	///
	/// # Errors
	///
	/// Fails if `operand` is invalid
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // add [rax],ebx
	/// let bytes = b"\x01\x18";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	/// let instr = decoder.decode();
	///
	/// assert_eq!(instr.op_count(), 2);
	/// assert_eq!(instr.try_op_kind(0).unwrap(), OpKind::Memory);
	/// assert_eq!(instr.memory_base(), Register::RAX);
	/// assert_eq!(instr.memory_index(), Register::None);
	/// assert_eq!(instr.try_op_kind(1).unwrap(), OpKind::Register);
	/// assert_eq!(instr.op_register(1), Register::EBX);
	/// ```
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn try_op_kind(&self, operand: u32) -> Result<OpKind, IcedError> {
		const_assert_eq!(IcedConstants::MAX_OP_COUNT, 5);
		match operand {
			0 => Ok(self.op0_kind()),
			1 => Ok(self.op1_kind()),
			2 => Ok(self.op2_kind()),
			3 => Ok(self.op3_kind()),
			4 => Ok(self.op4_kind()),
			_ => Err(IcedError::new("Invalid operand")),
		}
	}

	/// Sets an operand's kind
	///
	/// # Panics
	///
	/// Panics if `operand` or `op_kind` is invalid
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	/// * `op_kind`: Operand kind
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_set_op_kind() instead")]
	#[allow(clippy::unwrap_used)]
	#[inline]
	pub fn set_op_kind(&mut self, operand: u32, op_kind: OpKind) {
		self.try_set_op_kind(operand, op_kind).unwrap()
	}

	/// Sets an operand's kind
	///
	/// # Errors
	///
	/// Fails if `operand` or `op_kind` is invalid
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	/// * `op_kind`: Operand kind
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn try_set_op_kind(&mut self, operand: u32, op_kind: OpKind) -> Result<(), IcedError> {
		const_assert_eq!(IcedConstants::MAX_OP_COUNT, 5);
		match operand {
			0 => self.set_op0_kind(op_kind),
			1 => self.set_op1_kind(op_kind),
			2 => self.set_op2_kind(op_kind),
			3 => self.set_op3_kind(op_kind),
			4 => return self.try_set_op4_kind(op_kind),
			_ => return Err(IcedError::new("Invalid operand")),
		};
		Ok(())
	}

	/// Checks if the instruction has a segment override prefix, see [`segment_prefix()`]
	///
	/// [`segment_prefix()`]: #method.segment_prefix
	#[must_use]
	#[inline]
	pub fn has_segment_prefix(&self) -> bool {
		(((self.memory_flags as u32) >> MemoryFlags::SEGMENT_PREFIX_SHIFT) & MemoryFlags::SEGMENT_PREFIX_MASK).wrapping_sub(1) < 6
	}

	/// Gets the segment override prefix or [`Register::None`] if none. See also [`memory_segment()`].
	/// Use this method if the operand has kind [`OpKind::Memory`],
	/// [`OpKind::MemorySegSI`], [`OpKind::MemorySegESI`], [`OpKind::MemorySegRSI`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`memory_segment()`]: #method.memory_segment
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	/// [`OpKind::MemorySegSI`]: enum.OpKind.html#variant.MemorySegSI
	/// [`OpKind::MemorySegESI`]: enum.OpKind.html#variant.MemorySegESI
	/// [`OpKind::MemorySegESI`]: enum.OpKind.html#variant.MemorySegESI
	/// [`OpKind::MemorySegRSI`]: enum.OpKind.html#variant.MemorySegRSI
	#[must_use]
	#[inline]
	pub fn segment_prefix(&self) -> Register {
		let index = (((self.memory_flags as u32) >> MemoryFlags::SEGMENT_PREFIX_SHIFT) & MemoryFlags::SEGMENT_PREFIX_MASK).wrapping_sub(1);
		if index < 6 {
			unsafe { mem::transmute((Register::ES as u32 + index) as u8) }
		} else {
			Register::None
		}
	}

	/// Sets the segment override prefix or [`Register::None`] if none. See also [`memory_segment()`].
	/// Use this method if the operand has kind [`OpKind::Memory`],
	/// [`OpKind::MemorySegSI`], [`OpKind::MemorySegESI`], [`OpKind::MemorySegRSI`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`memory_segment()`]: #method.memory_segment
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	/// [`OpKind::MemorySegSI`]: enum.OpKind.html#variant.MemorySegSI
	/// [`OpKind::MemorySegESI`]: enum.OpKind.html#variant.MemorySegESI
	/// [`OpKind::MemorySegESI`]: enum.OpKind.html#variant.MemorySegESI
	/// [`OpKind::MemorySegRSI`]: enum.OpKind.html#variant.MemorySegRSI
	///
	/// # Arguments
	///
	/// * `new_value`: Segment register prefix
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn set_segment_prefix(&mut self, new_value: Register) {
		let enc_value =
			if new_value == Register::None { 0 } else { (((new_value as u32) - (Register::ES as u32)) + 1) & MemoryFlags::SEGMENT_PREFIX_MASK };
		self.memory_flags = (((self.memory_flags as u32) & !(MemoryFlags::SEGMENT_PREFIX_MASK << MemoryFlags::SEGMENT_PREFIX_SHIFT))
			| (enc_value << MemoryFlags::SEGMENT_PREFIX_SHIFT)) as u16;
	}

	/// Gets the effective segment register used to reference the memory location.
	/// Use this method if the operand has kind [`OpKind::Memory`],
	/// [`OpKind::MemorySegSI`], [`OpKind::MemorySegESI`], [`OpKind::MemorySegRSI`]
	///
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	/// [`OpKind::MemorySegSI`]: enum.OpKind.html#variant.MemorySegSI
	/// [`OpKind::MemorySegESI`]: enum.OpKind.html#variant.MemorySegESI
	/// [`OpKind::MemorySegRSI`]: enum.OpKind.html#variant.MemorySegRSI
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn memory_segment(&self) -> Register {
		let seg_reg = self.segment_prefix();
		if seg_reg != Register::None {
			return seg_reg;
		}
		match self.memory_base() {
			Register::BP | Register::EBP | Register::ESP | Register::RBP | Register::RSP => Register::SS,
			_ => Register::DS,
		}
	}

	/// Gets the size of the memory displacement in bytes. Valid values are `0`, `1` (16/32/64-bit), `2` (16-bit), `4` (32-bit), `8` (64-bit).
	/// Note that the return value can be 1 and [`memory_displacement64()`] may still not fit in
	/// a signed byte if it's an EVEX encoded instruction.
	/// Use this method if the operand has kind [`OpKind::Memory`]
	///
	/// [`memory_displacement64()`]: #method.memory_displacement64
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	#[must_use]
	#[inline]
	pub fn memory_displ_size(&self) -> u32 {
		let size = ((self.memory_flags as u32) >> MemoryFlags::DISPL_SIZE_SHIFT) & MemoryFlags::DISPL_SIZE_MASK;
		if size <= 2 {
			size
		} else if size == 3 {
			4
		} else {
			8
		}
	}

	/// Sets the size of the memory displacement in bytes. Valid values are `0`, `1` (16/32/64-bit), `2` (16-bit), `4` (32-bit), `8` (64-bit).
	/// Note that the return value can be 1 and [`memory_displacement64()`] may still not fit in
	/// a signed byte if it's an EVEX encoded instruction.
	/// Use this method if the operand has kind [`OpKind::Memory`]
	///
	/// [`memory_displacement64()`]: #method.memory_displacement64
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	///
	/// # Arguments
	///
	/// * `new_value`: Displacement size
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn set_memory_displ_size(&mut self, new_value: u32) {
		let enc_value = match new_value {
			0 => 0,
			1 => 1,
			2 => 2,
			4 => 3,
			_ => 4,
		};
		self.memory_flags = (((self.memory_flags as u32) & !(MemoryFlags::DISPL_SIZE_MASK << MemoryFlags::DISPL_SIZE_SHIFT))
			| (enc_value << MemoryFlags::DISPL_SIZE_SHIFT)) as u16;
	}

	/// `true` if the data is broadcasted (EVEX instructions only)
	#[must_use]
	#[inline]
	pub fn is_broadcast(&self) -> bool {
		(self.memory_flags & (MemoryFlags::BROADCAST as u16)) != 0
	}

	/// Sets the is broadcast flag (EVEX instructions only)
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_is_broadcast(&mut self, new_value: bool) {
		if new_value {
			self.memory_flags |= MemoryFlags::BROADCAST as u16;
		} else {
			self.memory_flags &= !(MemoryFlags::BROADCAST as u16);
		}
	}

	/// Gets the size of the memory location that is referenced by the operand. See also [`is_broadcast()`].
	/// Use this method if the operand has kind [`OpKind::Memory`],
	/// [`OpKind::MemorySegSI`], [`OpKind::MemorySegESI`], [`OpKind::MemorySegRSI`],
	/// [`OpKind::MemoryESDI`], [`OpKind::MemoryESEDI`], [`OpKind::MemoryESRDI`]
	///
	/// [`is_broadcast()`]: #method.is_broadcast
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	/// [`OpKind::MemorySegSI`]: enum.OpKind.html#variant.MemorySegSI
	/// [`OpKind::MemorySegESI`]: enum.OpKind.html#variant.MemorySegESI
	/// [`OpKind::MemorySegRSI`]: enum.OpKind.html#variant.MemorySegRSI
	/// [`OpKind::MemoryESDI`]: enum.OpKind.html#variant.MemoryESDI
	/// [`OpKind::MemoryESEDI`]: enum.OpKind.html#variant.MemoryESEDI
	/// [`OpKind::MemoryESRDI`]: enum.OpKind.html#variant.MemoryESRDI
	#[must_use]
	#[inline]
	pub fn memory_size(&self) -> MemorySize {
		let mut index = self.code() as usize;
		if self.is_broadcast() {
			index += IcedConstants::CODE_ENUM_COUNT;
		}
		// SAFETY: SIZES.len() == count(Code)*2 and all Code values have usize values 0-max < count(Code) and max+1+count(Code) == SIZES.len()
		unsafe { *instruction_memory_sizes::SIZES.get_unchecked(index) }
	}

	/// Gets the index register scale value, valid values are `*1`, `*2`, `*4`, `*8`. Use this method if the operand has kind [`OpKind::Memory`]
	///
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	#[must_use]
	#[inline]
	pub fn memory_index_scale(&self) -> u32 {
		1 << (self.memory_flags as u32 & MemoryFlags::SCALE_MASK)
	}

	/// Sets the index register scale value, valid values are `*1`, `*2`, `*4`, `*8`. Use this method if the operand has kind [`OpKind::Memory`]
	///
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	///
	/// # Arguments
	///
	/// * `new_value`: New value (1, 2, 4 or 8)
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn set_memory_index_scale(&mut self, new_value: u32) {
		match new_value {
			1 => self.memory_flags &= !3,
			2 => self.memory_flags = (self.memory_flags & !(MemoryFlags::SCALE_MASK as u16)) | 1,
			4 => self.memory_flags = (self.memory_flags & !(MemoryFlags::SCALE_MASK as u16)) | 2,
			_ => {
				debug_assert_eq!(new_value, 8);
				self.memory_flags |= 3;
			}
		}
	}

	/// Gets the memory operand's displacement or the 32-bit absolute address if it's
	/// an `EIP` or `RIP` relative memory operand.
	/// Use this method if the operand has kind [`OpKind::Memory`]
	///
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	#[must_use]
	#[inline]
	#[deprecated(since = "1.11.0", note = "Use memory_displacement32() or memory_displacement64() instead")]
	pub fn memory_displacement(&self) -> u32 {
		self.memory_displacement32()
	}

	/// Gets the memory operand's displacement or the 32-bit absolute address if it's
	/// an `EIP` or `RIP` relative memory operand.
	/// Use this method if the operand has kind [`OpKind::Memory`]
	///
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	#[deprecated(since = "1.11.0", note = "Use set_memory_displacement32() or set_memory_displacement64() instead")]
	pub fn set_memory_displacement(&mut self, new_value: u32) {
		self.set_memory_displacement32(new_value);
	}

	/// Gets the memory operand's displacement or the 32-bit absolute address if it's
	/// an `EIP` or `RIP` relative memory operand.
	/// Use this method if the operand has kind [`OpKind::Memory`]
	///
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	#[must_use]
	#[inline]
	pub fn memory_displacement32(&self) -> u32 {
		self.mem_displ
	}

	/// Gets the memory operand's displacement or the 32-bit absolute address if it's
	/// an `EIP` or `RIP` relative memory operand.
	/// Use this method if the operand has kind [`OpKind::Memory`]
	///
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_memory_displacement32(&mut self, new_value: u32) {
		self.mem_displ = new_value;
		self.mem_displ_hi = 0;
	}

	/// Gets the memory operand's displacement or the 64-bit absolute address if it's
	/// an `EIP` or `RIP` relative memory operand.
	/// Use this method if the operand has kind [`OpKind::Memory`]
	///
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	#[must_use]
	#[inline]
	pub fn memory_displacement64(&self) -> u64 {
		(self.mem_displ as u64) | ((self.mem_displ_hi as u64) << 32)
	}

	/// Gets the memory operand's displacement or the 64-bit absolute address if it's
	/// an `EIP` or `RIP` relative memory operand.
	/// Use this method if the operand has kind [`OpKind::Memory`]
	///
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_memory_displacement64(&mut self, new_value: u64) {
		self.mem_displ = new_value as u32;
		self.mem_displ_hi = (new_value >> 32) as u32;
	}

	/// Gets an operand's immediate value, or `None` if the operand is not immediate
	///
	/// # Errors
	///
	/// Fails if `operand` is invalid or not an immediate operand
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn try_immediate(&self, operand: u32) -> Result<u64, IcedError> {
		Ok(match self.try_op_kind(operand)? {
			OpKind::Immediate8 => self.immediate8() as u64,
			OpKind::Immediate8_2nd => self.immediate8_2nd() as u64,
			OpKind::Immediate16 => self.immediate16() as u64,
			OpKind::Immediate32 => self.immediate32() as u64,
			OpKind::Immediate64 => self.immediate64(),
			OpKind::Immediate8to16 => self.immediate8to16() as u64,
			OpKind::Immediate8to32 => self.immediate8to32() as u64,
			OpKind::Immediate8to64 => self.immediate8to64() as u64,
			OpKind::Immediate32to64 => self.immediate32to64() as u64,
			_ => return Err(IcedError::new("Not an immediate operand")),
		})
	}

	/// Gets an operand's immediate value
	///
	/// # Panics
	///
	/// Panics if `operand` is invalid or not immediate.
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	#[must_use]
	#[deprecated(since = "1.11.0", note = "This method can panic, use try_immediate() instead")]
	#[allow(clippy::unwrap_used)]
	#[inline]
	pub fn immediate(&self, operand: u32) -> u64 {
		self.try_immediate(operand).unwrap()
	}

	/// Sets an operand's immediate value
	///
	/// # Panics
	///
	/// Panics if `operand` is invalid or if it's not an immediate operand
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	/// * `new_value`: Immediate
	#[inline]
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_set_immediate_i32() instead")]
	#[allow(clippy::unwrap_used)]
	pub fn set_immediate_i32(&mut self, operand: u32, new_value: i32) {
		self.try_set_immediate_i32(operand, new_value).unwrap();
	}

	/// Sets an operand's immediate value
	///
	/// # Errors
	///
	/// Fails if `operand` is invalid or if it's not an immediate operand
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	/// * `new_value`: Immediate
	#[inline]
	pub fn try_set_immediate_i32(&mut self, operand: u32, new_value: i32) -> Result<(), IcedError> {
		self.try_set_immediate_u64(operand, new_value as u64)
	}

	/// Sets an operand's immediate value
	///
	/// # Panics
	///
	/// Panics if `operand` is invalid or if it's not an immediate operand
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	/// * `new_value`: Immediate
	#[inline]
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_set_immediate_u32() instead")]
	#[allow(clippy::unwrap_used)]
	pub fn set_immediate_u32(&mut self, operand: u32, new_value: u32) {
		self.try_set_immediate_u32(operand, new_value).unwrap();
	}

	/// Sets an operand's immediate value
	///
	/// # Errors
	///
	/// Fails if `operand` is invalid or if it's not an immediate operand
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	/// * `new_value`: Immediate
	#[inline]
	pub fn try_set_immediate_u32(&mut self, operand: u32, new_value: u32) -> Result<(), IcedError> {
		self.try_set_immediate_u64(operand, new_value as u64)
	}

	/// Sets an operand's immediate value
	///
	/// # Panics
	///
	/// Panics if `operand` is invalid or if it's not an immediate operand
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	/// * `new_value`: Immediate
	#[inline]
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_set_immediate_i64() instead")]
	#[allow(clippy::unwrap_used)]
	pub fn set_immediate_i64(&mut self, operand: u32, new_value: i64) {
		self.try_set_immediate_i64(operand, new_value).unwrap()
	}

	/// Sets an operand's immediate value
	///
	/// # Errors
	///
	/// Fails if `operand` is invalid or if it's not an immediate operand
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	/// * `new_value`: Immediate
	#[inline]
	pub fn try_set_immediate_i64(&mut self, operand: u32, new_value: i64) -> Result<(), IcedError> {
		self.try_set_immediate_u64(operand, new_value as u64)
	}

	/// Sets an operand's immediate value
	///
	/// # Panics
	///
	/// Panics if `operand` is invalid or if it's not an immediate operand
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	/// * `new_value`: Immediate
	#[inline]
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_set_immediate_u64() instead")]
	#[allow(clippy::unwrap_used)]
	pub fn set_immediate_u64(&mut self, operand: u32, new_value: u64) {
		self.try_set_immediate_u64(operand, new_value).unwrap()
	}

	/// Sets an operand's immediate value
	///
	/// # Errors
	///
	/// Fails if `operand` is invalid or if it's not an immediate operand
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	/// * `new_value`: Immediate
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn try_set_immediate_u64(&mut self, operand: u32, new_value: u64) -> Result<(), IcedError> {
		match self.try_op_kind(operand)? {
			OpKind::Immediate8 | OpKind::Immediate8to16 | OpKind::Immediate8to32 | OpKind::Immediate8to64 => self.immediate = new_value as u8 as u32,
			OpKind::Immediate8_2nd => self.mem_displ = new_value as u8 as u32,
			OpKind::Immediate16 => self.immediate = new_value as u16 as u32,
			OpKind::Immediate32to64 | OpKind::Immediate32 => self.immediate = new_value as u32,
			OpKind::Immediate64 => self.set_immediate64(new_value),
			_ => return Err(IcedError::new("Not an immediate operand")),
		}
		Ok(())
	}

	/// Gets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate8`]
	///
	/// [`OpKind::Immediate8`]: enum.OpKind.html#variant.Immediate8
	#[must_use]
	#[inline]
	pub fn immediate8(&self) -> u8 {
		self.immediate as u8
	}

	/// Sets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate8`]
	///
	/// [`OpKind::Immediate8`]: enum.OpKind.html#variant.Immediate8
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_immediate8(&mut self, new_value: u8) {
		self.immediate = new_value as u32;
	}

	/// Gets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate8_2nd`]
	///
	/// [`OpKind::Immediate8_2nd`]: enum.OpKind.html#variant.Immediate8_2nd
	#[must_use]
	#[inline]
	pub fn immediate8_2nd(&self) -> u8 {
		self.mem_displ as u8
	}

	/// Sets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate8_2nd`]
	///
	/// [`OpKind::Immediate8_2nd`]: enum.OpKind.html#variant.Immediate8_2nd
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_immediate8_2nd(&mut self, new_value: u8) {
		self.mem_displ = new_value as u32;
	}

	/// Gets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate16`]
	///
	/// [`OpKind::Immediate16`]: enum.OpKind.html#variant.Immediate16
	#[must_use]
	#[inline]
	pub fn immediate16(&self) -> u16 {
		self.immediate as u16
	}

	/// Sets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate16`]
	///
	/// [`OpKind::Immediate16`]: enum.OpKind.html#variant.Immediate16
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_immediate16(&mut self, new_value: u16) {
		self.immediate = new_value as u32;
	}

	/// Gets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate32`]
	///
	/// [`OpKind::Immediate32`]: enum.OpKind.html#variant.Immediate32
	#[must_use]
	#[inline]
	pub fn immediate32(&self) -> u32 {
		self.immediate
	}

	/// Sets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate32`]
	///
	/// [`OpKind::Immediate32`]: enum.OpKind.html#variant.Immediate32
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_immediate32(&mut self, new_value: u32) {
		self.immediate = new_value;
	}

	/// Gets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate64`]
	///
	/// [`OpKind::Immediate64`]: enum.OpKind.html#variant.Immediate64
	#[must_use]
	#[inline]
	pub fn immediate64(&self) -> u64 {
		((self.mem_displ as u64) << 32) | (self.immediate as u64)
	}

	/// Sets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate64`]
	///
	/// [`OpKind::Immediate64`]: enum.OpKind.html#variant.Immediate64
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_immediate64(&mut self, new_value: u64) {
		self.immediate = new_value as u32;
		self.mem_displ = (new_value >> 32) as u32;
	}

	/// Gets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate8to16`]
	///
	/// [`OpKind::Immediate8to16`]: enum.OpKind.html#variant.Immediate8to16
	#[must_use]
	#[inline]
	pub fn immediate8to16(&self) -> i16 {
		self.immediate as i8 as i16
	}

	/// Sets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate8to16`]
	///
	/// [`OpKind::Immediate8to16`]: enum.OpKind.html#variant.Immediate8to16
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_immediate8to16(&mut self, new_value: i16) {
		self.immediate = new_value as i8 as u32;
	}

	/// Gets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate8to32`]
	///
	/// [`OpKind::Immediate8to32`]: enum.OpKind.html#variant.Immediate8to32
	#[must_use]
	#[inline]
	pub fn immediate8to32(&self) -> i32 {
		self.immediate as i8 as i32
	}

	/// Sets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate8to32`]
	///
	/// [`OpKind::Immediate8to32`]: enum.OpKind.html#variant.Immediate8to32
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_immediate8to32(&mut self, new_value: i32) {
		self.immediate = new_value as i8 as u32;
	}

	/// Gets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate8to64`]
	///
	/// [`OpKind::Immediate8to64`]: enum.OpKind.html#variant.Immediate8to64
	#[must_use]
	#[inline]
	pub fn immediate8to64(&self) -> i64 {
		self.immediate as i8 as i64
	}

	/// Sets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate8to64`]
	///
	/// [`OpKind::Immediate8to64`]: enum.OpKind.html#variant.Immediate8to64
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_immediate8to64(&mut self, new_value: i64) {
		self.immediate = new_value as i8 as u32;
	}

	/// Gets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate32to64`]
	///
	/// [`OpKind::Immediate32to64`]: enum.OpKind.html#variant.Immediate32to64
	#[must_use]
	#[inline]
	pub fn immediate32to64(&self) -> i64 {
		self.immediate as i32 as i64
	}

	/// Sets the operand's immediate value. Use this method if the operand has kind [`OpKind::Immediate32to64`]
	///
	/// [`OpKind::Immediate32to64`]: enum.OpKind.html#variant.Immediate32to64
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_immediate32to64(&mut self, new_value: i64) {
		self.immediate = new_value as u32;
	}

	/// [`OpKind::Memory64`] is deprecated, this method does nothing now.
	///
	/// [`OpKind::Memory64`]: enum.OpKind.html#variant.Memory64
	#[must_use]
	#[allow(clippy::unused_self)]
	#[inline]
	#[deprecated(since = "1.11.0", note = "OpKind::Memory64 is deprecated, this method does nothing now. Use memory_displacement64() instead.")]
	pub fn memory_address64(&self) -> u64 {
		0
	}

	/// [`OpKind::Memory64`] is deprecated, this method does nothing now.
	///
	/// [`OpKind::Memory64`]: enum.OpKind.html#variant.Memory64
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[allow(clippy::unused_self)]
	#[inline]
	#[deprecated(since = "1.11.0", note = "OpKind::Memory64 is deprecated, this method does nothing now. Use set_memory_displacement64() instead.")]
	pub fn set_memory_address64(&mut self, _new_value: u64) {}

	/// Gets the operand's branch target. Use this method if the operand has kind [`OpKind::NearBranch16`]
	///
	/// [`OpKind::NearBranch16`]: enum.OpKind.html#variant.NearBranch16
	#[must_use]
	#[inline]
	pub fn near_branch16(&self) -> u16 {
		self.immediate as u16
	}

	/// Sets the operand's branch target. Use this method if the operand has kind [`OpKind::NearBranch16`]
	///
	/// [`OpKind::NearBranch16`]: enum.OpKind.html#variant.NearBranch16
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_near_branch16(&mut self, new_value: u16) {
		self.immediate = new_value as u32;
	}

	/// Gets the operand's branch target. Use this method if the operand has kind [`OpKind::NearBranch32`]
	///
	/// [`OpKind::NearBranch32`]: enum.OpKind.html#variant.NearBranch32
	#[must_use]
	#[inline]
	pub fn near_branch32(&self) -> u32 {
		self.immediate
	}

	/// Sets the operand's branch target. Use this method if the operand has kind [`OpKind::NearBranch32`]
	///
	/// [`OpKind::NearBranch32`]: enum.OpKind.html#variant.NearBranch32
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_near_branch32(&mut self, new_value: u32) {
		self.immediate = new_value;
	}

	/// Gets the operand's branch target. Use this method if the operand has kind [`OpKind::NearBranch64`]
	///
	/// [`OpKind::NearBranch64`]: enum.OpKind.html#variant.NearBranch64
	#[must_use]
	#[inline]
	pub fn near_branch64(&self) -> u64 {
		((self.mem_displ as u64) << 32) | self.immediate as u64
	}

	/// Sets the operand's branch target. Use this method if the operand has kind [`OpKind::NearBranch64`]
	///
	/// [`OpKind::NearBranch64`]: enum.OpKind.html#variant.NearBranch64
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_near_branch64(&mut self, new_value: u64) {
		self.immediate = new_value as u32;
		self.mem_displ = (new_value >> 32) as u32;
	}

	/// Gets the near branch target if it's a `CALL`/`JMP`/`Jcc` near branch instruction
	/// (i.e., if [`op0_kind()`] is [`OpKind::NearBranch16`], [`OpKind::NearBranch32`] or [`OpKind::NearBranch64`])
	///
	/// [`op0_kind()`]: #method.op0_kind
	/// [`OpKind::NearBranch16`]: enum.OpKind.html#variant.NearBranch16
	/// [`OpKind::NearBranch32`]: enum.OpKind.html#variant.NearBranch32
	/// [`OpKind::NearBranch64`]: enum.OpKind.html#variant.NearBranch64
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn near_branch_target(&self) -> u64 {
		match self.op0_kind() {
			OpKind::NearBranch16 => self.near_branch16() as u64,
			OpKind::NearBranch32 => self.near_branch32() as u64,
			OpKind::NearBranch64 => self.near_branch64(),
			_ => 0,
		}
	}

	/// Gets the operand's branch target. Use this method if the operand has kind [`OpKind::FarBranch16`]
	///
	/// [`OpKind::FarBranch16`]: enum.OpKind.html#variant.FarBranch16
	#[must_use]
	#[inline]
	pub fn far_branch16(&self) -> u16 {
		self.immediate as u16
	}

	/// Sets the operand's branch target. Use this method if the operand has kind [`OpKind::FarBranch16`]
	///
	/// [`OpKind::FarBranch16`]: enum.OpKind.html#variant.FarBranch16
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_far_branch16(&mut self, new_value: u16) {
		self.immediate = new_value as u32;
	}

	/// Gets the operand's branch target. Use this method if the operand has kind [`OpKind::FarBranch32`]
	///
	/// [`OpKind::FarBranch32`]: enum.OpKind.html#variant.FarBranch32
	#[must_use]
	#[inline]
	pub fn far_branch32(&self) -> u32 {
		self.immediate
	}

	/// Sets the operand's branch target. Use this method if the operand has kind [`OpKind::FarBranch32`]
	///
	/// [`OpKind::FarBranch32`]: enum.OpKind.html#variant.FarBranch32
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_far_branch32(&mut self, new_value: u32) {
		self.immediate = new_value;
	}

	/// Gets the operand's branch target selector. Use this method if the operand has kind [`OpKind::FarBranch16`] or [`OpKind::FarBranch32`]
	///
	/// [`OpKind::FarBranch16`]: enum.OpKind.html#variant.FarBranch16
	/// [`OpKind::FarBranch32`]: enum.OpKind.html#variant.FarBranch32
	#[must_use]
	#[inline]
	pub fn far_branch_selector(&self) -> u16 {
		self.mem_displ as u16
	}

	/// Sets the operand's branch target selector. Use this method if the operand has kind [`OpKind::FarBranch16`] or [`OpKind::FarBranch32`]
	///
	/// [`OpKind::FarBranch16`]: enum.OpKind.html#variant.FarBranch16
	/// [`OpKind::FarBranch32`]: enum.OpKind.html#variant.FarBranch32
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_far_branch_selector(&mut self, new_value: u16) {
		self.mem_displ = new_value as u32;
	}

	/// Gets the memory operand's base register or [`Register::None`] if none. Use this method if the operand has kind [`OpKind::Memory`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	#[must_use]
	#[inline]
	pub fn memory_base(&self) -> Register {
		unsafe { mem::transmute(self.mem_base_reg) }
	}

	/// Sets the memory operand's base register or [`Register::None`] if none. Use this method if the operand has kind [`OpKind::Memory`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_memory_base(&mut self, new_value: Register) {
		self.mem_base_reg = new_value as u8;
	}

	/// Gets the memory operand's index register or [`Register::None`] if none. Use this method if the operand has kind [`OpKind::Memory`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	#[must_use]
	#[inline]
	pub fn memory_index(&self) -> Register {
		unsafe { mem::transmute(self.mem_index_reg) }
	}

	/// Sets the memory operand's index register or [`Register::None`] if none. Use this method if the operand has kind [`OpKind::Memory`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`OpKind::Memory`]: enum.OpKind.html#variant.Memory
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_memory_index(&mut self, new_value: Register) {
		self.mem_index_reg = new_value as u8;
	}

	/// Gets operand #0's register value. Use this method if operand #0 ([`op0_kind()`]) has kind [`OpKind::Register`], see [`op_count()`] and [`try_op_register()`]
	///
	/// [`op0_kind()`]: #method.op0_kind
	/// [`op_count()`]: #method.op_count
	/// [`try_op_register()`]: #method.try_op_register
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`OpKind::Register`]: enum.OpKind.html#variant.Register
	#[must_use]
	#[inline]
	pub fn op0_register(&self) -> Register {
		unsafe { mem::transmute(self.reg0) }
	}

	/// Sets operand #0's register value. Use this method if operand #0 ([`op0_kind()`]) has kind [`OpKind::Register`], see [`op_count()`] and [`try_op_register()`]
	///
	/// [`op0_kind()`]: #method.op0_kind
	/// [`op_count()`]: #method.op_count
	/// [`try_op_register()`]: #method.try_op_register
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`OpKind::Register`]: enum.OpKind.html#variant.Register
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_op0_register(&mut self, new_value: Register) {
		self.reg0 = new_value as u8;
	}

	/// Gets operand #1's register value. Use this method if operand #1 ([`op0_kind()`]) has kind [`OpKind::Register`], see [`op_count()`] and [`try_op_register()`]
	///
	/// [`op0_kind()`]: #method.op0_kind
	/// [`op_count()`]: #method.op_count
	/// [`try_op_register()`]: #method.try_op_register
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`OpKind::Register`]: enum.OpKind.html#variant.Register
	#[must_use]
	#[inline]
	pub fn op1_register(&self) -> Register {
		unsafe { mem::transmute(self.reg1) }
	}

	/// Sets operand #1's register value. Use this method if operand #1 ([`op0_kind()`]) has kind [`OpKind::Register`], see [`op_count()`] and [`try_op_register()`]
	///
	/// [`op0_kind()`]: #method.op0_kind
	/// [`op_count()`]: #method.op_count
	/// [`try_op_register()`]: #method.try_op_register
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`OpKind::Register`]: enum.OpKind.html#variant.Register
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_op1_register(&mut self, new_value: Register) {
		self.reg1 = new_value as u8;
	}

	/// Gets operand #2's register value. Use this method if operand #2 ([`op0_kind()`]) has kind [`OpKind::Register`], see [`op_count()`] and [`try_op_register()`]
	///
	/// [`op0_kind()`]: #method.op0_kind
	/// [`op_count()`]: #method.op_count
	/// [`try_op_register()`]: #method.try_op_register
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`OpKind::Register`]: enum.OpKind.html#variant.Register
	#[must_use]
	#[inline]
	pub fn op2_register(&self) -> Register {
		unsafe { mem::transmute(self.reg2) }
	}

	/// Sets operand #2's register value. Use this method if operand #2 ([`op0_kind()`]) has kind [`OpKind::Register`], see [`op_count()`] and [`try_op_register()`]
	///
	/// [`op0_kind()`]: #method.op0_kind
	/// [`op_count()`]: #method.op_count
	/// [`try_op_register()`]: #method.try_op_register
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`OpKind::Register`]: enum.OpKind.html#variant.Register
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_op2_register(&mut self, new_value: Register) {
		self.reg2 = new_value as u8;
	}

	/// Gets operand #3's register value. Use this method if operand #3 ([`op0_kind()`]) has kind [`OpKind::Register`], see [`op_count()`] and [`try_op_register()`]
	///
	/// [`op0_kind()`]: #method.op0_kind
	/// [`op_count()`]: #method.op_count
	/// [`try_op_register()`]: #method.try_op_register
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`OpKind::Register`]: enum.OpKind.html#variant.Register
	#[must_use]
	#[inline]
	pub fn op3_register(&self) -> Register {
		unsafe { mem::transmute(self.reg3) }
	}

	/// Sets operand #3's register value. Use this method if operand #3 ([`op0_kind()`]) has kind [`OpKind::Register`], see [`op_count()`] and [`try_op_register()`]
	///
	/// [`op0_kind()`]: #method.op0_kind
	/// [`op_count()`]: #method.op_count
	/// [`try_op_register()`]: #method.try_op_register
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`OpKind::Register`]: enum.OpKind.html#variant.Register
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_op3_register(&mut self, new_value: Register) {
		self.reg3 = new_value as u8;
	}

	/// Gets operand #4's register value. Use this method if operand #4 ([`op0_kind()`]) has kind [`OpKind::Register`], see [`op_count()`] and [`try_op_register()`]
	///
	/// [`op0_kind()`]: #method.op0_kind
	/// [`op_count()`]: #method.op_count
	/// [`try_op_register()`]: #method.try_op_register
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`OpKind::Register`]: enum.OpKind.html#variant.Register
	#[allow(clippy::unused_self)]
	#[must_use]
	#[inline]
	pub fn op4_register(&self) -> Register {
		Register::None
	}

	/// Sets operand #4's register value. Use this method if operand #4 ([`op0_kind()`]) has kind [`OpKind::Register`], see [`op_count()`] and [`try_op_register()`]
	///
	/// [`op0_kind()`]: #method.op0_kind
	/// [`op_count()`]: #method.op_count
	/// [`try_op_register()`]: #method.try_op_register
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`OpKind::Register`]: enum.OpKind.html#variant.Register
	///
	/// # Panics
	///
	/// Panics if `new_value` is invalid
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_set_op4_register() instead")]
	#[allow(clippy::unwrap_used)]
	pub fn set_op4_register(&mut self, new_value: Register) {
		self.try_set_op4_register(new_value).unwrap();
	}

	/// Sets operand #4's register value. Use this method if operand #4 ([`op0_kind()`]) has kind [`OpKind::Register`], see [`op_count()`] and [`try_op_register()`]
	///
	/// [`op0_kind()`]: #method.op0_kind
	/// [`op_count()`]: #method.op_count
	/// [`try_op_register()`]: #method.try_op_register
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`OpKind::Register`]: enum.OpKind.html#variant.Register
	///
	/// # Errors
	///
	/// Fails if `new_value` is invalid
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[allow(clippy::unused_self)]
	#[inline]
	pub fn try_set_op4_register(&mut self, new_value: Register) -> Result<(), IcedError> {
		if new_value != Register::None {
			Err(IcedError::new("Invalid register"))
		} else {
			Ok(())
		}
	}

	/// Gets the operand's register value. Use this method if the operand has kind [`OpKind::Register`]
	///
	/// [`OpKind::Register`]: enum.OpKind.html#variant.Register
	///
	/// # Panics
	///
	/// Panics if `operand` is invalid
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // add [rax],ebx
	/// let bytes = b"\x01\x18";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	/// let instr = decoder.decode();
	///
	/// assert_eq!(instr.op_count(), 2);
	/// assert_eq!(instr.op_kind(0), OpKind::Memory);
	/// assert_eq!(instr.op_kind(1), OpKind::Register);
	/// assert_eq!(instr.op_register(1), Register::EBX);
	/// ```
	#[must_use]
	#[inline]
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_op_register() instead")]
	#[allow(clippy::unwrap_used)]
	pub fn op_register(&self, operand: u32) -> Register {
		self.try_op_register(operand).unwrap()
	}

	/// Gets the operand's register value. Use this method if the operand has kind [`OpKind::Register`]
	///
	/// [`OpKind::Register`]: enum.OpKind.html#variant.Register
	///
	/// # Errors
	///
	/// Fails if `operand` is invalid
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // add [rax],ebx
	/// let bytes = b"\x01\x18";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	/// let instr = decoder.decode();
	///
	/// assert_eq!(instr.op_count(), 2);
	/// assert_eq!(instr.op_kind(0), OpKind::Memory);
	/// assert_eq!(instr.op_kind(1), OpKind::Register);
	/// assert_eq!(instr.try_op_register(1).unwrap(), Register::EBX);
	/// ```
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn try_op_register(&self, operand: u32) -> Result<Register, IcedError> {
		const_assert_eq!(IcedConstants::MAX_OP_COUNT, 5);
		match operand {
			0 => Ok(self.op0_register()),
			1 => Ok(self.op1_register()),
			2 => Ok(self.op2_register()),
			3 => Ok(self.op3_register()),
			4 => Ok(self.op4_register()),
			_ => Err(IcedError::new("Invalid operand")),
		}
	}

	/// Sets the operand's register value. Use this method if the operand has kind [`OpKind::Register`]
	///
	/// [`OpKind::Register`]: enum.OpKind.html#variant.Register
	///
	/// # Panics
	///
	/// - Panics if `operand` is invalid
	/// - Panics if `new_value` is invalid
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	/// * `new_value`: New value
	#[inline]
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_set_op_register() instead")]
	#[allow(clippy::unwrap_used)]
	pub fn set_op_register(&mut self, operand: u32, new_value: Register) {
		self.try_set_op_register(operand, new_value).unwrap();
	}

	/// Sets the operand's register value. Use this method if the operand has kind [`OpKind::Register`]
	///
	/// [`OpKind::Register`]: enum.OpKind.html#variant.Register
	///
	/// # Errors
	///
	/// - Fails if `operand` is invalid
	/// - Fails if `new_value` is invalid
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4
	/// * `new_value`: New value
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn try_set_op_register(&mut self, operand: u32, new_value: Register) -> Result<(), IcedError> {
		const_assert_eq!(IcedConstants::MAX_OP_COUNT, 5);
		match operand {
			0 => self.set_op0_register(new_value),
			1 => self.set_op1_register(new_value),
			2 => self.set_op2_register(new_value),
			3 => self.set_op3_register(new_value),
			4 => return self.try_set_op4_register(new_value),
			_ => return Err(IcedError::new("Invalid operand")),
		}
		Ok(())
	}

	/// Gets the opmask register ([`Register::K1`] - [`Register::K7`]) or [`Register::None`] if none
	///
	/// [`Register::K1`]: enum.Register.html#variant.K1
	/// [`Register::K7`]: enum.Register.html#variant.K7
	/// [`Register::None`]: enum.Register.html#variant.None
	#[must_use]
	#[inline]
	pub fn op_mask(&self) -> Register {
		let r = (self.code_flags >> CodeFlags::OP_MASK_SHIFT) & CodeFlags::OP_MASK_MASK;
		if r == 0 {
			Register::None
		} else {
			unsafe { mem::transmute((r + Register::K0 as u32) as u8) }
		}
	}

	/// Sets the opmask register ([`Register::K1`] - [`Register::K7`]) or [`Register::None`] if none
	///
	/// [`Register::K1`]: enum.Register.html#variant.K1
	/// [`Register::K7`]: enum.Register.html#variant.K7
	/// [`Register::None`]: enum.Register.html#variant.None
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn set_op_mask(&mut self, new_value: Register) {
		let r = if new_value == Register::None { 0 } else { (new_value as u32 - Register::K0 as u32) & CodeFlags::OP_MASK_MASK };
		self.code_flags = (self.code_flags & !(CodeFlags::OP_MASK_MASK << CodeFlags::OP_MASK_SHIFT)) | (r << CodeFlags::OP_MASK_SHIFT);
	}

	/// Checks if there's an opmask register ([`op_mask()`])
	///
	/// [`op_mask()`]: #method.op_mask
	#[must_use]
	#[inline]
	pub fn has_op_mask(&self) -> bool {
		(self.code_flags & (CodeFlags::OP_MASK_MASK << CodeFlags::OP_MASK_SHIFT)) != 0
	}

	/// `true` if zeroing-masking, `false` if merging-masking.
	/// Only used by most EVEX encoded instructions that use opmask registers.
	#[must_use]
	#[inline]
	pub fn zeroing_masking(&self) -> bool {
		(self.code_flags & CodeFlags::ZEROING_MASKING) != 0
	}

	/// `true` if zeroing-masking, `false` if merging-masking.
	/// Only used by most EVEX encoded instructions that use opmask registers.
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_zeroing_masking(&mut self, new_value: bool) {
		if new_value {
			self.code_flags |= CodeFlags::ZEROING_MASKING;
		} else {
			self.code_flags &= !CodeFlags::ZEROING_MASKING;
		}
	}

	/// `true` if merging-masking, `false` if zeroing-masking.
	/// Only used by most EVEX encoded instructions that use opmask registers.
	#[must_use]
	#[inline]
	pub fn merging_masking(&self) -> bool {
		(self.code_flags & CodeFlags::ZEROING_MASKING) == 0
	}

	/// `true` if merging-masking, `false` if zeroing-masking.
	/// Only used by most EVEX encoded instructions that use opmask registers.
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_merging_masking(&mut self, new_value: bool) {
		if new_value {
			self.code_flags &= !CodeFlags::ZEROING_MASKING;
		} else {
			self.code_flags |= CodeFlags::ZEROING_MASKING;
		}
	}

	/// Gets the rounding control (SAE is implied but [`suppress_all_exceptions()`] still returns `false`)
	/// or [`RoundingControl::None`] if the instruction doesn't use it.
	///
	/// [`suppress_all_exceptions()`]: #method.suppress_all_exceptions
	/// [`RoundingControl::None`]: enum.RoundingControl.html#variant.None
	#[must_use]
	#[inline]
	pub fn rounding_control(&self) -> RoundingControl {
		unsafe { mem::transmute(((self.code_flags >> CodeFlags::ROUNDING_CONTROL_SHIFT) & CodeFlags::ROUNDING_CONTROL_MASK) as u8) }
	}

	/// Sets the rounding control (SAE is implied but [`suppress_all_exceptions()`] still returns `false`)
	/// or [`RoundingControl::None`] if the instruction doesn't use it.
	///
	/// [`suppress_all_exceptions()`]: #method.suppress_all_exceptions
	/// [`RoundingControl::None`]: enum.RoundingControl.html#variant.None
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_rounding_control(&mut self, new_value: RoundingControl) {
		self.code_flags = (self.code_flags & !(CodeFlags::ROUNDING_CONTROL_MASK << CodeFlags::ROUNDING_CONTROL_SHIFT))
			| ((new_value as u32 & CodeFlags::ROUNDING_CONTROL_MASK) << CodeFlags::ROUNDING_CONTROL_SHIFT);
	}

	/// Gets the number of elements in a `db`/`dw`/`dd`/`dq` directive.
	/// Can only be called if [`code()`] is [`Code::DeclareByte`], [`Code::DeclareWord`], [`Code::DeclareDword`], [`Code::DeclareQword`]
	///
	/// [`code()`]: #method.code
	/// [`Code::DeclareByte`]: enum.Code.html#variant.DeclareByte
	/// [`Code::DeclareWord`]: enum.Code.html#variant.DeclareWord
	/// [`Code::DeclareDword`]: enum.Code.html#variant.DeclareDword
	/// [`Code::DeclareQword`]: enum.Code.html#variant.DeclareQword
	#[must_use]
	#[inline]
	pub fn declare_data_len(&self) -> usize {
		(((self.op_kind_flags >> OpKindFlags::DATA_LENGTH_SHIFT) & OpKindFlags::DATA_LENGTH_MASK) + 1) as usize
	}

	/// Sets the number of elements in a `db`/`dw`/`dd`/`dq` directive.
	/// Can only be called if [`code()`] is [`Code::DeclareByte`], [`Code::DeclareWord`], [`Code::DeclareDword`], [`Code::DeclareQword`]
	///
	/// [`code()`]: #method.code
	/// [`Code::DeclareByte`]: enum.Code.html#variant.DeclareByte
	/// [`Code::DeclareWord`]: enum.Code.html#variant.DeclareWord
	/// [`Code::DeclareDword`]: enum.Code.html#variant.DeclareDword
	/// [`Code::DeclareQword`]: enum.Code.html#variant.DeclareQword
	///
	/// # Arguments
	///
	/// * `new_value`: New value: `db`: 1-16; `dw`: 1-8; `dd`: 1-4; `dq`: 1-2
	#[inline]
	pub fn set_declare_data_len(&mut self, new_value: usize) {
		self.op_kind_flags = (self.op_kind_flags & !(OpKindFlags::DATA_LENGTH_MASK << OpKindFlags::DATA_LENGTH_SHIFT))
			| ((((new_value as u32) - 1) & OpKindFlags::DATA_LENGTH_MASK) << OpKindFlags::DATA_LENGTH_SHIFT);
	}

	/// Sets a new `db` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareByte`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareByte`]: enum.Code.html#variant.DeclareByte
	///
	/// # Panics
	///
	/// - Panics if `index` is invalid
	/// - Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `index`: Index (0-15)
	/// * `new_value`: New value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_set_declare_byte_value_i8() instead")]
	#[allow(clippy::unwrap_used)]
	#[inline]
	pub fn set_declare_byte_value_i8(&mut self, index: usize, new_value: i8) {
		self.try_set_declare_byte_value_i8(index, new_value).unwrap();
	}

	/// Sets a new `db` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareByte`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareByte`]: enum.Code.html#variant.DeclareByte
	///
	/// # Errors
	///
	/// - Fails if `index` is invalid
	/// - Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `index`: Index (0-15)
	/// * `new_value`: New value
	#[inline]
	pub fn try_set_declare_byte_value_i8(&mut self, index: usize, new_value: i8) -> Result<(), IcedError> {
		self.try_set_declare_byte_value(index, new_value as u8)
	}

	/// Sets a new `db` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareByte`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareByte`]: enum.Code.html#variant.DeclareByte
	///
	/// # Panics
	///
	/// - Panics if `index` is invalid
	/// - Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `index`: Index (0-15)
	/// * `new_value`: New value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_set_declare_byte_value() instead")]
	#[allow(clippy::unwrap_used)]
	#[inline]
	pub fn set_declare_byte_value(&mut self, index: usize, new_value: u8) {
		self.try_set_declare_byte_value(index, new_value).unwrap();
	}

	/// Sets a new `db` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareByte`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareByte`]: enum.Code.html#variant.DeclareByte
	///
	/// # Errors
	///
	/// - Fails if `index` is invalid
	/// - Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `index`: Index (0-15)
	/// * `new_value`: New value
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn try_set_declare_byte_value(&mut self, index: usize, new_value: u8) -> Result<(), IcedError> {
		if cfg!(feature = "db") {
			match index {
				0 => self.reg0 = new_value,
				1 => self.reg1 = new_value,
				2 => self.reg2 = new_value,
				3 => self.reg3 = new_value,
				4 => self.immediate = (self.immediate & 0xFFFF_FF00) | new_value as u32,
				5 => self.immediate = (self.immediate & 0xFFFF_00FF) | ((new_value as u32) << 8),
				6 => self.immediate = (self.immediate & 0xFF00_FFFF) | ((new_value as u32) << 16),
				7 => self.immediate = (self.immediate & 0x00FF_FFFF) | ((new_value as u32) << 24),
				8 => self.mem_displ = (self.mem_displ & 0xFFFF_FF00) | new_value as u32,
				9 => self.mem_displ = (self.mem_displ & 0xFFFF_00FF) | ((new_value as u32) << 8),
				10 => self.mem_displ = (self.mem_displ & 0xFF00_FFFF) | ((new_value as u32) << 16),
				11 => self.mem_displ = (self.mem_displ & 0x00FF_FFFF) | ((new_value as u32) << 24),
				12 => self.mem_base_reg = new_value,
				13 => self.mem_index_reg = new_value,
				14 => self.op_kind_flags = (self.op_kind_flags & 0xFFFF_FF00) | new_value as u32,
				15 => self.op_kind_flags = (self.op_kind_flags & 0xFFFF_00FF) | ((new_value as u32) << 8),
				_ => return Err(IcedError::new("Invalid index")),
			}
			Ok(())
		} else {
			Err(IcedError::new("`db` feature wasn't enabled"))
		}
	}

	/// Gets a `db` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareByte`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareByte`]: enum.Code.html#variant.DeclareByte
	///
	/// # Panics
	///
	/// Panics if `index` is invalid
	///
	/// # Arguments
	///
	/// * `index`: Index (0-15)
	#[must_use]
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_get_declare_byte_value() instead")]
	#[allow(clippy::unwrap_used)]
	#[inline]
	pub fn get_declare_byte_value(&self, index: usize) -> u8 {
		self.try_get_declare_byte_value(index).unwrap()
	}

	/// Gets a `db` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareByte`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareByte`]: enum.Code.html#variant.DeclareByte
	///
	/// # Errors
	///
	/// Fails if `index` is invalid
	///
	/// # Arguments
	///
	/// * `index`: Index (0-15)
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn try_get_declare_byte_value(&self, index: usize) -> Result<u8, IcedError> {
		Ok(match index {
			0 => self.reg0,
			1 => self.reg1,
			2 => self.reg2,
			3 => self.reg3,
			4 => self.immediate as u8,
			5 => (self.immediate >> 8) as u8,
			6 => (self.immediate >> 16) as u8,
			7 => (self.immediate >> 24) as u8,
			8 => self.mem_displ as u8,
			9 => (self.mem_displ >> 8) as u8,
			10 => (self.mem_displ >> 16) as u8,
			11 => (self.mem_displ >> 24) as u8,
			12 => self.mem_base_reg,
			13 => self.mem_index_reg,
			14 => self.op_kind_flags as u8,
			15 => (self.op_kind_flags >> 8) as u8,
			_ => return Err(IcedError::new("Invalid index")),
		})
	}

	/// Sets a new `dw` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareWord`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareWord`]: enum.Code.html#variant.DeclareWord
	///
	/// # Panics
	///
	/// - Panics if `index` is invalid
	/// - Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `index`: Index (0-7)
	/// * `new_value`: New value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_set_declare_word_value_i16() instead")]
	#[allow(clippy::unwrap_used)]
	#[inline]
	pub fn set_declare_word_value_i16(&mut self, index: usize, new_value: i16) {
		self.try_set_declare_word_value_i16(index, new_value).unwrap();
	}

	/// Sets a new `dw` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareWord`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareWord`]: enum.Code.html#variant.DeclareWord
	///
	/// # Errors
	///
	/// - Fails if `index` is invalid
	/// - Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `index`: Index (0-7)
	/// * `new_value`: New value
	#[inline]
	pub fn try_set_declare_word_value_i16(&mut self, index: usize, new_value: i16) -> Result<(), IcedError> {
		self.try_set_declare_word_value(index, new_value as u16)
	}

	/// Sets a new `dw` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareWord`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareWord`]: enum.Code.html#variant.DeclareWord
	///
	/// # Panics
	///
	/// - Panics if `index` is invalid
	/// - Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `index`: Index (0-7)
	/// * `new_value`: New value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_set_declare_word_value() instead")]
	#[allow(clippy::unwrap_used)]
	#[inline]
	pub fn set_declare_word_value(&mut self, index: usize, new_value: u16) {
		self.try_set_declare_word_value(index, new_value).unwrap();
	}

	/// Sets a new `dw` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareWord`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareWord`]: enum.Code.html#variant.DeclareWord
	///
	/// # Errors
	///
	/// - Fails if `index` is invalid
	/// - Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `index`: Index (0-7)
	/// * `new_value`: New value
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn try_set_declare_word_value(&mut self, index: usize, new_value: u16) -> Result<(), IcedError> {
		if cfg!(feature = "db") {
			match index {
				0 => {
					self.reg0 = new_value as u8;
					self.reg1 = (new_value >> 8) as u8;
				}
				1 => {
					self.reg2 = new_value as u8;
					self.reg3 = (new_value >> 8) as u8;
				}
				2 => self.immediate = (self.immediate & 0xFFFF_0000) | new_value as u32,
				3 => self.immediate = self.immediate as u16 as u32 | (new_value as u32) << 16,
				4 => self.mem_displ = (self.mem_displ & 0xFFFF_0000) | new_value as u32,
				5 => self.mem_displ = self.mem_displ as u16 as u32 | (new_value as u32) << 16,
				6 => {
					self.mem_base_reg = new_value as u8;
					self.mem_index_reg = (new_value >> 8) as u8;
				}
				7 => self.op_kind_flags = (self.op_kind_flags & 0xFFFF_0000) | new_value as u32,
				_ => return Err(IcedError::new("Invalid index")),
			}
			Ok(())
		} else {
			Err(IcedError::new("`db` feature wasn't enabled"))
		}
	}

	/// Gets a `dw` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareWord`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareWord`]: enum.Code.html#variant.DeclareWord
	///
	/// # Panics
	///
	/// Panics if `index` is invalid
	///
	/// # Arguments
	///
	/// * `index`: Index (0-7)
	#[must_use]
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_get_declare_word_value() instead")]
	#[allow(clippy::unwrap_used)]
	#[inline]
	pub fn get_declare_word_value(&self, index: usize) -> u16 {
		self.try_get_declare_word_value(index).unwrap()
	}

	/// Gets a `dw` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareWord`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareWord`]: enum.Code.html#variant.DeclareWord
	///
	/// # Errors
	///
	/// Fails if `index` is invalid
	///
	/// # Arguments
	///
	/// * `index`: Index (0-7)
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn try_get_declare_word_value(&self, index: usize) -> Result<u16, IcedError> {
		Ok(match index {
			0 => self.reg0 as u16 | ((self.reg1 as u16) << 8),
			1 => self.reg2 as u16 | ((self.reg3 as u16) << 8),
			2 => self.immediate as u16,
			3 => (self.immediate >> 16) as u16,
			4 => self.mem_displ as u16,
			5 => (self.mem_displ >> 16) as u16,
			6 => self.mem_base_reg as u16 | ((self.mem_index_reg as u16) << 8),
			7 => self.op_kind_flags as u16,
			_ => return Err(IcedError::new("Invalid index")),
		})
	}

	/// Sets a new `dd` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareDword`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareDword`]: enum.Code.html#variant.DeclareDword
	///
	/// # Panics
	///
	/// - Panics if `index` is invalid
	/// - Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `index`: Index (0-3)
	/// * `new_value`: New value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_set_declare_dword_value_i32() instead")]
	#[allow(clippy::unwrap_used)]
	#[inline]
	pub fn set_declare_dword_value_i32(&mut self, index: usize, new_value: i32) {
		self.try_set_declare_dword_value_i32(index, new_value).unwrap();
	}

	/// Sets a new `dd` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareDword`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareDword`]: enum.Code.html#variant.DeclareDword
	///
	/// # Errors
	///
	/// - Fails if `index` is invalid
	/// - Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `index`: Index (0-3)
	/// * `new_value`: New value
	#[inline]
	pub fn try_set_declare_dword_value_i32(&mut self, index: usize, new_value: i32) -> Result<(), IcedError> {
		self.try_set_declare_dword_value(index, new_value as u32)
	}

	/// Sets a new `dd` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareDword`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareDword`]: enum.Code.html#variant.DeclareDword
	///
	/// # Panics
	///
	/// - Panics if `index` is invalid
	/// - Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `index`: Index (0-3)
	/// * `new_value`: New value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_set_declare_dword_value() instead")]
	#[allow(clippy::unwrap_used)]
	#[inline]
	pub fn set_declare_dword_value(&mut self, index: usize, new_value: u32) {
		self.try_set_declare_dword_value(index, new_value).unwrap();
	}

	/// Sets a new `dd` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareDword`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareDword`]: enum.Code.html#variant.DeclareDword
	///
	/// # Errors
	///
	/// - Fails if `index` is invalid
	/// - Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `index`: Index (0-3)
	/// * `new_value`: New value
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn try_set_declare_dword_value(&mut self, index: usize, new_value: u32) -> Result<(), IcedError> {
		if cfg!(feature = "db") {
			match index {
				0 => {
					self.reg0 = new_value as u8;
					self.reg1 = (new_value >> 8) as u8;
					self.reg2 = (new_value >> 16) as u8;
					self.reg3 = (new_value >> 24) as u8;
				}
				1 => self.immediate = new_value,
				2 => self.mem_displ = new_value,
				3 => {
					self.mem_base_reg = new_value as u8;
					self.mem_index_reg = (new_value >> 8) as u8;
					self.op_kind_flags = (self.op_kind_flags & 0xFFFF_0000) | (new_value >> 16);
				}
				_ => return Err(IcedError::new("Invalid index")),
			}
			Ok(())
		} else {
			Err(IcedError::new("`db` feature wasn't enabled"))
		}
	}

	/// Gets a `dd` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareDword`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareDword`]: enum.Code.html#variant.DeclareDword
	///
	/// # Panics
	///
	/// Panics if `index` is invalid
	///
	/// # Arguments
	///
	/// * `index`: Index (0-3)
	#[must_use]
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_get_declare_dword_value() instead")]
	#[allow(clippy::unwrap_used)]
	#[inline]
	pub fn get_declare_dword_value(&self, index: usize) -> u32 {
		self.try_get_declare_dword_value(index).unwrap()
	}

	/// Gets a `dd` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareDword`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareDword`]: enum.Code.html#variant.DeclareDword
	///
	/// # Errors
	///
	/// Fails if `index` is invalid
	///
	/// # Arguments
	///
	/// * `index`: Index (0-3)
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn try_get_declare_dword_value(&self, index: usize) -> Result<u32, IcedError> {
		Ok(match index {
			0 => self.reg0 as u32 | ((self.reg1 as u32) << 8) | ((self.reg2 as u32) << 16) | ((self.reg3 as u32) << 24),
			1 => self.immediate,
			2 => self.mem_displ,
			3 => self.mem_base_reg as u32 | ((self.mem_index_reg as u32) << 8) | (self.op_kind_flags << 16),
			_ => return Err(IcedError::new("Invalid index")),
		})
	}

	/// Sets a new `dq` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareQword`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareQword`]: enum.Code.html#variant.DeclareQword
	///
	/// # Panics
	///
	/// - Panics if `index` is invalid
	/// - Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `index`: Index (0-1)
	/// * `new_value`: New value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_set_declare_qword_value_i64() instead")]
	#[allow(clippy::unwrap_used)]
	#[inline]
	pub fn set_declare_qword_value_i64(&mut self, index: usize, new_value: i64) {
		self.try_set_declare_qword_value_i64(index, new_value).unwrap();
	}

	/// Sets a new `dq` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareQword`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareQword`]: enum.Code.html#variant.DeclareQword
	///
	/// # Errors
	///
	/// - Fails if `index` is invalid
	/// - Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `index`: Index (0-1)
	/// * `new_value`: New value
	#[inline]
	pub fn try_set_declare_qword_value_i64(&mut self, index: usize, new_value: i64) -> Result<(), IcedError> {
		self.try_set_declare_qword_value(index, new_value as u64)
	}

	/// Sets a new `dq` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareQword`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareQword`]: enum.Code.html#variant.DeclareQword
	///
	/// # Panics
	///
	/// - Panics if `index` is invalid
	/// - Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `index`: Index (0-1)
	/// * `new_value`: New value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_set_declare_qword_value() instead")]
	#[allow(clippy::unwrap_used)]
	#[inline]
	pub fn set_declare_qword_value(&mut self, index: usize, new_value: u64) {
		self.try_set_declare_qword_value(index, new_value).unwrap();
	}

	/// Sets a new `dq` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareQword`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareQword`]: enum.Code.html#variant.DeclareQword
	///
	/// # Errors
	///
	/// - Fails if `index` is invalid
	/// - Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `index`: Index (0-1)
	/// * `new_value`: New value
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn try_set_declare_qword_value(&mut self, index: usize, new_value: u64) -> Result<(), IcedError> {
		if cfg!(feature = "db") {
			match index {
				0 => {
					self.reg0 = new_value as u8;
					self.reg1 = (new_value >> 8) as u8;
					self.reg2 = (new_value >> 16) as u8;
					self.reg3 = (new_value >> 24) as u8;
					self.immediate = (new_value >> 32) as u32;
				}
				1 => {
					self.mem_displ = new_value as u32;
					self.mem_base_reg = (new_value >> 32) as u8;
					self.mem_index_reg = (new_value >> 40) as u8;
					self.op_kind_flags = (self.op_kind_flags & 0xFFFF_0000) | (new_value >> 48) as u32;
				}
				_ => return Err(IcedError::new("Invalid index")),
			}
			Ok(())
		} else {
			Err(IcedError::new("`db` feature wasn't enabled"))
		}
	}

	/// Gets a `dq` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareQword`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareQword`]: enum.Code.html#variant.DeclareQword
	///
	/// # Panics
	///
	/// Panics if `index` is invalid
	///
	/// # Arguments
	///
	/// * `index`: Index (0-1)
	#[must_use]
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_get_declare_qword_value() instead")]
	#[allow(clippy::unwrap_used)]
	#[inline]
	pub fn get_declare_qword_value(&self, index: usize) -> u64 {
		self.try_get_declare_qword_value(index).unwrap()
	}

	/// Gets a `dq` value, see also [`declare_data_len()`].
	/// Can only be called if [`code()`] is [`Code::DeclareQword`]
	///
	/// [`declare_data_len()`]: #method.declare_data_len
	/// [`code()`]: #method.code
	/// [`Code::DeclareQword`]: enum.Code.html#variant.DeclareQword
	///
	/// # Errors
	///
	/// Fails if `index` is invalid
	///
	/// # Arguments
	///
	/// * `index`: Index (0-1)
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn try_get_declare_qword_value(&self, index: usize) -> Result<u64, IcedError> {
		Ok(match index {
			0 => {
				self.reg0 as u64
					| ((self.reg1 as u64) << 8)
					| ((self.reg2 as u64) << 16)
					| ((self.reg3 as u64) << 24)
					| ((self.immediate as u64) << 32)
			}
			1 => {
				self.mem_displ as u64 | ((self.mem_base_reg as u64) << 32) | ((self.mem_index_reg as u64) << 40) | ((self.op_kind_flags as u64) << 48)
			}
			_ => return Err(IcedError::new("Invalid index")),
		})
	}

	/// Checks if this is a VSIB instruction, see also [`is_vsib32()`], [`is_vsib64()`]
	///
	/// [`is_vsib32()`]: #method.is_vsib32
	/// [`is_vsib64()`]: #method.is_vsib64
	#[must_use]
	#[inline]
	pub fn is_vsib(&self) -> bool {
		self.vsib().is_some()
	}

	/// VSIB instructions only ([`is_vsib()`]): `true` if it's using 32-bit indexes, `false` if it's using 64-bit indexes
	///
	/// [`is_vsib()`]: #method.is_vsib
	#[must_use]
	#[inline]
	pub fn is_vsib32(&self) -> bool {
		if let Some(is_vsib64) = self.vsib() {
			!is_vsib64
		} else {
			false
		}
	}

	/// VSIB instructions only ([`is_vsib()`]): `true` if it's using 64-bit indexes, `false` if it's using 32-bit indexes
	///
	/// [`is_vsib()`]: #method.is_vsib
	#[must_use]
	#[inline]
	pub fn is_vsib64(&self) -> bool {
		if let Some(is_vsib64) = self.vsib() {
			is_vsib64
		} else {
			false
		}
	}

	/// Checks if it's a vsib instruction.
	///
	/// # Returns
	///
	/// * `Some(true)` if it's a VSIB instruction with 64-bit indexes
	/// * `Some(false)` if it's a VSIB instruction with 32-bit indexes
	/// * `None` if it's not a VSIB instruction.
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	#[allow(clippy::match_single_binding)]
	pub fn vsib(&self) -> Option<bool> {
		#[cfg_attr(feature = "cargo-fmt", rustfmt::skip)]
		match self.code() {
			// GENERATOR-BEGIN: Vsib32
			// ⚠️This was generated by GENERATOR!🦹‍♂️
			Code::VEX_Vpgatherdd_xmm_vm32x_xmm
			| Code::VEX_Vpgatherdd_ymm_vm32y_ymm
			| Code::VEX_Vpgatherdq_xmm_vm32x_xmm
			| Code::VEX_Vpgatherdq_ymm_vm32x_ymm
			| Code::EVEX_Vpgatherdd_xmm_k1_vm32x
			| Code::EVEX_Vpgatherdd_ymm_k1_vm32y
			| Code::EVEX_Vpgatherdd_zmm_k1_vm32z
			| Code::EVEX_Vpgatherdq_xmm_k1_vm32x
			| Code::EVEX_Vpgatherdq_ymm_k1_vm32x
			| Code::EVEX_Vpgatherdq_zmm_k1_vm32y
			| Code::VEX_Vgatherdps_xmm_vm32x_xmm
			| Code::VEX_Vgatherdps_ymm_vm32y_ymm
			| Code::VEX_Vgatherdpd_xmm_vm32x_xmm
			| Code::VEX_Vgatherdpd_ymm_vm32x_ymm
			| Code::EVEX_Vgatherdps_xmm_k1_vm32x
			| Code::EVEX_Vgatherdps_ymm_k1_vm32y
			| Code::EVEX_Vgatherdps_zmm_k1_vm32z
			| Code::EVEX_Vgatherdpd_xmm_k1_vm32x
			| Code::EVEX_Vgatherdpd_ymm_k1_vm32x
			| Code::EVEX_Vgatherdpd_zmm_k1_vm32y
			| Code::EVEX_Vpscatterdd_vm32x_k1_xmm
			| Code::EVEX_Vpscatterdd_vm32y_k1_ymm
			| Code::EVEX_Vpscatterdd_vm32z_k1_zmm
			| Code::EVEX_Vpscatterdq_vm32x_k1_xmm
			| Code::EVEX_Vpscatterdq_vm32x_k1_ymm
			| Code::EVEX_Vpscatterdq_vm32y_k1_zmm
			| Code::EVEX_Vscatterdps_vm32x_k1_xmm
			| Code::EVEX_Vscatterdps_vm32y_k1_ymm
			| Code::EVEX_Vscatterdps_vm32z_k1_zmm
			| Code::EVEX_Vscatterdpd_vm32x_k1_xmm
			| Code::EVEX_Vscatterdpd_vm32x_k1_ymm
			| Code::EVEX_Vscatterdpd_vm32y_k1_zmm
			| Code::EVEX_Vgatherpf0dps_vm32z_k1
			| Code::EVEX_Vgatherpf0dpd_vm32y_k1
			| Code::EVEX_Vgatherpf1dps_vm32z_k1
			| Code::EVEX_Vgatherpf1dpd_vm32y_k1
			| Code::EVEX_Vscatterpf0dps_vm32z_k1
			| Code::EVEX_Vscatterpf0dpd_vm32y_k1
			| Code::EVEX_Vscatterpf1dps_vm32z_k1
			| Code::EVEX_Vscatterpf1dpd_vm32y_k1
			=> Some(false),
			// GENERATOR-END: Vsib32

			// GENERATOR-BEGIN: Vsib64
			// ⚠️This was generated by GENERATOR!🦹‍♂️
			Code::VEX_Vpgatherqd_xmm_vm64x_xmm
			| Code::VEX_Vpgatherqd_xmm_vm64y_xmm
			| Code::VEX_Vpgatherqq_xmm_vm64x_xmm
			| Code::VEX_Vpgatherqq_ymm_vm64y_ymm
			| Code::EVEX_Vpgatherqd_xmm_k1_vm64x
			| Code::EVEX_Vpgatherqd_xmm_k1_vm64y
			| Code::EVEX_Vpgatherqd_ymm_k1_vm64z
			| Code::EVEX_Vpgatherqq_xmm_k1_vm64x
			| Code::EVEX_Vpgatherqq_ymm_k1_vm64y
			| Code::EVEX_Vpgatherqq_zmm_k1_vm64z
			| Code::VEX_Vgatherqps_xmm_vm64x_xmm
			| Code::VEX_Vgatherqps_xmm_vm64y_xmm
			| Code::VEX_Vgatherqpd_xmm_vm64x_xmm
			| Code::VEX_Vgatherqpd_ymm_vm64y_ymm
			| Code::EVEX_Vgatherqps_xmm_k1_vm64x
			| Code::EVEX_Vgatherqps_xmm_k1_vm64y
			| Code::EVEX_Vgatherqps_ymm_k1_vm64z
			| Code::EVEX_Vgatherqpd_xmm_k1_vm64x
			| Code::EVEX_Vgatherqpd_ymm_k1_vm64y
			| Code::EVEX_Vgatherqpd_zmm_k1_vm64z
			| Code::EVEX_Vpscatterqd_vm64x_k1_xmm
			| Code::EVEX_Vpscatterqd_vm64y_k1_xmm
			| Code::EVEX_Vpscatterqd_vm64z_k1_ymm
			| Code::EVEX_Vpscatterqq_vm64x_k1_xmm
			| Code::EVEX_Vpscatterqq_vm64y_k1_ymm
			| Code::EVEX_Vpscatterqq_vm64z_k1_zmm
			| Code::EVEX_Vscatterqps_vm64x_k1_xmm
			| Code::EVEX_Vscatterqps_vm64y_k1_xmm
			| Code::EVEX_Vscatterqps_vm64z_k1_ymm
			| Code::EVEX_Vscatterqpd_vm64x_k1_xmm
			| Code::EVEX_Vscatterqpd_vm64y_k1_ymm
			| Code::EVEX_Vscatterqpd_vm64z_k1_zmm
			| Code::EVEX_Vgatherpf0qps_vm64z_k1
			| Code::EVEX_Vgatherpf0qpd_vm64z_k1
			| Code::EVEX_Vgatherpf1qps_vm64z_k1
			| Code::EVEX_Vgatherpf1qpd_vm64z_k1
			| Code::EVEX_Vscatterpf0qps_vm64z_k1
			| Code::EVEX_Vscatterpf0qpd_vm64z_k1
			| Code::EVEX_Vscatterpf1qps_vm64z_k1
			| Code::EVEX_Vscatterpf1qpd_vm64z_k1
			=> Some(true),
			// GENERATOR-END: Vsib64

			_ => None,
		}
	}

	/// Gets the suppress all exceptions flag (EVEX encoded instructions). Note that if [`rounding_control()`] is
	/// not [`RoundingControl::None`], SAE is implied but this method will still return `false`.
	///
	/// [`rounding_control()`]: #method.rounding_control
	/// [`RoundingControl::None`]: enum.RoundingControl.html#variant.None
	#[must_use]
	#[inline]
	pub fn suppress_all_exceptions(&self) -> bool {
		(self.code_flags & CodeFlags::SUPPRESS_ALL_EXCEPTIONS) != 0
	}

	/// Sets the suppress all exceptions flag (EVEX encoded instructions). Note that if [`rounding_control()`] is
	/// not [`RoundingControl::None`], SAE is implied but this method will still return `false`.
	///
	/// [`rounding_control()`]: #method.rounding_control
	/// [`RoundingControl::None`]: enum.RoundingControl.html#variant.None
	///
	/// # Arguments
	///
	/// * `new_value`: New value
	#[inline]
	pub fn set_suppress_all_exceptions(&mut self, new_value: bool) {
		if new_value {
			self.code_flags |= CodeFlags::SUPPRESS_ALL_EXCEPTIONS;
		} else {
			self.code_flags &= !CodeFlags::SUPPRESS_ALL_EXCEPTIONS;
		}
	}

	/// Checks if the memory operand is `RIP`/`EIP` relative
	#[must_use]
	#[inline]
	pub fn is_ip_rel_memory_operand(&self) -> bool {
		let base_reg = self.memory_base();
		base_reg == Register::RIP || base_reg == Register::EIP
	}

	/// Gets the `RIP`/`EIP` releative address ([`memory_displacement32()`] or [`memory_displacement64()`]).
	/// This method is only valid if there's a memory operand with `RIP`/`EIP` relative addressing, see [`is_ip_rel_memory_operand()`]
	///
	/// [`memory_displacement32()`]: #method.memory_displacement32
	/// [`memory_displacement64()`]: #method.memory_displacement64
	/// [`is_ip_rel_memory_operand()`]: #method.is_ip_rel_memory_operand
	#[must_use]
	#[inline]
	pub fn ip_rel_memory_address(&self) -> u64 {
		if self.memory_base() == Register::EIP {
			self.memory_displacement32() as u64
		} else {
			self.memory_displacement64()
		}
	}

	/// Gets the virtual address of a memory operand
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4, must be a memory operand
	/// * `element_index`: Only used if it's a vsib memory operand. This is the element index of the vector index register.
	/// * `get_register_value`: Function that returns the value of a register or the base address of a segment register, or `None` for unsupported
	///    registers.
	///
	/// # Call-back function args
	///
	/// * Arg 1: `register`: Register (GPR8, GPR16, GPR32, GPR64, XMM, YMM, ZMM, seg). If it's a segment register, the call-back function should return the segment's base address, not the segment's register value.
	/// * Arg 2: `element_index`: Only used if it's a vsib memory operand. This is the element index of the vector index register.
	/// * Arg 3: `element_size`: Only used if it's a vsib memory operand. Size in bytes of elements in vector index register (4 or 8).
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // add [rdi+r12*8-5AA5EDCCh],esi
	/// let bytes = b"\x42\x01\xB4\xE7\x34\x12\x5A\xA5";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	/// let instr = decoder.decode();
	///
	/// let va = instr.try_virtual_address(0, 0, |register, element_index, element_size| {
	///     match register {
	///         // The base address of ES, CS, SS and DS is always 0 in 64-bit mode
	///         Register::DS => Some(0x0000_0000_0000_0000),
	///         Register::RDI => Some(0x0000_0000_1000_0000),
	///         Register::R12 => Some(0x0000_0004_0000_0000),
	///         _ => None,
	///     }
	/// });
	/// assert_eq!(va, Some(0x0000_001F_B55A_1234));
	/// ```
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn try_virtual_address<F>(&self, operand: u32, element_index: usize, mut get_register_value: F) -> Option<u64>
	where
		F: FnMut(Register, usize, usize) -> Option<u64>,
	{
		let op_kind = self.try_op_kind(operand).ok()?;
		Some(match op_kind {
			OpKind::Register
			| OpKind::NearBranch16
			| OpKind::NearBranch32
			| OpKind::NearBranch64
			| OpKind::FarBranch16
			| OpKind::FarBranch32
			| OpKind::Immediate8
			| OpKind::Immediate8_2nd
			| OpKind::Immediate16
			| OpKind::Immediate32
			| OpKind::Immediate64
			| OpKind::Immediate8to16
			| OpKind::Immediate8to32
			| OpKind::Immediate8to64
			| OpKind::Immediate32to64 => 0,

			OpKind::MemorySegSI => {
				get_register_value(self.memory_segment(), 0, 0)?.wrapping_add(get_register_value(Register::SI, 0, 0)? as u16 as u64)
			}
			OpKind::MemorySegESI => {
				get_register_value(self.memory_segment(), 0, 0)?.wrapping_add(get_register_value(Register::ESI, 0, 0)? as u32 as u64)
			}
			OpKind::MemorySegRSI => get_register_value(self.memory_segment(), 0, 0)?.wrapping_add(get_register_value(Register::RSI, 0, 0)?),
			OpKind::MemorySegDI => {
				get_register_value(self.memory_segment(), 0, 0)?.wrapping_add(get_register_value(Register::DI, 0, 0)? as u16 as u64)
			}
			OpKind::MemorySegEDI => {
				get_register_value(self.memory_segment(), 0, 0)?.wrapping_add(get_register_value(Register::EDI, 0, 0)? as u32 as u64)
			}
			OpKind::MemorySegRDI => get_register_value(self.memory_segment(), 0, 0)?.wrapping_add(get_register_value(Register::RDI, 0, 0)?),
			OpKind::MemoryESDI => get_register_value(Register::ES, 0, 0)?.wrapping_add(get_register_value(Register::DI, 0, 0)? as u16 as u64),
			OpKind::MemoryESEDI => get_register_value(Register::ES, 0, 0)?.wrapping_add(get_register_value(Register::EDI, 0, 0)? as u32 as u64),
			OpKind::MemoryESRDI => get_register_value(Register::ES, 0, 0)?.wrapping_add(get_register_value(Register::RDI, 0, 0)?),
			#[allow(deprecated)]
			OpKind::Memory64 => return None,
			OpKind::Memory => {
				let base_reg = self.memory_base();
				let index_reg = self.memory_index();
				let addr_size = instruction_internal::get_address_size_in_bytes(base_reg, index_reg, self.memory_displ_size(), self.code_size());
				let mut offset = self.memory_displacement64();
				let offset_mask = match addr_size {
					8 => u64::MAX,
					4 => u32::MAX as u64,
					_ => {
						debug_assert_eq!(addr_size, 2);
						u16::MAX as u64
					}
				};
				match base_reg {
					Register::None | Register::EIP | Register::RIP => {}
					_ => offset = offset.wrapping_add(get_register_value(base_reg, 0, 0)?),
				}
				let code = self.code();
				if index_reg != Register::None && !code.ignores_index() && !code.is_tile_stride_index() {
					if let Some(is_vsib64) = self.vsib() {
						if is_vsib64 {
							offset = offset.wrapping_add(
								get_register_value(index_reg, element_index, 8)? << instruction_internal::internal_get_memory_index_scale(self),
							);
						} else {
							offset = offset.wrapping_add(
								(get_register_value(index_reg, element_index, 4)? as i32 as u64)
									<< instruction_internal::internal_get_memory_index_scale(self),
							);
						}
					} else {
						offset =
							offset.wrapping_add(get_register_value(index_reg, 0, 0)? << instruction_internal::internal_get_memory_index_scale(self));
					}
				}
				offset &= offset_mask;
				if !code.ignores_segment() {
					get_register_value(self.memory_segment(), 0, 0)?.wrapping_add(offset)
				} else {
					offset
				}
			}
		})
	}

	/// Gets the virtual address of a memory operand. See also [`try_virtual_address()`]
	///
	/// [`try_virtual_address()`]: #method.try_virtual_address
	///
	/// # Panics
	///
	/// Panics if `operand` is invalid
	///
	/// # Arguments
	///
	/// * `operand`: Operand number, 0-4, must be a memory operand
	/// * `element_index`: Only used if it's a vsib memory operand. This is the element index of the vector index register.
	/// * `get_register_value`: Function that returns the value of a register or the base address of a segment register.
	///
	/// # Call-back function args
	///
	/// * Arg 1: `register`: Register (GPR8, GPR16, GPR32, GPR64, XMM, YMM, ZMM, seg). If it's a segment register, the call-back function should return the segment's base address, not the segment's register value.
	/// * Arg 2: `element_index`: Only used if it's a vsib memory operand. This is the element index of the vector index register.
	/// * Arg 3: `element_size`: Only used if it's a vsib memory operand. Size in bytes of elements in vector index register (4 or 8).
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // add [rdi+r12*8-5AA5EDCCh],esi
	/// let bytes = b"\x42\x01\xB4\xE7\x34\x12\x5A\xA5";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	/// let instr = decoder.decode();
	///
	/// let va = instr.virtual_address(0, 0, |register, element_index, element_size| {
	///     match register {
	///         // The base address of ES, CS, SS and DS is always 0 in 64-bit mode
	///         Register::DS => 0x0000_0000_0000_0000,
	///         Register::RDI => 0x0000_0000_1000_0000,
	///         Register::R12 => 0x0000_0004_0000_0000,
	///         _ => unimplemented!(),
	///     }
	/// });
	/// assert_eq!(va, 0x0000_001F_B55A_1234);
	/// ```
	#[must_use]
	#[inline]
	#[deprecated(since = "1.11.0", note = "This method can panic, use try_virtual_address() instead")]
	#[allow(clippy::unwrap_used)]
	pub fn virtual_address<F>(&self, operand: u32, element_index: usize, mut get_register_value: F) -> u64
	where
		F: FnMut(Register, usize, usize) -> u64,
	{
		self.try_virtual_address(operand, element_index, |x, y, z| Some(get_register_value(x, y, z))).unwrap()
	}
}

/// Contains the FPU `TOP` increment, whether it's conditional and whether the instruction writes to `TOP`
#[cfg(feature = "instr_info")]
#[derive(Debug, Default, Copy, Clone, Eq, PartialEq, Hash)]
pub struct FpuStackIncrementInfo {
	increment: i32,
	conditional: bool,
	writes_top: bool,
}

#[cfg(feature = "instr_info")]
impl FpuStackIncrementInfo {
	/// Constructor
	#[must_use]
	#[inline]
	pub fn new(increment: i32, conditional: bool, writes_top: bool) -> Self {
		Self { increment, conditional, writes_top }
	}

	/// Used if [`writes_top()`] is `true`:
	///
	/// Value added to `TOP`.
	///
	/// This is negative if it pushes one or more values and positive if it pops one or more values
	/// and `0` if it writes to `TOP` (eg. `FLDENV`, etc) without pushing/popping anything.
	///
	/// [`writes_top()`]: #method.writes_top
	#[must_use]
	#[inline]
	pub fn increment(&self) -> i32 {
		self.increment
	}

	/// `true` if it's a conditional push/pop (eg. `FPTAN` or `FSINCOS`)
	#[must_use]
	#[inline]
	pub fn conditional(&self) -> bool {
		self.conditional
	}

	/// `true` if `TOP` is written (it's a conditional/unconditional push/pop, `FNSAVE`, `FLDENV`, etc)
	#[must_use]
	#[inline]
	pub fn writes_top(&self) -> bool {
		self.writes_top
	}
}

#[cfg(feature = "instr_info")]
impl Instruction {
	/// Gets the number of bytes added to `SP`/`ESP`/`RSP` or 0 if it's not an instruction that pushes or pops data. This method assumes
	/// the instruction doesn't change the privilege level (eg. `IRET/D/Q`). If it's the `LEAVE` instruction, this method returns 0.
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // pushfq
	/// let bytes = b"\x9C";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	/// let instr = decoder.decode();
	///
	/// assert!(instr.is_stack_instruction());
	/// assert_eq!(instr.stack_pointer_increment(), -8);
	/// ```
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn stack_pointer_increment(&self) -> i32 {
		#[cfg_attr(feature = "cargo-fmt", rustfmt::skip)]
		#[allow(clippy::match_single_binding)]
		match self.code() {
			// GENERATOR-BEGIN: StackPointerIncrementTable
			// ⚠️This was generated by GENERATOR!🦹‍♂️
			Code::Pushad => -32,
			Code::Pushaw
			| Code::Call_m1664 => -16,
			Code::Push_r64
			| Code::Pushq_imm32
			| Code::Pushq_imm8
			| Code::Call_ptr1632
			| Code::Pushfq
			| Code::Call_rel32_64
			| Code::Call_rm64
			| Code::Call_m1632
			| Code::Push_rm64
			| Code::Pushq_FS
			| Code::Pushq_GS => -8,
			Code::Pushd_ES
			| Code::Pushd_CS
			| Code::Pushd_SS
			| Code::Pushd_DS
			| Code::Push_r32
			| Code::Pushd_imm32
			| Code::Pushd_imm8
			| Code::Call_ptr1616
			| Code::Pushfd
			| Code::Call_rel32_32
			| Code::Call_rm32
			| Code::Call_m1616
			| Code::Push_rm32
			| Code::Pushd_FS
			| Code::Pushd_GS => -4,
			Code::Pushw_ES
			| Code::Pushw_CS
			| Code::Pushw_SS
			| Code::Pushw_DS
			| Code::Push_r16
			| Code::Push_imm16
			| Code::Pushw_imm8
			| Code::Pushfw
			| Code::Call_rel16
			| Code::Call_rm16
			| Code::Push_rm16
			| Code::Pushw_FS
			| Code::Pushw_GS => -2,
			Code::Popw_ES
			| Code::Popw_CS
			| Code::Popw_SS
			| Code::Popw_DS
			| Code::Pop_r16
			| Code::Pop_rm16
			| Code::Popfw
			| Code::Retnw
			| Code::Popw_FS
			| Code::Popw_GS => 2,
			Code::Popd_ES
			| Code::Popd_SS
			| Code::Popd_DS
			| Code::Pop_r32
			| Code::Pop_rm32
			| Code::Popfd
			| Code::Retnd
			| Code::Retfw
			| Code::Popd_FS
			| Code::Popd_GS => 4,
			Code::Pop_r64
			| Code::Pop_rm64
			| Code::Popfq
			| Code::Retnq
			| Code::Retfd
			| Code::Popq_FS
			| Code::Popq_GS => 8,
			Code::Popaw
			| Code::Retfq => 16,
			Code::Uiret => 24,
			Code::Popad => 32,
			Code::Iretq
			| Code::Eretu
			| Code::Erets => 40,
			Code::Enterw_imm16_imm8 => -(2 + (self.immediate8_2nd() as i32 & 0x1F) * 2 + self.immediate16() as i32),
			Code::Enterd_imm16_imm8 => -(4 + (self.immediate8_2nd() as i32 & 0x1F) * 4 + self.immediate16() as i32),
			Code::Enterq_imm16_imm8 => -(8 + (self.immediate8_2nd() as i32 & 0x1F) * 8 + self.immediate16() as i32),
			Code::Iretw => if self.code_size() == CodeSize::Code64 { 2 * 5 } else { 2 * 3 },
			Code::Iretd => if self.code_size() == CodeSize::Code64 { 4 * 5 } else { 4 * 3 },
			Code::Retnw_imm16 => 2 + self.immediate16() as i32,
			Code::Retnd_imm16
			| Code::Retfw_imm16 => 4 + self.immediate16() as i32,
			Code::Retnq_imm16
			| Code::Retfd_imm16 => 8 + self.immediate16() as i32,
			Code::Retfq_imm16 => 16 + self.immediate16() as i32,
			// GENERATOR-END: StackPointerIncrementTable
			_ => 0,
		}
	}

	/// Gets the FPU status word's `TOP` increment and whether it's a conditional or unconditional push/pop
	/// and whether `TOP` is written.
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // ficomp dword ptr [rax]
	/// let bytes = b"\xDA\x18";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	/// let instr = decoder.decode();
	///
	/// let info = instr.fpu_stack_increment_info();
	/// // It pops the stack once
	/// assert_eq!(info.increment(), 1);
	/// assert!(!info.conditional());
	/// assert!(info.writes_top());
	/// ```
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn fpu_stack_increment_info(&self) -> FpuStackIncrementInfo {
		#[cfg_attr(feature = "cargo-fmt", rustfmt::skip)]
		#[allow(clippy::match_single_binding)]
		match self.code() {
			// GENERATOR-BEGIN: FpuStackIncrementInfoTable
			// ⚠️This was generated by GENERATOR!🦹‍♂️
			Code::Fld_m32fp
			| Code::Fld_sti
			| Code::Fld1
			| Code::Fldl2t
			| Code::Fldl2e
			| Code::Fldpi
			| Code::Fldlg2
			| Code::Fldln2
			| Code::Fldz
			| Code::Fxtract
			| Code::Fdecstp
			| Code::Fild_m32int
			| Code::Fld_m80fp
			| Code::Fld_m64fp
			| Code::Fild_m16int
			| Code::Fbld_m80bcd
			| Code::Fild_m64int
			=> FpuStackIncrementInfo { increment: -1, conditional: false, writes_top: true },
			Code::Fptan
			| Code::Fsincos
			=> FpuStackIncrementInfo { increment: -1, conditional: true, writes_top: true },
			Code::Fldenv_m14byte
			| Code::Fldenv_m28byte
			| Code::Fninit
			| Code::Finit
			| Code::Frstor_m94byte
			| Code::Frstor_m108byte
			| Code::Fnsave_m94byte
			| Code::Fsave_m94byte
			| Code::Fnsave_m108byte
			| Code::Fsave_m108byte
			=> FpuStackIncrementInfo { increment: 0, conditional: false, writes_top: true },
			Code::Fcomp_m32fp
			| Code::Fcomp_st0_sti
			| Code::Fstp_m32fp
			| Code::Fstpnce_sti
			| Code::Fyl2x
			| Code::Fpatan
			| Code::Fincstp
			| Code::Fyl2xp1
			| Code::Ficomp_m32int
			| Code::Fisttp_m32int
			| Code::Fistp_m32int
			| Code::Fstp_m80fp
			| Code::Fcomp_m64fp
			| Code::Fcomp_st0_sti_DCD8
			| Code::Fisttp_m64int
			| Code::Fstp_m64fp
			| Code::Fstp_sti
			| Code::Fucomp_st0_sti
			| Code::Ficomp_m16int
			| Code::Faddp_sti_st0
			| Code::Fmulp_sti_st0
			| Code::Fcomp_st0_sti_DED0
			| Code::Fsubrp_sti_st0
			| Code::Fsubp_sti_st0
			| Code::Fdivrp_sti_st0
			| Code::Fdivp_sti_st0
			| Code::Fisttp_m16int
			| Code::Fistp_m16int
			| Code::Fbstp_m80bcd
			| Code::Fistp_m64int
			| Code::Ffreep_sti
			| Code::Fstp_sti_DFD0
			| Code::Fstp_sti_DFD8
			| Code::Fucomip_st0_sti
			| Code::Fcomip_st0_sti
			| Code::Ftstp
			=> FpuStackIncrementInfo { increment: 1, conditional: false, writes_top: true },
			Code::Fucompp
			| Code::Fcompp
			=> FpuStackIncrementInfo { increment: 2, conditional: false, writes_top: true },
			// GENERATOR-END: FpuStackIncrementInfoTable
			_ => FpuStackIncrementInfo::default(),
		}
	}

	/// Instruction encoding, eg. Legacy, 3DNow!, VEX, EVEX, XOP
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // vmovaps xmm1,xmm5
	/// let bytes = b"\xC5\xF8\x28\xCD";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	/// let instr = decoder.decode();
	///
	/// assert_eq!(instr.encoding(), EncodingKind::VEX);
	/// ```
	#[must_use]
	#[inline]
	pub fn encoding(&self) -> EncodingKind {
		self.code().encoding()
	}

	/// Gets the CPU or CPUID feature flags
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // vmovaps xmm1,xmm5
	/// // vmovaps xmm10{k3}{z},xmm19
	/// let bytes = b"\xC5\xF8\x28\xCD\x62\x31\x7C\x8B\x28\xD3";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	///
	/// // vmovaps xmm1,xmm5
	/// let instr = decoder.decode();
	/// let cpuid = instr.cpuid_features();
	/// assert_eq!(cpuid.len(), 1);
	/// assert_eq!(cpuid[0], CpuidFeature::AVX);
	///
	/// // vmovaps xmm10{k3}{z},xmm19
	/// let instr = decoder.decode();
	/// let cpuid = instr.cpuid_features();
	/// assert_eq!(cpuid.len(), 2);
	/// assert_eq!(cpuid[0], CpuidFeature::AVX512VL);
	/// assert_eq!(cpuid[1], CpuidFeature::AVX512F);
	/// ```
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	pub fn cpuid_features(&self) -> &'static [CpuidFeature] {
		// SAFETY: TABLE.len() == count(Code)*2 and for all Code values c, c as usize < count(Code)
		let flags2 = unsafe { *crate::info::info_table::TABLE.get_unchecked((self.code() as usize) * 2 + 1) };
		let index = ((flags2 >> InfoFlags2::CPUID_FEATURE_INTERNAL_SHIFT) & InfoFlags2::CPUID_FEATURE_INTERNAL_MASK) as usize;
		// SAFETY: generated and valid index
		unsafe { *crate::info::cpuid_table::CPUID.get_unchecked(index) }
	}

	/// Control flow info
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // or ecx,esi
	/// // ud0 rcx,rsi
	/// // call rcx
	/// let bytes = b"\x0B\xCE\x48\x0F\xFF\xCE\xFF\xD1";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	///
	/// // or ecx,esi
	/// let instr = decoder.decode();
	/// assert_eq!(instr.flow_control(), FlowControl::Next);
	///
	/// // ud0 rcx,rsi
	/// let instr = decoder.decode();
	/// assert_eq!(instr.flow_control(), FlowControl::Exception);
	///
	/// // call rcx
	/// let instr = decoder.decode();
	/// assert_eq!(instr.flow_control(), FlowControl::IndirectCall);
	/// ```
	#[must_use]
	#[inline]
	pub fn flow_control(&self) -> FlowControl {
		self.code().flow_control()
	}

	/// `true` if it's a privileged instruction (all CPL=0 instructions (except `VMCALL`) and IOPL instructions `IN`, `INS`, `OUT`, `OUTS`, `CLI`, `STI`)
	#[must_use]
	#[inline]
	pub fn is_privileged(&self) -> bool {
		self.code().is_privileged()
	}

	/// `true` if this is an instruction that implicitly uses the stack pointer (`SP`/`ESP`/`RSP`), eg. `CALL`, `PUSH`, `POP`, `RET`, etc.
	/// See also [`stack_pointer_increment()`]
	///
	/// [`stack_pointer_increment()`]: #method.stack_pointer_increment
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // or ecx,esi
	/// // push rax
	/// let bytes = b"\x0B\xCE\x50";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	///
	/// // or ecx,esi
	/// let instr = decoder.decode();
	/// assert!(!instr.is_stack_instruction());
	///
	/// // push rax
	/// let instr = decoder.decode();
	/// assert!(instr.is_stack_instruction());
	/// assert_eq!(instr.stack_pointer_increment(), -8);
	/// ```
	#[must_use]
	#[inline]
	pub fn is_stack_instruction(&self) -> bool {
		self.code().is_stack_instruction()
	}

	/// `true` if it's an instruction that saves or restores too many registers (eg. `FXRSTOR`, `XSAVE`, etc).
	#[must_use]
	#[inline]
	pub fn is_save_restore_instruction(&self) -> bool {
		self.code().is_save_restore_instruction()
	}

	#[must_use]
	fn rflags_info(&self) -> usize {
		// SAFETY: TABLE.len() == count(Code)*2 and for all Code values c, c as usize < count(Code)
		let flags1 = unsafe { *crate::info::info_table::TABLE.get_unchecked((self.code() as usize) * 2) };
		let implied_access = (flags1 >> InfoFlags1::IMPLIED_ACCESS_SHIFT) & InfoFlags1::IMPLIED_ACCESS_MASK;
		const_assert!(ImpliedAccess::Shift_Ib_MASK1FMOD9 as u32 + 1 == ImpliedAccess::Shift_Ib_MASK1FMOD11 as u32);
		const_assert!(ImpliedAccess::Shift_Ib_MASK1FMOD9 as u32 + 2 == ImpliedAccess::Shift_Ib_MASK1F as u32);
		const_assert!(ImpliedAccess::Shift_Ib_MASK1FMOD9 as u32 + 3 == ImpliedAccess::Shift_Ib_MASK3F as u32);
		const_assert!(ImpliedAccess::Shift_Ib_MASK1FMOD9 as u32 + 4 == ImpliedAccess::Clear_rflags as u32);
		let result = ((flags1 >> InfoFlags1::RFLAGS_INFO_SHIFT) & InfoFlags1::RFLAGS_INFO_MASK) as usize;
		let e = implied_access.wrapping_sub(ImpliedAccess::Shift_Ib_MASK1FMOD9 as u32);
		match e {
			0 | 1 => {
				const_assert_eq!(ImpliedAccess::Shift_Ib_MASK1FMOD9 as u32 - ImpliedAccess::Shift_Ib_MASK1FMOD9 as u32, 0);
				const_assert_eq!(ImpliedAccess::Shift_Ib_MASK1FMOD11 as u32 - ImpliedAccess::Shift_Ib_MASK1FMOD9 as u32, 1);
				let m = if e == 0 { 9 } else { 17 };
				match (self.immediate8() & 0x1F) % m {
					0 => return RflagsInfo::None as usize,
					1 => return RflagsInfo::R_c_W_co as usize,
					_ => {}
				}
			}
			2 | 3 => {
				const_assert_eq!(ImpliedAccess::Shift_Ib_MASK1F as u32 - ImpliedAccess::Shift_Ib_MASK1FMOD9 as u32, 2);
				const_assert_eq!(ImpliedAccess::Shift_Ib_MASK3F as u32 - ImpliedAccess::Shift_Ib_MASK1FMOD9 as u32, 3);
				let mask = if e == 2 { 0x1F } else { 0x3F };
				match self.immediate8() & mask {
					0 => return RflagsInfo::None as usize,
					1 => {
						if result == RflagsInfo::W_c_U_o as usize {
							return RflagsInfo::W_co as usize;
						} else if result == RflagsInfo::R_c_W_c_U_o as usize {
							return RflagsInfo::R_c_W_co as usize;
						} else {
							debug_assert_eq!(result, RflagsInfo::W_cpsz_U_ao as usize);
							return RflagsInfo::W_copsz_U_a as usize;
						}
					}
					_ => {}
				}
			}
			4 => {
				const_assert_eq!(ImpliedAccess::Clear_rflags as u32 - ImpliedAccess::Shift_Ib_MASK1FMOD9 as u32, 4);
				if self.op0_register() == self.op1_register() && self.op0_kind() == OpKind::Register && self.op1_kind() == OpKind::Register {
					if self.mnemonic() == Mnemonic::Xor {
						return RflagsInfo::C_cos_S_pz_U_a as usize;
					} else {
						return RflagsInfo::C_acos_S_pz as usize;
					}
				}
			}
			_ => {}
		}
		result
	}

	/// All flags that are read by the CPU when executing the instruction.
	/// This method returns a [`RflagsBits`] value. See also [`rflags_modified()`].
	///
	/// [`RflagsBits`]: struct.RflagsBits.html
	/// [`rflags_modified()`]: #method.rflags_modified
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // adc rsi,rcx
	/// // xor rdi,5Ah
	/// let bytes = b"\x48\x11\xCE\x48\x83\xF7\x5A";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	///
	/// // adc rsi,rcx
	/// let instr = decoder.decode();
	/// assert_eq!(instr.rflags_read(), RflagsBits::CF);
	/// assert_eq!(instr.rflags_written(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	/// assert_eq!(instr.rflags_cleared(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_set(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_undefined(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_modified(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	///
	/// // xor rdi,5Ah
	/// let instr = decoder.decode();
	/// assert_eq!(instr.rflags_read(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_written(), RflagsBits::SF | RflagsBits::ZF | RflagsBits::PF);
	/// assert_eq!(instr.rflags_cleared(), RflagsBits::OF | RflagsBits::CF);
	/// assert_eq!(instr.rflags_set(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_undefined(), RflagsBits::AF);
	/// assert_eq!(instr.rflags_modified(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	/// ```
	#[must_use]
	#[inline]
	pub fn rflags_read(&self) -> u32 {
		// SAFETY: index is an RflagsInfo which is a valid index into this table. The index is generated and valid
		unsafe { *crate::info::rflags_table::FLAGS_READ.get_unchecked(self.rflags_info()) as u32 }
	}

	/// All flags that are written by the CPU, except those flags that are known to be undefined, always set or always cleared.
	/// This method returns a [`RflagsBits`] value. See also [`rflags_modified()`].
	///
	/// [`RflagsBits`]: struct.RflagsBits.html
	/// [`rflags_modified()`]: #method.rflags_modified
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // adc rsi,rcx
	/// // xor rdi,5Ah
	/// let bytes = b"\x48\x11\xCE\x48\x83\xF7\x5A";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	///
	/// // adc rsi,rcx
	/// let instr = decoder.decode();
	/// assert_eq!(instr.rflags_read(), RflagsBits::CF);
	/// assert_eq!(instr.rflags_written(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	/// assert_eq!(instr.rflags_cleared(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_set(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_undefined(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_modified(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	///
	/// // xor rdi,5Ah
	/// let instr = decoder.decode();
	/// assert_eq!(instr.rflags_read(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_written(), RflagsBits::SF | RflagsBits::ZF | RflagsBits::PF);
	/// assert_eq!(instr.rflags_cleared(), RflagsBits::OF | RflagsBits::CF);
	/// assert_eq!(instr.rflags_set(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_undefined(), RflagsBits::AF);
	/// assert_eq!(instr.rflags_modified(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	/// ```
	#[must_use]
	#[inline]
	pub fn rflags_written(&self) -> u32 {
		// SAFETY: index is an RflagsInfo which is a valid index into this table. The index is generated and valid
		unsafe { *crate::info::rflags_table::FLAGS_WRITTEN.get_unchecked(self.rflags_info()) as u32 }
	}

	/// All flags that are always cleared by the CPU.
	/// This method returns a [`RflagsBits`] value. See also [`rflags_modified()`].
	///
	/// [`RflagsBits`]: struct.RflagsBits.html
	/// [`rflags_modified()`]: #method.rflags_modified
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // adc rsi,rcx
	/// // xor rdi,5Ah
	/// let bytes = b"\x48\x11\xCE\x48\x83\xF7\x5A";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	///
	/// // adc rsi,rcx
	/// let instr = decoder.decode();
	/// assert_eq!(instr.rflags_read(), RflagsBits::CF);
	/// assert_eq!(instr.rflags_written(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	/// assert_eq!(instr.rflags_cleared(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_set(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_undefined(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_modified(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	///
	/// // xor rdi,5Ah
	/// let instr = decoder.decode();
	/// assert_eq!(instr.rflags_read(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_written(), RflagsBits::SF | RflagsBits::ZF | RflagsBits::PF);
	/// assert_eq!(instr.rflags_cleared(), RflagsBits::OF | RflagsBits::CF);
	/// assert_eq!(instr.rflags_set(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_undefined(), RflagsBits::AF);
	/// assert_eq!(instr.rflags_modified(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	/// ```
	#[must_use]
	#[inline]
	pub fn rflags_cleared(&self) -> u32 {
		// SAFETY: index is an RflagsInfo which is a valid index into this table. The index is generated and valid
		unsafe { *crate::info::rflags_table::FLAGS_CLEARED.get_unchecked(self.rflags_info()) as u32 }
	}

	/// All flags that are always set by the CPU.
	/// This method returns a [`RflagsBits`] value. See also [`rflags_modified()`].
	///
	/// [`RflagsBits`]: struct.RflagsBits.html
	/// [`rflags_modified()`]: #method.rflags_modified
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // adc rsi,rcx
	/// // xor rdi,5Ah
	/// let bytes = b"\x48\x11\xCE\x48\x83\xF7\x5A";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	///
	/// // adc rsi,rcx
	/// let instr = decoder.decode();
	/// assert_eq!(instr.rflags_read(), RflagsBits::CF);
	/// assert_eq!(instr.rflags_written(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	/// assert_eq!(instr.rflags_cleared(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_set(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_undefined(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_modified(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	///
	/// // xor rdi,5Ah
	/// let instr = decoder.decode();
	/// assert_eq!(instr.rflags_read(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_written(), RflagsBits::SF | RflagsBits::ZF | RflagsBits::PF);
	/// assert_eq!(instr.rflags_cleared(), RflagsBits::OF | RflagsBits::CF);
	/// assert_eq!(instr.rflags_set(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_undefined(), RflagsBits::AF);
	/// assert_eq!(instr.rflags_modified(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	/// ```
	#[must_use]
	#[inline]
	pub fn rflags_set(&self) -> u32 {
		// SAFETY: index is an RflagsInfo which is a valid index into this table. The index is generated and valid
		unsafe { *crate::info::rflags_table::FLAGS_SET.get_unchecked(self.rflags_info()) as u32 }
	}

	/// All flags that are undefined after executing the instruction.
	/// This method returns a [`RflagsBits`] value. See also [`rflags_modified()`].
	///
	/// [`RflagsBits`]: struct.RflagsBits.html
	/// [`rflags_modified()`]: #method.rflags_modified
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // adc rsi,rcx
	/// // xor rdi,5Ah
	/// let bytes = b"\x48\x11\xCE\x48\x83\xF7\x5A";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	///
	/// // adc rsi,rcx
	/// let instr = decoder.decode();
	/// assert_eq!(instr.rflags_read(), RflagsBits::CF);
	/// assert_eq!(instr.rflags_written(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	/// assert_eq!(instr.rflags_cleared(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_set(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_undefined(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_modified(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	///
	/// // xor rdi,5Ah
	/// let instr = decoder.decode();
	/// assert_eq!(instr.rflags_read(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_written(), RflagsBits::SF | RflagsBits::ZF | RflagsBits::PF);
	/// assert_eq!(instr.rflags_cleared(), RflagsBits::OF | RflagsBits::CF);
	/// assert_eq!(instr.rflags_set(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_undefined(), RflagsBits::AF);
	/// assert_eq!(instr.rflags_modified(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	/// ```
	#[must_use]
	#[inline]
	pub fn rflags_undefined(&self) -> u32 {
		// SAFETY: index is an RflagsInfo which is a valid index into this table. The index is generated and valid
		unsafe { *crate::info::rflags_table::FLAGS_UNDEFINED.get_unchecked(self.rflags_info()) as u32 }
	}

	/// All flags that are modified by the CPU. This is `rflags_written() + rflags_cleared() + rflags_set() + rflags_undefined()`. This method returns a [`RflagsBits`] value.
	///
	/// [`RflagsBits`]: struct.RflagsBits.html
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // adc rsi,rcx
	/// // xor rdi,5Ah
	/// let bytes = b"\x48\x11\xCE\x48\x83\xF7\x5A";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	///
	/// // adc rsi,rcx
	/// let instr = decoder.decode();
	/// assert_eq!(instr.rflags_read(), RflagsBits::CF);
	/// assert_eq!(instr.rflags_written(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	/// assert_eq!(instr.rflags_cleared(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_set(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_undefined(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_modified(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	///
	/// // xor rdi,5Ah
	/// let instr = decoder.decode();
	/// assert_eq!(instr.rflags_read(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_written(), RflagsBits::SF | RflagsBits::ZF | RflagsBits::PF);
	/// assert_eq!(instr.rflags_cleared(), RflagsBits::OF | RflagsBits::CF);
	/// assert_eq!(instr.rflags_set(), RflagsBits::NONE);
	/// assert_eq!(instr.rflags_undefined(), RflagsBits::AF);
	/// assert_eq!(instr.rflags_modified(), RflagsBits::OF | RflagsBits::SF | RflagsBits::ZF | RflagsBits::AF | RflagsBits::CF | RflagsBits::PF);
	/// ```
	#[must_use]
	#[inline]
	pub fn rflags_modified(&self) -> u32 {
		// SAFETY: index is an RflagsInfo which is a valid index into this table. The index is generated and valid
		unsafe { *crate::info::rflags_table::FLAGS_MODIFIED.get_unchecked(self.rflags_info()) as u32 }
	}

	/// Checks if it's a `Jcc SHORT` or `Jcc NEAR` instruction
	#[must_use]
	#[inline]
	pub fn is_jcc_short_or_near(&self) -> bool {
		self.code().is_jcc_short_or_near()
	}

	/// Checks if it's a `Jcc NEAR` instruction
	#[must_use]
	#[inline]
	pub fn is_jcc_near(&self) -> bool {
		self.code().is_jcc_near()
	}

	/// Checks if it's a `Jcc SHORT` instruction
	#[must_use]
	#[inline]
	pub fn is_jcc_short(&self) -> bool {
		self.code().is_jcc_short()
	}

	/// Checks if it's a `JMP SHORT` instruction
	#[must_use]
	#[inline]
	pub fn is_jmp_short(&self) -> bool {
		self.code().is_jmp_short()
	}

	/// Checks if it's a `JMP NEAR` instruction
	#[must_use]
	#[inline]
	pub fn is_jmp_near(&self) -> bool {
		self.code().is_jmp_near()
	}

	/// Checks if it's a `JMP SHORT` or a `JMP NEAR` instruction
	#[must_use]
	#[inline]
	pub fn is_jmp_short_or_near(&self) -> bool {
		self.code().is_jmp_short_or_near()
	}

	/// Checks if it's a `JMP FAR` instruction
	#[must_use]
	#[inline]
	pub fn is_jmp_far(&self) -> bool {
		self.code().is_jmp_far()
	}

	/// Checks if it's a `CALL NEAR` instruction
	#[must_use]
	#[inline]
	pub fn is_call_near(&self) -> bool {
		self.code().is_call_near()
	}

	/// Checks if it's a `CALL FAR` instruction
	#[must_use]
	#[inline]
	pub fn is_call_far(&self) -> bool {
		self.code().is_call_far()
	}

	/// Checks if it's a `JMP NEAR reg/[mem]` instruction
	#[must_use]
	#[inline]
	pub fn is_jmp_near_indirect(&self) -> bool {
		self.code().is_jmp_near_indirect()
	}

	/// Checks if it's a `JMP FAR [mem]` instruction
	#[must_use]
	#[inline]
	pub fn is_jmp_far_indirect(&self) -> bool {
		self.code().is_jmp_far_indirect()
	}

	/// Checks if it's a `CALL NEAR reg/[mem]` instruction
	#[must_use]
	#[inline]
	pub fn is_call_near_indirect(&self) -> bool {
		self.code().is_call_near_indirect()
	}

	/// Checks if it's a `CALL FAR [mem]` instruction
	#[must_use]
	#[inline]
	pub fn is_call_far_indirect(&self) -> bool {
		self.code().is_call_far_indirect()
	}

	/// Negates the condition code, eg. `JE` -> `JNE`. Can be used if it's `Jcc`, `SETcc`, `CMOVcc`, `LOOPcc`
	/// and does nothing if the instruction doesn't have a condition code.
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // setbe al
	/// let bytes = b"\x0F\x96\xC0";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	///
	/// let mut instr = decoder.decode();
	/// assert_eq!(instr.code(), Code::Setbe_rm8);
	/// assert_eq!(instr.condition_code(), ConditionCode::be);
	/// instr.negate_condition_code();
	/// assert_eq!(instr.code(), Code::Seta_rm8);
	/// assert_eq!(instr.condition_code(), ConditionCode::a);
	/// ```
	#[inline]
	pub fn negate_condition_code(&mut self) {
		self.set_code(self.code().negate_condition_code())
	}

	/// Converts `Jcc/JMP NEAR` to `Jcc/JMP SHORT` and does nothing if it's not a `Jcc/JMP NEAR` instruction
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // jbe near ptr label
	/// let bytes = b"\x0F\x86\x5A\xA5\x12\x34";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	///
	/// let mut instr = decoder.decode();
	/// assert_eq!(instr.code(), Code::Jbe_rel32_64);
	/// instr.as_short_branch();
	/// assert_eq!(instr.code(), Code::Jbe_rel8_64);
	/// instr.as_short_branch();
	/// assert_eq!(instr.code(), Code::Jbe_rel8_64);
	/// ```
	#[inline]
	pub fn as_short_branch(&mut self) {
		self.set_code(self.code().as_short_branch())
	}

	/// Converts `Jcc/JMP SHORT` to `Jcc/JMP NEAR` and does nothing if it's not a `Jcc/JMP SHORT` instruction
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // jbe short label
	/// let bytes = b"\x76\x5A";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	///
	/// let mut instr = decoder.decode();
	/// assert_eq!(instr.code(), Code::Jbe_rel8_64);
	/// instr.as_near_branch();
	/// assert_eq!(instr.code(), Code::Jbe_rel32_64);
	/// instr.as_near_branch();
	/// assert_eq!(instr.code(), Code::Jbe_rel32_64);
	/// ```
	#[inline]
	pub fn as_near_branch(&mut self) {
		self.set_code(self.code().as_near_branch())
	}

	/// Gets the condition code if it's `Jcc`, `SETcc`, `CMOVcc`, `LOOPcc` else [`ConditionCode::None`] is returned
	///
	/// [`ConditionCode::None`]: enum.ConditionCode.html#variant.None
	///
	/// # Examples
	///
	/// ```
	/// use iced_x86::*;
	///
	/// // setbe al
	/// // jl short label
	/// // cmovne ecx,esi
	/// // nop
	/// let bytes = b"\x0F\x96\xC0\x7C\x5A\x0F\x45\xCE\x90";
	/// let mut decoder = Decoder::new(64, bytes, DecoderOptions::NONE);
	///
	/// // setbe al
	/// let instr = decoder.decode();
	/// assert_eq!(instr.condition_code(), ConditionCode::be);
	///
	/// // jl short label
	/// let instr = decoder.decode();
	/// assert_eq!(instr.condition_code(), ConditionCode::l);
	///
	/// // cmovne ecx,esi
	/// let instr = decoder.decode();
	/// assert_eq!(instr.condition_code(), ConditionCode::ne);
	///
	/// // nop
	/// let instr = decoder.decode();
	/// assert_eq!(instr.condition_code(), ConditionCode::None);
	/// ```
	#[must_use]
	#[inline]
	pub fn condition_code(&self) -> ConditionCode {
		self.code().condition_code()
	}
}

#[cfg(all(feature = "encoder", feature = "op_code_info"))]
impl Instruction {
	/// Gets the [`OpCodeInfo`]
	///
	/// [`OpCodeInfo`]: struct.OpCodeInfo.html
	#[must_use]
	#[inline]
	pub fn op_code(&self) -> &'static OpCodeInfo {
		self.code().op_code()
	}
}

#[cfg(feature = "encoder")]
impl Instruction {
	fn init_memory_operand(instruction: &mut Instruction, memory: &MemoryOperand) {
		instruction_internal::internal_set_memory_base(instruction, memory.base);
		instruction_internal::internal_set_memory_index(instruction, memory.index);
		instruction.set_memory_index_scale(memory.scale);
		instruction.set_memory_displ_size(memory.displ_size);
		let addr_size = instruction_internal::get_address_size_in_bytes(memory.base, memory.index, memory.displ_size, CodeSize::Unknown);
		if addr_size == 8 {
			instruction.set_memory_displacement64(memory.displacement as u64);
		} else {
			instruction_internal::internal_set_memory_displacement64_lo(instruction, memory.displacement as u32);
		}
		instruction.set_is_broadcast(memory.is_broadcast);
		instruction.set_segment_prefix(memory.segment_prefix);
	}

	// GENERATOR-BEGIN: Create
	// ⚠️This was generated by GENERATOR!🦹‍♂️
	/// Creates an instruction with no operands
	///
	/// # Arguments
	///
	/// * `code`: Code value
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn with(code: Code) -> Self {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		debug_assert_eq!(instruction.op_count(), 0);
		instruction
	}

	/// Creates an instruction with 1 operand
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn with_reg(code: Code, register: Register) -> Self {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register);

		debug_assert_eq!(instruction.op_count(), 1);
		instruction
	}

	/// Creates an instruction with 1 operand
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `immediate`: op0: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_i32(code: Code, immediate: i32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		instruction_internal::initialize_signed_immediate(&mut instruction, 0, immediate as i64)?;

		debug_assert_eq!(instruction.op_count(), 1);
		Ok(instruction)
	}

	/// Creates an instruction with 1 operand
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `immediate`: op0: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_i32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_i32(code: Code, immediate: i32) -> Self {
		Instruction::try_with_i32(code, immediate).unwrap()
	}

	/// Creates an instruction with 1 operand
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `immediate`: op0: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_u32(code: Code, immediate: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 0, immediate as u64)?;

		debug_assert_eq!(instruction.op_count(), 1);
		Ok(instruction)
	}

	/// Creates an instruction with 1 operand
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `immediate`: op0: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_u32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_u32(code: Code, immediate: u32) -> Self {
		Instruction::try_with_u32(code, immediate).unwrap()
	}

	/// Creates an instruction with 1 operand
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `memory`: op0: Memory operand
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn with_mem(code: Code, memory: MemoryOperand) -> Self {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		debug_assert_eq!(instruction.op_count(), 1);
		instruction
	}

	/// Creates an instruction with 2 operands
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn with_reg_reg(code: Code, register1: Register, register2: Register) -> Self {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		debug_assert_eq!(instruction.op_count(), 2);
		instruction
	}

	/// Creates an instruction with 2 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	/// * `immediate`: op1: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_i32(code: Code, register: Register, immediate: i32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register);

		instruction_internal::initialize_signed_immediate(&mut instruction, 1, immediate as i64)?;

		debug_assert_eq!(instruction.op_count(), 2);
		Ok(instruction)
	}

	/// Creates an instruction with 2 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	/// * `immediate`: op1: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_i32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_i32(code: Code, register: Register, immediate: i32) -> Self {
		Instruction::try_with_reg_i32(code, register, immediate).unwrap()
	}

	/// Creates an instruction with 2 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	/// * `immediate`: op1: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_u32(code: Code, register: Register, immediate: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register);

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 1, immediate as u64)?;

		debug_assert_eq!(instruction.op_count(), 2);
		Ok(instruction)
	}

	/// Creates an instruction with 2 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	/// * `immediate`: op1: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_u32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_u32(code: Code, register: Register, immediate: u32) -> Self {
		Instruction::try_with_reg_u32(code, register, immediate).unwrap()
	}

	/// Creates an instruction with 2 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	/// * `immediate`: op1: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_i64(code: Code, register: Register, immediate: i64) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register);

		instruction_internal::initialize_signed_immediate(&mut instruction, 1, immediate)?;

		debug_assert_eq!(instruction.op_count(), 2);
		Ok(instruction)
	}

	/// Creates an instruction with 2 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	/// * `immediate`: op1: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_i64() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_i64(code: Code, register: Register, immediate: i64) -> Self {
		Instruction::try_with_reg_i64(code, register, immediate).unwrap()
	}

	/// Creates an instruction with 2 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	/// * `immediate`: op1: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_u64(code: Code, register: Register, immediate: u64) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register);

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 1, immediate)?;

		debug_assert_eq!(instruction.op_count(), 2);
		Ok(instruction)
	}

	/// Creates an instruction with 2 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	/// * `immediate`: op1: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_u64() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_u64(code: Code, register: Register, immediate: u64) -> Self {
		Instruction::try_with_reg_u64(code, register, immediate).unwrap()
	}

	/// Creates an instruction with 2 operands
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	/// * `memory`: op1: Memory operand
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn with_reg_mem(code: Code, register: Register, memory: MemoryOperand) -> Self {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register);

		instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		debug_assert_eq!(instruction.op_count(), 2);
		instruction
	}

	/// Creates an instruction with 2 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `immediate`: op0: Immediate value
	/// * `register`: op1: Register
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_i32_reg(code: Code, immediate: i32, register: Register) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		instruction_internal::initialize_signed_immediate(&mut instruction, 0, immediate as i64)?;

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register);

		debug_assert_eq!(instruction.op_count(), 2);
		Ok(instruction)
	}

	/// Creates an instruction with 2 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `immediate`: op0: Immediate value
	/// * `register`: op1: Register
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_i32_reg() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_i32_reg(code: Code, immediate: i32, register: Register) -> Self {
		Instruction::try_with_i32_reg(code, immediate, register).unwrap()
	}

	/// Creates an instruction with 2 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `immediate`: op0: Immediate value
	/// * `register`: op1: Register
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_u32_reg(code: Code, immediate: u32, register: Register) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 0, immediate as u64)?;

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register);

		debug_assert_eq!(instruction.op_count(), 2);
		Ok(instruction)
	}

	/// Creates an instruction with 2 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `immediate`: op0: Immediate value
	/// * `register`: op1: Register
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_u32_reg() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_u32_reg(code: Code, immediate: u32, register: Register) -> Self {
		Instruction::try_with_u32_reg(code, immediate, register).unwrap()
	}

	/// Creates an instruction with 2 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `immediate1`: op0: Immediate value
	/// * `immediate2`: op1: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_i32_i32(code: Code, immediate1: i32, immediate2: i32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		instruction_internal::initialize_signed_immediate(&mut instruction, 0, immediate1 as i64)?;

		instruction_internal::initialize_signed_immediate(&mut instruction, 1, immediate2 as i64)?;

		debug_assert_eq!(instruction.op_count(), 2);
		Ok(instruction)
	}

	/// Creates an instruction with 2 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `immediate1`: op0: Immediate value
	/// * `immediate2`: op1: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_i32_i32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_i32_i32(code: Code, immediate1: i32, immediate2: i32) -> Self {
		Instruction::try_with_i32_i32(code, immediate1, immediate2).unwrap()
	}

	/// Creates an instruction with 2 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `immediate1`: op0: Immediate value
	/// * `immediate2`: op1: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_u32_u32(code: Code, immediate1: u32, immediate2: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 0, immediate1 as u64)?;

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 1, immediate2 as u64)?;

		debug_assert_eq!(instruction.op_count(), 2);
		Ok(instruction)
	}

	/// Creates an instruction with 2 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `immediate1`: op0: Immediate value
	/// * `immediate2`: op1: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_u32_u32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_u32_u32(code: Code, immediate1: u32, immediate2: u32) -> Self {
		Instruction::try_with_u32_u32(code, immediate1, immediate2).unwrap()
	}

	/// Creates an instruction with 2 operands
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `memory`: op0: Memory operand
	/// * `register`: op1: Register
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn with_mem_reg(code: Code, memory: MemoryOperand, register: Register) -> Self {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register);

		debug_assert_eq!(instruction.op_count(), 2);
		instruction
	}

	/// Creates an instruction with 2 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `memory`: op0: Memory operand
	/// * `immediate`: op1: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_mem_i32(code: Code, memory: MemoryOperand, immediate: i32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		instruction_internal::initialize_signed_immediate(&mut instruction, 1, immediate as i64)?;

		debug_assert_eq!(instruction.op_count(), 2);
		Ok(instruction)
	}

	/// Creates an instruction with 2 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `memory`: op0: Memory operand
	/// * `immediate`: op1: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_mem_i32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_mem_i32(code: Code, memory: MemoryOperand, immediate: i32) -> Self {
		Instruction::try_with_mem_i32(code, memory, immediate).unwrap()
	}

	/// Creates an instruction with 2 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `memory`: op0: Memory operand
	/// * `immediate`: op1: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_mem_u32(code: Code, memory: MemoryOperand, immediate: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 1, immediate as u64)?;

		debug_assert_eq!(instruction.op_count(), 2);
		Ok(instruction)
	}

	/// Creates an instruction with 2 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `memory`: op0: Memory operand
	/// * `immediate`: op1: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_mem_u32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_mem_u32(code: Code, memory: MemoryOperand, immediate: u32) -> Self {
		Instruction::try_with_mem_u32(code, memory, immediate).unwrap()
	}

	/// Creates an instruction with 3 operands
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `register3`: op2: Register
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn with_reg_reg_reg(code: Code, register1: Register, register2: Register, register3: Register) -> Self {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op2_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op2_register(&mut instruction, register3);

		debug_assert_eq!(instruction.op_count(), 3);
		instruction
	}

	/// Creates an instruction with 3 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `immediate`: op2: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_reg_i32(code: Code, register1: Register, register2: Register, immediate: i32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		instruction_internal::initialize_signed_immediate(&mut instruction, 2, immediate as i64)?;

		debug_assert_eq!(instruction.op_count(), 3);
		Ok(instruction)
	}

	/// Creates an instruction with 3 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `immediate`: op2: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_reg_i32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_reg_i32(code: Code, register1: Register, register2: Register, immediate: i32) -> Self {
		Instruction::try_with_reg_reg_i32(code, register1, register2, immediate).unwrap()
	}

	/// Creates an instruction with 3 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `immediate`: op2: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_reg_u32(code: Code, register1: Register, register2: Register, immediate: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 2, immediate as u64)?;

		debug_assert_eq!(instruction.op_count(), 3);
		Ok(instruction)
	}

	/// Creates an instruction with 3 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `immediate`: op2: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_reg_u32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_reg_u32(code: Code, register1: Register, register2: Register, immediate: u32) -> Self {
		Instruction::try_with_reg_reg_u32(code, register1, register2, immediate).unwrap()
	}

	/// Creates an instruction with 3 operands
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `memory`: op2: Memory operand
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn with_reg_reg_mem(code: Code, register1: Register, register2: Register, memory: MemoryOperand) -> Self {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		instruction_internal::internal_set_op2_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		debug_assert_eq!(instruction.op_count(), 3);
		instruction
	}

	/// Creates an instruction with 3 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	/// * `immediate1`: op1: Immediate value
	/// * `immediate2`: op2: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_i32_i32(code: Code, register: Register, immediate1: i32, immediate2: i32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register);

		instruction_internal::initialize_signed_immediate(&mut instruction, 1, immediate1 as i64)?;

		instruction_internal::initialize_signed_immediate(&mut instruction, 2, immediate2 as i64)?;

		debug_assert_eq!(instruction.op_count(), 3);
		Ok(instruction)
	}

	/// Creates an instruction with 3 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	/// * `immediate1`: op1: Immediate value
	/// * `immediate2`: op2: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_i32_i32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_i32_i32(code: Code, register: Register, immediate1: i32, immediate2: i32) -> Self {
		Instruction::try_with_reg_i32_i32(code, register, immediate1, immediate2).unwrap()
	}

	/// Creates an instruction with 3 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	/// * `immediate1`: op1: Immediate value
	/// * `immediate2`: op2: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_u32_u32(code: Code, register: Register, immediate1: u32, immediate2: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register);

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 1, immediate1 as u64)?;

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 2, immediate2 as u64)?;

		debug_assert_eq!(instruction.op_count(), 3);
		Ok(instruction)
	}

	/// Creates an instruction with 3 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	/// * `immediate1`: op1: Immediate value
	/// * `immediate2`: op2: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_u32_u32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_u32_u32(code: Code, register: Register, immediate1: u32, immediate2: u32) -> Self {
		Instruction::try_with_reg_u32_u32(code, register, immediate1, immediate2).unwrap()
	}

	/// Creates an instruction with 3 operands
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `memory`: op1: Memory operand
	/// * `register2`: op2: Register
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn with_reg_mem_reg(code: Code, register1: Register, memory: MemoryOperand, register2: Register) -> Self {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op2_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op2_register(&mut instruction, register2);

		debug_assert_eq!(instruction.op_count(), 3);
		instruction
	}

	/// Creates an instruction with 3 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	/// * `memory`: op1: Memory operand
	/// * `immediate`: op2: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_mem_i32(code: Code, register: Register, memory: MemoryOperand, immediate: i32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register);

		instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		instruction_internal::initialize_signed_immediate(&mut instruction, 2, immediate as i64)?;

		debug_assert_eq!(instruction.op_count(), 3);
		Ok(instruction)
	}

	/// Creates an instruction with 3 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	/// * `memory`: op1: Memory operand
	/// * `immediate`: op2: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_mem_i32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_mem_i32(code: Code, register: Register, memory: MemoryOperand, immediate: i32) -> Self {
		Instruction::try_with_reg_mem_i32(code, register, memory, immediate).unwrap()
	}

	/// Creates an instruction with 3 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	/// * `memory`: op1: Memory operand
	/// * `immediate`: op2: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_mem_u32(code: Code, register: Register, memory: MemoryOperand, immediate: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register);

		instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 2, immediate as u64)?;

		debug_assert_eq!(instruction.op_count(), 3);
		Ok(instruction)
	}

	/// Creates an instruction with 3 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: op0: Register
	/// * `memory`: op1: Memory operand
	/// * `immediate`: op2: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_mem_u32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_mem_u32(code: Code, register: Register, memory: MemoryOperand, immediate: u32) -> Self {
		Instruction::try_with_reg_mem_u32(code, register, memory, immediate).unwrap()
	}

	/// Creates an instruction with 3 operands
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `memory`: op0: Memory operand
	/// * `register1`: op1: Register
	/// * `register2`: op2: Register
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn with_mem_reg_reg(code: Code, memory: MemoryOperand, register1: Register, register2: Register) -> Self {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op2_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op2_register(&mut instruction, register2);

		debug_assert_eq!(instruction.op_count(), 3);
		instruction
	}

	/// Creates an instruction with 3 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `memory`: op0: Memory operand
	/// * `register`: op1: Register
	/// * `immediate`: op2: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_mem_reg_i32(code: Code, memory: MemoryOperand, register: Register, immediate: i32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register);

		instruction_internal::initialize_signed_immediate(&mut instruction, 2, immediate as i64)?;

		debug_assert_eq!(instruction.op_count(), 3);
		Ok(instruction)
	}

	/// Creates an instruction with 3 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `memory`: op0: Memory operand
	/// * `register`: op1: Register
	/// * `immediate`: op2: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_mem_reg_i32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_mem_reg_i32(code: Code, memory: MemoryOperand, register: Register, immediate: i32) -> Self {
		Instruction::try_with_mem_reg_i32(code, memory, register, immediate).unwrap()
	}

	/// Creates an instruction with 3 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `memory`: op0: Memory operand
	/// * `register`: op1: Register
	/// * `immediate`: op2: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_mem_reg_u32(code: Code, memory: MemoryOperand, register: Register, immediate: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register);

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 2, immediate as u64)?;

		debug_assert_eq!(instruction.op_count(), 3);
		Ok(instruction)
	}

	/// Creates an instruction with 3 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `memory`: op0: Memory operand
	/// * `register`: op1: Register
	/// * `immediate`: op2: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_mem_reg_u32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_mem_reg_u32(code: Code, memory: MemoryOperand, register: Register, immediate: u32) -> Self {
		Instruction::try_with_mem_reg_u32(code, memory, register, immediate).unwrap()
	}

	/// Creates an instruction with 4 operands
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `register3`: op2: Register
	/// * `register4`: op3: Register
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn with_reg_reg_reg_reg(code: Code, register1: Register, register2: Register, register3: Register, register4: Register) -> Self {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op2_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op2_register(&mut instruction, register3);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op3_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op3_register(&mut instruction, register4);

		debug_assert_eq!(instruction.op_count(), 4);
		instruction
	}

	/// Creates an instruction with 4 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `register3`: op2: Register
	/// * `immediate`: op3: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_reg_reg_i32(code: Code, register1: Register, register2: Register, register3: Register, immediate: i32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op2_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op2_register(&mut instruction, register3);

		instruction_internal::initialize_signed_immediate(&mut instruction, 3, immediate as i64)?;

		debug_assert_eq!(instruction.op_count(), 4);
		Ok(instruction)
	}

	/// Creates an instruction with 4 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `register3`: op2: Register
	/// * `immediate`: op3: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_reg_reg_i32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_reg_reg_i32(code: Code, register1: Register, register2: Register, register3: Register, immediate: i32) -> Self {
		Instruction::try_with_reg_reg_reg_i32(code, register1, register2, register3, immediate).unwrap()
	}

	/// Creates an instruction with 4 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `register3`: op2: Register
	/// * `immediate`: op3: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_reg_reg_u32(code: Code, register1: Register, register2: Register, register3: Register, immediate: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op2_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op2_register(&mut instruction, register3);

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 3, immediate as u64)?;

		debug_assert_eq!(instruction.op_count(), 4);
		Ok(instruction)
	}

	/// Creates an instruction with 4 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `register3`: op2: Register
	/// * `immediate`: op3: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_reg_reg_u32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_reg_reg_u32(code: Code, register1: Register, register2: Register, register3: Register, immediate: u32) -> Self {
		Instruction::try_with_reg_reg_reg_u32(code, register1, register2, register3, immediate).unwrap()
	}

	/// Creates an instruction with 4 operands
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `register3`: op2: Register
	/// * `memory`: op3: Memory operand
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn with_reg_reg_reg_mem(code: Code, register1: Register, register2: Register, register3: Register, memory: MemoryOperand) -> Self {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op2_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op2_register(&mut instruction, register3);

		instruction_internal::internal_set_op3_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		debug_assert_eq!(instruction.op_count(), 4);
		instruction
	}

	/// Creates an instruction with 4 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `immediate1`: op2: Immediate value
	/// * `immediate2`: op3: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_reg_i32_i32(code: Code, register1: Register, register2: Register, immediate1: i32, immediate2: i32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		instruction_internal::initialize_signed_immediate(&mut instruction, 2, immediate1 as i64)?;

		instruction_internal::initialize_signed_immediate(&mut instruction, 3, immediate2 as i64)?;

		debug_assert_eq!(instruction.op_count(), 4);
		Ok(instruction)
	}

	/// Creates an instruction with 4 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `immediate1`: op2: Immediate value
	/// * `immediate2`: op3: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_reg_i32_i32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_reg_i32_i32(code: Code, register1: Register, register2: Register, immediate1: i32, immediate2: i32) -> Self {
		Instruction::try_with_reg_reg_i32_i32(code, register1, register2, immediate1, immediate2).unwrap()
	}

	/// Creates an instruction with 4 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `immediate1`: op2: Immediate value
	/// * `immediate2`: op3: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_reg_u32_u32(code: Code, register1: Register, register2: Register, immediate1: u32, immediate2: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 2, immediate1 as u64)?;

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 3, immediate2 as u64)?;

		debug_assert_eq!(instruction.op_count(), 4);
		Ok(instruction)
	}

	/// Creates an instruction with 4 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `immediate1`: op2: Immediate value
	/// * `immediate2`: op3: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_reg_u32_u32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_reg_u32_u32(code: Code, register1: Register, register2: Register, immediate1: u32, immediate2: u32) -> Self {
		Instruction::try_with_reg_reg_u32_u32(code, register1, register2, immediate1, immediate2).unwrap()
	}

	/// Creates an instruction with 4 operands
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `memory`: op2: Memory operand
	/// * `register3`: op3: Register
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn with_reg_reg_mem_reg(code: Code, register1: Register, register2: Register, memory: MemoryOperand, register3: Register) -> Self {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		instruction_internal::internal_set_op2_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op3_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op3_register(&mut instruction, register3);

		debug_assert_eq!(instruction.op_count(), 4);
		instruction
	}

	/// Creates an instruction with 4 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `memory`: op2: Memory operand
	/// * `immediate`: op3: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_reg_mem_i32(code: Code, register1: Register, register2: Register, memory: MemoryOperand, immediate: i32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		instruction_internal::internal_set_op2_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		instruction_internal::initialize_signed_immediate(&mut instruction, 3, immediate as i64)?;

		debug_assert_eq!(instruction.op_count(), 4);
		Ok(instruction)
	}

	/// Creates an instruction with 4 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `memory`: op2: Memory operand
	/// * `immediate`: op3: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_reg_mem_i32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_reg_mem_i32(code: Code, register1: Register, register2: Register, memory: MemoryOperand, immediate: i32) -> Self {
		Instruction::try_with_reg_reg_mem_i32(code, register1, register2, memory, immediate).unwrap()
	}

	/// Creates an instruction with 4 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `memory`: op2: Memory operand
	/// * `immediate`: op3: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_reg_mem_u32(code: Code, register1: Register, register2: Register, memory: MemoryOperand, immediate: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		instruction_internal::internal_set_op2_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 3, immediate as u64)?;

		debug_assert_eq!(instruction.op_count(), 4);
		Ok(instruction)
	}

	/// Creates an instruction with 4 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `memory`: op2: Memory operand
	/// * `immediate`: op3: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_reg_mem_u32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_reg_mem_u32(code: Code, register1: Register, register2: Register, memory: MemoryOperand, immediate: u32) -> Self {
		Instruction::try_with_reg_reg_mem_u32(code, register1, register2, memory, immediate).unwrap()
	}

	/// Creates an instruction with 5 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `register3`: op2: Register
	/// * `register4`: op3: Register
	/// * `immediate`: op4: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_reg_reg_reg_i32(code: Code, register1: Register, register2: Register, register3: Register, register4: Register, immediate: i32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op2_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op2_register(&mut instruction, register3);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op3_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op3_register(&mut instruction, register4);

		instruction_internal::initialize_signed_immediate(&mut instruction, 4, immediate as i64)?;

		debug_assert_eq!(instruction.op_count(), 5);
		Ok(instruction)
	}

	/// Creates an instruction with 5 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `register3`: op2: Register
	/// * `register4`: op3: Register
	/// * `immediate`: op4: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_reg_reg_reg_i32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_reg_reg_reg_i32(code: Code, register1: Register, register2: Register, register3: Register, register4: Register, immediate: i32) -> Self {
		Instruction::try_with_reg_reg_reg_reg_i32(code, register1, register2, register3, register4, immediate).unwrap()
	}

	/// Creates an instruction with 5 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `register3`: op2: Register
	/// * `register4`: op3: Register
	/// * `immediate`: op4: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_reg_reg_reg_u32(code: Code, register1: Register, register2: Register, register3: Register, register4: Register, immediate: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op2_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op2_register(&mut instruction, register3);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op3_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op3_register(&mut instruction, register4);

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 4, immediate as u64)?;

		debug_assert_eq!(instruction.op_count(), 5);
		Ok(instruction)
	}

	/// Creates an instruction with 5 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `register3`: op2: Register
	/// * `register4`: op3: Register
	/// * `immediate`: op4: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_reg_reg_reg_u32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_reg_reg_reg_u32(code: Code, register1: Register, register2: Register, register3: Register, register4: Register, immediate: u32) -> Self {
		Instruction::try_with_reg_reg_reg_reg_u32(code, register1, register2, register3, register4, immediate).unwrap()
	}

	/// Creates an instruction with 5 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `register3`: op2: Register
	/// * `memory`: op3: Memory operand
	/// * `immediate`: op4: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_reg_reg_mem_i32(code: Code, register1: Register, register2: Register, register3: Register, memory: MemoryOperand, immediate: i32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op2_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op2_register(&mut instruction, register3);

		instruction_internal::internal_set_op3_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		instruction_internal::initialize_signed_immediate(&mut instruction, 4, immediate as i64)?;

		debug_assert_eq!(instruction.op_count(), 5);
		Ok(instruction)
	}

	/// Creates an instruction with 5 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `register3`: op2: Register
	/// * `memory`: op3: Memory operand
	/// * `immediate`: op4: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_reg_reg_mem_i32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_reg_reg_mem_i32(code: Code, register1: Register, register2: Register, register3: Register, memory: MemoryOperand, immediate: i32) -> Self {
		Instruction::try_with_reg_reg_reg_mem_i32(code, register1, register2, register3, memory, immediate).unwrap()
	}

	/// Creates an instruction with 5 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `register3`: op2: Register
	/// * `memory`: op3: Memory operand
	/// * `immediate`: op4: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_reg_reg_mem_u32(code: Code, register1: Register, register2: Register, register3: Register, memory: MemoryOperand, immediate: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op2_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op2_register(&mut instruction, register3);

		instruction_internal::internal_set_op3_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 4, immediate as u64)?;

		debug_assert_eq!(instruction.op_count(), 5);
		Ok(instruction)
	}

	/// Creates an instruction with 5 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `register3`: op2: Register
	/// * `memory`: op3: Memory operand
	/// * `immediate`: op4: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_reg_reg_mem_u32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_reg_reg_mem_u32(code: Code, register1: Register, register2: Register, register3: Register, memory: MemoryOperand, immediate: u32) -> Self {
		Instruction::try_with_reg_reg_reg_mem_u32(code, register1, register2, register3, memory, immediate).unwrap()
	}

	/// Creates an instruction with 5 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `memory`: op2: Memory operand
	/// * `register3`: op3: Register
	/// * `immediate`: op4: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_reg_mem_reg_i32(code: Code, register1: Register, register2: Register, memory: MemoryOperand, register3: Register, immediate: i32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		instruction_internal::internal_set_op2_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op3_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op3_register(&mut instruction, register3);

		instruction_internal::initialize_signed_immediate(&mut instruction, 4, immediate as i64)?;

		debug_assert_eq!(instruction.op_count(), 5);
		Ok(instruction)
	}

	/// Creates an instruction with 5 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `memory`: op2: Memory operand
	/// * `register3`: op3: Register
	/// * `immediate`: op4: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_reg_mem_reg_i32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_reg_mem_reg_i32(code: Code, register1: Register, register2: Register, memory: MemoryOperand, register3: Register, immediate: i32) -> Self {
		Instruction::try_with_reg_reg_mem_reg_i32(code, register1, register2, memory, register3, immediate).unwrap()
	}

	/// Creates an instruction with 5 operands
	///
	/// # Errors
	///
	/// Fails if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `memory`: op2: Memory operand
	/// * `register3`: op3: Register
	/// * `immediate`: op4: Immediate value
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_reg_reg_mem_reg_u32(code: Code, register1: Register, register2: Register, memory: MemoryOperand, register3: Register, immediate: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op0_register(&mut instruction, register1);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op1_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op1_register(&mut instruction, register2);

		instruction_internal::internal_set_op2_kind(&mut instruction, OpKind::Memory);
		Instruction::init_memory_operand(&mut instruction, &memory);

		const_assert_eq!(OpKind::Register as u32, 0);
		//instruction_internal::internal_set_op3_kind(&mut instruction, OpKind::Register);
		instruction_internal::internal_set_op3_register(&mut instruction, register3);

		instruction_internal::initialize_unsigned_immediate(&mut instruction, 4, immediate as u64)?;

		debug_assert_eq!(instruction.op_count(), 5);
		Ok(instruction)
	}

	/// Creates an instruction with 5 operands
	///
	/// # Panics
	///
	/// Panics if the immediate is invalid
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register1`: op0: Register
	/// * `register2`: op1: Register
	/// * `memory`: op2: Memory operand
	/// * `register3`: op3: Register
	/// * `immediate`: op4: Immediate value
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_reg_reg_mem_reg_u32() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_reg_mem_reg_u32(code: Code, register1: Register, register2: Register, memory: MemoryOperand, register3: Register, immediate: u32) -> Self {
		Instruction::try_with_reg_reg_mem_reg_u32(code, register1, register2, memory, register3, immediate).unwrap()
	}

	/// Creates a new near/short branch instruction
	///
	/// # Errors
	///
	/// Fails if the created instruction doesn't have a near branch operand
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `target`: Target address
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_branch(code: Code, target: u64) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		instruction_internal::internal_set_op0_kind(&mut instruction, instruction_internal::get_near_branch_op_kind(code, 0)?);
		instruction.set_near_branch64(target);

		debug_assert_eq!(instruction.op_count(), 1);
		Ok(instruction)
	}

	/// Creates a new near/short branch instruction
	///
	/// # Panics
	///
	/// Panics if the created instruction doesn't have a near branch operand
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `target`: Target address
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_branch() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_branch(code: Code, target: u64) -> Self {
		Instruction::try_with_branch(code, target).unwrap()
	}

	/// Creates a new far branch instruction
	///
	/// # Errors
	///
	/// Fails if the created instruction doesn't have a far branch operand
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `selector`: Selector/segment value
	/// * `offset`: Offset
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_far_branch(code: Code, selector: u16, offset: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, code);

		instruction_internal::internal_set_op0_kind(&mut instruction, instruction_internal::get_far_branch_op_kind(code, 0)?);
		instruction.set_far_branch_selector(selector);
		instruction.set_far_branch32(offset);

		debug_assert_eq!(instruction.op_count(), 1);
		Ok(instruction)
	}

	/// Creates a new far branch instruction
	///
	/// # Panics
	///
	/// Panics if the created instruction doesn't have a far branch operand
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `selector`: Selector/segment value
	/// * `offset`: Offset
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_far_branch() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_far_branch(code: Code, selector: u16, offset: u32) -> Self {
		Instruction::try_with_far_branch(code, selector, offset).unwrap()
	}

	/// Creates a new `XBEGIN` instruction
	///
	/// # Errors
	///
	/// Fails if `bitness` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `bitness`: 16, 32, or 64
	/// * `target`: Target address
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_xbegin(bitness: u32, target: u64) -> Result<Self, IcedError> {
		let mut instruction = Self::default();

		match bitness {
			16 => {
				instruction_internal::internal_set_code(&mut instruction, Code::Xbegin_rel16);
				instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::NearBranch32);
				instruction.set_near_branch32(target as u32);
			}

			32 => {
				instruction_internal::internal_set_code(&mut instruction, Code::Xbegin_rel32);
				instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::NearBranch32);
				instruction.set_near_branch32(target as u32);
			}

			64 => {
				instruction_internal::internal_set_code(&mut instruction, Code::Xbegin_rel32);
				instruction_internal::internal_set_op0_kind(&mut instruction, OpKind::NearBranch64);
				instruction.set_near_branch64(target);
			}

			_ => return Err(IcedError::new("Invalid bitness")),
		}

		debug_assert_eq!(instruction.op_count(), 1);
		Ok(instruction)
	}

	/// Creates a new `XBEGIN` instruction
	///
	/// # Panics
	///
	/// Panics if `bitness` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `bitness`: 16, 32, or 64
	/// * `target`: Target address
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_xbegin() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_xbegin(bitness: u32, target: u64) -> Self {
		Instruction::try_with_xbegin(bitness, target).unwrap()
	}

	/// Creates an instruction with a 64-bit memory offset as the second operand, eg. `mov al,[123456789ABCDEF0]`
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `register`: Register (`AL`, `AX`, `EAX`, `RAX`)
	/// * `address`: 64-bit address
	/// * `segment_prefix`: Segment override or [`Register::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	#[deprecated(since = "1.11.0", note = "Use with_reg_mem() with a MemoryOperand arg instead")]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_reg_mem64(code: Code, register: Register, address: u64, segment_prefix: Register) -> Self {
		Instruction::with_reg_mem(code, register, MemoryOperand::with_base_displ_size_bcst_seg(Register::None, address as i64, 8, false, segment_prefix))
	}

	/// Creates an instruction with a 64-bit memory offset as the first operand, eg. `mov [123456789ABCDEF0],al`
	///
	/// # Arguments
	///
	/// * `code`: Code value
	/// * `address`: 64-bit address
	/// * `register`: Register (`AL`, `AX`, `EAX`, `RAX`)
	/// * `segment_prefix`: Segment override or [`Register::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	#[deprecated(since = "1.11.0", note = "Use with_mem_reg() with a MemoryOperand arg instead")]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_mem64_reg(code: Code, address: u64, register: Register, segment_prefix: Register) -> Self {
		Instruction::with_mem_reg(code, MemoryOperand::with_base_displ_size_bcst_seg(Register::None, address as i64, 8, false, segment_prefix), register)
	}

	/// Creates a `OUTSB` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_outsb(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_segrsi(Code::Outsb_DX_m8, address_size, Register::DX, segment_prefix, rep_prefix)
	}

	/// Creates a `OUTSB` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_outsb() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_outsb(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_outsb(address_size, segment_prefix, rep_prefix).unwrap()
	}

	/// Creates a `REP OUTSB` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_outsb(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_segrsi(Code::Outsb_DX_m8, address_size, Register::DX, Register::None, RepPrefixKind::Repe)
	}

	/// Creates a `REP OUTSB` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_outsb() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_outsb(address_size: u32) -> Self {
		Instruction::try_with_rep_outsb(address_size).unwrap()
	}

	/// Creates a `OUTSW` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_outsw(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_segrsi(Code::Outsw_DX_m16, address_size, Register::DX, segment_prefix, rep_prefix)
	}

	/// Creates a `OUTSW` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_outsw() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_outsw(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_outsw(address_size, segment_prefix, rep_prefix).unwrap()
	}

	/// Creates a `REP OUTSW` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_outsw(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_segrsi(Code::Outsw_DX_m16, address_size, Register::DX, Register::None, RepPrefixKind::Repe)
	}

	/// Creates a `REP OUTSW` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_outsw() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_outsw(address_size: u32) -> Self {
		Instruction::try_with_rep_outsw(address_size).unwrap()
	}

	/// Creates a `OUTSD` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_outsd(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_segrsi(Code::Outsd_DX_m32, address_size, Register::DX, segment_prefix, rep_prefix)
	}

	/// Creates a `OUTSD` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_outsd() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_outsd(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_outsd(address_size, segment_prefix, rep_prefix).unwrap()
	}

	/// Creates a `REP OUTSD` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_outsd(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_segrsi(Code::Outsd_DX_m32, address_size, Register::DX, Register::None, RepPrefixKind::Repe)
	}

	/// Creates a `REP OUTSD` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_outsd() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_outsd(address_size: u32) -> Self {
		Instruction::try_with_rep_outsd(address_size).unwrap()
	}

	/// Creates a `LODSB` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_lodsb(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_segrsi(Code::Lodsb_AL_m8, address_size, Register::AL, segment_prefix, rep_prefix)
	}

	/// Creates a `LODSB` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_lodsb() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_lodsb(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_lodsb(address_size, segment_prefix, rep_prefix).unwrap()
	}

	/// Creates a `REP LODSB` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_lodsb(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_segrsi(Code::Lodsb_AL_m8, address_size, Register::AL, Register::None, RepPrefixKind::Repe)
	}

	/// Creates a `REP LODSB` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_lodsb() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_lodsb(address_size: u32) -> Self {
		Instruction::try_with_rep_lodsb(address_size).unwrap()
	}

	/// Creates a `LODSW` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_lodsw(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_segrsi(Code::Lodsw_AX_m16, address_size, Register::AX, segment_prefix, rep_prefix)
	}

	/// Creates a `LODSW` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_lodsw() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_lodsw(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_lodsw(address_size, segment_prefix, rep_prefix).unwrap()
	}

	/// Creates a `REP LODSW` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_lodsw(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_segrsi(Code::Lodsw_AX_m16, address_size, Register::AX, Register::None, RepPrefixKind::Repe)
	}

	/// Creates a `REP LODSW` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_lodsw() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_lodsw(address_size: u32) -> Self {
		Instruction::try_with_rep_lodsw(address_size).unwrap()
	}

	/// Creates a `LODSD` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_lodsd(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_segrsi(Code::Lodsd_EAX_m32, address_size, Register::EAX, segment_prefix, rep_prefix)
	}

	/// Creates a `LODSD` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_lodsd() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_lodsd(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_lodsd(address_size, segment_prefix, rep_prefix).unwrap()
	}

	/// Creates a `REP LODSD` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_lodsd(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_segrsi(Code::Lodsd_EAX_m32, address_size, Register::EAX, Register::None, RepPrefixKind::Repe)
	}

	/// Creates a `REP LODSD` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_lodsd() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_lodsd(address_size: u32) -> Self {
		Instruction::try_with_rep_lodsd(address_size).unwrap()
	}

	/// Creates a `LODSQ` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_lodsq(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_segrsi(Code::Lodsq_RAX_m64, address_size, Register::RAX, segment_prefix, rep_prefix)
	}

	/// Creates a `LODSQ` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_lodsq() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_lodsq(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_lodsq(address_size, segment_prefix, rep_prefix).unwrap()
	}

	/// Creates a `REP LODSQ` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_lodsq(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_segrsi(Code::Lodsq_RAX_m64, address_size, Register::RAX, Register::None, RepPrefixKind::Repe)
	}

	/// Creates a `REP LODSQ` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_lodsq() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_lodsq(address_size: u32) -> Self {
		Instruction::try_with_rep_lodsq(address_size).unwrap()
	}

	/// Creates a `SCASB` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_scasb(address_size: u32, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_esrdi(Code::Scasb_AL_m8, address_size, Register::AL, rep_prefix)
	}

	/// Creates a `SCASB` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_scasb() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_scasb(address_size: u32, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_scasb(address_size, rep_prefix).unwrap()
	}

	/// Creates a `REPE SCASB` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_repe_scasb(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_esrdi(Code::Scasb_AL_m8, address_size, Register::AL, RepPrefixKind::Repe)
	}

	/// Creates a `REPE SCASB` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_repe_scasb() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_repe_scasb(address_size: u32) -> Self {
		Instruction::try_with_repe_scasb(address_size).unwrap()
	}

	/// Creates a `REPNE SCASB` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_repne_scasb(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_esrdi(Code::Scasb_AL_m8, address_size, Register::AL, RepPrefixKind::Repne)
	}

	/// Creates a `REPNE SCASB` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_repne_scasb() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_repne_scasb(address_size: u32) -> Self {
		Instruction::try_with_repne_scasb(address_size).unwrap()
	}

	/// Creates a `SCASW` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_scasw(address_size: u32, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_esrdi(Code::Scasw_AX_m16, address_size, Register::AX, rep_prefix)
	}

	/// Creates a `SCASW` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_scasw() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_scasw(address_size: u32, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_scasw(address_size, rep_prefix).unwrap()
	}

	/// Creates a `REPE SCASW` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_repe_scasw(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_esrdi(Code::Scasw_AX_m16, address_size, Register::AX, RepPrefixKind::Repe)
	}

	/// Creates a `REPE SCASW` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_repe_scasw() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_repe_scasw(address_size: u32) -> Self {
		Instruction::try_with_repe_scasw(address_size).unwrap()
	}

	/// Creates a `REPNE SCASW` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_repne_scasw(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_esrdi(Code::Scasw_AX_m16, address_size, Register::AX, RepPrefixKind::Repne)
	}

	/// Creates a `REPNE SCASW` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_repne_scasw() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_repne_scasw(address_size: u32) -> Self {
		Instruction::try_with_repne_scasw(address_size).unwrap()
	}

	/// Creates a `SCASD` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_scasd(address_size: u32, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_esrdi(Code::Scasd_EAX_m32, address_size, Register::EAX, rep_prefix)
	}

	/// Creates a `SCASD` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_scasd() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_scasd(address_size: u32, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_scasd(address_size, rep_prefix).unwrap()
	}

	/// Creates a `REPE SCASD` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_repe_scasd(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_esrdi(Code::Scasd_EAX_m32, address_size, Register::EAX, RepPrefixKind::Repe)
	}

	/// Creates a `REPE SCASD` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_repe_scasd() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_repe_scasd(address_size: u32) -> Self {
		Instruction::try_with_repe_scasd(address_size).unwrap()
	}

	/// Creates a `REPNE SCASD` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_repne_scasd(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_esrdi(Code::Scasd_EAX_m32, address_size, Register::EAX, RepPrefixKind::Repne)
	}

	/// Creates a `REPNE SCASD` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_repne_scasd() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_repne_scasd(address_size: u32) -> Self {
		Instruction::try_with_repne_scasd(address_size).unwrap()
	}

	/// Creates a `SCASQ` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_scasq(address_size: u32, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_esrdi(Code::Scasq_RAX_m64, address_size, Register::RAX, rep_prefix)
	}

	/// Creates a `SCASQ` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_scasq() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_scasq(address_size: u32, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_scasq(address_size, rep_prefix).unwrap()
	}

	/// Creates a `REPE SCASQ` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_repe_scasq(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_esrdi(Code::Scasq_RAX_m64, address_size, Register::RAX, RepPrefixKind::Repe)
	}

	/// Creates a `REPE SCASQ` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_repe_scasq() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_repe_scasq(address_size: u32) -> Self {
		Instruction::try_with_repe_scasq(address_size).unwrap()
	}

	/// Creates a `REPNE SCASQ` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_repne_scasq(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_reg_esrdi(Code::Scasq_RAX_m64, address_size, Register::RAX, RepPrefixKind::Repne)
	}

	/// Creates a `REPNE SCASQ` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_repne_scasq() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_repne_scasq(address_size: u32) -> Self {
		Instruction::try_with_repne_scasq(address_size).unwrap()
	}

	/// Creates a `INSB` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_insb(address_size: u32, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_reg(Code::Insb_m8_DX, address_size, Register::DX, rep_prefix)
	}

	/// Creates a `INSB` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_insb() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_insb(address_size: u32, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_insb(address_size, rep_prefix).unwrap()
	}

	/// Creates a `REP INSB` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_insb(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_reg(Code::Insb_m8_DX, address_size, Register::DX, RepPrefixKind::Repe)
	}

	/// Creates a `REP INSB` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_insb() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_insb(address_size: u32) -> Self {
		Instruction::try_with_rep_insb(address_size).unwrap()
	}

	/// Creates a `INSW` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_insw(address_size: u32, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_reg(Code::Insw_m16_DX, address_size, Register::DX, rep_prefix)
	}

	/// Creates a `INSW` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_insw() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_insw(address_size: u32, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_insw(address_size, rep_prefix).unwrap()
	}

	/// Creates a `REP INSW` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_insw(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_reg(Code::Insw_m16_DX, address_size, Register::DX, RepPrefixKind::Repe)
	}

	/// Creates a `REP INSW` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_insw() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_insw(address_size: u32) -> Self {
		Instruction::try_with_rep_insw(address_size).unwrap()
	}

	/// Creates a `INSD` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_insd(address_size: u32, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_reg(Code::Insd_m32_DX, address_size, Register::DX, rep_prefix)
	}

	/// Creates a `INSD` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_insd() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_insd(address_size: u32, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_insd(address_size, rep_prefix).unwrap()
	}

	/// Creates a `REP INSD` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_insd(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_reg(Code::Insd_m32_DX, address_size, Register::DX, RepPrefixKind::Repe)
	}

	/// Creates a `REP INSD` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_insd() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_insd(address_size: u32) -> Self {
		Instruction::try_with_rep_insd(address_size).unwrap()
	}

	/// Creates a `STOSB` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_stosb(address_size: u32, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_reg(Code::Stosb_m8_AL, address_size, Register::AL, rep_prefix)
	}

	/// Creates a `STOSB` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_stosb() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_stosb(address_size: u32, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_stosb(address_size, rep_prefix).unwrap()
	}

	/// Creates a `REP STOSB` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_stosb(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_reg(Code::Stosb_m8_AL, address_size, Register::AL, RepPrefixKind::Repe)
	}

	/// Creates a `REP STOSB` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_stosb() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_stosb(address_size: u32) -> Self {
		Instruction::try_with_rep_stosb(address_size).unwrap()
	}

	/// Creates a `STOSW` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_stosw(address_size: u32, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_reg(Code::Stosw_m16_AX, address_size, Register::AX, rep_prefix)
	}

	/// Creates a `STOSW` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_stosw() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_stosw(address_size: u32, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_stosw(address_size, rep_prefix).unwrap()
	}

	/// Creates a `REP STOSW` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_stosw(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_reg(Code::Stosw_m16_AX, address_size, Register::AX, RepPrefixKind::Repe)
	}

	/// Creates a `REP STOSW` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_stosw() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_stosw(address_size: u32) -> Self {
		Instruction::try_with_rep_stosw(address_size).unwrap()
	}

	/// Creates a `STOSD` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_stosd(address_size: u32, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_reg(Code::Stosd_m32_EAX, address_size, Register::EAX, rep_prefix)
	}

	/// Creates a `STOSD` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_stosd() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_stosd(address_size: u32, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_stosd(address_size, rep_prefix).unwrap()
	}

	/// Creates a `REP STOSD` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_stosd(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_reg(Code::Stosd_m32_EAX, address_size, Register::EAX, RepPrefixKind::Repe)
	}

	/// Creates a `REP STOSD` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_stosd() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_stosd(address_size: u32) -> Self {
		Instruction::try_with_rep_stosd(address_size).unwrap()
	}

	/// Creates a `STOSQ` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_stosq(address_size: u32, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_reg(Code::Stosq_m64_RAX, address_size, Register::RAX, rep_prefix)
	}

	/// Creates a `STOSQ` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_stosq() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_stosq(address_size: u32, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_stosq(address_size, rep_prefix).unwrap()
	}

	/// Creates a `REP STOSQ` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_stosq(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_reg(Code::Stosq_m64_RAX, address_size, Register::RAX, RepPrefixKind::Repe)
	}

	/// Creates a `REP STOSQ` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_stosq() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_stosq(address_size: u32) -> Self {
		Instruction::try_with_rep_stosq(address_size).unwrap()
	}

	/// Creates a `CMPSB` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_cmpsb(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_segrsi_esrdi(Code::Cmpsb_m8_m8, address_size, segment_prefix, rep_prefix)
	}

	/// Creates a `CMPSB` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_cmpsb() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_cmpsb(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_cmpsb(address_size, segment_prefix, rep_prefix).unwrap()
	}

	/// Creates a `REPE CMPSB` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_repe_cmpsb(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_segrsi_esrdi(Code::Cmpsb_m8_m8, address_size, Register::None, RepPrefixKind::Repe)
	}

	/// Creates a `REPE CMPSB` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_repe_cmpsb() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_repe_cmpsb(address_size: u32) -> Self {
		Instruction::try_with_repe_cmpsb(address_size).unwrap()
	}

	/// Creates a `REPNE CMPSB` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_repne_cmpsb(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_segrsi_esrdi(Code::Cmpsb_m8_m8, address_size, Register::None, RepPrefixKind::Repne)
	}

	/// Creates a `REPNE CMPSB` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_repne_cmpsb() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_repne_cmpsb(address_size: u32) -> Self {
		Instruction::try_with_repne_cmpsb(address_size).unwrap()
	}

	/// Creates a `CMPSW` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_cmpsw(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_segrsi_esrdi(Code::Cmpsw_m16_m16, address_size, segment_prefix, rep_prefix)
	}

	/// Creates a `CMPSW` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_cmpsw() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_cmpsw(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_cmpsw(address_size, segment_prefix, rep_prefix).unwrap()
	}

	/// Creates a `REPE CMPSW` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_repe_cmpsw(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_segrsi_esrdi(Code::Cmpsw_m16_m16, address_size, Register::None, RepPrefixKind::Repe)
	}

	/// Creates a `REPE CMPSW` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_repe_cmpsw() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_repe_cmpsw(address_size: u32) -> Self {
		Instruction::try_with_repe_cmpsw(address_size).unwrap()
	}

	/// Creates a `REPNE CMPSW` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_repne_cmpsw(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_segrsi_esrdi(Code::Cmpsw_m16_m16, address_size, Register::None, RepPrefixKind::Repne)
	}

	/// Creates a `REPNE CMPSW` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_repne_cmpsw() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_repne_cmpsw(address_size: u32) -> Self {
		Instruction::try_with_repne_cmpsw(address_size).unwrap()
	}

	/// Creates a `CMPSD` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_cmpsd(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_segrsi_esrdi(Code::Cmpsd_m32_m32, address_size, segment_prefix, rep_prefix)
	}

	/// Creates a `CMPSD` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_cmpsd() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_cmpsd(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_cmpsd(address_size, segment_prefix, rep_prefix).unwrap()
	}

	/// Creates a `REPE CMPSD` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_repe_cmpsd(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_segrsi_esrdi(Code::Cmpsd_m32_m32, address_size, Register::None, RepPrefixKind::Repe)
	}

	/// Creates a `REPE CMPSD` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_repe_cmpsd() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_repe_cmpsd(address_size: u32) -> Self {
		Instruction::try_with_repe_cmpsd(address_size).unwrap()
	}

	/// Creates a `REPNE CMPSD` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_repne_cmpsd(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_segrsi_esrdi(Code::Cmpsd_m32_m32, address_size, Register::None, RepPrefixKind::Repne)
	}

	/// Creates a `REPNE CMPSD` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_repne_cmpsd() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_repne_cmpsd(address_size: u32) -> Self {
		Instruction::try_with_repne_cmpsd(address_size).unwrap()
	}

	/// Creates a `CMPSQ` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_cmpsq(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_segrsi_esrdi(Code::Cmpsq_m64_m64, address_size, segment_prefix, rep_prefix)
	}

	/// Creates a `CMPSQ` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_cmpsq() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_cmpsq(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_cmpsq(address_size, segment_prefix, rep_prefix).unwrap()
	}

	/// Creates a `REPE CMPSQ` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_repe_cmpsq(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_segrsi_esrdi(Code::Cmpsq_m64_m64, address_size, Register::None, RepPrefixKind::Repe)
	}

	/// Creates a `REPE CMPSQ` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_repe_cmpsq() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_repe_cmpsq(address_size: u32) -> Self {
		Instruction::try_with_repe_cmpsq(address_size).unwrap()
	}

	/// Creates a `REPNE CMPSQ` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_repne_cmpsq(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_segrsi_esrdi(Code::Cmpsq_m64_m64, address_size, Register::None, RepPrefixKind::Repne)
	}

	/// Creates a `REPNE CMPSQ` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_repne_cmpsq() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_repne_cmpsq(address_size: u32) -> Self {
		Instruction::try_with_repne_cmpsq(address_size).unwrap()
	}

	/// Creates a `MOVSB` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_movsb(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_segrsi(Code::Movsb_m8_m8, address_size, segment_prefix, rep_prefix)
	}

	/// Creates a `MOVSB` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_movsb() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_movsb(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_movsb(address_size, segment_prefix, rep_prefix).unwrap()
	}

	/// Creates a `REP MOVSB` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_movsb(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_segrsi(Code::Movsb_m8_m8, address_size, Register::None, RepPrefixKind::Repe)
	}

	/// Creates a `REP MOVSB` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_movsb() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_movsb(address_size: u32) -> Self {
		Instruction::try_with_rep_movsb(address_size).unwrap()
	}

	/// Creates a `MOVSW` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_movsw(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_segrsi(Code::Movsw_m16_m16, address_size, segment_prefix, rep_prefix)
	}

	/// Creates a `MOVSW` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_movsw() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_movsw(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_movsw(address_size, segment_prefix, rep_prefix).unwrap()
	}

	/// Creates a `REP MOVSW` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_movsw(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_segrsi(Code::Movsw_m16_m16, address_size, Register::None, RepPrefixKind::Repe)
	}

	/// Creates a `REP MOVSW` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_movsw() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_movsw(address_size: u32) -> Self {
		Instruction::try_with_rep_movsw(address_size).unwrap()
	}

	/// Creates a `MOVSD` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_movsd(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_segrsi(Code::Movsd_m32_m32, address_size, segment_prefix, rep_prefix)
	}

	/// Creates a `MOVSD` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_movsd() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_movsd(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_movsd(address_size, segment_prefix, rep_prefix).unwrap()
	}

	/// Creates a `REP MOVSD` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_movsd(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_segrsi(Code::Movsd_m32_m32, address_size, Register::None, RepPrefixKind::Repe)
	}

	/// Creates a `REP MOVSD` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_movsd() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_movsd(address_size: u32) -> Self {
		Instruction::try_with_rep_movsd(address_size).unwrap()
	}

	/// Creates a `MOVSQ` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_movsq(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_segrsi(Code::Movsq_m64_m64, address_size, segment_prefix, rep_prefix)
	}

	/// Creates a `MOVSQ` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `segment_prefix`: Segment override or [`Register::None`]
	/// * `rep_prefix`: Rep prefix or [`RepPrefixKind::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	/// [`RepPrefixKind::None`]: enum.RepPrefixKind.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_movsq() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_movsq(address_size: u32, segment_prefix: Register, rep_prefix: RepPrefixKind) -> Self {
		Instruction::try_with_movsq(address_size, segment_prefix, rep_prefix).unwrap()
	}

	/// Creates a `REP MOVSQ` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_rep_movsq(address_size: u32) -> Result<Self, IcedError> {
		instruction_internal::with_string_esrdi_segrsi(Code::Movsq_m64_m64, address_size, Register::None, RepPrefixKind::Repe)
	}

	/// Creates a `REP MOVSQ` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_rep_movsq() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_rep_movsq(address_size: u32) -> Self {
		Instruction::try_with_rep_movsq(address_size).unwrap()
	}

	/// Creates a `MASKMOVQ` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `register1`: Register
	/// * `register2`: Register
	/// * `segment_prefix`: Segment override or [`Register::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_maskmovq(address_size: u32, register1: Register, register2: Register, segment_prefix: Register) -> Result<Self, IcedError> {
		instruction_internal::with_maskmov(Code::Maskmovq_rDI_mm_mm, address_size, register1, register2, segment_prefix)
	}

	/// Creates a `MASKMOVQ` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `register1`: Register
	/// * `register2`: Register
	/// * `segment_prefix`: Segment override or [`Register::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_maskmovq() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_maskmovq(address_size: u32, register1: Register, register2: Register, segment_prefix: Register) -> Self {
		Instruction::try_with_maskmovq(address_size, register1, register2, segment_prefix).unwrap()
	}

	/// Creates a `MASKMOVDQU` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `register1`: Register
	/// * `register2`: Register
	/// * `segment_prefix`: Segment override or [`Register::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_maskmovdqu(address_size: u32, register1: Register, register2: Register, segment_prefix: Register) -> Result<Self, IcedError> {
		instruction_internal::with_maskmov(Code::Maskmovdqu_rDI_xmm_xmm, address_size, register1, register2, segment_prefix)
	}

	/// Creates a `MASKMOVDQU` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `register1`: Register
	/// * `register2`: Register
	/// * `segment_prefix`: Segment override or [`Register::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_maskmovdqu() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_maskmovdqu(address_size: u32, register1: Register, register2: Register, segment_prefix: Register) -> Self {
		Instruction::try_with_maskmovdqu(address_size, register1, register2, segment_prefix).unwrap()
	}

	/// Creates a `VMASKMOVDQU` instruction
	///
	/// # Errors
	///
	/// Fails if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `register1`: Register
	/// * `register2`: Register
	/// * `segment_prefix`: Segment override or [`Register::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	#[inline]
	#[rustfmt::skip]
	pub fn try_with_vmaskmovdqu(address_size: u32, register1: Register, register2: Register, segment_prefix: Register) -> Result<Self, IcedError> {
		instruction_internal::with_maskmov(Code::VEX_Vmaskmovdqu_rDI_xmm_xmm, address_size, register1, register2, segment_prefix)
	}

	/// Creates a `VMASKMOVDQU` instruction
	///
	/// # Panics
	///
	/// Panics if `address_size` is not one of 16, 32, 64.
	///
	/// # Arguments
	///
	/// * `address_size`: 16, 32, or 64
	/// * `register1`: Register
	/// * `register2`: Register
	/// * `segment_prefix`: Segment override or [`Register::None`]
	///
	/// [`Register::None`]: enum.Register.html#variant.None
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_vmaskmovdqu() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_vmaskmovdqu(address_size: u32, register1: Register, register2: Register, segment_prefix: Register) -> Self {
		Instruction::try_with_vmaskmovdqu(address_size, register1, register2, segment_prefix).unwrap()
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_byte_1(b0: u8) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareByte);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 1);

		instruction.try_set_declare_byte_value(0, b0)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_byte_1() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_byte_1(b0: u8) -> Self {
		Instruction::try_with_declare_byte_1(b0).unwrap()
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_byte_2(b0: u8, b1: u8) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareByte);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 2);

		instruction.try_set_declare_byte_value(0, b0)?;
		instruction.try_set_declare_byte_value(1, b1)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_byte_2() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_byte_2(b0: u8, b1: u8) -> Self {
		Instruction::try_with_declare_byte_2(b0, b1).unwrap()
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_byte_3(b0: u8, b1: u8, b2: u8) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareByte);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 3);

		instruction.try_set_declare_byte_value(0, b0)?;
		instruction.try_set_declare_byte_value(1, b1)?;
		instruction.try_set_declare_byte_value(2, b2)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_byte_3() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_byte_3(b0: u8, b1: u8, b2: u8) -> Self {
		Instruction::try_with_declare_byte_3(b0, b1, b2).unwrap()
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_byte_4(b0: u8, b1: u8, b2: u8, b3: u8) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareByte);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 4);

		instruction.try_set_declare_byte_value(0, b0)?;
		instruction.try_set_declare_byte_value(1, b1)?;
		instruction.try_set_declare_byte_value(2, b2)?;
		instruction.try_set_declare_byte_value(3, b3)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_byte_4() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_byte_4(b0: u8, b1: u8, b2: u8, b3: u8) -> Self {
		Instruction::try_with_declare_byte_4(b0, b1, b2, b3).unwrap()
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_byte_5(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareByte);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 5);

		instruction.try_set_declare_byte_value(0, b0)?;
		instruction.try_set_declare_byte_value(1, b1)?;
		instruction.try_set_declare_byte_value(2, b2)?;
		instruction.try_set_declare_byte_value(3, b3)?;
		instruction.try_set_declare_byte_value(4, b4)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_byte_5() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_byte_5(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8) -> Self {
		Instruction::try_with_declare_byte_5(b0, b1, b2, b3, b4).unwrap()
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_byte_6(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareByte);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 6);

		instruction.try_set_declare_byte_value(0, b0)?;
		instruction.try_set_declare_byte_value(1, b1)?;
		instruction.try_set_declare_byte_value(2, b2)?;
		instruction.try_set_declare_byte_value(3, b3)?;
		instruction.try_set_declare_byte_value(4, b4)?;
		instruction.try_set_declare_byte_value(5, b5)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_byte_6() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_byte_6(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8) -> Self {
		Instruction::try_with_declare_byte_6(b0, b1, b2, b3, b4, b5).unwrap()
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_byte_7(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareByte);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 7);

		instruction.try_set_declare_byte_value(0, b0)?;
		instruction.try_set_declare_byte_value(1, b1)?;
		instruction.try_set_declare_byte_value(2, b2)?;
		instruction.try_set_declare_byte_value(3, b3)?;
		instruction.try_set_declare_byte_value(4, b4)?;
		instruction.try_set_declare_byte_value(5, b5)?;
		instruction.try_set_declare_byte_value(6, b6)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_byte_7() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_byte_7(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8) -> Self {
		Instruction::try_with_declare_byte_7(b0, b1, b2, b3, b4, b5, b6).unwrap()
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_byte_8(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareByte);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 8);

		instruction.try_set_declare_byte_value(0, b0)?;
		instruction.try_set_declare_byte_value(1, b1)?;
		instruction.try_set_declare_byte_value(2, b2)?;
		instruction.try_set_declare_byte_value(3, b3)?;
		instruction.try_set_declare_byte_value(4, b4)?;
		instruction.try_set_declare_byte_value(5, b5)?;
		instruction.try_set_declare_byte_value(6, b6)?;
		instruction.try_set_declare_byte_value(7, b7)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_byte_8() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_byte_8(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8) -> Self {
		Instruction::try_with_declare_byte_8(b0, b1, b2, b3, b4, b5, b6, b7).unwrap()
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	/// * `b8`: Byte 8
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_byte_9(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8, b8: u8) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareByte);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 9);

		instruction.try_set_declare_byte_value(0, b0)?;
		instruction.try_set_declare_byte_value(1, b1)?;
		instruction.try_set_declare_byte_value(2, b2)?;
		instruction.try_set_declare_byte_value(3, b3)?;
		instruction.try_set_declare_byte_value(4, b4)?;
		instruction.try_set_declare_byte_value(5, b5)?;
		instruction.try_set_declare_byte_value(6, b6)?;
		instruction.try_set_declare_byte_value(7, b7)?;
		instruction.try_set_declare_byte_value(8, b8)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	/// * `b8`: Byte 8
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_byte_9() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_byte_9(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8, b8: u8) -> Self {
		Instruction::try_with_declare_byte_9(b0, b1, b2, b3, b4, b5, b6, b7, b8).unwrap()
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	/// * `b8`: Byte 8
	/// * `b9`: Byte 9
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_byte_10(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8, b8: u8, b9: u8) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareByte);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 10);

		instruction.try_set_declare_byte_value(0, b0)?;
		instruction.try_set_declare_byte_value(1, b1)?;
		instruction.try_set_declare_byte_value(2, b2)?;
		instruction.try_set_declare_byte_value(3, b3)?;
		instruction.try_set_declare_byte_value(4, b4)?;
		instruction.try_set_declare_byte_value(5, b5)?;
		instruction.try_set_declare_byte_value(6, b6)?;
		instruction.try_set_declare_byte_value(7, b7)?;
		instruction.try_set_declare_byte_value(8, b8)?;
		instruction.try_set_declare_byte_value(9, b9)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	/// * `b8`: Byte 8
	/// * `b9`: Byte 9
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_byte_10() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_byte_10(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8, b8: u8, b9: u8) -> Self {
		Instruction::try_with_declare_byte_10(b0, b1, b2, b3, b4, b5, b6, b7, b8, b9).unwrap()
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	/// * `b8`: Byte 8
	/// * `b9`: Byte 9
	/// * `b10`: Byte 10
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_byte_11(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8, b8: u8, b9: u8, b10: u8) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareByte);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 11);

		instruction.try_set_declare_byte_value(0, b0)?;
		instruction.try_set_declare_byte_value(1, b1)?;
		instruction.try_set_declare_byte_value(2, b2)?;
		instruction.try_set_declare_byte_value(3, b3)?;
		instruction.try_set_declare_byte_value(4, b4)?;
		instruction.try_set_declare_byte_value(5, b5)?;
		instruction.try_set_declare_byte_value(6, b6)?;
		instruction.try_set_declare_byte_value(7, b7)?;
		instruction.try_set_declare_byte_value(8, b8)?;
		instruction.try_set_declare_byte_value(9, b9)?;
		instruction.try_set_declare_byte_value(10, b10)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	/// * `b8`: Byte 8
	/// * `b9`: Byte 9
	/// * `b10`: Byte 10
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_byte_11() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_byte_11(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8, b8: u8, b9: u8, b10: u8) -> Self {
		Instruction::try_with_declare_byte_11(b0, b1, b2, b3, b4, b5, b6, b7, b8, b9, b10).unwrap()
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	/// * `b8`: Byte 8
	/// * `b9`: Byte 9
	/// * `b10`: Byte 10
	/// * `b11`: Byte 11
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_byte_12(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8, b8: u8, b9: u8, b10: u8, b11: u8) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareByte);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 12);

		instruction.try_set_declare_byte_value(0, b0)?;
		instruction.try_set_declare_byte_value(1, b1)?;
		instruction.try_set_declare_byte_value(2, b2)?;
		instruction.try_set_declare_byte_value(3, b3)?;
		instruction.try_set_declare_byte_value(4, b4)?;
		instruction.try_set_declare_byte_value(5, b5)?;
		instruction.try_set_declare_byte_value(6, b6)?;
		instruction.try_set_declare_byte_value(7, b7)?;
		instruction.try_set_declare_byte_value(8, b8)?;
		instruction.try_set_declare_byte_value(9, b9)?;
		instruction.try_set_declare_byte_value(10, b10)?;
		instruction.try_set_declare_byte_value(11, b11)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	/// * `b8`: Byte 8
	/// * `b9`: Byte 9
	/// * `b10`: Byte 10
	/// * `b11`: Byte 11
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_byte_12() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_byte_12(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8, b8: u8, b9: u8, b10: u8, b11: u8) -> Self {
		Instruction::try_with_declare_byte_12(b0, b1, b2, b3, b4, b5, b6, b7, b8, b9, b10, b11).unwrap()
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	/// * `b8`: Byte 8
	/// * `b9`: Byte 9
	/// * `b10`: Byte 10
	/// * `b11`: Byte 11
	/// * `b12`: Byte 12
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_byte_13(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8, b8: u8, b9: u8, b10: u8, b11: u8, b12: u8) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareByte);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 13);

		instruction.try_set_declare_byte_value(0, b0)?;
		instruction.try_set_declare_byte_value(1, b1)?;
		instruction.try_set_declare_byte_value(2, b2)?;
		instruction.try_set_declare_byte_value(3, b3)?;
		instruction.try_set_declare_byte_value(4, b4)?;
		instruction.try_set_declare_byte_value(5, b5)?;
		instruction.try_set_declare_byte_value(6, b6)?;
		instruction.try_set_declare_byte_value(7, b7)?;
		instruction.try_set_declare_byte_value(8, b8)?;
		instruction.try_set_declare_byte_value(9, b9)?;
		instruction.try_set_declare_byte_value(10, b10)?;
		instruction.try_set_declare_byte_value(11, b11)?;
		instruction.try_set_declare_byte_value(12, b12)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	/// * `b8`: Byte 8
	/// * `b9`: Byte 9
	/// * `b10`: Byte 10
	/// * `b11`: Byte 11
	/// * `b12`: Byte 12
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_byte_13() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_byte_13(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8, b8: u8, b9: u8, b10: u8, b11: u8, b12: u8) -> Self {
		Instruction::try_with_declare_byte_13(b0, b1, b2, b3, b4, b5, b6, b7, b8, b9, b10, b11, b12).unwrap()
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	/// * `b8`: Byte 8
	/// * `b9`: Byte 9
	/// * `b10`: Byte 10
	/// * `b11`: Byte 11
	/// * `b12`: Byte 12
	/// * `b13`: Byte 13
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_byte_14(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8, b8: u8, b9: u8, b10: u8, b11: u8, b12: u8, b13: u8) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareByte);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 14);

		instruction.try_set_declare_byte_value(0, b0)?;
		instruction.try_set_declare_byte_value(1, b1)?;
		instruction.try_set_declare_byte_value(2, b2)?;
		instruction.try_set_declare_byte_value(3, b3)?;
		instruction.try_set_declare_byte_value(4, b4)?;
		instruction.try_set_declare_byte_value(5, b5)?;
		instruction.try_set_declare_byte_value(6, b6)?;
		instruction.try_set_declare_byte_value(7, b7)?;
		instruction.try_set_declare_byte_value(8, b8)?;
		instruction.try_set_declare_byte_value(9, b9)?;
		instruction.try_set_declare_byte_value(10, b10)?;
		instruction.try_set_declare_byte_value(11, b11)?;
		instruction.try_set_declare_byte_value(12, b12)?;
		instruction.try_set_declare_byte_value(13, b13)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	/// * `b8`: Byte 8
	/// * `b9`: Byte 9
	/// * `b10`: Byte 10
	/// * `b11`: Byte 11
	/// * `b12`: Byte 12
	/// * `b13`: Byte 13
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_byte_14() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_byte_14(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8, b8: u8, b9: u8, b10: u8, b11: u8, b12: u8, b13: u8) -> Self {
		Instruction::try_with_declare_byte_14(b0, b1, b2, b3, b4, b5, b6, b7, b8, b9, b10, b11, b12, b13).unwrap()
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	/// * `b8`: Byte 8
	/// * `b9`: Byte 9
	/// * `b10`: Byte 10
	/// * `b11`: Byte 11
	/// * `b12`: Byte 12
	/// * `b13`: Byte 13
	/// * `b14`: Byte 14
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_byte_15(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8, b8: u8, b9: u8, b10: u8, b11: u8, b12: u8, b13: u8, b14: u8) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareByte);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 15);

		instruction.try_set_declare_byte_value(0, b0)?;
		instruction.try_set_declare_byte_value(1, b1)?;
		instruction.try_set_declare_byte_value(2, b2)?;
		instruction.try_set_declare_byte_value(3, b3)?;
		instruction.try_set_declare_byte_value(4, b4)?;
		instruction.try_set_declare_byte_value(5, b5)?;
		instruction.try_set_declare_byte_value(6, b6)?;
		instruction.try_set_declare_byte_value(7, b7)?;
		instruction.try_set_declare_byte_value(8, b8)?;
		instruction.try_set_declare_byte_value(9, b9)?;
		instruction.try_set_declare_byte_value(10, b10)?;
		instruction.try_set_declare_byte_value(11, b11)?;
		instruction.try_set_declare_byte_value(12, b12)?;
		instruction.try_set_declare_byte_value(13, b13)?;
		instruction.try_set_declare_byte_value(14, b14)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	/// * `b8`: Byte 8
	/// * `b9`: Byte 9
	/// * `b10`: Byte 10
	/// * `b11`: Byte 11
	/// * `b12`: Byte 12
	/// * `b13`: Byte 13
	/// * `b14`: Byte 14
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_byte_15() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_byte_15(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8, b8: u8, b9: u8, b10: u8, b11: u8, b12: u8, b13: u8, b14: u8) -> Self {
		Instruction::try_with_declare_byte_15(b0, b1, b2, b3, b4, b5, b6, b7, b8, b9, b10, b11, b12, b13, b14).unwrap()
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	/// * `b8`: Byte 8
	/// * `b9`: Byte 9
	/// * `b10`: Byte 10
	/// * `b11`: Byte 11
	/// * `b12`: Byte 12
	/// * `b13`: Byte 13
	/// * `b14`: Byte 14
	/// * `b15`: Byte 15
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_byte_16(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8, b8: u8, b9: u8, b10: u8, b11: u8, b12: u8, b13: u8, b14: u8, b15: u8) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareByte);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 16);

		instruction.try_set_declare_byte_value(0, b0)?;
		instruction.try_set_declare_byte_value(1, b1)?;
		instruction.try_set_declare_byte_value(2, b2)?;
		instruction.try_set_declare_byte_value(3, b3)?;
		instruction.try_set_declare_byte_value(4, b4)?;
		instruction.try_set_declare_byte_value(5, b5)?;
		instruction.try_set_declare_byte_value(6, b6)?;
		instruction.try_set_declare_byte_value(7, b7)?;
		instruction.try_set_declare_byte_value(8, b8)?;
		instruction.try_set_declare_byte_value(9, b9)?;
		instruction.try_set_declare_byte_value(10, b10)?;
		instruction.try_set_declare_byte_value(11, b11)?;
		instruction.try_set_declare_byte_value(12, b12)?;
		instruction.try_set_declare_byte_value(13, b13)?;
		instruction.try_set_declare_byte_value(14, b14)?;
		instruction.try_set_declare_byte_value(15, b15)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `b0`: Byte 0
	/// * `b1`: Byte 1
	/// * `b2`: Byte 2
	/// * `b3`: Byte 3
	/// * `b4`: Byte 4
	/// * `b5`: Byte 5
	/// * `b6`: Byte 6
	/// * `b7`: Byte 7
	/// * `b8`: Byte 8
	/// * `b9`: Byte 9
	/// * `b10`: Byte 10
	/// * `b11`: Byte 11
	/// * `b12`: Byte 12
	/// * `b13`: Byte 13
	/// * `b14`: Byte 14
	/// * `b15`: Byte 15
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_byte_16() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_byte_16(b0: u8, b1: u8, b2: u8, b3: u8, b4: u8, b5: u8, b6: u8, b7: u8, b8: u8, b9: u8, b10: u8, b11: u8, b12: u8, b13: u8, b14: u8, b15: u8) -> Self {
		Instruction::try_with_declare_byte_16(b0, b1, b2, b3, b4, b5, b6, b7, b8, b9, b10, b11, b12, b13, b14, b15).unwrap()
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Errors
	///
	/// - Fails if `data.len()` is not 1-16
	/// - Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `data`: Data
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_byte(data: &[u8]) -> Result<Self, IcedError> {
		if data.len().wrapping_sub(1) > 16 - 1 {
			return Err(IcedError::new("Invalid slice length"));
		}

		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareByte);
		instruction_internal::internal_set_declare_data_len(&mut instruction, data.len() as u32);

		for i in data.iter().enumerate() {
			instruction.try_set_declare_byte_value(i.0, *i.1)?;
		}

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `db`/`.byte` asm directive
	///
	/// # Panics
	///
	/// - Panics if `data.len()` is not 1-16
	/// - Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `data`: Data
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_byte() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_byte(data: &[u8]) -> Self {
		Instruction::try_with_declare_byte(data).unwrap()
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `w0`: Word 0
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_word_1(w0: u16) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareWord);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 1);

		instruction.try_set_declare_word_value(0, w0)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `w0`: Word 0
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_word_1() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_word_1(w0: u16) -> Self {
		Instruction::try_with_declare_word_1(w0).unwrap()
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `w0`: Word 0
	/// * `w1`: Word 1
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_word_2(w0: u16, w1: u16) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareWord);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 2);

		instruction.try_set_declare_word_value(0, w0)?;
		instruction.try_set_declare_word_value(1, w1)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `w0`: Word 0
	/// * `w1`: Word 1
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_word_2() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_word_2(w0: u16, w1: u16) -> Self {
		Instruction::try_with_declare_word_2(w0, w1).unwrap()
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `w0`: Word 0
	/// * `w1`: Word 1
	/// * `w2`: Word 2
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_word_3(w0: u16, w1: u16, w2: u16) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareWord);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 3);

		instruction.try_set_declare_word_value(0, w0)?;
		instruction.try_set_declare_word_value(1, w1)?;
		instruction.try_set_declare_word_value(2, w2)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `w0`: Word 0
	/// * `w1`: Word 1
	/// * `w2`: Word 2
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_word_3() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_word_3(w0: u16, w1: u16, w2: u16) -> Self {
		Instruction::try_with_declare_word_3(w0, w1, w2).unwrap()
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `w0`: Word 0
	/// * `w1`: Word 1
	/// * `w2`: Word 2
	/// * `w3`: Word 3
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_word_4(w0: u16, w1: u16, w2: u16, w3: u16) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareWord);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 4);

		instruction.try_set_declare_word_value(0, w0)?;
		instruction.try_set_declare_word_value(1, w1)?;
		instruction.try_set_declare_word_value(2, w2)?;
		instruction.try_set_declare_word_value(3, w3)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `w0`: Word 0
	/// * `w1`: Word 1
	/// * `w2`: Word 2
	/// * `w3`: Word 3
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_word_4() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_word_4(w0: u16, w1: u16, w2: u16, w3: u16) -> Self {
		Instruction::try_with_declare_word_4(w0, w1, w2, w3).unwrap()
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `w0`: Word 0
	/// * `w1`: Word 1
	/// * `w2`: Word 2
	/// * `w3`: Word 3
	/// * `w4`: Word 4
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_word_5(w0: u16, w1: u16, w2: u16, w3: u16, w4: u16) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareWord);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 5);

		instruction.try_set_declare_word_value(0, w0)?;
		instruction.try_set_declare_word_value(1, w1)?;
		instruction.try_set_declare_word_value(2, w2)?;
		instruction.try_set_declare_word_value(3, w3)?;
		instruction.try_set_declare_word_value(4, w4)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `w0`: Word 0
	/// * `w1`: Word 1
	/// * `w2`: Word 2
	/// * `w3`: Word 3
	/// * `w4`: Word 4
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_word_5() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_word_5(w0: u16, w1: u16, w2: u16, w3: u16, w4: u16) -> Self {
		Instruction::try_with_declare_word_5(w0, w1, w2, w3, w4).unwrap()
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `w0`: Word 0
	/// * `w1`: Word 1
	/// * `w2`: Word 2
	/// * `w3`: Word 3
	/// * `w4`: Word 4
	/// * `w5`: Word 5
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_word_6(w0: u16, w1: u16, w2: u16, w3: u16, w4: u16, w5: u16) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareWord);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 6);

		instruction.try_set_declare_word_value(0, w0)?;
		instruction.try_set_declare_word_value(1, w1)?;
		instruction.try_set_declare_word_value(2, w2)?;
		instruction.try_set_declare_word_value(3, w3)?;
		instruction.try_set_declare_word_value(4, w4)?;
		instruction.try_set_declare_word_value(5, w5)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `w0`: Word 0
	/// * `w1`: Word 1
	/// * `w2`: Word 2
	/// * `w3`: Word 3
	/// * `w4`: Word 4
	/// * `w5`: Word 5
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_word_6() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_word_6(w0: u16, w1: u16, w2: u16, w3: u16, w4: u16, w5: u16) -> Self {
		Instruction::try_with_declare_word_6(w0, w1, w2, w3, w4, w5).unwrap()
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `w0`: Word 0
	/// * `w1`: Word 1
	/// * `w2`: Word 2
	/// * `w3`: Word 3
	/// * `w4`: Word 4
	/// * `w5`: Word 5
	/// * `w6`: Word 6
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_word_7(w0: u16, w1: u16, w2: u16, w3: u16, w4: u16, w5: u16, w6: u16) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareWord);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 7);

		instruction.try_set_declare_word_value(0, w0)?;
		instruction.try_set_declare_word_value(1, w1)?;
		instruction.try_set_declare_word_value(2, w2)?;
		instruction.try_set_declare_word_value(3, w3)?;
		instruction.try_set_declare_word_value(4, w4)?;
		instruction.try_set_declare_word_value(5, w5)?;
		instruction.try_set_declare_word_value(6, w6)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `w0`: Word 0
	/// * `w1`: Word 1
	/// * `w2`: Word 2
	/// * `w3`: Word 3
	/// * `w4`: Word 4
	/// * `w5`: Word 5
	/// * `w6`: Word 6
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_word_7() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_word_7(w0: u16, w1: u16, w2: u16, w3: u16, w4: u16, w5: u16, w6: u16) -> Self {
		Instruction::try_with_declare_word_7(w0, w1, w2, w3, w4, w5, w6).unwrap()
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `w0`: Word 0
	/// * `w1`: Word 1
	/// * `w2`: Word 2
	/// * `w3`: Word 3
	/// * `w4`: Word 4
	/// * `w5`: Word 5
	/// * `w6`: Word 6
	/// * `w7`: Word 7
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_word_8(w0: u16, w1: u16, w2: u16, w3: u16, w4: u16, w5: u16, w6: u16, w7: u16) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareWord);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 8);

		instruction.try_set_declare_word_value(0, w0)?;
		instruction.try_set_declare_word_value(1, w1)?;
		instruction.try_set_declare_word_value(2, w2)?;
		instruction.try_set_declare_word_value(3, w3)?;
		instruction.try_set_declare_word_value(4, w4)?;
		instruction.try_set_declare_word_value(5, w5)?;
		instruction.try_set_declare_word_value(6, w6)?;
		instruction.try_set_declare_word_value(7, w7)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `w0`: Word 0
	/// * `w1`: Word 1
	/// * `w2`: Word 2
	/// * `w3`: Word 3
	/// * `w4`: Word 4
	/// * `w5`: Word 5
	/// * `w6`: Word 6
	/// * `w7`: Word 7
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_word_8() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_word_8(w0: u16, w1: u16, w2: u16, w3: u16, w4: u16, w5: u16, w6: u16, w7: u16) -> Self {
		Instruction::try_with_declare_word_8(w0, w1, w2, w3, w4, w5, w6, w7).unwrap()
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Errors
	///
	/// - Fails if `data.len()` is not 2-16 or not a multiple of 2
	/// - Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `data`: Data
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	#[allow(trivial_casts)]
	pub fn try_with_declare_word_slice_u8(data: &[u8]) -> Result<Self, IcedError> {
		if data.len().wrapping_sub(1) > 16 - 1 || (data.len() & 1) != 0 {
			return Err(IcedError::new("Invalid slice length"));
		}

		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareWord);
		instruction_internal::internal_set_declare_data_len(&mut instruction, data.len() as u32 / 2);

		for i in 0..data.len() / 2 {
			let v = unsafe { u16::from_le(ptr::read_unaligned(data.get_unchecked(i * 2) as *const _ as *const u16)) };
			instruction.try_set_declare_word_value(i, v)?;
		}

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Panics
	///
	/// - Panics if `data.len()` is not 2-16 or not a multiple of 2
	/// - Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `data`: Data
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_word_slice_u8() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_word_slice_u8(data: &[u8]) -> Self {
		Instruction::try_with_declare_word_slice_u8(data).unwrap()
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Errors
	///
	/// - Fails if `data.len()` is not 1-8
	/// - Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `data`: Data
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_word(data: &[u16]) -> Result<Self, IcedError> {
		if data.len().wrapping_sub(1) > 8 - 1 {
			return Err(IcedError::new("Invalid slice length"));
		}

		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareWord);
		instruction_internal::internal_set_declare_data_len(&mut instruction, data.len() as u32);

		for i in data.iter().enumerate() {
			instruction.try_set_declare_word_value(i.0, *i.1)?;
		}

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dw`/`.word` asm directive
	///
	/// # Panics
	///
	/// - Panics if `data.len()` is not 1-8
	/// - Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `data`: Data
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_word() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_word(data: &[u16]) -> Self {
		Instruction::try_with_declare_word(data).unwrap()
	}

	/// Creates a `dd`/`.int` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `d0`: Dword 0
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_dword_1(d0: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareDword);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 1);

		instruction.try_set_declare_dword_value(0, d0)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dd`/`.int` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `d0`: Dword 0
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_dword_1() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_dword_1(d0: u32) -> Self {
		Instruction::try_with_declare_dword_1(d0).unwrap()
	}

	/// Creates a `dd`/`.int` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `d0`: Dword 0
	/// * `d1`: Dword 1
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_dword_2(d0: u32, d1: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareDword);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 2);

		instruction.try_set_declare_dword_value(0, d0)?;
		instruction.try_set_declare_dword_value(1, d1)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dd`/`.int` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `d0`: Dword 0
	/// * `d1`: Dword 1
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_dword_2() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_dword_2(d0: u32, d1: u32) -> Self {
		Instruction::try_with_declare_dword_2(d0, d1).unwrap()
	}

	/// Creates a `dd`/`.int` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `d0`: Dword 0
	/// * `d1`: Dword 1
	/// * `d2`: Dword 2
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_dword_3(d0: u32, d1: u32, d2: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareDword);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 3);

		instruction.try_set_declare_dword_value(0, d0)?;
		instruction.try_set_declare_dword_value(1, d1)?;
		instruction.try_set_declare_dword_value(2, d2)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dd`/`.int` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `d0`: Dword 0
	/// * `d1`: Dword 1
	/// * `d2`: Dword 2
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_dword_3() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_dword_3(d0: u32, d1: u32, d2: u32) -> Self {
		Instruction::try_with_declare_dword_3(d0, d1, d2).unwrap()
	}

	/// Creates a `dd`/`.int` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `d0`: Dword 0
	/// * `d1`: Dword 1
	/// * `d2`: Dword 2
	/// * `d3`: Dword 3
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_dword_4(d0: u32, d1: u32, d2: u32, d3: u32) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareDword);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 4);

		instruction.try_set_declare_dword_value(0, d0)?;
		instruction.try_set_declare_dword_value(1, d1)?;
		instruction.try_set_declare_dword_value(2, d2)?;
		instruction.try_set_declare_dword_value(3, d3)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dd`/`.int` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `d0`: Dword 0
	/// * `d1`: Dword 1
	/// * `d2`: Dword 2
	/// * `d3`: Dword 3
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_dword_4() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_dword_4(d0: u32, d1: u32, d2: u32, d3: u32) -> Self {
		Instruction::try_with_declare_dword_4(d0, d1, d2, d3).unwrap()
	}

	/// Creates a `dd`/`.int` asm directive
	///
	/// # Errors
	///
	/// - Fails if `data.len()` is not 4-16 or not a multiple of 4
	/// - Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `data`: Data
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	#[allow(trivial_casts)]
	pub fn try_with_declare_dword_slice_u8(data: &[u8]) -> Result<Self, IcedError> {
		if data.len().wrapping_sub(1) > 16 - 1 || (data.len() & 3) != 0 {
			return Err(IcedError::new("Invalid slice length"));
		}

		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareDword);
		instruction_internal::internal_set_declare_data_len(&mut instruction, data.len() as u32 / 4);

		for i in 0..data.len() / 4 {
			let v = unsafe { u32::from_le(ptr::read_unaligned(data.get_unchecked(i * 4) as *const _ as *const u32)) };
			instruction.try_set_declare_dword_value(i, v)?;
		}

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dd`/`.int` asm directive
	///
	/// # Panics
	///
	/// - Panics if `data.len()` is not 4-16 or not a multiple of 4
	/// - Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `data`: Data
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_dword_slice_u8() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_dword_slice_u8(data: &[u8]) -> Self {
		Instruction::try_with_declare_dword_slice_u8(data).unwrap()
	}

	/// Creates a `dd`/`.int` asm directive
	///
	/// # Errors
	///
	/// - Fails if `data.len()` is not 1-4
	/// - Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `data`: Data
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_dword(data: &[u32]) -> Result<Self, IcedError> {
		if data.len().wrapping_sub(1) > 4 - 1 {
			return Err(IcedError::new("Invalid slice length"));
		}

		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareDword);
		instruction_internal::internal_set_declare_data_len(&mut instruction, data.len() as u32);

		for i in data.iter().enumerate() {
			instruction.try_set_declare_dword_value(i.0, *i.1)?;
		}

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dd`/`.int` asm directive
	///
	/// # Panics
	///
	/// - Panics if `data.len()` is not 1-4
	/// - Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `data`: Data
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_dword() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_dword(data: &[u32]) -> Self {
		Instruction::try_with_declare_dword(data).unwrap()
	}

	/// Creates a `dq`/`.quad` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `q0`: Qword 0
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_qword_1(q0: u64) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareQword);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 1);

		instruction.try_set_declare_qword_value(0, q0)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dq`/`.quad` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `q0`: Qword 0
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_qword_1() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_qword_1(q0: u64) -> Self {
		Instruction::try_with_declare_qword_1(q0).unwrap()
	}

	/// Creates a `dq`/`.quad` asm directive
	///
	/// # Errors
	///
	/// Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `q0`: Qword 0
	/// * `q1`: Qword 1
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_qword_2(q0: u64, q1: u64) -> Result<Self, IcedError> {
		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareQword);
		instruction_internal::internal_set_declare_data_len(&mut instruction, 2);

		instruction.try_set_declare_qword_value(0, q0)?;
		instruction.try_set_declare_qword_value(1, q1)?;

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dq`/`.quad` asm directive
	///
	/// # Panics
	///
	/// Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `q0`: Qword 0
	/// * `q1`: Qword 1
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_qword_2() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_qword_2(q0: u64, q1: u64) -> Self {
		Instruction::try_with_declare_qword_2(q0, q1).unwrap()
	}

	/// Creates a `dq`/`.quad` asm directive
	///
	/// # Errors
	///
	/// - Fails if `data.len()` is not 8-16 or not a multiple of 8
	/// - Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `data`: Data
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	#[allow(trivial_casts)]
	pub fn try_with_declare_qword_slice_u8(data: &[u8]) -> Result<Self, IcedError> {
		if data.len().wrapping_sub(1) > 16 - 1 || (data.len() & 7) != 0 {
			return Err(IcedError::new("Invalid slice length"));
		}

		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareQword);
		instruction_internal::internal_set_declare_data_len(&mut instruction, data.len() as u32 / 8);

		for i in 0..data.len() / 8 {
			let v = unsafe { u64::from_le(ptr::read_unaligned(data.get_unchecked(i * 8) as *const _ as *const u64)) };
			instruction.try_set_declare_qword_value(i, v)?;
		}

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dq`/`.quad` asm directive
	///
	/// # Panics
	///
	/// - Panics if `data.len()` is not 8-16 or not a multiple of 8
	/// - Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `data`: Data
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_qword_slice_u8() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_qword_slice_u8(data: &[u8]) -> Self {
		Instruction::try_with_declare_qword_slice_u8(data).unwrap()
	}

	/// Creates a `dq`/`.quad` asm directive
	///
	/// # Errors
	///
	/// - Fails if `data.len()` is not 1-2
	/// - Fails if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `data`: Data
	#[allow(clippy::missing_inline_in_public_items)]
	#[rustfmt::skip]
	pub fn try_with_declare_qword(data: &[u64]) -> Result<Self, IcedError> {
		if data.len().wrapping_sub(1) > 2 - 1 {
			return Err(IcedError::new("Invalid slice length"));
		}

		let mut instruction = Self::default();
		instruction_internal::internal_set_code(&mut instruction, Code::DeclareQword);
		instruction_internal::internal_set_declare_data_len(&mut instruction, data.len() as u32);

		for i in data.iter().enumerate() {
			instruction.try_set_declare_qword_value(i.0, *i.1)?;
		}

		debug_assert_eq!(instruction.op_count(), 0);
		Ok(instruction)
	}

	/// Creates a `dq`/`.quad` asm directive
	///
	/// # Panics
	///
	/// - Panics if `data.len()` is not 1-2
	/// - Panics if `db` feature wasn't enabled
	///
	/// # Arguments
	///
	/// * `data`: Data
	#[deprecated(since = "1.10.0", note = "This method can panic, use try_with_declare_qword() instead")]
	#[allow(clippy::unwrap_used)]
	#[must_use]
	#[inline]
	#[rustfmt::skip]
	pub fn with_declare_qword(data: &[u64]) -> Self {
		Instruction::try_with_declare_qword(data).unwrap()
	}
	// GENERATOR-END: Create
}

impl Eq for Instruction {}

impl PartialEq<Instruction> for Instruction {
	#[must_use]
	#[allow(clippy::missing_inline_in_public_items)]
	fn eq(&self, other: &Self) -> bool {
		((self.code_flags ^ other.code_flags) & !CodeFlags::EQUALS_IGNORE_MASK) == 0
			&& ((self.op_kind_flags ^ other.op_kind_flags) & !OpKindFlags::EQUALS_IGNORE_MASK) == 0
			&& self.immediate == other.immediate
			&& self.mem_displ == other.mem_displ
			&& self.mem_displ_hi == other.mem_displ_hi
			&& self.memory_flags == other.memory_flags
			&& self.mem_base_reg == other.mem_base_reg
			&& self.mem_index_reg == other.mem_index_reg
			&& self.reg0 == other.reg0
			&& self.reg1 == other.reg1
			&& self.reg2 == other.reg2
			&& self.reg3 == other.reg3
	}
}

impl Hash for Instruction {
	#[allow(clippy::missing_inline_in_public_items)]
	fn hash<H: Hasher>(&self, state: &mut H) {
		state.write_u32(self.code_flags & !CodeFlags::EQUALS_IGNORE_MASK);
		state.write_u32(self.op_kind_flags & !OpKindFlags::EQUALS_IGNORE_MASK);
		state.write_u32(self.immediate);
		state.write_u32(self.mem_displ);
		state.write_u32(self.mem_displ_hi);
		state.write_u16(self.memory_flags);
		state.write_u8(self.mem_base_reg);
		state.write_u8(self.mem_index_reg);
		state.write_u8(self.reg0);
		state.write_u8(self.reg1);
		state.write_u8(self.reg2);
		state.write_u8(self.reg3);
	}
}

#[cfg(any(feature = "gas", feature = "intel", feature = "masm", feature = "nasm"))]
struct FmtFormatterOutput<'a, 'b> {
	f: &'a mut fmt::Formatter<'b>,
	result: fmt::Result,
}
#[cfg(any(feature = "gas", feature = "intel", feature = "masm", feature = "nasm"))]
impl<'a, 'b> FmtFormatterOutput<'a, 'b> {
	fn new(f: &'a mut fmt::Formatter<'b>) -> Self {
		Self { f, result: Ok(()) }
	}
}
#[cfg(any(feature = "gas", feature = "intel", feature = "masm", feature = "nasm"))]
impl FormatterOutput for FmtFormatterOutput<'_, '_> {
	fn write(&mut self, text: &str, _kind: FormatterTextKind) {
		if self.result.is_ok() {
			self.result = self.f.write_str(text);
		}
	}
}

#[cfg(any(feature = "gas", feature = "intel", feature = "masm", feature = "nasm", feature = "fast_fmt"))]
impl fmt::Display for Instruction {
	#[allow(clippy::missing_inline_in_public_items)]
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		//
		// if the order of #[cfg()] checks gets updated, also update the `display_trait()` test method
		//
		#[cfg(any(feature = "gas", feature = "intel", feature = "masm", feature = "nasm"))]
		{
			#[cfg(feature = "masm")]
			let mut formatter = MasmFormatter::new();

			#[cfg(all(not(feature = "masm"), feature = "nasm"))]
			let mut formatter = NasmFormatter::new();

			#[cfg(all(not(feature = "masm"), not(feature = "nasm"), feature = "intel"))]
			let mut formatter = IntelFormatter::new();

			#[cfg(all(not(feature = "masm"), not(feature = "nasm"), not(feature = "intel"), feature = "gas"))]
			let mut formatter = GasFormatter::new();

			let mut output = FmtFormatterOutput::new(f);
			formatter.format(self, &mut output);
			output.result
		}
		#[cfg(not(any(feature = "gas", feature = "intel", feature = "masm", feature = "nasm")))]
		{
			let mut formatter = FastFormatter::new();

			let mut output = alloc::string::String::new();
			formatter.format(self, &mut output);
			f.write_str(&output)
		}
	}
}

struct OpKindIterator {
	count: u8,
	index: u8,
	op_kinds: [OpKind; IcedConstants::MAX_OP_COUNT],
}

impl OpKindIterator {
	fn new(instruction: &Instruction) -> Self {
		// `index`/`count` are `u8`
		const_assert!(IcedConstants::MAX_OP_COUNT <= core::u8::MAX as usize);
		OpKindIterator {
			count: instruction.op_count() as u8,
			index: 0,
			op_kinds: [instruction.op0_kind(), instruction.op1_kind(), instruction.op2_kind(), instruction.op3_kind(), instruction.op4_kind()],
		}
	}
}

impl Iterator for OpKindIterator {
	type Item = OpKind;

	#[inline]
	fn next(&mut self) -> Option<Self::Item> {
		let index = self.index;
		if index < self.count {
			self.index = index + 1;
			Some(self.op_kinds[index as usize])
		} else {
			None
		}
	}

	#[inline]
	fn size_hint(&self) -> (usize, Option<usize>) {
		let len = self.count as usize - self.index as usize;
		(len, Some(len))
	}
}

impl ExactSizeIterator for OpKindIterator {}
impl FusedIterator for OpKindIterator {}
