// SPDX-License-Identifier: MIT
// Copyright (C) 2018-present iced project and contributors

use crate::block_enc::tests::*;
use core::u32;

const BITNESS: u32 = 64;
const ORIG_RIP: u64 = 0x8000;
const NEW_RIP: u64 = 0x8000_0000_0000_0000;

#[test]
fn jcc_short_fwd() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x70, 0x3E,// jo short 0000000000000040h
		/*0002*/ 0xB0, 0x00,// mov al,0
		/*0004*/ 0x71, 0x3A,// jno short 0000000000000040h
		/*0006*/ 0xB0, 0x01,// mov al,1
		/*0008*/ 0x72, 0x36,// jb short 0000000000000040h
		/*000A*/ 0xB0, 0x02,// mov al,2
		/*000C*/ 0x73, 0x32,// jae short 0000000000000040h
		/*000E*/ 0xB0, 0x03,// mov al,3
		/*0010*/ 0x74, 0x2E,// je short 0000000000000040h
		/*0012*/ 0xB0, 0x04,// mov al,4
		/*0014*/ 0x75, 0x2A,// jne short 0000000000000040h
		/*0016*/ 0xB0, 0x05,// mov al,5
		/*0018*/ 0x76, 0x26,// jbe short 0000000000000040h
		/*001A*/ 0xB0, 0x06,// mov al,6
		/*001C*/ 0x77, 0x22,// ja short 0000000000000040h
		/*001E*/ 0xB0, 0x07,// mov al,7
		/*0020*/ 0x78, 0x1E,// js short 0000000000000040h
		/*0022*/ 0xB0, 0x08,// mov al,8
		/*0024*/ 0x79, 0x1A,// jns short 0000000000000040h
		/*0026*/ 0xB0, 0x09,// mov al,9
		/*0028*/ 0x7A, 0x16,// jp short 0000000000000040h
		/*002A*/ 0xB0, 0x0A,// mov al,0Ah
		/*002C*/ 0x7B, 0x12,// jnp short 0000000000000040h
		/*002E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0030*/ 0x7C, 0x0E,// jl short 0000000000000040h
		/*0032*/ 0xB0, 0x0C,// mov al,0Ch
		/*0034*/ 0x7D, 0x0A,// jge short 0000000000000040h
		/*0036*/ 0xB0, 0x0D,// mov al,0Dh
		/*0038*/ 0x7E, 0x06,// jle short 0000000000000040h
		/*003A*/ 0xB0, 0x0E,// mov al,0Eh
		/*003C*/ 0x7F, 0x02,// jg short 0000000000000040h
		/*003E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0040*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x70, 0x3E,// jo short 0000000000000040h
		/*0002*/ 0xB0, 0x00,// mov al,0
		/*0004*/ 0x71, 0x3A,// jno short 0000000000000040h
		/*0006*/ 0xB0, 0x01,// mov al,1
		/*0008*/ 0x72, 0x36,// jb short 0000000000000040h
		/*000A*/ 0xB0, 0x02,// mov al,2
		/*000C*/ 0x73, 0x32,// jae short 0000000000000040h
		/*000E*/ 0xB0, 0x03,// mov al,3
		/*0010*/ 0x74, 0x2E,// je short 0000000000000040h
		/*0012*/ 0xB0, 0x04,// mov al,4
		/*0014*/ 0x75, 0x2A,// jne short 0000000000000040h
		/*0016*/ 0xB0, 0x05,// mov al,5
		/*0018*/ 0x76, 0x26,// jbe short 0000000000000040h
		/*001A*/ 0xB0, 0x06,// mov al,6
		/*001C*/ 0x77, 0x22,// ja short 0000000000000040h
		/*001E*/ 0xB0, 0x07,// mov al,7
		/*0020*/ 0x78, 0x1E,// js short 0000000000000040h
		/*0022*/ 0xB0, 0x08,// mov al,8
		/*0024*/ 0x79, 0x1A,// jns short 0000000000000040h
		/*0026*/ 0xB0, 0x09,// mov al,9
		/*0028*/ 0x7A, 0x16,// jp short 0000000000000040h
		/*002A*/ 0xB0, 0x0A,// mov al,0Ah
		/*002C*/ 0x7B, 0x12,// jnp short 0000000000000040h
		/*002E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0030*/ 0x7C, 0x0E,// jl short 0000000000000040h
		/*0032*/ 0xB0, 0x0C,// mov al,0Ch
		/*0034*/ 0x7D, 0x0A,// jge short 0000000000000040h
		/*0036*/ 0xB0, 0x0D,// mov al,0Dh
		/*0038*/ 0x7E, 0x06,// jle short 0000000000000040h
		/*003A*/ 0xB0, 0x0E,// mov al,0Eh
		/*003C*/ 0x7F, 0x02,// jg short 0000000000000040h
		/*003E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0040*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0002,
		0x0004,
		0x0006,
		0x0008,
		0x000A,
		0x000C,
		0x000E,
		0x0010,
		0x0012,
		0x0014,
		0x0016,
		0x0018,
		0x001A,
		0x001C,
		0x001E,
		0x0020,
		0x0022,
		0x0024,
		0x0026,
		0x0028,
		0x002A,
		0x002C,
		0x002E,
		0x0030,
		0x0032,
		0x0034,
		0x0036,
		0x0038,
		0x003A,
		0x003C,
		0x003E,
		0x0040,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_short_fwd_os() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x66, 0x70, 0x4D,// jo short 0050h
		/*0003*/ 0xB0, 0x00,// mov al,0
		/*0005*/ 0x66, 0x71, 0x48,// jno short 0050h
		/*0008*/ 0xB0, 0x01,// mov al,1
		/*000A*/ 0x66, 0x72, 0x43,// jb short 0050h
		/*000D*/ 0xB0, 0x02,// mov al,2
		/*000F*/ 0x66, 0x73, 0x3E,// jae short 0050h
		/*0012*/ 0xB0, 0x03,// mov al,3
		/*0014*/ 0x66, 0x74, 0x39,// je short 0050h
		/*0017*/ 0xB0, 0x04,// mov al,4
		/*0019*/ 0x66, 0x75, 0x34,// jne short 0050h
		/*001C*/ 0xB0, 0x05,// mov al,5
		/*001E*/ 0x66, 0x76, 0x2F,// jbe short 0050h
		/*0021*/ 0xB0, 0x06,// mov al,6
		/*0023*/ 0x66, 0x77, 0x2A,// ja short 0050h
		/*0026*/ 0xB0, 0x07,// mov al,7
		/*0028*/ 0x66, 0x78, 0x25,// js short 0050h
		/*002B*/ 0xB0, 0x08,// mov al,8
		/*002D*/ 0x66, 0x79, 0x20,// jns short 0050h
		/*0030*/ 0xB0, 0x09,// mov al,9
		/*0032*/ 0x66, 0x7A, 0x1B,// jp short 0050h
		/*0035*/ 0xB0, 0x0A,// mov al,0Ah
		/*0037*/ 0x66, 0x7B, 0x16,// jnp short 0050h
		/*003A*/ 0xB0, 0x0B,// mov al,0Bh
		/*003C*/ 0x66, 0x7C, 0x11,// jl short 0050h
		/*003F*/ 0xB0, 0x0C,// mov al,0Ch
		/*0041*/ 0x66, 0x7D, 0x0C,// jge short 0050h
		/*0044*/ 0xB0, 0x0D,// mov al,0Dh
		/*0046*/ 0x66, 0x7E, 0x07,// jle short 0050h
		/*0049*/ 0xB0, 0x0E,// mov al,0Eh
		/*004B*/ 0x66, 0x7F, 0x02,// jg short 0050h
		/*004E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0050*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x66, 0x70, 0x4D,// jo short 0050h
		/*0003*/ 0xB0, 0x00,// mov al,0
		/*0005*/ 0x66, 0x71, 0x48,// jno short 0050h
		/*0008*/ 0xB0, 0x01,// mov al,1
		/*000A*/ 0x66, 0x72, 0x43,// jb short 0050h
		/*000D*/ 0xB0, 0x02,// mov al,2
		/*000F*/ 0x66, 0x73, 0x3E,// jae short 0050h
		/*0012*/ 0xB0, 0x03,// mov al,3
		/*0014*/ 0x66, 0x74, 0x39,// je short 0050h
		/*0017*/ 0xB0, 0x04,// mov al,4
		/*0019*/ 0x66, 0x75, 0x34,// jne short 0050h
		/*001C*/ 0xB0, 0x05,// mov al,5
		/*001E*/ 0x66, 0x76, 0x2F,// jbe short 0050h
		/*0021*/ 0xB0, 0x06,// mov al,6
		/*0023*/ 0x66, 0x77, 0x2A,// ja short 0050h
		/*0026*/ 0xB0, 0x07,// mov al,7
		/*0028*/ 0x66, 0x78, 0x25,// js short 0050h
		/*002B*/ 0xB0, 0x08,// mov al,8
		/*002D*/ 0x66, 0x79, 0x20,// jns short 0050h
		/*0030*/ 0xB0, 0x09,// mov al,9
		/*0032*/ 0x66, 0x7A, 0x1B,// jp short 0050h
		/*0035*/ 0xB0, 0x0A,// mov al,0Ah
		/*0037*/ 0x66, 0x7B, 0x16,// jnp short 0050h
		/*003A*/ 0xB0, 0x0B,// mov al,0Bh
		/*003C*/ 0x66, 0x7C, 0x11,// jl short 0050h
		/*003F*/ 0xB0, 0x0C,// mov al,0Ch
		/*0041*/ 0x66, 0x7D, 0x0C,// jge short 0050h
		/*0044*/ 0xB0, 0x0D,// mov al,0Dh
		/*0046*/ 0x66, 0x7E, 0x07,// jle short 0050h
		/*0049*/ 0xB0, 0x0E,// mov al,0Eh
		/*004B*/ 0x66, 0x7F, 0x02,// jg short 0050h
		/*004E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0050*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0003,
		0x0005,
		0x0008,
		0x000A,
		0x000D,
		0x000F,
		0x0012,
		0x0014,
		0x0017,
		0x0019,
		0x001C,
		0x001E,
		0x0021,
		0x0023,
		0x0026,
		0x0028,
		0x002B,
		0x002D,
		0x0030,
		0x0032,
		0x0035,
		0x0037,
		0x003A,
		0x003C,
		0x003F,
		0x0041,
		0x0044,
		0x0046,
		0x0049,
		0x004B,
		0x004E,
		0x0050,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS | DecoderOptions::AMD,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_short_bwd() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0xB0, 0x10,// mov al,10h
		/*0002*/ 0x70, 0xFC,// jo short 0
		/*0004*/ 0xB0, 0x00,// mov al,0
		/*0006*/ 0x71, 0xF8,// jno short 0
		/*0008*/ 0xB0, 0x01,// mov al,1
		/*000A*/ 0x72, 0xF4,// jb short 0
		/*000C*/ 0xB0, 0x02,// mov al,2
		/*000E*/ 0x73, 0xF0,// jae short 0
		/*0010*/ 0xB0, 0x03,// mov al,3
		/*0012*/ 0x74, 0xEC,// je short 0
		/*0014*/ 0xB0, 0x04,// mov al,4
		/*0016*/ 0x75, 0xE8,// jne short 0
		/*0018*/ 0xB0, 0x05,// mov al,5
		/*001A*/ 0x76, 0xE4,// jbe short 0
		/*001C*/ 0xB0, 0x06,// mov al,6
		/*001E*/ 0x77, 0xE0,// ja short 0
		/*0020*/ 0xB0, 0x07,// mov al,7
		/*0022*/ 0x78, 0xDC,// js short 0
		/*0024*/ 0xB0, 0x08,// mov al,8
		/*0026*/ 0x79, 0xD8,// jns short 0
		/*0028*/ 0xB0, 0x09,// mov al,9
		/*002A*/ 0x7A, 0xD4,// jp short 0
		/*002C*/ 0xB0, 0x0A,// mov al,0Ah
		/*002E*/ 0x7B, 0xD0,// jnp short 0
		/*0030*/ 0xB0, 0x0B,// mov al,0Bh
		/*0032*/ 0x7C, 0xCC,// jl short 0
		/*0034*/ 0xB0, 0x0C,// mov al,0Ch
		/*0036*/ 0x7D, 0xC8,// jge short 0
		/*0038*/ 0xB0, 0x0D,// mov al,0Dh
		/*003A*/ 0x7E, 0xC4,// jle short 0
		/*003C*/ 0xB0, 0x0E,// mov al,0Eh
		/*003E*/ 0x7F, 0xC0,// jg short 0
		/*0040*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0xB0, 0x10,// mov al,10h
		/*0002*/ 0x70, 0xFC,// jo short 0
		/*0004*/ 0xB0, 0x00,// mov al,0
		/*0006*/ 0x71, 0xF8,// jno short 0
		/*0008*/ 0xB0, 0x01,// mov al,1
		/*000A*/ 0x72, 0xF4,// jb short 0
		/*000C*/ 0xB0, 0x02,// mov al,2
		/*000E*/ 0x73, 0xF0,// jae short 0
		/*0010*/ 0xB0, 0x03,// mov al,3
		/*0012*/ 0x74, 0xEC,// je short 0
		/*0014*/ 0xB0, 0x04,// mov al,4
		/*0016*/ 0x75, 0xE8,// jne short 0
		/*0018*/ 0xB0, 0x05,// mov al,5
		/*001A*/ 0x76, 0xE4,// jbe short 0
		/*001C*/ 0xB0, 0x06,// mov al,6
		/*001E*/ 0x77, 0xE0,// ja short 0
		/*0020*/ 0xB0, 0x07,// mov al,7
		/*0022*/ 0x78, 0xDC,// js short 0
		/*0024*/ 0xB0, 0x08,// mov al,8
		/*0026*/ 0x79, 0xD8,// jns short 0
		/*0028*/ 0xB0, 0x09,// mov al,9
		/*002A*/ 0x7A, 0xD4,// jp short 0
		/*002C*/ 0xB0, 0x0A,// mov al,0Ah
		/*002E*/ 0x7B, 0xD0,// jnp short 0
		/*0030*/ 0xB0, 0x0B,// mov al,0Bh
		/*0032*/ 0x7C, 0xCC,// jl short 0
		/*0034*/ 0xB0, 0x0C,// mov al,0Ch
		/*0036*/ 0x7D, 0xC8,// jge short 0
		/*0038*/ 0xB0, 0x0D,// mov al,0Dh
		/*003A*/ 0x7E, 0xC4,// jle short 0
		/*003C*/ 0xB0, 0x0E,// mov al,0Eh
		/*003E*/ 0x7F, 0xC0,// jg short 0
		/*0040*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0002,
		0x0004,
		0x0006,
		0x0008,
		0x000A,
		0x000C,
		0x000E,
		0x0010,
		0x0012,
		0x0014,
		0x0016,
		0x0018,
		0x001A,
		0x001C,
		0x001E,
		0x0020,
		0x0022,
		0x0024,
		0x0026,
		0x0028,
		0x002A,
		0x002C,
		0x002E,
		0x0030,
		0x0032,
		0x0034,
		0x0036,
		0x0038,
		0x003A,
		0x003C,
		0x003E,
		0x0040,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_short_bwd_os() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0xB0, 0x10,// mov al,10h
		/*0002*/ 0x66, 0x70, 0xFB,// jo short 0
		/*0005*/ 0xB0, 0x00,// mov al,0
		/*0007*/ 0x66, 0x71, 0xF6,// jno short 0
		/*000A*/ 0xB0, 0x01,// mov al,1
		/*000C*/ 0x66, 0x72, 0xF1,// jb short 0
		/*000F*/ 0xB0, 0x02,// mov al,2
		/*0011*/ 0x66, 0x73, 0xEC,// jae short 0
		/*0014*/ 0xB0, 0x03,// mov al,3
		/*0016*/ 0x66, 0x74, 0xE7,// je short 0
		/*0019*/ 0xB0, 0x04,// mov al,4
		/*001B*/ 0x66, 0x75, 0xE2,// jne short 0
		/*001E*/ 0xB0, 0x05,// mov al,5
		/*0020*/ 0x66, 0x76, 0xDD,// jbe short 0
		/*0023*/ 0xB0, 0x06,// mov al,6
		/*0025*/ 0x66, 0x77, 0xD8,// ja short 0
		/*0028*/ 0xB0, 0x07,// mov al,7
		/*002A*/ 0x66, 0x78, 0xD3,// js short 0
		/*002D*/ 0xB0, 0x08,// mov al,8
		/*002F*/ 0x66, 0x79, 0xCE,// jns short 0
		/*0032*/ 0xB0, 0x09,// mov al,9
		/*0034*/ 0x66, 0x7A, 0xC9,// jp short 0
		/*0037*/ 0xB0, 0x0A,// mov al,0Ah
		/*0039*/ 0x66, 0x7B, 0xC4,// jnp short 0
		/*003C*/ 0xB0, 0x0B,// mov al,0Bh
		/*003E*/ 0x66, 0x7C, 0xBF,// jl short 0
		/*0041*/ 0xB0, 0x0C,// mov al,0Ch
		/*0043*/ 0x66, 0x7D, 0xBA,// jge short 0
		/*0046*/ 0xB0, 0x0D,// mov al,0Dh
		/*0048*/ 0x66, 0x7E, 0xB5,// jle short 0
		/*004B*/ 0xB0, 0x0E,// mov al,0Eh
		/*004D*/ 0x66, 0x7F, 0xB0,// jg short 0
		/*0050*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0xB0, 0x10,// mov al,10h
		/*0002*/ 0x66, 0x70, 0xFB,// jo short 0
		/*0005*/ 0xB0, 0x00,// mov al,0
		/*0007*/ 0x66, 0x71, 0xF6,// jno short 0
		/*000A*/ 0xB0, 0x01,// mov al,1
		/*000C*/ 0x66, 0x72, 0xF1,// jb short 0
		/*000F*/ 0xB0, 0x02,// mov al,2
		/*0011*/ 0x66, 0x73, 0xEC,// jae short 0
		/*0014*/ 0xB0, 0x03,// mov al,3
		/*0016*/ 0x66, 0x74, 0xE7,// je short 0
		/*0019*/ 0xB0, 0x04,// mov al,4
		/*001B*/ 0x66, 0x75, 0xE2,// jne short 0
		/*001E*/ 0xB0, 0x05,// mov al,5
		/*0020*/ 0x66, 0x76, 0xDD,// jbe short 0
		/*0023*/ 0xB0, 0x06,// mov al,6
		/*0025*/ 0x66, 0x77, 0xD8,// ja short 0
		/*0028*/ 0xB0, 0x07,// mov al,7
		/*002A*/ 0x66, 0x78, 0xD3,// js short 0
		/*002D*/ 0xB0, 0x08,// mov al,8
		/*002F*/ 0x66, 0x79, 0xCE,// jns short 0
		/*0032*/ 0xB0, 0x09,// mov al,9
		/*0034*/ 0x66, 0x7A, 0xC9,// jp short 0
		/*0037*/ 0xB0, 0x0A,// mov al,0Ah
		/*0039*/ 0x66, 0x7B, 0xC4,// jnp short 0
		/*003C*/ 0xB0, 0x0B,// mov al,0Bh
		/*003E*/ 0x66, 0x7C, 0xBF,// jl short 0
		/*0041*/ 0xB0, 0x0C,// mov al,0Ch
		/*0043*/ 0x66, 0x7D, 0xBA,// jge short 0
		/*0046*/ 0xB0, 0x0D,// mov al,0Dh
		/*0048*/ 0x66, 0x7E, 0xB5,// jle short 0
		/*004B*/ 0xB0, 0x0E,// mov al,0Eh
		/*004D*/ 0x66, 0x7F, 0xB0,// jg short 0
		/*0050*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0002,
		0x0005,
		0x0007,
		0x000A,
		0x000C,
		0x000F,
		0x0011,
		0x0014,
		0x0016,
		0x0019,
		0x001B,
		0x001E,
		0x0020,
		0x0023,
		0x0025,
		0x0028,
		0x002A,
		0x002D,
		0x002F,
		0x0032,
		0x0034,
		0x0037,
		0x0039,
		0x003C,
		0x003E,
		0x0041,
		0x0043,
		0x0046,
		0x0048,
		0x004B,
		0x004D,
		0x0050,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS | DecoderOptions::AMD,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_short_other_short() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x70, 0x3E,// jo short 0000000000000040h
		/*0002*/ 0xB0, 0x00,// mov al,0
		/*0004*/ 0x71, 0x3A,// jno short 0000000000000040h
		/*0006*/ 0xB0, 0x01,// mov al,1
		/*0008*/ 0x72, 0x36,// jb short 0000000000000040h
		/*000A*/ 0xB0, 0x02,// mov al,2
		/*000C*/ 0x73, 0x32,// jae short 0000000000000040h
		/*000E*/ 0xB0, 0x03,// mov al,3
		/*0010*/ 0x74, 0x2E,// je short 0000000000000040h
		/*0012*/ 0xB0, 0x04,// mov al,4
		/*0014*/ 0x75, 0x2A,// jne short 0000000000000040h
		/*0016*/ 0xB0, 0x05,// mov al,5
		/*0018*/ 0x76, 0x26,// jbe short 0000000000000040h
		/*001A*/ 0xB0, 0x06,// mov al,6
		/*001C*/ 0x77, 0x22,// ja short 0000000000000040h
		/*001E*/ 0xB0, 0x07,// mov al,7
		/*0020*/ 0x78, 0x1E,// js short 0000000000000040h
		/*0022*/ 0xB0, 0x08,// mov al,8
		/*0024*/ 0x79, 0x1A,// jns short 0000000000000040h
		/*0026*/ 0xB0, 0x09,// mov al,9
		/*0028*/ 0x7A, 0x16,// jp short 0000000000000040h
		/*002A*/ 0xB0, 0x0A,// mov al,0Ah
		/*002C*/ 0x7B, 0x12,// jnp short 0000000000000040h
		/*002E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0030*/ 0x7C, 0x0E,// jl short 0000000000000040h
		/*0032*/ 0xB0, 0x0C,// mov al,0Ch
		/*0034*/ 0x7D, 0x0A,// jge short 0000000000000040h
		/*0036*/ 0xB0, 0x0D,// mov al,0Dh
		/*0038*/ 0x7E, 0x06,// jle short 0000000000000040h
		/*003A*/ 0xB0, 0x0E,// mov al,0Eh
		/*003C*/ 0x7F, 0x02,// jg short 0000000000000040h
		/*003E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x70, 0x3F,// jo short 0000000000000041h
		/*0002*/ 0xB0, 0x00,// mov al,0
		/*0004*/ 0x71, 0x3B,// jno short 0000000000000041h
		/*0006*/ 0xB0, 0x01,// mov al,1
		/*0008*/ 0x72, 0x37,// jb short 0000000000000041h
		/*000A*/ 0xB0, 0x02,// mov al,2
		/*000C*/ 0x73, 0x33,// jae short 0000000000000041h
		/*000E*/ 0xB0, 0x03,// mov al,3
		/*0010*/ 0x74, 0x2F,// je short 0000000000000041h
		/*0012*/ 0xB0, 0x04,// mov al,4
		/*0014*/ 0x75, 0x2B,// jne short 0000000000000041h
		/*0016*/ 0xB0, 0x05,// mov al,5
		/*0018*/ 0x76, 0x27,// jbe short 0000000000000041h
		/*001A*/ 0xB0, 0x06,// mov al,6
		/*001C*/ 0x77, 0x23,// ja short 0000000000000041h
		/*001E*/ 0xB0, 0x07,// mov al,7
		/*0020*/ 0x78, 0x1F,// js short 0000000000000041h
		/*0022*/ 0xB0, 0x08,// mov al,8
		/*0024*/ 0x79, 0x1B,// jns short 0000000000000041h
		/*0026*/ 0xB0, 0x09,// mov al,9
		/*0028*/ 0x7A, 0x17,// jp short 0000000000000041h
		/*002A*/ 0xB0, 0x0A,// mov al,0Ah
		/*002C*/ 0x7B, 0x13,// jnp short 0000000000000041h
		/*002E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0030*/ 0x7C, 0x0F,// jl short 0000000000000041h
		/*0032*/ 0xB0, 0x0C,// mov al,0Ch
		/*0034*/ 0x7D, 0x0B,// jge short 0000000000000041h
		/*0036*/ 0xB0, 0x0D,// mov al,0Dh
		/*0038*/ 0x7E, 0x07,// jle short 0000000000000041h
		/*003A*/ 0xB0, 0x0E,// mov al,0Eh
		/*003C*/ 0x7F, 0x03,// jg short 0000000000000041h
		/*003E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0002,
		0x0004,
		0x0006,
		0x0008,
		0x000A,
		0x000C,
		0x000E,
		0x0010,
		0x0012,
		0x0014,
		0x0016,
		0x0018,
		0x001A,
		0x001C,
		0x001E,
		0x0020,
		0x0022,
		0x0024,
		0x0026,
		0x0028,
		0x002A,
		0x002C,
		0x002E,
		0x0030,
		0x0032,
		0x0034,
		0x0036,
		0x0038,
		0x003A,
		0x003C,
		0x003E,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		ORIG_RIP - 1,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_short_other_near() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x70, 0x3E,// jo short 0000000000000040h
		/*0002*/ 0xB0, 0x00,// mov al,0
		/*0004*/ 0x71, 0x3A,// jno short 0000000000000040h
		/*0006*/ 0xB0, 0x01,// mov al,1
		/*0008*/ 0x72, 0x36,// jb short 0000000000000040h
		/*000A*/ 0xB0, 0x02,// mov al,2
		/*000C*/ 0x73, 0x32,// jae short 0000000000000040h
		/*000E*/ 0xB0, 0x03,// mov al,3
		/*0010*/ 0x74, 0x2E,// je short 0000000000000040h
		/*0012*/ 0xB0, 0x04,// mov al,4
		/*0014*/ 0x75, 0x2A,// jne short 0000000000000040h
		/*0016*/ 0xB0, 0x05,// mov al,5
		/*0018*/ 0x76, 0x26,// jbe short 0000000000000040h
		/*001A*/ 0xB0, 0x06,// mov al,6
		/*001C*/ 0x77, 0x22,// ja short 0000000000000040h
		/*001E*/ 0xB0, 0x07,// mov al,7
		/*0020*/ 0x78, 0x1E,// js short 0000000000000040h
		/*0022*/ 0xB0, 0x08,// mov al,8
		/*0024*/ 0x79, 0x1A,// jns short 0000000000000040h
		/*0026*/ 0xB0, 0x09,// mov al,9
		/*0028*/ 0x7A, 0x16,// jp short 0000000000000040h
		/*002A*/ 0xB0, 0x0A,// mov al,0Ah
		/*002C*/ 0x7B, 0x12,// jnp short 0000000000000040h
		/*002E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0030*/ 0x7C, 0x0E,// jl short 0000000000000040h
		/*0032*/ 0xB0, 0x0C,// mov al,0Ch
		/*0034*/ 0x7D, 0x0A,// jge short 0000000000000040h
		/*0036*/ 0xB0, 0x0D,// mov al,0Dh
		/*0038*/ 0x7E, 0x06,// jle short 0000000000000040h
		/*003A*/ 0xB0, 0x0E,// mov al,0Eh
		/*003C*/ 0x7F, 0x02,// jg short 0000000000000040h
		/*003E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x0F, 0x80, 0x3A, 0xF0, 0xFF, 0xFF,// jo near ptr 0FFFFFFFFFFFFF040h
		/*0006*/ 0xB0, 0x00,// mov al,0
		/*0008*/ 0x0F, 0x81, 0x32, 0xF0, 0xFF, 0xFF,// jno near ptr 0FFFFFFFFFFFFF040h
		/*000E*/ 0xB0, 0x01,// mov al,1
		/*0010*/ 0x0F, 0x82, 0x2A, 0xF0, 0xFF, 0xFF,// jb near ptr 0FFFFFFFFFFFFF040h
		/*0016*/ 0xB0, 0x02,// mov al,2
		/*0018*/ 0x0F, 0x83, 0x22, 0xF0, 0xFF, 0xFF,// jae near ptr 0FFFFFFFFFFFFF040h
		/*001E*/ 0xB0, 0x03,// mov al,3
		/*0020*/ 0x0F, 0x84, 0x1A, 0xF0, 0xFF, 0xFF,// je near ptr 0FFFFFFFFFFFFF040h
		/*0026*/ 0xB0, 0x04,// mov al,4
		/*0028*/ 0x0F, 0x85, 0x12, 0xF0, 0xFF, 0xFF,// jne near ptr 0FFFFFFFFFFFFF040h
		/*002E*/ 0xB0, 0x05,// mov al,5
		/*0030*/ 0x0F, 0x86, 0x0A, 0xF0, 0xFF, 0xFF,// jbe near ptr 0FFFFFFFFFFFFF040h
		/*0036*/ 0xB0, 0x06,// mov al,6
		/*0038*/ 0x0F, 0x87, 0x02, 0xF0, 0xFF, 0xFF,// ja near ptr 0FFFFFFFFFFFFF040h
		/*003E*/ 0xB0, 0x07,// mov al,7
		/*0040*/ 0x0F, 0x88, 0xFA, 0xEF, 0xFF, 0xFF,// js near ptr 0FFFFFFFFFFFFF040h
		/*0046*/ 0xB0, 0x08,// mov al,8
		/*0048*/ 0x0F, 0x89, 0xF2, 0xEF, 0xFF, 0xFF,// jns near ptr 0FFFFFFFFFFFFF040h
		/*004E*/ 0xB0, 0x09,// mov al,9
		/*0050*/ 0x0F, 0x8A, 0xEA, 0xEF, 0xFF, 0xFF,// jp near ptr 0FFFFFFFFFFFFF040h
		/*0056*/ 0xB0, 0x0A,// mov al,0Ah
		/*0058*/ 0x0F, 0x8B, 0xE2, 0xEF, 0xFF, 0xFF,// jnp near ptr 0FFFFFFFFFFFFF040h
		/*005E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0060*/ 0x0F, 0x8C, 0xDA, 0xEF, 0xFF, 0xFF,// jl near ptr 0FFFFFFFFFFFFF040h
		/*0066*/ 0xB0, 0x0C,// mov al,0Ch
		/*0068*/ 0x0F, 0x8D, 0xD2, 0xEF, 0xFF, 0xFF,// jge near ptr 0FFFFFFFFFFFFF040h
		/*006E*/ 0xB0, 0x0D,// mov al,0Dh
		/*0070*/ 0x0F, 0x8E, 0xCA, 0xEF, 0xFF, 0xFF,// jle near ptr 0FFFFFFFFFFFFF040h
		/*0076*/ 0xB0, 0x0E,// mov al,0Eh
		/*0078*/ 0x0F, 0x8F, 0xC2, 0xEF, 0xFF, 0xFF,// jg near ptr 0FFFFFFFFFFFFF040h
		/*007E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0006,
		0x0008,
		0x000E,
		0x0010,
		0x0016,
		0x0018,
		0x001E,
		0x0020,
		0x0026,
		0x0028,
		0x002E,
		0x0030,
		0x0036,
		0x0038,
		0x003E,
		0x0040,
		0x0046,
		0x0048,
		0x004E,
		0x0050,
		0x0056,
		0x0058,
		0x005E,
		0x0060,
		0x0066,
		0x0068,
		0x006E,
		0x0070,
		0x0076,
		0x0078,
		0x007E,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		ORIG_RIP + 0x1000,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_short_other_short_os() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x66, 0x70, 0x4D,// jo short 8050h
		/*0003*/ 0xB0, 0x00,// mov al,0
		/*0005*/ 0x66, 0x71, 0x49,// jno short 8051h
		/*0008*/ 0xB0, 0x01,// mov al,1
		/*000A*/ 0x66, 0x72, 0x45,// jb short 8052h
		/*000D*/ 0xB0, 0x02,// mov al,2
		/*000F*/ 0x66, 0x73, 0x41,// jae short 8053h
		/*0012*/ 0xB0, 0x03,// mov al,3
		/*0014*/ 0x66, 0x74, 0x3D,// je short 8054h
		/*0017*/ 0xB0, 0x04,// mov al,4
		/*0019*/ 0x66, 0x75, 0x39,// jne short 8055h
		/*001C*/ 0xB0, 0x05,// mov al,5
		/*001E*/ 0x66, 0x76, 0x35,// jbe short 8056h
		/*0021*/ 0xB0, 0x06,// mov al,6
		/*0023*/ 0x66, 0x77, 0x31,// ja short 8057h
		/*0026*/ 0xB0, 0x07,// mov al,7
		/*0028*/ 0x66, 0x78, 0x2D,// js short 8058h
		/*002B*/ 0xB0, 0x08,// mov al,8
		/*002D*/ 0x66, 0x79, 0x29,// jns short 8059h
		/*0030*/ 0xB0, 0x09,// mov al,9
		/*0032*/ 0x66, 0x7A, 0x25,// jp short 805Ah
		/*0035*/ 0xB0, 0x0A,// mov al,0Ah
		/*0037*/ 0x66, 0x7B, 0x21,// jnp short 805Bh
		/*003A*/ 0xB0, 0x0B,// mov al,0Bh
		/*003C*/ 0x66, 0x7C, 0x1D,// jl short 805Ch
		/*003F*/ 0xB0, 0x0C,// mov al,0Ch
		/*0041*/ 0x66, 0x7D, 0x19,// jge short 805Dh
		/*0044*/ 0xB0, 0x0D,// mov al,0Dh
		/*0046*/ 0x66, 0x7E, 0x15,// jle short 805Eh
		/*0049*/ 0xB0, 0x0E,// mov al,0Eh
		/*004B*/ 0x66, 0x7F, 0x11,// jg short 805Fh
		/*004E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x66, 0x70, 0x4E,// jo short 8050h
		/*0003*/ 0xB0, 0x00,// mov al,0
		/*0005*/ 0x66, 0x71, 0x4A,// jno short 8051h
		/*0008*/ 0xB0, 0x01,// mov al,1
		/*000A*/ 0x66, 0x72, 0x46,// jb short 8052h
		/*000D*/ 0xB0, 0x02,// mov al,2
		/*000F*/ 0x66, 0x73, 0x42,// jae short 8053h
		/*0012*/ 0xB0, 0x03,// mov al,3
		/*0014*/ 0x66, 0x74, 0x3E,// je short 8054h
		/*0017*/ 0xB0, 0x04,// mov al,4
		/*0019*/ 0x66, 0x75, 0x3A,// jne short 8055h
		/*001C*/ 0xB0, 0x05,// mov al,5
		/*001E*/ 0x66, 0x76, 0x36,// jbe short 8056h
		/*0021*/ 0xB0, 0x06,// mov al,6
		/*0023*/ 0x66, 0x77, 0x32,// ja short 8057h
		/*0026*/ 0xB0, 0x07,// mov al,7
		/*0028*/ 0x66, 0x78, 0x2E,// js short 8058h
		/*002B*/ 0xB0, 0x08,// mov al,8
		/*002D*/ 0x66, 0x79, 0x2A,// jns short 8059h
		/*0030*/ 0xB0, 0x09,// mov al,9
		/*0032*/ 0x66, 0x7A, 0x26,// jp short 805Ah
		/*0035*/ 0xB0, 0x0A,// mov al,0Ah
		/*0037*/ 0x66, 0x7B, 0x22,// jnp short 805Bh
		/*003A*/ 0xB0, 0x0B,// mov al,0Bh
		/*003C*/ 0x66, 0x7C, 0x1E,// jl short 805Ch
		/*003F*/ 0xB0, 0x0C,// mov al,0Ch
		/*0041*/ 0x66, 0x7D, 0x1A,// jge short 805Dh
		/*0044*/ 0xB0, 0x0D,// mov al,0Dh
		/*0046*/ 0x66, 0x7E, 0x16,// jle short 805Eh
		/*0049*/ 0xB0, 0x0E,// mov al,0Eh
		/*004B*/ 0x66, 0x7F, 0x12,// jg short 805Fh
		/*004E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0003,
		0x0005,
		0x0008,
		0x000A,
		0x000D,
		0x000F,
		0x0012,
		0x0014,
		0x0017,
		0x0019,
		0x001C,
		0x001E,
		0x0021,
		0x0023,
		0x0026,
		0x0028,
		0x002B,
		0x002D,
		0x0030,
		0x0032,
		0x0035,
		0x0037,
		0x003A,
		0x003C,
		0x003F,
		0x0041,
		0x0044,
		0x0046,
		0x0049,
		0x004B,
		0x004E,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		ORIG_RIP - 1,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS | DecoderOptions::AMD,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_short_other_near_os() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x66, 0x70, 0x4D,// jo short 8050h
		/*0003*/ 0xB0, 0x00,// mov al,0
		/*0005*/ 0x66, 0x71, 0x49,// jno short 8051h
		/*0008*/ 0xB0, 0x01,// mov al,1
		/*000A*/ 0x66, 0x72, 0x45,// jb short 8052h
		/*000D*/ 0xB0, 0x02,// mov al,2
		/*000F*/ 0x66, 0x73, 0x41,// jae short 8053h
		/*0012*/ 0xB0, 0x03,// mov al,3
		/*0014*/ 0x66, 0x74, 0x3D,// je short 8054h
		/*0017*/ 0xB0, 0x04,// mov al,4
		/*0019*/ 0x66, 0x75, 0x39,// jne short 8055h
		/*001C*/ 0xB0, 0x05,// mov al,5
		/*001E*/ 0x66, 0x76, 0x35,// jbe short 8056h
		/*0021*/ 0xB0, 0x06,// mov al,6
		/*0023*/ 0x66, 0x77, 0x31,// ja short 8057h
		/*0026*/ 0xB0, 0x07,// mov al,7
		/*0028*/ 0x66, 0x78, 0x2D,// js short 8058h
		/*002B*/ 0xB0, 0x08,// mov al,8
		/*002D*/ 0x66, 0x79, 0x29,// jns short 8059h
		/*0030*/ 0xB0, 0x09,// mov al,9
		/*0032*/ 0x66, 0x7A, 0x25,// jp short 805Ah
		/*0035*/ 0xB0, 0x0A,// mov al,0Ah
		/*0037*/ 0x66, 0x7B, 0x21,// jnp short 805Bh
		/*003A*/ 0xB0, 0x0B,// mov al,0Bh
		/*003C*/ 0x66, 0x7C, 0x1D,// jl short 805Ch
		/*003F*/ 0xB0, 0x0C,// mov al,0Ch
		/*0041*/ 0x66, 0x7D, 0x19,// jge short 805Dh
		/*0044*/ 0xB0, 0x0D,// mov al,0Dh
		/*0046*/ 0x66, 0x7E, 0x15,// jle short 805Eh
		/*0049*/ 0xB0, 0x0E,// mov al,0Eh
		/*004B*/ 0x66, 0x7F, 0x11,// jg short 805Fh
		/*004E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x66, 0x0F, 0x80, 0x4B, 0xF0,// jo near ptr 8050h
		/*0005*/ 0xB0, 0x00,// mov al,0
		/*0007*/ 0x66, 0x0F, 0x81, 0x45, 0xF0,// jno near ptr 8051h
		/*000C*/ 0xB0, 0x01,// mov al,1
		/*000E*/ 0x66, 0x0F, 0x82, 0x3F, 0xF0,// jb near ptr 8052h
		/*0013*/ 0xB0, 0x02,// mov al,2
		/*0015*/ 0x66, 0x0F, 0x83, 0x39, 0xF0,// jae near ptr 8053h
		/*001A*/ 0xB0, 0x03,// mov al,3
		/*001C*/ 0x66, 0x0F, 0x84, 0x33, 0xF0,// je near ptr 8054h
		/*0021*/ 0xB0, 0x04,// mov al,4
		/*0023*/ 0x66, 0x0F, 0x85, 0x2D, 0xF0,// jne near ptr 8055h
		/*0028*/ 0xB0, 0x05,// mov al,5
		/*002A*/ 0x66, 0x0F, 0x86, 0x27, 0xF0,// jbe near ptr 8056h
		/*002F*/ 0xB0, 0x06,// mov al,6
		/*0031*/ 0x66, 0x0F, 0x87, 0x21, 0xF0,// ja near ptr 8057h
		/*0036*/ 0xB0, 0x07,// mov al,7
		/*0038*/ 0x66, 0x0F, 0x88, 0x1B, 0xF0,// js near ptr 8058h
		/*003D*/ 0xB0, 0x08,// mov al,8
		/*003F*/ 0x66, 0x0F, 0x89, 0x15, 0xF0,// jns near ptr 8059h
		/*0044*/ 0xB0, 0x09,// mov al,9
		/*0046*/ 0x66, 0x0F, 0x8A, 0x0F, 0xF0,// jp near ptr 805Ah
		/*004B*/ 0xB0, 0x0A,// mov al,0Ah
		/*004D*/ 0x66, 0x0F, 0x8B, 0x09, 0xF0,// jnp near ptr 805Bh
		/*0052*/ 0xB0, 0x0B,// mov al,0Bh
		/*0054*/ 0x66, 0x0F, 0x8C, 0x03, 0xF0,// jl near ptr 805Ch
		/*0059*/ 0xB0, 0x0C,// mov al,0Ch
		/*005B*/ 0x66, 0x0F, 0x8D, 0xFD, 0xEF,// jge near ptr 805Dh
		/*0060*/ 0xB0, 0x0D,// mov al,0Dh
		/*0062*/ 0x66, 0x0F, 0x8E, 0xF7, 0xEF,// jle near ptr 805Eh
		/*0067*/ 0xB0, 0x0E,// mov al,0Eh
		/*0069*/ 0x66, 0x0F, 0x8F, 0xF1, 0xEF,// jg near ptr 805Fh
		/*006E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0005,
		0x0007,
		0x000C,
		0x000E,
		0x0013,
		0x0015,
		0x001A,
		0x001C,
		0x0021,
		0x0023,
		0x0028,
		0x002A,
		0x002F,
		0x0031,
		0x0036,
		0x0038,
		0x003D,
		0x003F,
		0x0044,
		0x0046,
		0x004B,
		0x004D,
		0x0052,
		0x0054,
		0x0059,
		0x005B,
		0x0060,
		0x0062,
		0x0067,
		0x0069,
		0x006E,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		ORIG_RIP + 0x1000,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS | DecoderOptions::AMD,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_short_other_long() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x70, 0x3F,// jo short 123456789ABCDE41h
		/*0002*/ 0xB0, 0x00,// mov al,0
		/*0004*/ 0x71, 0x3C,// jno short 123456789ABCDE42h
		/*0006*/ 0xB0, 0x01,// mov al,1
		/*0008*/ 0x72, 0x39,// jb short 123456789ABCDE43h
		/*000A*/ 0xB0, 0x02,// mov al,2
		/*000C*/ 0x73, 0x36,// jae short 123456789ABCDE44h
		/*000E*/ 0xB0, 0x03,// mov al,3
		/*0010*/ 0x74, 0x33,// je short 123456789ABCDE45h
		/*0012*/ 0xB0, 0x04,// mov al,4
		/*0014*/ 0x75, 0x30,// jne short 123456789ABCDE46h
		/*0016*/ 0xB0, 0x05,// mov al,5
		/*0018*/ 0x76, 0x2D,// jbe short 123456789ABCDE47h
		/*001A*/ 0xB0, 0x06,// mov al,6
		/*001C*/ 0x77, 0x2A,// ja short 123456789ABCDE48h
		/*001E*/ 0xB0, 0x07,// mov al,7
		/*0020*/ 0x78, 0x27,// js short 123456789ABCDE49h
		/*0022*/ 0xB0, 0x08,// mov al,8
		/*0024*/ 0x79, 0x24,// jns short 123456789ABCDE4Ah
		/*0026*/ 0xB0, 0x09,// mov al,9
		/*0028*/ 0x7A, 0x21,// jp short 123456789ABCDE4Bh
		/*002A*/ 0xB0, 0x0A,// mov al,0Ah
		/*002C*/ 0x7B, 0x1E,// jnp short 123456789ABCDE4Ch
		/*002E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0030*/ 0x7C, 0x1B,// jl short 123456789ABCDE4Dh
		/*0032*/ 0xB0, 0x0C,// mov al,0Ch
		/*0034*/ 0x7D, 0x18,// jge short 123456789ABCDE4Eh
		/*0036*/ 0xB0, 0x0D,// mov al,0Dh
		/*0038*/ 0x7E, 0x15,// jle short 123456789ABCDE4Fh
		/*003A*/ 0xB0, 0x0E,// mov al,0Eh
		/*003C*/ 0x7F, 0x12,// jg short 123456789ABCDE50h
		/*003E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0040*/ 0x90,// nop
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x71, 0x06,// jno short 8000000000000008h
		/*0002*/ 0xFF, 0x25, 0xA0, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000A8h]
		/*0008*/ 0xB0, 0x00,// mov al,0
		/*000A*/ 0x70, 0x06,// jo short 8000000000000012h
		/*000C*/ 0xFF, 0x25, 0x9E, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000B0h]
		/*0012*/ 0xB0, 0x01,// mov al,1
		/*0014*/ 0x73, 0x06,// jae short 800000000000001Ch
		/*0016*/ 0xFF, 0x25, 0x9C, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000B8h]
		/*001C*/ 0xB0, 0x02,// mov al,2
		/*001E*/ 0x72, 0x06,// jb short 8000000000000026h
		/*0020*/ 0xFF, 0x25, 0x9A, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000C0h]
		/*0026*/ 0xB0, 0x03,// mov al,3
		/*0028*/ 0x75, 0x06,// jne short 8000000000000030h
		/*002A*/ 0xFF, 0x25, 0x98, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000C8h]
		/*0030*/ 0xB0, 0x04,// mov al,4
		/*0032*/ 0x74, 0x06,// je short 800000000000003Ah
		/*0034*/ 0xFF, 0x25, 0x96, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000D0h]
		/*003A*/ 0xB0, 0x05,// mov al,5
		/*003C*/ 0x77, 0x06,// ja short 8000000000000044h
		/*003E*/ 0xFF, 0x25, 0x94, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000D8h]
		/*0044*/ 0xB0, 0x06,// mov al,6
		/*0046*/ 0x76, 0x06,// jbe short 800000000000004Eh
		/*0048*/ 0xFF, 0x25, 0x92, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000E0h]
		/*004E*/ 0xB0, 0x07,// mov al,7
		/*0050*/ 0x79, 0x06,// jns short 8000000000000058h
		/*0052*/ 0xFF, 0x25, 0x90, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000E8h]
		/*0058*/ 0xB0, 0x08,// mov al,8
		/*005A*/ 0x78, 0x06,// js short 8000000000000062h
		/*005C*/ 0xFF, 0x25, 0x8E, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000F0h]
		/*0062*/ 0xB0, 0x09,// mov al,9
		/*0064*/ 0x7B, 0x06,// jnp short 800000000000006Ch
		/*0066*/ 0xFF, 0x25, 0x8C, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000F8h]
		/*006C*/ 0xB0, 0x0A,// mov al,0Ah
		/*006E*/ 0x7A, 0x06,// jp short 8000000000000076h
		/*0070*/ 0xFF, 0x25, 0x8A, 0x00, 0x00, 0x00,// jmp qword ptr [8000000000000100h]
		/*0076*/ 0xB0, 0x0B,// mov al,0Bh
		/*0078*/ 0x7D, 0x06,// jge short 8000000000000080h
		/*007A*/ 0xFF, 0x25, 0x88, 0x00, 0x00, 0x00,// jmp qword ptr [8000000000000108h]
		/*0080*/ 0xB0, 0x0C,// mov al,0Ch
		/*0082*/ 0x7C, 0x06,// jl short 800000000000008Ah
		/*0084*/ 0xFF, 0x25, 0x86, 0x00, 0x00, 0x00,// jmp qword ptr [8000000000000110h]
		/*008A*/ 0xB0, 0x0D,// mov al,0Dh
		/*008C*/ 0x7F, 0x06,// jg short 8000000000000094h
		/*008E*/ 0xFF, 0x25, 0x84, 0x00, 0x00, 0x00,// jmp qword ptr [8000000000000118h]
		/*0094*/ 0xB0, 0x0E,// mov al,0Eh
		/*0096*/ 0x7E, 0x06,// jle short 800000000000009Eh
		/*0098*/ 0xFF, 0x25, 0x82, 0x00, 0x00, 0x00,// jmp qword ptr [8000000000000120h]
		/*009E*/ 0xB0, 0x0F,// mov al,0Fh
		/*00A0*/ 0x90,// nop
		/*00A1*/ 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC,
		/*00A8*/ 0x41, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00B0*/ 0x42, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00B8*/ 0x43, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00C0*/ 0x44, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00C8*/ 0x45, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00D0*/ 0x46, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00D8*/ 0x47, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00E0*/ 0x48, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00E8*/ 0x49, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00F0*/ 0x4A, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00F8*/ 0x4B, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*0100*/ 0x4C, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*0108*/ 0x4D, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*0110*/ 0x4E, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*0118*/ 0x4F, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*0120*/ 0x50, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		u32::MAX,
		0x0008,
		u32::MAX,
		0x0012,
		u32::MAX,
		0x001C,
		u32::MAX,
		0x0026,
		u32::MAX,
		0x0030,
		u32::MAX,
		0x003A,
		u32::MAX,
		0x0044,
		u32::MAX,
		0x004E,
		u32::MAX,
		0x0058,
		u32::MAX,
		0x0062,
		u32::MAX,
		0x006C,
		u32::MAX,
		0x0076,
		u32::MAX,
		0x0080,
		u32::MAX,
		0x008A,
		u32::MAX,
		0x0094,
		u32::MAX,
		0x009E,
		0x00A0,
	];
	#[rustfmt::skip]
	let expected_reloc_infos = [
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00A8),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00B0),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00B8),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00C0),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00C8),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00D0),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00D8),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00E0),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00E8),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00F0),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00F8),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_0100),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_0108),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_0110),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_0118),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_0120),
	];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	const ORIG_RIP: u64 = 0x1234_5678_9ABC_DE00;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_near_fwd_short() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x0F, 0x80, 0x7A, 0x00, 0x00, 0x00,// jo near ptr 0000000000008080h
		/*0006*/ 0xB0, 0x00,// mov al,0
		/*0008*/ 0x0F, 0x81, 0x72, 0x00, 0x00, 0x00,// jno near ptr 0000000000008080h
		/*000E*/ 0xB0, 0x01,// mov al,1
		/*0010*/ 0x0F, 0x82, 0x6A, 0x00, 0x00, 0x00,// jb near ptr 0000000000008080h
		/*0016*/ 0xB0, 0x02,// mov al,2
		/*0018*/ 0x0F, 0x83, 0x62, 0x00, 0x00, 0x00,// jae near ptr 0000000000008080h
		/*001E*/ 0xB0, 0x03,// mov al,3
		/*0020*/ 0x0F, 0x84, 0x5A, 0x00, 0x00, 0x00,// je near ptr 0000000000008080h
		/*0026*/ 0xB0, 0x04,// mov al,4
		/*0028*/ 0x0F, 0x85, 0x52, 0x00, 0x00, 0x00,// jne near ptr 0000000000008080h
		/*002E*/ 0xB0, 0x05,// mov al,5
		/*0030*/ 0x0F, 0x86, 0x4A, 0x00, 0x00, 0x00,// jbe near ptr 0000000000008080h
		/*0036*/ 0xB0, 0x06,// mov al,6
		/*0038*/ 0x0F, 0x87, 0x42, 0x00, 0x00, 0x00,// ja near ptr 0000000000008080h
		/*003E*/ 0xB0, 0x07,// mov al,7
		/*0040*/ 0x0F, 0x88, 0x3A, 0x00, 0x00, 0x00,// js near ptr 0000000000008080h
		/*0046*/ 0xB0, 0x08,// mov al,8
		/*0048*/ 0x0F, 0x89, 0x32, 0x00, 0x00, 0x00,// jns near ptr 0000000000008080h
		/*004E*/ 0xB0, 0x09,// mov al,9
		/*0050*/ 0x0F, 0x8A, 0x2A, 0x00, 0x00, 0x00,// jp near ptr 0000000000008080h
		/*0056*/ 0xB0, 0x0A,// mov al,0Ah
		/*0058*/ 0x0F, 0x8B, 0x22, 0x00, 0x00, 0x00,// jnp near ptr 0000000000008080h
		/*005E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0060*/ 0x0F, 0x8C, 0x1A, 0x00, 0x00, 0x00,// jl near ptr 0000000000008080h
		/*0066*/ 0xB0, 0x0C,// mov al,0Ch
		/*0068*/ 0x0F, 0x8D, 0x12, 0x00, 0x00, 0x00,// jge near ptr 0000000000008080h
		/*006E*/ 0xB0, 0x0D,// mov al,0Dh
		/*0070*/ 0x0F, 0x8E, 0x0A, 0x00, 0x00, 0x00,// jle near ptr 0000000000008080h
		/*0076*/ 0xB0, 0x0E,// mov al,0Eh
		/*0078*/ 0x0F, 0x8F, 0x02, 0x00, 0x00, 0x00,// jg near ptr 0000000000008080h
		/*007E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0080*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x70, 0x3E,// jo short 8000000000000040h
		/*0002*/ 0xB0, 0x00,// mov al,0
		/*0004*/ 0x71, 0x3A,// jno short 8000000000000040h
		/*0006*/ 0xB0, 0x01,// mov al,1
		/*0008*/ 0x72, 0x36,// jb short 8000000000000040h
		/*000A*/ 0xB0, 0x02,// mov al,2
		/*000C*/ 0x73, 0x32,// jae short 8000000000000040h
		/*000E*/ 0xB0, 0x03,// mov al,3
		/*0010*/ 0x74, 0x2E,// je short 8000000000000040h
		/*0012*/ 0xB0, 0x04,// mov al,4
		/*0014*/ 0x75, 0x2A,// jne short 8000000000000040h
		/*0016*/ 0xB0, 0x05,// mov al,5
		/*0018*/ 0x76, 0x26,// jbe short 8000000000000040h
		/*001A*/ 0xB0, 0x06,// mov al,6
		/*001C*/ 0x77, 0x22,// ja short 8000000000000040h
		/*001E*/ 0xB0, 0x07,// mov al,7
		/*0020*/ 0x78, 0x1E,// js short 8000000000000040h
		/*0022*/ 0xB0, 0x08,// mov al,8
		/*0024*/ 0x79, 0x1A,// jns short 8000000000000040h
		/*0026*/ 0xB0, 0x09,// mov al,9
		/*0028*/ 0x7A, 0x16,// jp short 8000000000000040h
		/*002A*/ 0xB0, 0x0A,// mov al,0Ah
		/*002C*/ 0x7B, 0x12,// jnp short 8000000000000040h
		/*002E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0030*/ 0x7C, 0x0E,// jl short 8000000000000040h
		/*0032*/ 0xB0, 0x0C,// mov al,0Ch
		/*0034*/ 0x7D, 0x0A,// jge short 8000000000000040h
		/*0036*/ 0xB0, 0x0D,// mov al,0Dh
		/*0038*/ 0x7E, 0x06,// jle short 8000000000000040h
		/*003A*/ 0xB0, 0x0E,// mov al,0Eh
		/*003C*/ 0x7F, 0x02,// jg short 8000000000000040h
		/*003E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0040*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0002,
		0x0004,
		0x0006,
		0x0008,
		0x000A,
		0x000C,
		0x000E,
		0x0010,
		0x0012,
		0x0014,
		0x0016,
		0x0018,
		0x001A,
		0x001C,
		0x001E,
		0x0020,
		0x0022,
		0x0024,
		0x0026,
		0x0028,
		0x002A,
		0x002C,
		0x002E,
		0x0030,
		0x0032,
		0x0034,
		0x0036,
		0x0038,
		0x003A,
		0x003C,
		0x003E,
		0x0040,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_near_fwd_near() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x0F, 0x80, 0x14, 0x01, 0x00, 0x00,// jo near ptr 000000000000811Ah
		/*0006*/ 0xB0, 0x00,// mov al,0
		/*0008*/ 0x0F, 0x81, 0x0C, 0x01, 0x00, 0x00,// jno near ptr 000000000000811Ah
		/*000E*/ 0xB0, 0x01,// mov al,1
		/*0010*/ 0x0F, 0x82, 0x04, 0x01, 0x00, 0x00,// jb near ptr 000000000000811Ah
		/*0016*/ 0xB0, 0x02,// mov al,2
		/*0018*/ 0x0F, 0x83, 0xFC, 0x00, 0x00, 0x00,// jae near ptr 000000000000811Ah
		/*001E*/ 0xB0, 0x03,// mov al,3
		/*0020*/ 0x0F, 0x84, 0xF4, 0x00, 0x00, 0x00,// je near ptr 000000000000811Ah
		/*0026*/ 0xB0, 0x04,// mov al,4
		/*0028*/ 0x0F, 0x85, 0xEC, 0x00, 0x00, 0x00,// jne near ptr 000000000000811Ah
		/*002E*/ 0xB0, 0x05,// mov al,5
		/*0030*/ 0x0F, 0x86, 0xE4, 0x00, 0x00, 0x00,// jbe near ptr 000000000000811Ah
		/*0036*/ 0xB0, 0x06,// mov al,6
		/*0038*/ 0x0F, 0x87, 0xDC, 0x00, 0x00, 0x00,// ja near ptr 000000000000811Ah
		/*003E*/ 0xB0, 0x07,// mov al,7
		/*0040*/ 0x0F, 0x88, 0xD4, 0x00, 0x00, 0x00,// js near ptr 000000000000811Ah
		/*0046*/ 0xB0, 0x08,// mov al,8
		/*0048*/ 0x0F, 0x89, 0xCC, 0x00, 0x00, 0x00,// jns near ptr 000000000000811Ah
		/*004E*/ 0xB0, 0x09,// mov al,9
		/*0050*/ 0x0F, 0x8A, 0xC4, 0x00, 0x00, 0x00,// jp near ptr 000000000000811Ah
		/*0056*/ 0xB0, 0x0A,// mov al,0Ah
		/*0058*/ 0x0F, 0x8B, 0xBC, 0x00, 0x00, 0x00,// jnp near ptr 000000000000811Ah
		/*005E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0060*/ 0x0F, 0x8C, 0xB4, 0x00, 0x00, 0x00,// jl near ptr 000000000000811Ah
		/*0066*/ 0xB0, 0x0C,// mov al,0Ch
		/*0068*/ 0x0F, 0x8D, 0xAC, 0x00, 0x00, 0x00,// jge near ptr 000000000000811Ah
		/*006E*/ 0xB0, 0x0D,// mov al,0Dh
		/*0070*/ 0x0F, 0x8E, 0xA4, 0x00, 0x00, 0x00,// jle near ptr 000000000000811Ah
		/*0076*/ 0xB0, 0x0E,// mov al,0Eh
		/*0078*/ 0x0F, 0x8F, 0x9C, 0x00, 0x00, 0x00,// jg near ptr 000000000000811Ah
		/*007E*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*008B*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0098*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00A5*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00B2*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00BF*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00CC*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00D9*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00E6*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00F3*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0100*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*010D*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*011A*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x0F, 0x80, 0x14, 0x01, 0x00, 0x00,// jo near ptr 800000000000011Ah
		/*0006*/ 0xB0, 0x00,// mov al,0
		/*0008*/ 0x0F, 0x81, 0x0C, 0x01, 0x00, 0x00,// jno near ptr 800000000000011Ah
		/*000E*/ 0xB0, 0x01,// mov al,1
		/*0010*/ 0x0F, 0x82, 0x04, 0x01, 0x00, 0x00,// jb near ptr 800000000000011Ah
		/*0016*/ 0xB0, 0x02,// mov al,2
		/*0018*/ 0x0F, 0x83, 0xFC, 0x00, 0x00, 0x00,// jae near ptr 800000000000011Ah
		/*001E*/ 0xB0, 0x03,// mov al,3
		/*0020*/ 0x0F, 0x84, 0xF4, 0x00, 0x00, 0x00,// je near ptr 800000000000011Ah
		/*0026*/ 0xB0, 0x04,// mov al,4
		/*0028*/ 0x0F, 0x85, 0xEC, 0x00, 0x00, 0x00,// jne near ptr 800000000000011Ah
		/*002E*/ 0xB0, 0x05,// mov al,5
		/*0030*/ 0x0F, 0x86, 0xE4, 0x00, 0x00, 0x00,// jbe near ptr 800000000000011Ah
		/*0036*/ 0xB0, 0x06,// mov al,6
		/*0038*/ 0x0F, 0x87, 0xDC, 0x00, 0x00, 0x00,// ja near ptr 800000000000011Ah
		/*003E*/ 0xB0, 0x07,// mov al,7
		/*0040*/ 0x0F, 0x88, 0xD4, 0x00, 0x00, 0x00,// js near ptr 800000000000011Ah
		/*0046*/ 0xB0, 0x08,// mov al,8
		/*0048*/ 0x0F, 0x89, 0xCC, 0x00, 0x00, 0x00,// jns near ptr 800000000000011Ah
		/*004E*/ 0xB0, 0x09,// mov al,9
		/*0050*/ 0x0F, 0x8A, 0xC4, 0x00, 0x00, 0x00,// jp near ptr 800000000000011Ah
		/*0056*/ 0xB0, 0x0A,// mov al,0Ah
		/*0058*/ 0x0F, 0x8B, 0xBC, 0x00, 0x00, 0x00,// jnp near ptr 800000000000011Ah
		/*005E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0060*/ 0x0F, 0x8C, 0xB4, 0x00, 0x00, 0x00,// jl near ptr 800000000000011Ah
		/*0066*/ 0xB0, 0x0C,// mov al,0Ch
		/*0068*/ 0x0F, 0x8D, 0xAC, 0x00, 0x00, 0x00,// jge near ptr 800000000000011Ah
		/*006E*/ 0xB0, 0x0D,// mov al,0Dh
		/*0070*/ 0x0F, 0x8E, 0xA4, 0x00, 0x00, 0x00,// jle near ptr 800000000000011Ah
		/*0076*/ 0xB0, 0x0E,// mov al,0Eh
		/*0078*/ 0x0F, 0x8F, 0x9C, 0x00, 0x00, 0x00,// jg near ptr 800000000000011Ah
		/*007E*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*008B*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0098*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00A5*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00B2*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00BF*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00CC*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00D9*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00E6*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*00F3*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0100*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*010D*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*011A*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0006,
		0x0008,
		0x000E,
		0x0010,
		0x0016,
		0x0018,
		0x001E,
		0x0020,
		0x0026,
		0x0028,
		0x002E,
		0x0030,
		0x0036,
		0x0038,
		0x003E,
		0x0040,
		0x0046,
		0x0048,
		0x004E,
		0x0050,
		0x0056,
		0x0058,
		0x005E,
		0x0060,
		0x0066,
		0x0068,
		0x006E,
		0x0070,
		0x0076,
		0x0078,
		0x007E,
		0x008B,
		0x0098,
		0x00A5,
		0x00B2,
		0x00BF,
		0x00CC,
		0x00D9,
		0x00E6,
		0x00F3,
		0x0100,
		0x010D,
		0x011A,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_near_bwd_short() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x0F, 0x80, 0xFA, 0xFF, 0xFF, 0xFF,// jo near ptr 0000000000008000h
		/*0006*/ 0xB0, 0x00,// mov al,0
		/*0008*/ 0x0F, 0x81, 0xF2, 0xFF, 0xFF, 0xFF,// jno near ptr 0000000000008000h
		/*000E*/ 0xB0, 0x01,// mov al,1
		/*0010*/ 0x0F, 0x82, 0xEA, 0xFF, 0xFF, 0xFF,// jb near ptr 0000000000008000h
		/*0016*/ 0xB0, 0x02,// mov al,2
		/*0018*/ 0x0F, 0x83, 0xE2, 0xFF, 0xFF, 0xFF,// jae near ptr 0000000000008000h
		/*001E*/ 0xB0, 0x03,// mov al,3
		/*0020*/ 0x0F, 0x84, 0xDA, 0xFF, 0xFF, 0xFF,// je near ptr 0000000000008000h
		/*0026*/ 0xB0, 0x04,// mov al,4
		/*0028*/ 0x0F, 0x85, 0xD2, 0xFF, 0xFF, 0xFF,// jne near ptr 0000000000008000h
		/*002E*/ 0xB0, 0x05,// mov al,5
		/*0030*/ 0x0F, 0x86, 0xCA, 0xFF, 0xFF, 0xFF,// jbe near ptr 0000000000008000h
		/*0036*/ 0xB0, 0x06,// mov al,6
		/*0038*/ 0x0F, 0x87, 0xC2, 0xFF, 0xFF, 0xFF,// ja near ptr 0000000000008000h
		/*003E*/ 0xB0, 0x07,// mov al,7
		/*0040*/ 0x0F, 0x88, 0xBA, 0xFF, 0xFF, 0xFF,// js near ptr 0000000000008000h
		/*0046*/ 0xB0, 0x08,// mov al,8
		/*0048*/ 0x0F, 0x89, 0xB2, 0xFF, 0xFF, 0xFF,// jns near ptr 0000000000008000h
		/*004E*/ 0xB0, 0x09,// mov al,9
		/*0050*/ 0x0F, 0x8A, 0xAA, 0xFF, 0xFF, 0xFF,// jp near ptr 0000000000008000h
		/*0056*/ 0xB0, 0x0A,// mov al,0Ah
		/*0058*/ 0x0F, 0x8B, 0xA2, 0xFF, 0xFF, 0xFF,// jnp near ptr 0000000000008000h
		/*005E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0060*/ 0x0F, 0x8C, 0x9A, 0xFF, 0xFF, 0xFF,// jl near ptr 0000000000008000h
		/*0066*/ 0xB0, 0x0C,// mov al,0Ch
		/*0068*/ 0x0F, 0x8D, 0x92, 0xFF, 0xFF, 0xFF,// jge near ptr 0000000000008000h
		/*006E*/ 0xB0, 0x0D,// mov al,0Dh
		/*0070*/ 0x0F, 0x8E, 0x8A, 0xFF, 0xFF, 0xFF,// jle near ptr 0000000000008000h
		/*0076*/ 0xB0, 0x0E,// mov al,0Eh
		/*0078*/ 0x0F, 0x8F, 0x82, 0xFF, 0xFF, 0xFF,// jg near ptr 0000000000008000h
		/*007E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x70, 0xFE,// jo short 8000000000000000h
		/*0002*/ 0xB0, 0x00,// mov al,0
		/*0004*/ 0x71, 0xFA,// jno short 8000000000000000h
		/*0006*/ 0xB0, 0x01,// mov al,1
		/*0008*/ 0x72, 0xF6,// jb short 8000000000000000h
		/*000A*/ 0xB0, 0x02,// mov al,2
		/*000C*/ 0x73, 0xF2,// jae short 8000000000000000h
		/*000E*/ 0xB0, 0x03,// mov al,3
		/*0010*/ 0x74, 0xEE,// je short 8000000000000000h
		/*0012*/ 0xB0, 0x04,// mov al,4
		/*0014*/ 0x75, 0xEA,// jne short 8000000000000000h
		/*0016*/ 0xB0, 0x05,// mov al,5
		/*0018*/ 0x76, 0xE6,// jbe short 8000000000000000h
		/*001A*/ 0xB0, 0x06,// mov al,6
		/*001C*/ 0x77, 0xE2,// ja short 8000000000000000h
		/*001E*/ 0xB0, 0x07,// mov al,7
		/*0020*/ 0x78, 0xDE,// js short 8000000000000000h
		/*0022*/ 0xB0, 0x08,// mov al,8
		/*0024*/ 0x79, 0xDA,// jns short 8000000000000000h
		/*0026*/ 0xB0, 0x09,// mov al,9
		/*0028*/ 0x7A, 0xD6,// jp short 8000000000000000h
		/*002A*/ 0xB0, 0x0A,// mov al,0Ah
		/*002C*/ 0x7B, 0xD2,// jnp short 8000000000000000h
		/*002E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0030*/ 0x7C, 0xCE,// jl short 8000000000000000h
		/*0032*/ 0xB0, 0x0C,// mov al,0Ch
		/*0034*/ 0x7D, 0xCA,// jge short 8000000000000000h
		/*0036*/ 0xB0, 0x0D,// mov al,0Dh
		/*0038*/ 0x7E, 0xC6,// jle short 8000000000000000h
		/*003A*/ 0xB0, 0x0E,// mov al,0Eh
		/*003C*/ 0x7F, 0xC2,// jg short 8000000000000000h
		/*003E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0002,
		0x0004,
		0x0006,
		0x0008,
		0x000A,
		0x000C,
		0x000E,
		0x0010,
		0x0012,
		0x0014,
		0x0016,
		0x0018,
		0x001A,
		0x001C,
		0x001E,
		0x0020,
		0x0022,
		0x0024,
		0x0026,
		0x0028,
		0x002A,
		0x002C,
		0x002E,
		0x0030,
		0x0032,
		0x0034,
		0x0036,
		0x0038,
		0x003A,
		0x003C,
		0x003E,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_near_bwd_near() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*000D*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*001A*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0027*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0034*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0041*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*004E*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*005B*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0068*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0075*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0082*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*008F*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*009C*/ 0x0F, 0x80, 0x5E, 0xFF, 0xFF, 0xFF,// jo near ptr 0000000000008000h
		/*00A2*/ 0xB0, 0x00,// mov al,0
		/*00A4*/ 0x0F, 0x81, 0x56, 0xFF, 0xFF, 0xFF,// jno near ptr 0000000000008000h
		/*00AA*/ 0xB0, 0x01,// mov al,1
		/*00AC*/ 0x0F, 0x82, 0x4E, 0xFF, 0xFF, 0xFF,// jb near ptr 0000000000008000h
		/*00B2*/ 0xB0, 0x02,// mov al,2
		/*00B4*/ 0x0F, 0x83, 0x46, 0xFF, 0xFF, 0xFF,// jae near ptr 0000000000008000h
		/*00BA*/ 0xB0, 0x03,// mov al,3
		/*00BC*/ 0x0F, 0x84, 0x3E, 0xFF, 0xFF, 0xFF,// je near ptr 0000000000008000h
		/*00C2*/ 0xB0, 0x04,// mov al,4
		/*00C4*/ 0x0F, 0x85, 0x36, 0xFF, 0xFF, 0xFF,// jne near ptr 0000000000008000h
		/*00CA*/ 0xB0, 0x05,// mov al,5
		/*00CC*/ 0x0F, 0x86, 0x2E, 0xFF, 0xFF, 0xFF,// jbe near ptr 0000000000008000h
		/*00D2*/ 0xB0, 0x06,// mov al,6
		/*00D4*/ 0x0F, 0x87, 0x26, 0xFF, 0xFF, 0xFF,// ja near ptr 0000000000008000h
		/*00DA*/ 0xB0, 0x07,// mov al,7
		/*00DC*/ 0x0F, 0x88, 0x1E, 0xFF, 0xFF, 0xFF,// js near ptr 0000000000008000h
		/*00E2*/ 0xB0, 0x08,// mov al,8
		/*00E4*/ 0x0F, 0x89, 0x16, 0xFF, 0xFF, 0xFF,// jns near ptr 0000000000008000h
		/*00EA*/ 0xB0, 0x09,// mov al,9
		/*00EC*/ 0x0F, 0x8A, 0x0E, 0xFF, 0xFF, 0xFF,// jp near ptr 0000000000008000h
		/*00F2*/ 0xB0, 0x0A,// mov al,0Ah
		/*00F4*/ 0x0F, 0x8B, 0x06, 0xFF, 0xFF, 0xFF,// jnp near ptr 0000000000008000h
		/*00FA*/ 0xB0, 0x0B,// mov al,0Bh
		/*00FC*/ 0x0F, 0x8C, 0xFE, 0xFE, 0xFF, 0xFF,// jl near ptr 0000000000008000h
		/*0102*/ 0xB0, 0x0C,// mov al,0Ch
		/*0104*/ 0x0F, 0x8D, 0xF6, 0xFE, 0xFF, 0xFF,// jge near ptr 0000000000008000h
		/*010A*/ 0xB0, 0x0D,// mov al,0Dh
		/*010C*/ 0x0F, 0x8E, 0xEE, 0xFE, 0xFF, 0xFF,// jle near ptr 0000000000008000h
		/*0112*/ 0xB0, 0x0E,// mov al,0Eh
		/*0114*/ 0x0F, 0x8F, 0xE6, 0xFE, 0xFF, 0xFF,// jg near ptr 0000000000008000h
		/*011A*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*000D*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*001A*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0027*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0034*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0041*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*004E*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*005B*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0068*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0075*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*0082*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*008F*/ 0x64, 0x67, 0xC7, 0x84, 0x0B, 0x78, 0x56, 0x34, 0x12, 0x78, 0x56, 0x34, 0x12,// mov dword ptr fs:[ebx+ecx+12345678h],12345678h
		/*009C*/ 0x0F, 0x80, 0x5E, 0xFF, 0xFF, 0xFF,// jo near ptr 8000000000000000h
		/*00A2*/ 0xB0, 0x00,// mov al,0
		/*00A4*/ 0x0F, 0x81, 0x56, 0xFF, 0xFF, 0xFF,// jno near ptr 8000000000000000h
		/*00AA*/ 0xB0, 0x01,// mov al,1
		/*00AC*/ 0x0F, 0x82, 0x4E, 0xFF, 0xFF, 0xFF,// jb near ptr 8000000000000000h
		/*00B2*/ 0xB0, 0x02,// mov al,2
		/*00B4*/ 0x0F, 0x83, 0x46, 0xFF, 0xFF, 0xFF,// jae near ptr 8000000000000000h
		/*00BA*/ 0xB0, 0x03,// mov al,3
		/*00BC*/ 0x0F, 0x84, 0x3E, 0xFF, 0xFF, 0xFF,// je near ptr 8000000000000000h
		/*00C2*/ 0xB0, 0x04,// mov al,4
		/*00C4*/ 0x0F, 0x85, 0x36, 0xFF, 0xFF, 0xFF,// jne near ptr 8000000000000000h
		/*00CA*/ 0xB0, 0x05,// mov al,5
		/*00CC*/ 0x0F, 0x86, 0x2E, 0xFF, 0xFF, 0xFF,// jbe near ptr 8000000000000000h
		/*00D2*/ 0xB0, 0x06,// mov al,6
		/*00D4*/ 0x0F, 0x87, 0x26, 0xFF, 0xFF, 0xFF,// ja near ptr 8000000000000000h
		/*00DA*/ 0xB0, 0x07,// mov al,7
		/*00DC*/ 0x0F, 0x88, 0x1E, 0xFF, 0xFF, 0xFF,// js near ptr 8000000000000000h
		/*00E2*/ 0xB0, 0x08,// mov al,8
		/*00E4*/ 0x0F, 0x89, 0x16, 0xFF, 0xFF, 0xFF,// jns near ptr 8000000000000000h
		/*00EA*/ 0xB0, 0x09,// mov al,9
		/*00EC*/ 0x0F, 0x8A, 0x0E, 0xFF, 0xFF, 0xFF,// jp near ptr 8000000000000000h
		/*00F2*/ 0xB0, 0x0A,// mov al,0Ah
		/*00F4*/ 0x0F, 0x8B, 0x06, 0xFF, 0xFF, 0xFF,// jnp near ptr 8000000000000000h
		/*00FA*/ 0xB0, 0x0B,// mov al,0Bh
		/*00FC*/ 0x0F, 0x8C, 0xFE, 0xFE, 0xFF, 0xFF,// jl near ptr 8000000000000000h
		/*0102*/ 0xB0, 0x0C,// mov al,0Ch
		/*0104*/ 0x0F, 0x8D, 0xF6, 0xFE, 0xFF, 0xFF,// jge near ptr 8000000000000000h
		/*010A*/ 0xB0, 0x0D,// mov al,0Dh
		/*010C*/ 0x0F, 0x8E, 0xEE, 0xFE, 0xFF, 0xFF,// jle near ptr 8000000000000000h
		/*0112*/ 0xB0, 0x0E,// mov al,0Eh
		/*0114*/ 0x0F, 0x8F, 0xE6, 0xFE, 0xFF, 0xFF,// jg near ptr 8000000000000000h
		/*011A*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x000D,
		0x001A,
		0x0027,
		0x0034,
		0x0041,
		0x004E,
		0x005B,
		0x0068,
		0x0075,
		0x0082,
		0x008F,
		0x009C,
		0x00A2,
		0x00A4,
		0x00AA,
		0x00AC,
		0x00B2,
		0x00B4,
		0x00BA,
		0x00BC,
		0x00C2,
		0x00C4,
		0x00CA,
		0x00CC,
		0x00D2,
		0x00D4,
		0x00DA,
		0x00DC,
		0x00E2,
		0x00E4,
		0x00EA,
		0x00EC,
		0x00F2,
		0x00F4,
		0x00FA,
		0x00FC,
		0x0102,
		0x0104,
		0x010A,
		0x010C,
		0x0112,
		0x0114,
		0x011A,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_near_other_short() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x0F, 0x80, 0x7A, 0x00, 0x00, 0x00,// jo near ptr 0000000000008080h
		/*0006*/ 0xB0, 0x00,// mov al,0
		/*0008*/ 0x0F, 0x81, 0x73, 0x00, 0x00, 0x00,// jno near ptr 0000000000008081h
		/*000E*/ 0xB0, 0x01,// mov al,1
		/*0010*/ 0x0F, 0x82, 0x6C, 0x00, 0x00, 0x00,// jb near ptr 0000000000008082h
		/*0016*/ 0xB0, 0x02,// mov al,2
		/*0018*/ 0x0F, 0x83, 0x65, 0x00, 0x00, 0x00,// jae near ptr 0000000000008083h
		/*001E*/ 0xB0, 0x03,// mov al,3
		/*0020*/ 0x0F, 0x84, 0x5E, 0x00, 0x00, 0x00,// je near ptr 0000000000008084h
		/*0026*/ 0xB0, 0x04,// mov al,4
		/*0028*/ 0x0F, 0x85, 0x57, 0x00, 0x00, 0x00,// jne near ptr 0000000000008085h
		/*002E*/ 0xB0, 0x05,// mov al,5
		/*0030*/ 0x0F, 0x86, 0x50, 0x00, 0x00, 0x00,// jbe near ptr 0000000000008086h
		/*0036*/ 0xB0, 0x06,// mov al,6
		/*0038*/ 0x0F, 0x87, 0x49, 0x00, 0x00, 0x00,// ja near ptr 0000000000008087h
		/*003E*/ 0xB0, 0x07,// mov al,7
		/*0040*/ 0x0F, 0x88, 0x42, 0x00, 0x00, 0x00,// js near ptr 0000000000008088h
		/*0046*/ 0xB0, 0x08,// mov al,8
		/*0048*/ 0x0F, 0x89, 0x3B, 0x00, 0x00, 0x00,// jns near ptr 0000000000008089h
		/*004E*/ 0xB0, 0x09,// mov al,9
		/*0050*/ 0x0F, 0x8A, 0x34, 0x00, 0x00, 0x00,// jp near ptr 000000000000808Ah
		/*0056*/ 0xB0, 0x0A,// mov al,0Ah
		/*0058*/ 0x0F, 0x8B, 0x2D, 0x00, 0x00, 0x00,// jnp near ptr 000000000000808Bh
		/*005E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0060*/ 0x0F, 0x8C, 0x26, 0x00, 0x00, 0x00,// jl near ptr 000000000000808Ch
		/*0066*/ 0xB0, 0x0C,// mov al,0Ch
		/*0068*/ 0x0F, 0x8D, 0x1F, 0x00, 0x00, 0x00,// jge near ptr 000000000000808Dh
		/*006E*/ 0xB0, 0x0D,// mov al,0Dh
		/*0070*/ 0x0F, 0x8E, 0x18, 0x00, 0x00, 0x00,// jle near ptr 000000000000808Eh
		/*0076*/ 0xB0, 0x0E,// mov al,0Eh
		/*0078*/ 0x0F, 0x8F, 0x11, 0x00, 0x00, 0x00,// jg near ptr 000000000000808Fh
		/*007E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x70, 0x7F,// jo short 0000000000008080h
		/*0002*/ 0xB0, 0x00,// mov al,0
		/*0004*/ 0x71, 0x7C,// jno short 0000000000008081h
		/*0006*/ 0xB0, 0x01,// mov al,1
		/*0008*/ 0x72, 0x79,// jb short 0000000000008082h
		/*000A*/ 0xB0, 0x02,// mov al,2
		/*000C*/ 0x73, 0x76,// jae short 0000000000008083h
		/*000E*/ 0xB0, 0x03,// mov al,3
		/*0010*/ 0x74, 0x73,// je short 0000000000008084h
		/*0012*/ 0xB0, 0x04,// mov al,4
		/*0014*/ 0x75, 0x70,// jne short 0000000000008085h
		/*0016*/ 0xB0, 0x05,// mov al,5
		/*0018*/ 0x76, 0x6D,// jbe short 0000000000008086h
		/*001A*/ 0xB0, 0x06,// mov al,6
		/*001C*/ 0x77, 0x6A,// ja short 0000000000008087h
		/*001E*/ 0xB0, 0x07,// mov al,7
		/*0020*/ 0x78, 0x67,// js short 0000000000008088h
		/*0022*/ 0xB0, 0x08,// mov al,8
		/*0024*/ 0x79, 0x64,// jns short 0000000000008089h
		/*0026*/ 0xB0, 0x09,// mov al,9
		/*0028*/ 0x7A, 0x61,// jp short 000000000000808Ah
		/*002A*/ 0xB0, 0x0A,// mov al,0Ah
		/*002C*/ 0x7B, 0x5E,// jnp short 000000000000808Bh
		/*002E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0030*/ 0x7C, 0x5B,// jl short 000000000000808Ch
		/*0032*/ 0xB0, 0x0C,// mov al,0Ch
		/*0034*/ 0x7D, 0x58,// jge short 000000000000808Dh
		/*0036*/ 0xB0, 0x0D,// mov al,0Dh
		/*0038*/ 0x7E, 0x55,// jle short 000000000000808Eh
		/*003A*/ 0xB0, 0x0E,// mov al,0Eh
		/*003C*/ 0x7F, 0x52,// jg short 000000000000808Fh
		/*003E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0002,
		0x0004,
		0x0006,
		0x0008,
		0x000A,
		0x000C,
		0x000E,
		0x0010,
		0x0012,
		0x0014,
		0x0016,
		0x0018,
		0x001A,
		0x001C,
		0x001E,
		0x0020,
		0x0022,
		0x0024,
		0x0026,
		0x0028,
		0x002A,
		0x002C,
		0x002E,
		0x0030,
		0x0032,
		0x0034,
		0x0036,
		0x0038,
		0x003A,
		0x003C,
		0x003E,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		ORIG_RIP - 1,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_near_other_near() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x0F, 0x80, 0x7A, 0x00, 0x00, 0x00,// jo near ptr 0000000000008080h
		/*0006*/ 0xB0, 0x00,// mov al,0
		/*0008*/ 0x0F, 0x81, 0x73, 0x00, 0x00, 0x00,// jno near ptr 0000000000008081h
		/*000E*/ 0xB0, 0x01,// mov al,1
		/*0010*/ 0x0F, 0x82, 0x6C, 0x00, 0x00, 0x00,// jb near ptr 0000000000008082h
		/*0016*/ 0xB0, 0x02,// mov al,2
		/*0018*/ 0x0F, 0x83, 0x65, 0x00, 0x00, 0x00,// jae near ptr 0000000000008083h
		/*001E*/ 0xB0, 0x03,// mov al,3
		/*0020*/ 0x0F, 0x84, 0x5E, 0x00, 0x00, 0x00,// je near ptr 0000000000008084h
		/*0026*/ 0xB0, 0x04,// mov al,4
		/*0028*/ 0x0F, 0x85, 0x57, 0x00, 0x00, 0x00,// jne near ptr 0000000000008085h
		/*002E*/ 0xB0, 0x05,// mov al,5
		/*0030*/ 0x0F, 0x86, 0x50, 0x00, 0x00, 0x00,// jbe near ptr 0000000000008086h
		/*0036*/ 0xB0, 0x06,// mov al,6
		/*0038*/ 0x0F, 0x87, 0x49, 0x00, 0x00, 0x00,// ja near ptr 0000000000008087h
		/*003E*/ 0xB0, 0x07,// mov al,7
		/*0040*/ 0x0F, 0x88, 0x42, 0x00, 0x00, 0x00,// js near ptr 0000000000008088h
		/*0046*/ 0xB0, 0x08,// mov al,8
		/*0048*/ 0x0F, 0x89, 0x3B, 0x00, 0x00, 0x00,// jns near ptr 0000000000008089h
		/*004E*/ 0xB0, 0x09,// mov al,9
		/*0050*/ 0x0F, 0x8A, 0x34, 0x00, 0x00, 0x00,// jp near ptr 000000000000808Ah
		/*0056*/ 0xB0, 0x0A,// mov al,0Ah
		/*0058*/ 0x0F, 0x8B, 0x2D, 0x00, 0x00, 0x00,// jnp near ptr 000000000000808Bh
		/*005E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0060*/ 0x0F, 0x8C, 0x26, 0x00, 0x00, 0x00,// jl near ptr 000000000000808Ch
		/*0066*/ 0xB0, 0x0C,// mov al,0Ch
		/*0068*/ 0x0F, 0x8D, 0x1F, 0x00, 0x00, 0x00,// jge near ptr 000000000000808Dh
		/*006E*/ 0xB0, 0x0D,// mov al,0Dh
		/*0070*/ 0x0F, 0x8E, 0x18, 0x00, 0x00, 0x00,// jle near ptr 000000000000808Eh
		/*0076*/ 0xB0, 0x0E,// mov al,0Eh
		/*0078*/ 0x0F, 0x8F, 0x11, 0x00, 0x00, 0x00,// jg near ptr 000000000000808Fh
		/*007E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x0F, 0x80, 0x7A, 0xF0, 0xFF, 0xFF,// jo near ptr 0000000000008080h
		/*0006*/ 0xB0, 0x00,// mov al,0
		/*0008*/ 0x0F, 0x81, 0x73, 0xF0, 0xFF, 0xFF,// jno near ptr 0000000000008081h
		/*000E*/ 0xB0, 0x01,// mov al,1
		/*0010*/ 0x0F, 0x82, 0x6C, 0xF0, 0xFF, 0xFF,// jb near ptr 0000000000008082h
		/*0016*/ 0xB0, 0x02,// mov al,2
		/*0018*/ 0x0F, 0x83, 0x65, 0xF0, 0xFF, 0xFF,// jae near ptr 0000000000008083h
		/*001E*/ 0xB0, 0x03,// mov al,3
		/*0020*/ 0x0F, 0x84, 0x5E, 0xF0, 0xFF, 0xFF,// je near ptr 0000000000008084h
		/*0026*/ 0xB0, 0x04,// mov al,4
		/*0028*/ 0x0F, 0x85, 0x57, 0xF0, 0xFF, 0xFF,// jne near ptr 0000000000008085h
		/*002E*/ 0xB0, 0x05,// mov al,5
		/*0030*/ 0x0F, 0x86, 0x50, 0xF0, 0xFF, 0xFF,// jbe near ptr 0000000000008086h
		/*0036*/ 0xB0, 0x06,// mov al,6
		/*0038*/ 0x0F, 0x87, 0x49, 0xF0, 0xFF, 0xFF,// ja near ptr 0000000000008087h
		/*003E*/ 0xB0, 0x07,// mov al,7
		/*0040*/ 0x0F, 0x88, 0x42, 0xF0, 0xFF, 0xFF,// js near ptr 0000000000008088h
		/*0046*/ 0xB0, 0x08,// mov al,8
		/*0048*/ 0x0F, 0x89, 0x3B, 0xF0, 0xFF, 0xFF,// jns near ptr 0000000000008089h
		/*004E*/ 0xB0, 0x09,// mov al,9
		/*0050*/ 0x0F, 0x8A, 0x34, 0xF0, 0xFF, 0xFF,// jp near ptr 000000000000808Ah
		/*0056*/ 0xB0, 0x0A,// mov al,0Ah
		/*0058*/ 0x0F, 0x8B, 0x2D, 0xF0, 0xFF, 0xFF,// jnp near ptr 000000000000808Bh
		/*005E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0060*/ 0x0F, 0x8C, 0x26, 0xF0, 0xFF, 0xFF,// jl near ptr 000000000000808Ch
		/*0066*/ 0xB0, 0x0C,// mov al,0Ch
		/*0068*/ 0x0F, 0x8D, 0x1F, 0xF0, 0xFF, 0xFF,// jge near ptr 000000000000808Dh
		/*006E*/ 0xB0, 0x0D,// mov al,0Dh
		/*0070*/ 0x0F, 0x8E, 0x18, 0xF0, 0xFF, 0xFF,// jle near ptr 000000000000808Eh
		/*0076*/ 0xB0, 0x0E,// mov al,0Eh
		/*0078*/ 0x0F, 0x8F, 0x11, 0xF0, 0xFF, 0xFF,// jg near ptr 000000000000808Fh
		/*007E*/ 0xB0, 0x0F,// mov al,0Fh
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0006,
		0x0008,
		0x000E,
		0x0010,
		0x0016,
		0x0018,
		0x001E,
		0x0020,
		0x0026,
		0x0028,
		0x002E,
		0x0030,
		0x0036,
		0x0038,
		0x003E,
		0x0040,
		0x0046,
		0x0048,
		0x004E,
		0x0050,
		0x0056,
		0x0058,
		0x005E,
		0x0060,
		0x0066,
		0x0068,
		0x006E,
		0x0070,
		0x0076,
		0x0078,
		0x007E,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		ORIG_RIP + 0x1000,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_near_other_long() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x0F, 0x80, 0x7B, 0x00, 0x00, 0x00,// jo near ptr 123456789ABCDE81h
		/*0006*/ 0xB0, 0x00,// mov al,0
		/*0008*/ 0x0F, 0x81, 0x74, 0x00, 0x00, 0x00,// jno near ptr 123456789ABCDE82h
		/*000E*/ 0xB0, 0x01,// mov al,1
		/*0010*/ 0x0F, 0x82, 0x6D, 0x00, 0x00, 0x00,// jb near ptr 123456789ABCDE83h
		/*0016*/ 0xB0, 0x02,// mov al,2
		/*0018*/ 0x0F, 0x83, 0x66, 0x00, 0x00, 0x00,// jae near ptr 123456789ABCDE84h
		/*001E*/ 0xB0, 0x03,// mov al,3
		/*0020*/ 0x0F, 0x84, 0x5F, 0x00, 0x00, 0x00,// je near ptr 123456789ABCDE85h
		/*0026*/ 0xB0, 0x04,// mov al,4
		/*0028*/ 0x0F, 0x85, 0x58, 0x00, 0x00, 0x00,// jne near ptr 123456789ABCDE86h
		/*002E*/ 0xB0, 0x05,// mov al,5
		/*0030*/ 0x0F, 0x86, 0x51, 0x00, 0x00, 0x00,// jbe near ptr 123456789ABCDE87h
		/*0036*/ 0xB0, 0x06,// mov al,6
		/*0038*/ 0x0F, 0x87, 0x4A, 0x00, 0x00, 0x00,// ja near ptr 123456789ABCDE88h
		/*003E*/ 0xB0, 0x07,// mov al,7
		/*0040*/ 0x0F, 0x88, 0x43, 0x00, 0x00, 0x00,// js near ptr 123456789ABCDE89h
		/*0046*/ 0xB0, 0x08,// mov al,8
		/*0048*/ 0x0F, 0x89, 0x3C, 0x00, 0x00, 0x00,// jns near ptr 123456789ABCDE8Ah
		/*004E*/ 0xB0, 0x09,// mov al,9
		/*0050*/ 0x0F, 0x8A, 0x35, 0x00, 0x00, 0x00,// jp near ptr 123456789ABCDE8Bh
		/*0056*/ 0xB0, 0x0A,// mov al,0Ah
		/*0058*/ 0x0F, 0x8B, 0x2E, 0x00, 0x00, 0x00,// jnp near ptr 123456789ABCDE8Ch
		/*005E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0060*/ 0x0F, 0x8C, 0x27, 0x00, 0x00, 0x00,// jl near ptr 123456789ABCDE8Dh
		/*0066*/ 0xB0, 0x0C,// mov al,0Ch
		/*0068*/ 0x0F, 0x8D, 0x20, 0x00, 0x00, 0x00,// jge near ptr 123456789ABCDE8Eh
		/*006E*/ 0xB0, 0x0D,// mov al,0Dh
		/*0070*/ 0x0F, 0x8E, 0x19, 0x00, 0x00, 0x00,// jle near ptr 123456789ABCDE8Fh
		/*0076*/ 0xB0, 0x0E,// mov al,0Eh
		/*0078*/ 0x0F, 0x8F, 0x12, 0x00, 0x00, 0x00,// jg near ptr 123456789ABCDE90h
		/*007E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0080*/ 0x90,// nop
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x71, 0x06,// jno short 8000000000000008h
		/*0002*/ 0xFF, 0x25, 0xA0, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000A8h]
		/*0008*/ 0xB0, 0x00,// mov al,0
		/*000A*/ 0x70, 0x06,// jo short 8000000000000012h
		/*000C*/ 0xFF, 0x25, 0x9E, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000B0h]
		/*0012*/ 0xB0, 0x01,// mov al,1
		/*0014*/ 0x73, 0x06,// jae short 800000000000001Ch
		/*0016*/ 0xFF, 0x25, 0x9C, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000B8h]
		/*001C*/ 0xB0, 0x02,// mov al,2
		/*001E*/ 0x72, 0x06,// jb short 8000000000000026h
		/*0020*/ 0xFF, 0x25, 0x9A, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000C0h]
		/*0026*/ 0xB0, 0x03,// mov al,3
		/*0028*/ 0x75, 0x06,// jne short 8000000000000030h
		/*002A*/ 0xFF, 0x25, 0x98, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000C8h]
		/*0030*/ 0xB0, 0x04,// mov al,4
		/*0032*/ 0x74, 0x06,// je short 800000000000003Ah
		/*0034*/ 0xFF, 0x25, 0x96, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000D0h]
		/*003A*/ 0xB0, 0x05,// mov al,5
		/*003C*/ 0x77, 0x06,// ja short 8000000000000044h
		/*003E*/ 0xFF, 0x25, 0x94, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000D8h]
		/*0044*/ 0xB0, 0x06,// mov al,6
		/*0046*/ 0x76, 0x06,// jbe short 800000000000004Eh
		/*0048*/ 0xFF, 0x25, 0x92, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000E0h]
		/*004E*/ 0xB0, 0x07,// mov al,7
		/*0050*/ 0x79, 0x06,// jns short 8000000000000058h
		/*0052*/ 0xFF, 0x25, 0x90, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000E8h]
		/*0058*/ 0xB0, 0x08,// mov al,8
		/*005A*/ 0x78, 0x06,// js short 8000000000000062h
		/*005C*/ 0xFF, 0x25, 0x8E, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000F0h]
		/*0062*/ 0xB0, 0x09,// mov al,9
		/*0064*/ 0x7B, 0x06,// jnp short 800000000000006Ch
		/*0066*/ 0xFF, 0x25, 0x8C, 0x00, 0x00, 0x00,// jmp qword ptr [80000000000000F8h]
		/*006C*/ 0xB0, 0x0A,// mov al,0Ah
		/*006E*/ 0x7A, 0x06,// jp short 8000000000000076h
		/*0070*/ 0xFF, 0x25, 0x8A, 0x00, 0x00, 0x00,// jmp qword ptr [8000000000000100h]
		/*0076*/ 0xB0, 0x0B,// mov al,0Bh
		/*0078*/ 0x7D, 0x06,// jge short 8000000000000080h
		/*007A*/ 0xFF, 0x25, 0x88, 0x00, 0x00, 0x00,// jmp qword ptr [8000000000000108h]
		/*0080*/ 0xB0, 0x0C,// mov al,0Ch
		/*0082*/ 0x7C, 0x06,// jl short 800000000000008Ah
		/*0084*/ 0xFF, 0x25, 0x86, 0x00, 0x00, 0x00,// jmp qword ptr [8000000000000110h]
		/*008A*/ 0xB0, 0x0D,// mov al,0Dh
		/*008C*/ 0x7F, 0x06,// jg short 8000000000000094h
		/*008E*/ 0xFF, 0x25, 0x84, 0x00, 0x00, 0x00,// jmp qword ptr [8000000000000118h]
		/*0094*/ 0xB0, 0x0E,// mov al,0Eh
		/*0096*/ 0x7E, 0x06,// jle short 800000000000009Eh
		/*0098*/ 0xFF, 0x25, 0x82, 0x00, 0x00, 0x00,// jmp qword ptr [8000000000000120h]
		/*009E*/ 0xB0, 0x0F,// mov al,0Fh
		/*00A0*/ 0x90,// nop
		/*00A1*/ 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC,
		/*00A8*/ 0x81, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00B0*/ 0x82, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00B8*/ 0x83, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00C0*/ 0x84, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00C8*/ 0x85, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00D0*/ 0x86, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00D8*/ 0x87, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00E0*/ 0x88, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00E8*/ 0x89, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00F0*/ 0x8A, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*00F8*/ 0x8B, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*0100*/ 0x8C, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*0108*/ 0x8D, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*0110*/ 0x8E, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*0118*/ 0x8F, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
		/*0120*/ 0x90, 0xDE, 0xBC, 0x9A, 0x78, 0x56, 0x34, 0x12,
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		u32::MAX,
		0x0008,
		u32::MAX,
		0x0012,
		u32::MAX,
		0x001C,
		u32::MAX,
		0x0026,
		u32::MAX,
		0x0030,
		u32::MAX,
		0x003A,
		u32::MAX,
		0x0044,
		u32::MAX,
		0x004E,
		u32::MAX,
		0x0058,
		u32::MAX,
		0x0062,
		u32::MAX,
		0x006C,
		u32::MAX,
		0x0076,
		u32::MAX,
		0x0080,
		u32::MAX,
		0x008A,
		u32::MAX,
		0x0094,
		u32::MAX,
		0x009E,
		0x00A0,
	];
	#[rustfmt::skip]
	let expected_reloc_infos = [
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00A8),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00B0),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00B8),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00C0),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00C8),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00D0),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00D8),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00E0),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00E8),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00F0),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_00F8),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_0100),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_0108),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_0110),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_0118),
		RelocInfo::new(RelocKind::Offset64, 0x8000_0000_0000_0120),
	];
	const OPTIONS: u32 = BlockEncoderOptions::NONE;
	const ORIG_RIP: u64 = 0x1234_5678_9ABC_DE00;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}

#[test]
fn jcc_near_fwd_short_no_opt() {
	#[rustfmt::skip]
	let original_data = [
		/*0000*/ 0x0F, 0x80, 0x7A, 0x00, 0x00, 0x00,// jo near ptr 0000000000008080h
		/*0006*/ 0xB0, 0x00,// mov al,0
		/*0008*/ 0x0F, 0x81, 0x72, 0x00, 0x00, 0x00,// jno near ptr 0000000000008080h
		/*000E*/ 0xB0, 0x01,// mov al,1
		/*0010*/ 0x0F, 0x82, 0x6A, 0x00, 0x00, 0x00,// jb near ptr 0000000000008080h
		/*0016*/ 0xB0, 0x02,// mov al,2
		/*0018*/ 0x0F, 0x83, 0x62, 0x00, 0x00, 0x00,// jae near ptr 0000000000008080h
		/*001E*/ 0xB0, 0x03,// mov al,3
		/*0020*/ 0x0F, 0x84, 0x5A, 0x00, 0x00, 0x00,// je near ptr 0000000000008080h
		/*0026*/ 0xB0, 0x04,// mov al,4
		/*0028*/ 0x0F, 0x85, 0x52, 0x00, 0x00, 0x00,// jne near ptr 0000000000008080h
		/*002E*/ 0xB0, 0x05,// mov al,5
		/*0030*/ 0x0F, 0x86, 0x4A, 0x00, 0x00, 0x00,// jbe near ptr 0000000000008080h
		/*0036*/ 0xB0, 0x06,// mov al,6
		/*0038*/ 0x0F, 0x87, 0x42, 0x00, 0x00, 0x00,// ja near ptr 0000000000008080h
		/*003E*/ 0xB0, 0x07,// mov al,7
		/*0040*/ 0x0F, 0x88, 0x3A, 0x00, 0x00, 0x00,// js near ptr 0000000000008080h
		/*0046*/ 0xB0, 0x08,// mov al,8
		/*0048*/ 0x0F, 0x89, 0x32, 0x00, 0x00, 0x00,// jns near ptr 0000000000008080h
		/*004E*/ 0xB0, 0x09,// mov al,9
		/*0050*/ 0x0F, 0x8A, 0x2A, 0x00, 0x00, 0x00,// jp near ptr 0000000000008080h
		/*0056*/ 0xB0, 0x0A,// mov al,0Ah
		/*0058*/ 0x0F, 0x8B, 0x22, 0x00, 0x00, 0x00,// jnp near ptr 0000000000008080h
		/*005E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0060*/ 0x0F, 0x8C, 0x1A, 0x00, 0x00, 0x00,// jl near ptr 0000000000008080h
		/*0066*/ 0xB0, 0x0C,// mov al,0Ch
		/*0068*/ 0x0F, 0x8D, 0x12, 0x00, 0x00, 0x00,// jge near ptr 0000000000008080h
		/*006E*/ 0xB0, 0x0D,// mov al,0Dh
		/*0070*/ 0x0F, 0x8E, 0x0A, 0x00, 0x00, 0x00,// jle near ptr 0000000000008080h
		/*0076*/ 0xB0, 0x0E,// mov al,0Eh
		/*0078*/ 0x0F, 0x8F, 0x02, 0x00, 0x00, 0x00,// jg near ptr 0000000000008080h
		/*007E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0080*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let new_data = [
		/*0000*/ 0x0F, 0x80, 0x7A, 0x00, 0x00, 0x00,// jo near ptr 0000000000008080h
		/*0006*/ 0xB0, 0x00,// mov al,0
		/*0008*/ 0x0F, 0x81, 0x72, 0x00, 0x00, 0x00,// jno near ptr 0000000000008080h
		/*000E*/ 0xB0, 0x01,// mov al,1
		/*0010*/ 0x0F, 0x82, 0x6A, 0x00, 0x00, 0x00,// jb near ptr 0000000000008080h
		/*0016*/ 0xB0, 0x02,// mov al,2
		/*0018*/ 0x0F, 0x83, 0x62, 0x00, 0x00, 0x00,// jae near ptr 0000000000008080h
		/*001E*/ 0xB0, 0x03,// mov al,3
		/*0020*/ 0x0F, 0x84, 0x5A, 0x00, 0x00, 0x00,// je near ptr 0000000000008080h
		/*0026*/ 0xB0, 0x04,// mov al,4
		/*0028*/ 0x0F, 0x85, 0x52, 0x00, 0x00, 0x00,// jne near ptr 0000000000008080h
		/*002E*/ 0xB0, 0x05,// mov al,5
		/*0030*/ 0x0F, 0x86, 0x4A, 0x00, 0x00, 0x00,// jbe near ptr 0000000000008080h
		/*0036*/ 0xB0, 0x06,// mov al,6
		/*0038*/ 0x0F, 0x87, 0x42, 0x00, 0x00, 0x00,// ja near ptr 0000000000008080h
		/*003E*/ 0xB0, 0x07,// mov al,7
		/*0040*/ 0x0F, 0x88, 0x3A, 0x00, 0x00, 0x00,// js near ptr 0000000000008080h
		/*0046*/ 0xB0, 0x08,// mov al,8
		/*0048*/ 0x0F, 0x89, 0x32, 0x00, 0x00, 0x00,// jns near ptr 0000000000008080h
		/*004E*/ 0xB0, 0x09,// mov al,9
		/*0050*/ 0x0F, 0x8A, 0x2A, 0x00, 0x00, 0x00,// jp near ptr 0000000000008080h
		/*0056*/ 0xB0, 0x0A,// mov al,0Ah
		/*0058*/ 0x0F, 0x8B, 0x22, 0x00, 0x00, 0x00,// jnp near ptr 0000000000008080h
		/*005E*/ 0xB0, 0x0B,// mov al,0Bh
		/*0060*/ 0x0F, 0x8C, 0x1A, 0x00, 0x00, 0x00,// jl near ptr 0000000000008080h
		/*0066*/ 0xB0, 0x0C,// mov al,0Ch
		/*0068*/ 0x0F, 0x8D, 0x12, 0x00, 0x00, 0x00,// jge near ptr 0000000000008080h
		/*006E*/ 0xB0, 0x0D,// mov al,0Dh
		/*0070*/ 0x0F, 0x8E, 0x0A, 0x00, 0x00, 0x00,// jle near ptr 0000000000008080h
		/*0076*/ 0xB0, 0x0E,// mov al,0Eh
		/*0078*/ 0x0F, 0x8F, 0x02, 0x00, 0x00, 0x00,// jg near ptr 0000000000008080h
		/*007E*/ 0xB0, 0x0F,// mov al,0Fh
		/*0080*/ 0xB0, 0x10,// mov al,10h
	];
	#[rustfmt::skip]
	let expected_instruction_offsets = [
		0x0000,
		0x0006,
		0x0008,
		0x000E,
		0x0010,
		0x0016,
		0x0018,
		0x001E,
		0x0020,
		0x0026,
		0x0028,
		0x002E,
		0x0030,
		0x0036,
		0x0038,
		0x003E,
		0x0040,
		0x0046,
		0x0048,
		0x004E,
		0x0050,
		0x0056,
		0x0058,
		0x005E,
		0x0060,
		0x0066,
		0x0068,
		0x006E,
		0x0070,
		0x0076,
		0x0078,
		0x007E,
		0x0080,
	];
	let expected_reloc_infos = [];
	const OPTIONS: u32 = BlockEncoderOptions::DONT_FIX_BRANCHES;
	encode_test(
		BITNESS,
		ORIG_RIP,
		&original_data,
		NEW_RIP,
		&new_data,
		OPTIONS,
		DECODER_OPTIONS,
		&expected_instruction_offsets,
		&expected_reloc_infos,
	);
}
