use std::{
    hash::{Hash, Hasher},
    ops::{Deref, DerefMut},
};

use super::{
    parametric::{FromParameterized, IntoParameterized, Parameterize},
    space::{Ciexyz, ColorVecSpace, Oklab, Srgb},
};

#[derive(Debug)]
pub struct Color<S: ColorVecSpace + ?Sized>(S::Vector);

impl<S: ColorVecSpace + ?Sized> Color<S> {
    pub fn new(vec: S::Vector) -> Self { Self(vec) }

    pub fn into_inner(self) -> S::Vector { self.0 }

    pub fn from_param<P: Parameterize>(param: ParamColor<P>) -> Self
    where S: FromParameterized<P> {
        S::from_param(param)
    }

    pub fn into_param<P: Parameterize>(self) -> ParamColor<P>
    where S: IntoParameterized<P> {
        S::into_param(self)
    }
}

impl<S: ColorVecSpace + ?Sized> Clone for Color<S>
where S::Vector: Clone
{
    fn clone(&self) -> Self { Self(self.0.clone()) }
}

impl<S: ColorVecSpace + ?Sized> Copy for Color<S> where S::Vector: Copy {}

impl<S: ColorVecSpace + ?Sized> PartialEq for Color<S>
where S::Vector: PartialEq
{
    fn eq(&self, rhs: &Self) -> bool { self.0.eq(&rhs.0) }
}

impl<S: ColorVecSpace + ?Sized> Eq for Color<S> where S::Vector: Eq {}

impl<S: ColorVecSpace + ?Sized> Hash for Color<S>
where S::Vector: Hash
{
    fn hash<H: Hasher>(&self, h: &mut H) { self.0.hash(h); }
}

impl<S: ColorVecSpace + ?Sized> Deref for Color<S> {
    type Target = S::Vector;

    fn deref(&self) -> &Self::Target { &self.0 }
}

impl<S: ColorVecSpace + ?Sized> DerefMut for Color<S> {
    fn deref_mut(&mut self) -> &mut Self::Target { &mut self.0 }
}

#[derive(Debug)]
pub struct ParamColor<P: Parameterize + ?Sized>(P::Value);

impl<P: Parameterize + ?Sized> ParamColor<P> {
    pub fn new(val: P::Value) -> Self { Self(val) }

    pub fn into_inner(self) -> P::Value { self.0 }

    pub fn from_color<S: IntoParameterized<P>>(clr: Color<S>) -> Self {
        S::into_param(clr)
    }

    pub fn into_color<S: FromParameterized<P>>(self) -> Color<S> {
        S::from_param(self)
    }
}

impl<P: Parameterize + ?Sized> Clone for ParamColor<P>
where P::Value: Clone
{
    fn clone(&self) -> Self { Self(self.0.clone()) }
}

impl<P: Parameterize + ?Sized> Copy for ParamColor<P> where P::Value: Copy {}

impl<P: Parameterize + ?Sized> PartialEq for ParamColor<P>
where P::Value: PartialEq
{
    fn eq(&self, rhs: &Self) -> bool { self.0.eq(&rhs.0) }
}

impl<P: Parameterize + ?Sized> Eq for ParamColor<P> where P::Value: Eq {}

impl<P: Parameterize + ?Sized> Hash for ParamColor<P>
where P::Value: Hash
{
    fn hash<H: Hasher>(&self, h: &mut H) { self.0.hash(h); }
}

impl<P: Parameterize + ?Sized> Deref for ParamColor<P> {
    type Target = P::Value;

    fn deref(&self) -> &Self::Target { &self.0 }
}

impl<P: Parameterize + ?Sized> DerefMut for ParamColor<P> {
    fn deref_mut(&mut self) -> &mut Self::Target { &mut self.0 }
}

#[derive(Debug, Clone, Copy, PartialEq)]
pub enum OmniColor {
    Ciexyz(Color<Ciexyz>),
    Oklab(Color<Oklab>),
    Srgb(Color<Srgb>),
}
